-----------------------------------
require("modules/module_utils")
require("scripts/globals/trust")
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("trusts")

local rovKIBattlefieldIDs = set{
    5,    -- Shattering Stars (WAR LB5)
    6,    -- Shattering Stars (BLM LB5)
    7,    -- Shattering Stars (RNG LB5)
    70,   -- Shattering Stars (RDM LB5)
    71,   -- Shattering Stars (THF LB5)
    72,   -- Shattering Stars (BST LB5)
    101,  -- Shattering Stars (MNK LB5)
    102,  -- Shattering Stars (WHM LB5)
    103,  -- Shattering Stars (SMN LB5)
    163,  -- Survival of the Wisest (SCH LB5)
    194,  -- Shattering Stars (SAM LB5)
    195,  -- Shattering Stars (NIN LB5)
    196,  -- Shattering Stars (DRG LB5)
    517,  -- Shattering Stars (PLD LB5)
    518,  -- Shattering Stars (DRK LB5)
    519,  -- Shattering Stars (BRD LB5)
    530,  -- A Furious Finale (DNC LB5)
    1091, -- Breaking the Bonds of Fate (COR LB5)
    1123, -- Achieving True Power (PUP LB5)
    1154, -- The Beast Within (BLU LB5)
-- TODO: GEO LB5
-- TODO: RUN LB5
}

local domains =
{
    [xi.zone.CEIZAK_BATTLEGROUNDS]  = 1,
    [xi.zone.FORET_DE_HENNETIEL]    = 1,
    [xi.zone.MORIMAR_BASALT_FIELDS] = 1,
    [xi.zone.YORCIA_WEALD]          = 1,
    [xi.zone.MARJAMI_RAVINE]        = 1,
    [xi.zone.KAMIHR_DRIFTS]         = 1,
    [xi.zone.RAKAZNAR_TURRIS]       = 1,
}

-- Prevent DNC trust from spamming errors
m:addOverride("xi.job_utils.dancer.useViolentFlourishAbility", function(player, target, ability, action)
    if player:isPC() then
        super(player, target, ability, action)
    end
end)

m:addOverride("xi.trust.canCast", function(caster, spell, notAllowedTrustIds)
    -- Trusts must be enabled in settings
    if xi.settings.main.ENABLE_TRUST_CASTING == 0 then
        return xi.msg.basic.TRUST_NO_CAST_TRUST
    end

    -- GMs can do what they want (as long as ENABLE_TRUST_CASTING is enabled)
    if caster:getGMLevel() > 0 then
        return 0
    end

    -- Trusts not allowed in an alliance
    if caster:checkSoloPartyAlliance() == 2 then
        return xi.msg.basic.TRUST_NO_CAST_TRUST
    end

    -- Trusts only allowed in certain zones (Remove this for trusts everywhere)
    if not caster:canUseMisc(xi.zoneMisc.TRUST) then
        return xi.msg.basic.TRUST_NO_CALL_AE
    end

    -- You can only summon trusts if you are the party leader or solo
    local leader = caster:getPartyLeader()
    if leader and caster:getID() ~= leader:getID() then
        caster:messageSystem(xi.msg.system.TRUST_SOLO_OR_LEADER)
        return -1
    end

    -- Block summoning trusts if seeking a party
    if caster:isSeekingParty() then
        caster:messageSystem(xi.msg.system.TRUST_NO_SEEKING_PARTY)
        return -1
    end

    -- Cannot summon trusts if NO_TRUSTS is set under level restriction
    if
        caster:getLocalVar("NO_TRUSTS") == 1 and
        caster:hasStatusEffect(xi.effect.LEVEL_RESTRICTION)
    then
        return xi.msg.basic.TRUST_NO_CAST_TRUST
    end

    -- Block summoning trusts if someone recently joined party (120s)
    local lastPartyMemberAddedTime = caster:getPartyLastMemberJoinedTime()
    if os.time() - lastPartyMemberAddedTime < 120 then
        caster:messageSystem(xi.msg.system.TRUST_DELAY_NEW_PARTY_MEMBER)
        return -1
    end

    -- Trusts cannot be summoned if you have hate (Except inside Domain Invasion areas)
    local zoneID = caster:getZoneID()

    if
        caster:hasEnmity() and
        domains[zoneID] == nil
    then
        caster:messageSystem(xi.msg.system.TRUST_NO_ENMITY)
        return -1
    end

    -- Check party for trusts
    local numPt     = 0
    local numTrusts = 0
    local party     = caster:getPartyWithTrusts()

    for _, member in pairs(party) do
        if member:getObjType() == xi.objType.TRUST then
            -- Check for same trust
            if member:getTrustID() == spell:getID() then
                caster:messageSystem(xi.msg.system.TRUST_ALREADY_CALLED)
                return -1
            -- Check not allowed trust combinations (Shantotto I vs Shantotto II)
            elseif type(notAllowedTrustIds) == "number" then
                if member:getTrustID() == notAllowedTrustIds then
                    caster:messageSystem(xi.msg.system.TRUST_ALREADY_CALLED)
                    return -1
                end
            elseif type(notAllowedTrustIds) == "table" then
                for _, v in pairs(notAllowedTrustIds) do
                    if type(v) == "number" then
                        if member:getTrustID() == v then
                            caster:messageSystem(xi.msg.system.TRUST_ALREADY_CALLED)
                            return -1
                        end
                    end
                end
            end

            numTrusts = numTrusts + 1
        end

        numPt = numPt + 1
    end

    -- Max party size
    if numPt >= 6 then
        caster:messageSystem(xi.msg.system.TRUST_MAXIMUM_NUMBER)
        return -1
    end

    -- Some battlefields allow trusts after you get this ROV Key Item
    local casterBattlefieldID = caster:getBattlefieldID()
    if
        rovKIBattlefieldIDs[casterBattlefieldID] and
        not caster:hasKeyItem(xi.ki.RHAPSODY_IN_UMBER)
    then
        return xi.msg.basic.TRUST_NO_CAST_TRUST
    end

    -- Limits set by ROV Key Items
    if
        numTrusts >= 3 and
        not caster:hasKeyItem(xi.ki.RHAPSODY_IN_WHITE) and
        not caster:isCrystalWarrior()
    then
        caster:messageSystem(xi.msg.system.TRUST_MAXIMUM_NUMBER)
        return -1
    elseif
        numTrusts >= 4 and
        not caster:hasKeyItem(xi.ki.RHAPSODY_IN_CRIMSON) and
        not caster:isCrystalWarrior()
    then
        caster:messageSystem(xi.msg.system.TRUST_MAXIMUM_NUMBER)
        return -1
    end

    if not xi.trust.checkBattlefieldTrustCount(caster) then
        return xi.msg.basic.TRUST_NO_CAST_TRUST
    end

    return 0
end)

local function getBonus(player)
    local domainsCompleted = math.floor(player:getCharVar("[Domain]Completed") / 5)
    local dailiesCompleted = player:getCharVar("[DQ]Completed")

    return utils.clamp(domainsCompleted + dailiesCompleted, 0, 5)
end

m:addOverride("xi.trust.spawn", function(caster, spell)
    local trust  = caster:spawnTrust(spell:getID())
    local zoneID = caster:getZoneID()

    if domains[zoneID] ~= nil then
        local dailyBonus = getBonus(caster)

        trust:addMod(xi.mod.HPP, 10 * dailyBonus)
        trust:addMod(xi.mod.MPP, 10 * dailyBonus)
        trust:updateHealth()
        trust:setHP(trust:getMaxHP())
        trust:setMP(trust:getMaxMP())

        if dailyBonus > 4 then
            trust:addStatusEffect(xi.effect.REFRESH, 3, 0, 0)
        elseif dailyBonus > 3 then
            trust:addStatusEffect(xi.effect.REFRESH, 2, 0, 0)
        elseif dailyBonus > 2 then
            trust:addStatusEffect(xi.effect.REFRESH, 1, 0, 0)
        end
    end

    -- Records of Eminence: Call Forth an Alter Ego
    if caster:getEminenceProgress(932) then
        xi.roe.onRecordTrigger(caster, 932)
    end

    return 0
end)

return m
