-------------------------------------------
-- Login Campaign
-------------------------------------------
require('modules/module_utils')
require('scripts/globals/pathfind')
-------------------------------------------
local m = Module:new('pirates_global')

xi.pirates = xi.pirates or {}

xi.pirates.status =
{
    IDLE       = 0,
    PREPARING  = 1,
    ARRIVING   = 2,
    SPAWNING   = 3,
    ATTACKING  = 4,
    DESPAWNING = 5,
    DEPARTING  = 6,
    FINISHED   = 7,
}

local function SetZoneMusic(zone, block, track)
    for _, player in pairs(zone:getPlayers()) do
        player:changeMusic(block, track)
    end
end

m:addOverride("xi.pirates.init", function(zoneID)
    print('[PIRATES] Event initializing')
    local ship = GetNPCByID(zones[zoneID].npc.PIRATE_SHIP.id)

    if ship ~= nil then
        ship:setAnimPath(zones[zoneID].npc.PIRATE_SHIP.anim_path)
        ship:setStatus(xi.status.DISAPPEAR)
    end

    for k, pirate in pairs(zones[zoneID].npc.PIRATES) do
        local npc = GetNPCByID(k)
        npc:setPos(pirate.start_pos)
        npc:setStatus(xi.status.DISAPPEAR)
        npc:initNpcAi()
    end
end)

m:addOverride("xi.pirates.start", function(zoneID)
    local ship = GetNPCByID(zones[zoneID].npc.PIRATE_SHIP.id)
    local hqChance = math.random(0, 100)
    print('[PIRATES] Event starting')
    ship:setLocalVar("pirateStatus", xi.pirates.status.IDLE)
    xi.pirates.despawnMobs(zoneID)
    xi.pirates.reset(zoneID)

    if hqChance >= 80 then
        print('[PIRATES] HQ Pirate Attack!')
        ship:setLocalVar('HQ', 1)
    else
        ship:setLocalVar('HQ', 0) -- Resetting to 0 as a fallback
    end
end)

m:addOverride("xi.pirates.reset", function(zoneID)
    local ship = GetNPCByID(zones[zoneID].npc.PIRATE_SHIP.id)
    print('[PIRATES] Event Resetting')
    ship:setAnimation(0)
    ship:setStatus(xi.status.DISAPPEAR)
    xi.pirates.setShipPosition(ship, zones[zoneID].npc.PIRATE_SHIP.start_pos)
    GetMobByID(zones[zoneID].mob.CROSSBONES + 5):setLocalVar("killed", 0)
end)

m:addOverride("xi.pirates.spawnMob", function(mobId)
    local x = math.random(-4, 4)
    local y = -7.263
    local z = math.random(10, 22)
    local rot = math.random(0, 255)
    local mob = GetMobByID(mobId)

    mob:setSpawn(x, y, z, rot)
    mob:spawn()
end)

m:addOverride("xi.pirates.spawnMobs", function(zoneID)
    local ship = GetNPCByID(zones[zoneID].npc.PIRATE_SHIP.id)

    for i = 0, 3 do
        local xbone = GetMobByID(zones[zoneID].mob.CROSSBONES + i)
        if
            not xbone:isSpawned() and
            os.time() > xbone:getLocalVar("respawnTime")
        then
            xi.pirates.spawnMob(zones[zoneID].mob.CROSSBONES + i)
        end
    end

    local wight = GetMobByID(zones[zoneID].mob.CROSSBONES + 4)
    local nm = GetMobByID(zones[zoneID].mob.CROSSBONES + 5)

    if
        not wight:isSpawned() and
        not nm:isSpawned()
    then
        if os.time() > wight:getLocalVar("respawnTime") then
            if
                nm:getLocalVar("killed") == 0 and
                ship:getLocalVar('HQ') == 1 and
                math.random(0, 100) > 30
            then
                xi.pirates.spawnMob(zones[zoneID].mob.CROSSBONES + 5)
            else
                xi.pirates.spawnMob(zones[zoneID].mob.CROSSBONES + 4)
            end
        end
    end
end)

m:addOverride("xi.pirates.despawnMobs", function(zoneID)
    for i = 0, 3 do
        if GetMobByID(zones[zoneID].mob.CROSSBONES + i):isSpawned() then
            if not GetMobByID(zones[zoneID].mob.CROSSBONES + i):isEngaged() then
                DespawnMob(zones[zoneID].mob.CROSSBONES + i)
            end
        end
    end

    if GetMobByID(zones[zoneID].mob.CROSSBONES + 4):isSpawned() then
        if not GetMobByID(zones[zoneID].mob.CROSSBONES + 4):isEngaged() then
            DespawnMob(zones[zoneID].mob.CROSSBONES + 4)
        end
    end

    if GetMobByID(zones[zoneID].mob.CROSSBONES + 5):isSpawned() then
        if not GetMobByID(zones[zoneID].mob.CROSSBONES + 5):isEngaged() then
            DespawnMob(zones[zoneID].mob.CROSSBONES + 5)
        end
    end
end)

m:addOverride("xi.pirates.despawnNPCs", function(zoneID)
    print('[PIRATES] Despawning NPCs')
    for k in pairs(zones[zoneID].npc.PIRATES) do
        local npc = GetNPCByID(k)
        npc:setLocalVar("summoning", 0)
        npc:setStatus(xi.status.DISAPPEAR)
    end
end)

m:addOverride("xi.pirates.setShipPosition", function(ship, pos)
    ship:setPos(pos.x, pos.y, pos.z, pos.rot)
    print('[PIRATES] Setting ship position')
end)

m:addOverride("xi.pirates.summonAnimations", function(zoneID, firstcast)
    for k in pairs(zones[zoneID].npc.PIRATES) do
        local npc = GetNPCByID(k)
        if
            npc:getLocalVar("castmode") == 1 and
            npc:getLocalVar("summoning") == 0
        then
            npc:setLocalVar("summoning", 1)
            local startTime = 0

            if not firstcast then
                startTime = 2000 + math.random(2000, 3000)
            end

            npc:timer(startTime, function(npcArg)
                if npcArg:getLocalVar("castmode") == 1 then
                    npcArg:entityAnimationPacket("casm")
                    local randomSummonTime = 2000 + math.random(0, 2000)
                    npcArg:timer(randomSummonTime, function(npcArg2)
                        npcArg2:entityAnimationPacket("shsm")
                        npcArg2:setLocalVar("summoning", 0)
                    end)
                end
            end)
        end
    end
end)

m:addOverride("xi.pirates.update", function(zoneID, zone, tripTime)
    local ship = GetNPCByID(zones[zoneID].npc.PIRATE_SHIP.id)
    local pirateStatus = ship:getLocalVar("pirateStatus")

    switch (pirateStatus): caseof
    {
        [0] = function(x)
            if tripTime >= 200 then
                SetZoneMusic(zone, 0, 0)
                SetZoneMusic(zone, 1, 0)
                ship:setLocalVar("pirateStatus", xi.pirates.status.PREPARING)
                print('[PIRATES] Pirates are preparing ...')
            end
        end,

        [1] = function(x)
            if tripTime >= 240 then
                ship:setAnimPath(zones[zoneID].npc.PIRATE_SHIP.anim_path)
                ship:setAnimation(18)
                ship:setAnimStart(true)
                ship:setAnimBegin(VanadielTime())
                ship:setStatus(xi.status.NORMAL)
                xi.pirates.setShipPosition(ship, zones[zoneID].npc.PIRATE_SHIP.start_pos)
                ship:sendUpdateToZoneCharsInRange(2000)
                ship:setLocalVar("pirateStatus", xi.pirates.status.ARRIVING)
                print('[PIRATES] Pirates are arriving ...')
            end
        end,

        [2] = function(x)
            if tripTime >= 290 then
                for k, pirate in pairs(zones[zoneID].npc.PIRATES) do
                    local npc = GetNPCByID(k)
                    local piratePos = pirate.position

                    if piratePos == 2 then
                        if ship:getLocalVar('HQ') == 1 then
                            npc:setModelId(47, 5) -- v. cloak in body slot
                        else
                            npc:setModelId(46, 5) -- basic cloak in body slot
                        end
                    end

                    npc:hideName(true)
                    npc:setPos(xi.path.first(pirate.enter_path))
                    npc:setStatus(xi.status.NORMAL)
                    local point = xi.path.last(pirate.enter_path)
                    npc:pathTo(point[1], point[2], point[3])
                    npc:sendUpdateToZoneCharsInRange(2000)
                end

                SetZoneMusic(zone, 0, 170)
                SetZoneMusic(zone, 1, 170)
                xi.pirates.setShipPosition(ship, zones[zoneID].npc.PIRATE_SHIP.event_pos)
                ship:sendUpdateToZoneCharsInRange(2000)
                ship:setLocalVar("pirateStatus", xi.pirates.status.SPAWNING)
                print('[PIRATES] Pirates are spawning')

            end
        end,

        [3] = function(x)
            if tripTime >= 297 then
                xi.pirates.setShipPosition(ship, zones[zoneID].npc.PIRATE_SHIP.event_pos)
                ship:setAnimation(0)
                ship:sendUpdateToZoneCharsInRange(2000)
                ship:setLocalVar("pirateStatus", xi.pirates.status.ATTACKING)
                print('[PIRATES] Pirates are attacking!')
                for k, pirate in pairs(zones[zoneID].npc.PIRATES) do
                    local npc = GetNPCByID(k)
                    npc:setLocalVar("castmode", 1)
                    npc:lookAt(pirate.look_at)
                    npc:sendUpdateToZoneCharsInRange(2000)
                end

                xi.pirates.summonAnimations(zoneID, true)
            else
                for k, pirate in pairs(zones[zoneID].npc.PIRATES) do
                    local npc = GetNPCByID(k)
                    npc:lookAt(pirate.look_at)
                    npc:sendUpdateToZoneCharsInRange(2000)
                end
            end
        end,

        [4] = function(x)
            if tripTime >= 695 then
                for k, pirate in pairs(zones[zoneID].npc.PIRATES) do
                    local npc = GetNPCByID(k)
                    local piratePos = pirate.position

                    npc:setLocalVar("castmode", 0)
                    npc:setPos(xi.path.first(pirate.enter_path))
                    npc:setStatus(xi.status.NORMAL)
                    npc:pathTo(xi.path.last(pirate.exit_path)[1], xi.path.last(pirate.exit_path)[2], xi.path.last(pirate.exit_path)[3])
                    npc:sendUpdateToZoneCharsInRange(2000)
                end

                xi.pirates.despawnMobs(zoneID)
                ship:setLocalVar("pirateStatus", xi.pirates.status.DESPAWNING)
                print('[PIRATES] Pirates are despawning')
            elseif tripTime <= 685 and tripTime >= 304 then
                xi.pirates.summonAnimations(zoneID, false)
                xi.pirates.spawnMobs(zoneID)
                SetZoneMusic(zone, 0, 170)
                SetZoneMusic(zone, 1, 170)
            end
        end,

        [5] = function(x)
            if tripTime >= 706 then
                xi.pirates.despawnNPCs(zoneID)
                ship:setAnimPath(zones[zoneID].npc.PIRATE_SHIP.anim_path)
                ship:setAnimation(19)
                ship:setAnimStart(true)
                ship:setAnimBegin(VanadielTime())
                xi.pirates.setShipPosition(ship, zones[zoneID].npc.PIRATE_SHIP.start_pos)
                ship:sendUpdateToZoneCharsInRange(2000)
                print('[PIRATES] Pirates are departing')
                ship:setLocalVar("pirateStatus", xi.pirates.status.DEPARTING)
            end
        end,

        [6] = function(x)
            if tripTime >= 760 then
                SetZoneMusic(zone, 0, 106)
                SetZoneMusic(zone, 1, 106)
                ship:setAnimation(0)
                ship:setStatus(xi.status.DISAPPEAR)
                ship:sendUpdateToZoneCharsInRange(2000)
                ship:setLocalVar("pirateStatus", xi.pirates.status.FINISHED)
                print('[PIRATES] Event Finished')

            end
        end,

        [7] = function(x)
            xi.pirates.despawnMobs(zoneID)
        end
    }
end)

xi.sea_creatures = xi.sea_creatures or {}

m:addOverride("xi.sea_creatures.checkSpawns", function(zoneID, percentChance, total)
    for i = 0, 4 do
        local mobId = zones[zoneID].mob.SEA_CREATURES + i
        local mob = GetMobByID(mobId)
        if
            mob:isSpawned() and
            os.time() > mob:getLocalVar("despawnTime")
        then
            if not mob:isEngaged() then
                DespawnMob(mobId)
            end
        end
    end

    if xi.sea_creatures.shouldSpawn(zoneID, percentChance, total) then
        local mobIdPos = math.random(0, 4)
        local mob = GetMobByID(zones[zoneID].mob.SEA_CREATURES + mobIdPos)
        if not mob:isSpawned() then
            mob:spawn()
            mob:setLocalVar("despawnTime", os.time() + math.random(180, 240)) -- 3-4 mins despawn if roaming
        end
    end
end)

m:addOverride("xi.sea_creatures.despawn", function(zoneID)
    for i = 0, 4 do
        local mobId = zones[zoneID].mob.SEA_CREATURES + i
        DespawnMob(mobId)
    end
end)

m:addOverride("xi.sea_creatures.shouldSpawn", function(zoneID, percentChance, total)
    if math.random(1, 100) > percentChance then
        return false
    end

    local count = 0
    for i = 0, 4 do
        local mobId = zones[zoneID].mob.SEA_CREATURES + i
        if GetMobByID(mobId):isSpawned() then
            count = count + 1
            if count == total then
                return false
            end
        end
    end

    return true
end)

return m
