-----------------------------------
require("modules/module_utils")
require("scripts/globals/gambits")
require("scripts/globals/magic")
require("scripts/globals/trust")
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("cxi_trusts")

local trustOverride =
{
    ["joachim"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())

            -- Records of Eminence: Alter Ego: Joachim
            if caster:getEminenceProgress(937) then
                xi.roe.onRecordTrigger(caster, 937)
            end

            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.POISON, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.POISONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PARALYSIS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.PARALYNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.BLINDNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SILENCE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.SILENA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PETRIFICATION, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.DISEASE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.VIRUNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.CURSE_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURSNA)

            -- TODO: BRD trusts need better logic and major overhaul, for now they compliment each other
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.MARCH, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.MARCH)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.BALLAD, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.MAGES_BALLAD)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.ELEGY, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.ELEGY)

            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)

            -- Try and ranged attack every 60s
            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS, 0, ai.r.RATTACK, 0, 0, 60)

            trust:setAutoAttackEnabled(false)

            trust:setMobMod(xi.mobMod.TRUST_DISTANCE, xi.trust.movementType.MID_RANGE)
        end,
    },

    ["ulmia"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())

            trust:addSimpleGambit(ai.t.TARGET, ai.c.STATUS_FLAG, xi.effectFlag.DISPELABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.MAGIC_FINALE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.MADRIGAL, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.MADRIGAL)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.MINUET, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.VALOR_MINUET)

            trust:setAutoAttackEnabled(false)

            trust:setMobMod(xi.mobMod.TRUST_DISTANCE, xi.trust.movementType.MID_RANGE)
        end,
    },

    ["aaev"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.FLASH, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.FLASH)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.RAMPART, ai.r.JA, ai.s.SPECIFIC, xi.ja.RAMPART)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.PALISADE, ai.r.JA, ai.s.SPECIFIC, xi.ja.PALISADE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_HAS_TOP_ENMITY, 0, xi.effect.SENTINEL, ai.r.JA, ai.s.SPECIFIC, xi.ja.SENTINEL)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 50, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.PHALANX, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PHALANX)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.ENLIGHT, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.ENLIGHT)
            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 5)
            trust:addMod(xi.mod.REGEN, 1)
            trust:addMod(xi.mod.ABSORB_DMG_TO_MP, 5)
            trust:addMod(xi.mod.FASTCAST, power / 2)
            trust:addMod(xi.mod.CURE_POTENCY, power)
            trust:addMod(xi.mod.ENMITY, power * 111)
        end,
    },

    ["elivira"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.EVOKERS_ROLL, ai.r.JA, ai.s.SPECIFIC, xi.ja.EVOKERS_ROLL)  -- Refresh
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.WARLOCKS_ROLL, ai.r.JA, ai.s.SPECIFIC, xi.ja.WARLOCKS_ROLL) -- Magic Accuracy
            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS, 0, ai.r.RATTACK, 0, 0, 10)

            trust:addMod(xi.mod.WSACC, power*3)
            trust:addMod(xi.mod.ACC, power*2)

            -- Notable: Uses a balance of melee and ranged attacks.
            -- TODO: Observe his WS behaviour on retail
            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["halver"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_HAS_TOP_ENMITY, 0, xi.effect.SENTINEL, ai.r.JA, ai.s.SPECIFIC, xi.ja.SENTINEL)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 50, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.FLASH, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.FLASH)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_HAS_TOP_ENMITY, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.PROVOKE)
            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)

            if not caster:isCrystalWarrior() then
                trust:addMod(xi.mod.MATT, power)
                trust:addMod(xi.mod.MACC, power)
                trust:addMod(xi.mod.DEF, power)
                trust:addMod(xi.mod.MDEF, power)
                trust:addMod(xi.mod.ATT, power)
                trust:addMod(xi.mod.ACC, power * 4)
                trust:addMod(xi.mod.ENMITY, power * 25)
            end

        end,
    },

    ["tenzen"] =
    {
        cwEnabled = true,
        cwOnly    = true,

        onSpellCast = function(caster, target, spell)
            if caster:isCrystalWarrior() then
                local trust = caster:spawnTrust(spell:getID())
                local power = trust:getMainLvl()

                trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)

                trust:addMod(xi.mod.ACC,     power * 3)
                trust:addMod(xi.mod.STORETP,        15)
            end
        end,
    },

    ["rainemard"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.COMPOSURE, ai.r.JA, ai.s.SPECIFIC, xi.ja.COMPOSURE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.MPP_LT, 5, ai.r.JA, ai.s.SPECIFIC, xi.ja.CONVERT)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 50, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.MELEE, ai.c.NOT_STATUS, xi.effect.HASTE, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.HASTE)
            trust:addSimpleGambit(ai.t.CASTER, ai.c.NOT_STATUS, xi.effect.REFRESH, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.REFRESH)
            trust:addSimpleGambit(ai.t.TANK, ai.c.NOT_STATUS, xi.effect.REFRESH, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.REFRESH)
            trust:addSimpleGambit(ai.t.RANGED, ai.c.NOT_STATUS, xi.effect.FLURRY, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.FLURRY)
            trust:addSimpleGambit(ai.t.TOP_ENMITY, ai.c.NOT_STATUS, xi.effect.PHALANX, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PHALANX)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.STATUS_FLAG, xi.effectFlag.DISPELABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.DISPEL)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.ENBLIZZARD, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.ENBLIZZARD, 60)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.DIA, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.DIA, 60)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.SLOW, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SLOW, 60)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.EVASION_DOWN, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.DISTRACT, 60)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power * 2)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ENSPELL_DMG, power / 4)
            trust:addMod(xi.mod.HASTE_MAGIC, power / 3)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 2)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["zeid_ii"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_WS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_MS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_JA, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.CASTING_MA, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            -- trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.SOULEATER) -- TODO: Check if this is still broken
            trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.LAST_RESORT)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 3)

            trust:setTrustTPSkillSettings(ai.tp.CLOSER_UNTIL_TP, ai.s.RANDOM, 3000)
        end,
    },

    ["gessho"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.NAJA_SALAHEEM] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.ABQUHBAH] = xi.trust.messageOffset.TEAMWORK_2,
            })

            trust:addListener("WEAPONSKILL_USE", "GESSHO_WEAPONSKILL_USE", function(mobArg, target, wsid, tp, action)
                if wsid == 3257 then -- Shibaraku
                    -- You have left me no choice. Prepare yourself!
                    xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            -- Shadows are represented by xi.effect.COPY_IMAGE, but with different icons depending on the tier
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.COPY_IMAGE, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.UTSUSEMI)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.KURAYAMI, 60)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.SLOW, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.HOJO, 60)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.YONIN, ai.r.JA, ai.s.SPECIFIC, xi.ja.YONIN)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_HAS_TOP_ENMITY, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.PROVOKE)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 4)
            trust:addMod(xi.mod.DOUBLE_ATTACK, 15)
            trust:addMod(xi.mod.ENMITY, power * 60)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["maximilian"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.COPY_IMAGE, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.UTSUSEMI)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 4)
            trust:addMod(xi.mod.DOUBLE_ATTACK, 15)

            trust:setTrustTPSkillSettings(ai.tp.OPENER, ai.s.HIGHEST)
        end,
    },

    ["semih_lafihna"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.STAR_SIBYL] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.AJIDO_MARUJIDO] = xi.trust.messageOffset.TEAMWORK_2,
            })

            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.BARRAGE, ai.r.JA, ai.s.SPECIFIC, xi.ja.BARRAGE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.SHARPSHOT, ai.r.JA, ai.s.SPECIFIC, xi.ja.SHARPSHOT)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.DOUBLE_SHOT, ai.r.JA, ai.s.SPECIFIC, xi.ja.DOUBLE_SHOT)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS, 0, ai.r.RATTACK, 0, 0)

            trust:addListener("WEAPONSKILL_USE", "SEMIH_LAFIHNA_WEAPONSKILL_USE", function(mobArg, target, wsid, tp, action)
                if wsid == 3490 then -- Stellar Arrow
                    -- I'll show you no quarter!
                    xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            trust:addMod(xi.mod.STORETP, 40)

            trust:setAutoAttackEnabled(false)

            trust:setMobMod(xi.mobMod.TRUST_DISTANCE, xi.trust.movementType.LONG_RANGE)
        end,
    },

    ["maat"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.MANTRA)

            trust:addListener("WEAPONSKILL_USE", "MAAT_WEAPONSKILL_USE", function(mobArg, target, wsid, tp, action)
                if wsid == 3263 then -- Bear Killer
                    --  Heh heh heh
                    xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 3)
            trust:addMod(xi.mod.DOUBLE_ATTACK, 15)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.HIGHEST)
        end,
    },

    ["mayakov"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())

            local healingJobs =
            {
                xi.job.WHM,
                xi.job.RDM,
                xi.job.SCH,
                xi.job.PLD,
            }

            xi.trust.message(trust, xi.trust.messageOffset.SPAWN)

            trust:addListener('WEAPONSKILL_USE', 'COMING_UP_ROSES_USE', function(mobArg, target, wsid, tp, action)
                if wsid == 3454 then -- Coming Up Roses
                    xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            for i = 1, #healingJobs do
                local master  = trust:getMaster()
                if master:getMainJob() == healingJobs[i] then
                    trust:addSimpleGambit(ai.t.SELF, ai.c.NO_SAMBA, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.HASTE_SAMBA)
                end
            end

            -- TP use and return (this logic is level 80)
            -- trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS, xi.effect.FINISHING_MOVE_5, ai.r.JA, ai.s.SPECIFIC, xi.ja.CLIMACTIC_FLOURISH, 60)

            -- Ecosystem check to swap to Haste samba if the target is undead
            trust:addSimpleGambit(ai.t.TARGET, ai.c.IS_ECOSYSTEM, xi.ecosystem.UNDEAD, ai.r.JA, ai.s.SPECIFIC, xi.ja.HASTE_SAMBA)

            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.BEWILDERED_DAZE_5, ai.r.JA, ai.s.SPECIFIC, xi.ja.FEATHER_STEP, 20)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 65, ai.r.JA, ai.s.HIGHEST_WALTZ, xi.ja.CURING_WALTZ)
            trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS_FLAG, xi.effectFlag.WALTZABLE, ai.r.JA, ai.s.SPECIFIC, xi.ja.HEALING_WALTZ)

            trust:addSimpleGambit(ai.t.SELF, ai.c.NO_SAMBA, 0, ai.r.JA, ai.s.BEST_SAMBA, xi.ja.DRAIN_SAMBA)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.SABER_DANCE, ai.r.JA, ai.s.SPECIFIC, xi.ja.SABER_DANCE, 180)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)

            if not caster:isCrystalWarrior() then
                local power = trust:getMainLvl()
                trust:addMod(xi.mod.MATT, power)
                trust:addMod(xi.mod.MACC, power)
                trust:addMod(xi.mod.DEF, power)
                trust:addMod(xi.mod.MDEF, power)
                trust:addMod(xi.mod.ATT, power)
                trust:addMod(xi.mod.ACC, power * 4)
                trust:addMod(xi.mod.STORETP, 10)
                trust:addMod(xi.mod.DOUBLE_ATTACK, 15)
            end
        end,
    },

    ["volker"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.NAJI] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.CID] = xi.trust.messageOffset.TEAMWORK_2,
                [xi.magic.spell.KLARA] = xi.trust.messageOffset.TEAMWORK_3,
            })

            trust:addSimpleGambit(ai.t.MASTER, ai.c.HPP_LT, 50, ai.r.JA, ai.s.SPECIFIC, xi.ja.PROVOKE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.WARCRY, ai.r.JA, ai.s.SPECIFIC, xi.ja.WARCRY)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.AGGRESSOR, ai.r.JA, ai.s.SPECIFIC, xi.ja.AGGRESSOR)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.BERSERK, ai.r.JA, ai.s.SPECIFIC, xi.ja.BERSERK)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.RETALIATION, ai.r.JA, ai.s.SPECIFIC, xi.ja.RETALIATION)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 4)
            trust:addMod(xi.mod.ENMITY, power * 25)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["uka_totlihn"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            local healingJobs =
            {
                xi.job.WHM,
                xi.job.RDM,
                xi.job.SCH,
                xi.job.PLD,
            }

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.MUMOR   ] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.ULLEGORE] = xi.trust.messageOffset.TEAMWORK_2,
            })

            -- Dynamic modifier that checks party member list on tick to apply synergy
            trust:addListener('COMBAT_TICK', 'UKA_TOTLIHN_CTICK', function(mobArg)
                local waltzPotencyBoost = 0
                local party = mobArg:getMaster():getPartyWithTrusts()
                for _, member in pairs(party) do
                    if member:getObjType() == xi.objType.TRUST then
                        if
                            member:getTrustID() == xi.magic.spell.MUMOR or
                            member:getTrustID() == xi.magic.spell.MUMOR_II
                        then
                            waltzPotencyBoost = 10
                        end
                    end
                end

                -- Always set the boost, even if Mumor wasn't found.
                -- This accounts for her being in the party and giving the boost
                -- and also if she dies and the boost goes away.
                mobArg:setMod(xi.mod.WALTZ_POTENCY, waltzPotencyBoost)
            end)

            for i = 1, #healingJobs do
                local master  = trust:getMaster()
                if master:getMainJob() == healingJobs[i] then
                    trust:addSimpleGambit(ai.t.SELF, ai.c.NO_SAMBA, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.HASTE_SAMBA)
                end
            end

            -- Step Interactions:
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.LETHARGIC_DAZE_5, ai.r.JA, ai.s.SPECIFIC, xi.ja.QUICKSTEP, 20)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_WS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.VIOLENT_FLOURISH)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_MS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.VIOLENT_FLOURISH)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_JA, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.VIOLENT_FLOURISH)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.CASTING_MA, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.VIOLENT_FLOURISH)

            -- Ecosystem check to swap to Haste samba if the target is undead
            trust:addSimpleGambit(ai.t.TARGET, ai.c.IS_ECOSYSTEM, xi.ecosystem.UNDEAD, ai.r.JA, ai.s.SPECIFIC, xi.ja.HASTE_SAMBA)

            -- Healing logic
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 50, ai.r.JA, ai.s.HIGHEST_WALTZ, xi.ja.CURING_WALTZ)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NO_SAMBA, 0, ai.r.JA, ai.s.BEST_SAMBA, xi.ja.DRAIN_SAMBA)
            trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS_FLAG, xi.effectFlag.WALTZABLE, ai.r.JA, ai.s.SPECIFIC, xi.ja.HEALING_WALTZ)

            -- TP use and return
            trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS, xi.effect.FINISHING_MOVE_5, ai.r.JA, ai.s.SPECIFIC, xi.ja.REVERSE_FLOURISH, 60)
            trust:setTrustTPSkillSettings(ai.tp.CLOSER_UNTIL_TP, ai.s.RANDOM, 2000)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 3)
            trust:addMod(xi.mod.STORETP, 10)
        end,
    },

    ["naja_salaheem"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.GESSHO] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.RONGELOUTS] = xi.trust.messageOffset.TEAMWORK_2,
                [xi.magic.spell.ABQUHBAH] = xi.trust.messageOffset.TEAMWORK_3,
            })

            trust:addListener("WEAPONSKILL_USE", "NAJA_WEAPONSKILL_USE", function(mobArg, target, wsid, tp, action)
                if wsid == 3215 then -- Peacebreaker
                    --  Cha-ching! Thirty gold coins!
                    xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.FOCUS)
            trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.DODGE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.HAS_TOP_ENMITY, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.COUNTERSTANCE)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power*3)
            trust:addMod(xi.mod.STORETP, 10)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["nanaa_mihgo"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.ROMAA_MIHGO] = xi.trust.messageOffset.TEAMWORK_1,
            })

            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.DESPOIL)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power*3)
            trust:addMod(xi.mod.STORETP, 10)

            trust:setTrustTPSkillSettings(ai.tp.OPENER, ai.s.HIGHEST)
        end,
    },

    ["aldo"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS,             0, ai.r.JA, ai.s.SPECIFIC, xi.ja.DESPOIL)
            trust:addSimpleGambit(ai.t.SELF,   ai.c.NOT_HAS_TOP_ENMITY, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.SNEAK_ATTACK)

            trust:addMod(xi.mod.MATT,      power)
            trust:addMod(xi.mod.MACC,      power)
            trust:addMod(xi.mod.DEF,       power)
            trust:addMod(xi.mod.MDEF,      power)
            trust:addMod(xi.mod.ATT,       power)
            trust:addMod(xi.mod.ACC,     power*3)
            trust:addMod(xi.mod.STORETP,      10)

            trust:setTrustTPSkillSettings(ai.tp.OPENER, ai.s.HIGHEST)
        end,
    },

    ["qultada"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl() / 3

            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.SAMURAI_ROLL, ai.r.JA, ai.s.SPECIFIC, xi.ja.SAMURAI_ROLL)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.FIGHTERS_ROLL, ai.r.JA, ai.s.SPECIFIC, xi.ja.FIGHTERS_ROLL)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS, 0, ai.r.RATTACK, 0, 0, 10)

            trust:addMod(xi.mod.MACC, power)

            trust:setTrustTPSkillSettings(ai.tp.OPENER, ai.s.RANDOM)
        end,
    },

    ["kupofried"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())

            trust:addStatusEffectEx(xi.effect.COLURE_ACTIVE, xi.effect.COLURE_ACTIVE, 6, 3, 0, xi.effect.CORSAIRS_ROLL, 120, xi.auraTarget.ALLIES, xi.effectFlag.AURA)
            trust:setAutoAttackEnabled(false)
        end,
    },

    ["ullegore"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local trustLevel  = trust:getMainLvl()
            local power       = math.ceil(trustLevel / 5)

            xi.trust.message(trust, xi.trust.messageOffset.SPAWN)

            -- Bored to Tears
            trust:addListener("WEAPONSKILL_USE", "ULLEGORE_WEAPONSKILL_USE", function(mobArg, target, wsid, tp, action)
                if wsid == 3627 then -- Bored to Tears
               -- Yes, now you will experience true boredom! Mwahahahaha!
               xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            -- Stun
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_WS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.ALWAYS, 0, ai.r.MA, ai.s.RANDOM, xi.magic.spellFamily.NONE, math.random(25, 35))

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.INT,  power)
            trust:setMod(xi.mod.HPP, 30)
            trust:setMod(xi.mod.MPP, 300)
            trust:updateHealth()
            trust:addHP(trust:getMaxHP())
            trust:addMP(trust:getMaxMP())

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["apururu_uc"] =
    {
        -- Aprururu logic is LSB base, this module is correcting protectra spam
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust       = caster:spawnTrust(spell:getID())

            local isWearingApururuShirt = function(player)
                local wearingBody = player:getEquipID(xi.slot.BODY) == xi.item.APURURU_UNITY_SHIRT
                return wearingBody
            end

            local master = trust:getMaster()
            if isWearingApururuShirt(master) then
                xi.trust.message(trust, xi.trust.messageOffset.TEAMWORK_2)
            else
                xi.trust.message(trust, xi.trust.messageOffset.SPAWN)
            end

            trust:addSimpleGambit(ai.t.SELF, ai.c.MPP_LT, 25, ai.r.JA, ai.s.SPECIFIC, xi.ja.CONVERT)

            -- BG WIKI (Uses Devotion on party members that have <20% MP.)
            -- Will not work, calls jobutil
            -- trust:addSimpleGambit(ai.t.PARTY, ai.c.MPP_LT, 25, ai.r.JA, ai.s.SPECIFIC, xi.ja.DEVOTION)

            -- LSB has logic for 75% and 25% HPP to use the same gambit logic below
            -- trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 25, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)

            -- Adds Curaga if multiple members are below HP threshold
            -- party_multi is not working as intended?
            --trust:addSimpleGambit(ai.t.PARTY_MULTI, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spell.CURAGA)

            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURAGA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_II, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURAGA)

            -- Check if the MASTER has protect before we spam it endlessly (members out of range)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.PROTECT, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PROTECTRA)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.SHELL, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SHELLRA)

            -- Cast single target protect/shell if the member does not have it
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.PROTECT, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PROTECT)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.SHELL, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SHELL)

            trust:addSimpleGambit(ai.t.MELEE, ai.c.NOT_STATUS, xi.effect.HASTE, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.HASTE)

            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.POISON, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.POISONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PARALYSIS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.PARALYNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.BLINDNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SILENCE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.SILENA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PETRIFICATION, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.DISEASE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.VIRUNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.CURSE_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURSNA)

            trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS_FLAG, xi.effectFlag.ERASABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.ERASE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS_FLAG, xi.effectFlag.ERASABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.ERASE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.STONESKIN, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STONESKIN)
            -- BGwiki states 75/tick regain.  Only used for Nott WS.
            trust:addMod(xi.mod.REGAIN, 75)

            trust:setAutoAttackEnabled(false)

            trust:setMobMod(xi.mobMod.TRUST_DISTANCE, xi.trust.movementType.MID_RANGE)
        end,
    },

    ["kupipi"] =
    {
        -- Kupipi logic is LSB base, this module is correcting protectra spam
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local windurstFirstTrust = caster:getCharVar('WindurstFirstTrust')
            local zone = caster:getZoneID()

            if
                windurstFirstTrust == 1 and
                (zone == xi.zone.EAST_SARUTABARUTA or zone == xi.zone.WEST_SARUTABARUTA)
            then
                caster:setCharVar('WindurstFirstTrust', 2)
            end

            -- Records of Eminence: Call Forth an Alter Ego
            if caster:getEminenceProgress(932) then
                xi.roe.onRecordTrigger(caster, 932)
            end

            local trust = caster:spawnTrust(spell:getID())

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.SHANTOTTO] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.STAR_SIBYL] = xi.trust.messageOffset.TEAMWORK_2,
            })

            -- LSB has logic for 75% and 25% HPP to use the same gambit logic below
            -- Logical update to prioritize Cure on Tanks (bgwiki)
            -- trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 25, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            -- LSB has a logical issue defining TANK (only PLD/RUN)
            -- trust:addSimpleGambit(ai.t.TANK, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.TOP_ENMITY, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 50, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)

            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_II, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURE)

            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.PROTECT, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PROTECTRA)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.SHELL, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SHELLRA)

            -- Cast single target protect/shell if the member does not have it
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.PROTECT, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PROTECT)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.SHELL, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SHELL)

            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.POISON, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.POISONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PARALYSIS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.PARALYNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.BLINDNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SILENCE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.SILENA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PETRIFICATION, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.DISEASE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.VIRUNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.CURSE_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURSNA)

            trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS_FLAG, xi.effectFlag.ERASABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.ERASE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS_FLAG, xi.effectFlag.ERASABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.ERASE)

            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.PARALYSIS, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PARALYZE, 60)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.SLOW, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SLOW, 60)

            -- Flash is not retail behavior, but it's a nice addition on LSB
            -- trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.FLASH, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.FLASH, 60)

            trust:setMobMod(xi.mobMod.TRUST_DISTANCE, xi.trust.movementType.NO_MOVE)
        end,
    },

    ["prishe_ii"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local psychoAnima  = 3539
            local trust        = caster:spawnTrust(spell:getID())

            xi.trust.teamworkMessage(trust, {
                [xi.magic.spell.TENZEN_II] = xi.trust.messageOffset.TEAMWORK_1,
                [xi.magic.spell.NASHMEIRA_II] = xi.trust.messageOffset.TEAMWORK_2,
                [xi.magic.spell.LILISETTE_II] = xi.trust.messageOffset.TEAMWORK_3,
                [xi.magic.spell.ARCIELA_II] = xi.trust.messageOffset.TEAMWORK_4,
                [xi.magic.spell.IROHA_II] = xi.trust.messageOffset.TEAMWORK_5,
            })

            local itemOneGambit = trust:addSimpleGambit(ai.t.SELF, ai.c.HPP_LT, 35, ai.r.MS, ai.s.SPECIFIC, psychoAnima)
            -- TODO: Add additional logic for Hysteroanima
            -- local itemTwoGambit = mob:addSimpleGambit(ai.t.TARGET, ai.c.CASTING_MA, 35, ai.r.MS, ai.s.SPECIFIC, 3540)
            local healingGambit = trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 25, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURAGA)

            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURAGA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_II, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURAGA)

            -- Trust Synergy: If Ulmia in the party, Prishe will cast Curaga when party members are in the yellow instead of red.
            -- TODO: Prishe will use Cure spells exclusively on Ulmia (Only Cure I-IV)
            -- https://ffxiclopedia.fandom.com/wiki/Trust:_Prishe_II
            trust:addListener('COMBAT_TICK', 'PRISHE_CTICK', function(mobArg)
                local party = mobArg:getMaster():getPartyWithTrusts()
                for _, member in pairs(party) do
                    if member:getObjType() == xi.objType.TRUST then
                        if member:getTrustID() == xi.magic.spell.ULMIA then
                            -- We replace our originally defined gambit if this condition is met
                            mobArg:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURAGA)
                            mobArg:removeSimpleGambit(healingGambit)
                        end
                    end
                end
            end)

            trust:addListener('WEAPONSKILL_USE', 'PRISHE_II_WEAPONSKILL_USE', function(mobArg, target, wsid, tp, action)
                if wsid == 3234 then -- Nullifying Dropkick
                    -- Welcome to Painville!
                    xi.trust.message(mobArg, xi.trust.messageOffset.SPECIAL_MOVE_1)
                end
            end)

            trust:addListener('WEAPONSKILL_STATE_EXIT', 'ANIMA_USED', function(mobArg, wsid)
                if wsid == psychoAnima then
                    mobArg:removeSimpleGambit(itemOneGambit)
                end

                --[[
                if wsid == hysteroAnima then
                    mobArg:removeSimpleGambit(itemTwoGambit)
                end
                ]]--
            end)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    -----------------------------------
    -- Replacement Trusts
    -----------------------------------

    -- lhu_mhakaracca -> Matsui-P
    ["lhu_mhakaracca"] =
    {
        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:setModelId(3121) -- Trust: Matsui-P
            trust:renameEntity("Matsui-P")

            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.COPY_IMAGE, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.UTSUSEMI)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.KURAYAMI)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.SLOW, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.HOJO)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.MB_AVAILABLE, 0, ai.r.MA, ai.s.MB_ELEMENT, xi.magic.spellFamily.NONE)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_SC_AVAILABLE, 0, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.NONE, 45)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_WS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_MS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_JA, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.CASTING_MA, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.INNIN, ai.r.JA, ai.s.SPECIFIC, xi.ja.INNIN)

            trust:addMod(xi.mod.MATT, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.ATT, power)
            trust:addMod(xi.mod.ACC, power * 4)
            trust:addMod(xi.mod.DOUBLE_ATTACK, 25)
            trust:addMod(xi.mod.MP, 150)

            trust:setTrustTPSkillSettings(ai.tp.OPENER, ai.s.RANDOM)
        end,

        onMobSpawn = function(mob)
            for _, member in ipairs(mob:getMaster():getParty()) do
                if member:isPC() then
                    member:printToPlayer("Matsui-P, Lets get this party started", 4, "Matsui-P") --4: MESSAGE_PARTY
                end
            end
        end,
    },

    -- kuyin_hathdenna -> Cornelia
    ["kuyin_hathdenna"] =
    {
        cwEnabled = true,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())

            -- Records of Eminence: Call Forth an Alter Ego
            if caster:getEminenceProgress(932) then
                xi.roe.onRecordTrigger(caster, 932)
            end

            trust:setModelId(3119) -- Trust: Cornelia
            trust:renameEntity("Cornelia")
            trust:addStatusEffectEx(xi.effect.COLURE_ACTIVE, xi.effect.COLURE_ACTIVE, 6, 3, 0, xi.effect.GEO_HASTE, 1200, xi.auraTarget.ALLIES, xi.effectFlag.AURA)
            trust:setAutoAttackEnabled(false)
            trust:setUnkillable(true)

            trust:setLocalVar("MASTER_ID", trust:getMaster():getID())
        end,

        onMobSpawn = function(mob)
            for _, member in ipairs(mob:getMaster():getParty()) do
                if member:isPC() then
                    member:printToPlayer("Cornelia, at your service.", 4, "Cornelia") -- 4: MESSAGE_PARTY
                end
            end
        end,

        onMobDespawn = function(mob)
            local masterId = mob:getLocalVar("MASTER_ID")
            local master = GetPlayerByID(masterId)

            if master == nil then
                return
            end

            for _, member in ipairs(master:getParty()) do
                if member:isPC() then
                    member:printToPlayer("Remember: never give up!", 4, "Cornelia") -- 4: MESSAGE_PARTY
                end
            end
        end,

        onMobDeath = function(mob)
            -- intentionally left blank to prevent conflictions with base
        end,
    },

    -- DISABLED Trusts (GM ONLY)
    ["aahm"] =
    {
        gmOnly = true,

        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl() / 3

            -- Shadows are represented by xi.effect.COPY_IMAGE, but with different icons depending on the tier
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_STATUS, xi.effect.COPY_IMAGE, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.UTSUSEMI)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.KURAYAMI, 60)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_STATUS, xi.effect.SLOW, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.HOJO, 60)

            -- DD Mode
            trust:addSimpleGambit(ai.t.SELF, ai.c.PT_HAS_TANK, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.BERSERK)
            trust:addSimpleGambit(ai.t.SELF, ai.c.PT_HAS_TANK, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.INNIN)
            trust:addSimpleGambit(ai.t.TANK, ai.c.HPP_LT, 50, ai.r.JA, ai.s.SPECIFIC, xi.ja.PROVOKE)

            -- Tank Mode
            trust:addSimpleGambit(ai.t.TARGET, ai.c.NOT_PT_HAS_TANK, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.PROVOKE)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_PT_HAS_TANK, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.WARCRY)
            trust:addSimpleGambit(ai.t.SELF, ai.c.NOT_PT_HAS_TANK, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.YONIN)
            trust:addSimpleGambit(ai.t.MASTER, ai.c.HPP_LT, 50, ai.r.JA, ai.s.SPECIFIC, xi.ja.PROVOKE)

            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.ATT, power * 3)
            trust:addMod(xi.mod.ACC, power * 5)
            trust:addMod(xi.mod.UTSUSEMI_BONUS, 1)


            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["leonoyne"] =
    {
        gmOnly = true,

        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_WS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_MS, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.READYING_JA, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.CASTING_MA, 0, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STUN)
            trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.SOULEATER)
            trust:addSimpleGambit(ai.t.SELF, ai.c.ALWAYS, 0, ai.r.JA, ai.s.SPECIFIC, xi.ja.LAST_RESORT)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.MB_AVAILABLE, 0, ai.r.MA, ai.s.MB_ELEMENT, xi.magic.spellFamily.BLIZZARD)
            trust:addSimpleGambit(ai.t.TARGET, ai.c.MB_AVAILABLE, 0, ai.r.MA, ai.s.MB_ELEMENT, xi.magic.spellFamily.THUNDER)

            trust:addMod(xi.mod.MATT, power * 2)
            trust:addMod(xi.mod.MACC, power * 2)
            trust:addMod(xi.mod.DEF, power * 2)
            trust:addMod(xi.mod.MDEF, power * 2)
            trust:addMod(xi.mod.ATT, power * 3)
            trust:addMod(xi.mod.ACC, power * 5)
            trust:addMod(xi.mod.DOUBLE_ATTACK, 20)
            trust:addMod(xi.mod.TRIPLE_ATTACK, 9)
            trust:addMod(xi.mod.STORETP, 100)
            trust:addMod(xi.mod.REFRESH, 3)

            trust:setTrustTPSkillSettings(ai.tp.ASAP, ai.s.RANDOM)
        end,
    },

    ["ygnas"] =
    {
        gmOnly = true,

        cwEnabled = false,

        onSpellCast = function(caster, target, spell)
            local trust = caster:spawnTrust(spell:getID())
            local power = trust:getMainLvl()

            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 25, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_I, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SLEEP_II, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.HPP_LT, 75, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.CURE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.PROTECT, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.PROTECTRA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.NOT_STATUS, xi.effect.SHELL, ai.r.MA, ai.s.HIGHEST, xi.magic.spellFamily.SHELLRA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.POISON, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.POISONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PARALYSIS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.PARALYNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.BLINDNESS, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.BLINDNA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.SILENCE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.SILENA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.PETRIFICATION, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.STONA)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS, xi.effect.DISEASE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.VIRUNA)
            trust:addSimpleGambit(ai.t.SELF, ai.c.STATUS_FLAG, xi.effectFlag.ERASABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.ERASE)
            trust:addSimpleGambit(ai.t.PARTY, ai.c.STATUS_FLAG, xi.effectFlag.ERASABLE, ai.r.MA, ai.s.SPECIFIC, xi.magic.spell.ERASE)

            trust:addMod(xi.mod.CURE_CAST_TIME, power)
            trust:addMod(xi.mod.MACC, power)
            trust:addMod(xi.mod.CURE_POTENCY, power)
            trust:addMod(xi.mod.DEF, power)
            trust:addMod(xi.mod.MDEF, power)
            trust:addMod(xi.mod.REFRESH, power)
            trust:addMod(xi.mod.HASTE_MAGIC, power)

            trust:setAutoAttackEnabled(false)
        end,
    },
}

for trustName, trustTable in pairs(trustOverride) do
local trustToOverride = string.format("xi.actions.spells.trust.%s", trustName)

    m:addOverride(trustToOverride .. ".onSpellCast", function(caster, target, spell)
        if trustTable.gmOnly ~= nil then
            if target:getGMLevel() > 4 then
                trustTable.onSpellCast(caster, target, spell)
                return
            else
                return
            end
        end

        if caster:isCrystalWarrior() then
            if trustTable.cwEnabled then
                trustTable.onSpellCast(caster, target, spell)
            else
                super(caster, target, spell)
            end
        else
            if trustTable.cwOnly then
                super(caster, target, spell)
            else
                trustTable.onSpellCast(caster, target, spell)
            end
        end
    end)

    m:addOverride(trustToOverride .. ".onMobSpawn", function(mob)
        local master   = mob:getMaster()
        local cwCheck  = master:isCrystalWarrior()

        if cwCheck and trustTable.cwEnabled == false then
            super(mob)
        else
            return
        end
    end)

    m:addOverride(trustToOverride .. ".onMobDespawn", function(mob)
        if trustTable.onMobDespawn ~= nil then
            trustTable.onMobDespawn(mob)
        else
            super(mob)
        end
    end)

    m:addOverride(trustToOverride .. ".onMobDeath", function(mob)
        if trustTable.onMobDeath ~= nil then
            trustTable.onMobDeath(mob)
        else
            super(mob)
        end
    end)
end



return m
