-----------------------------------
require("modules/module_utils")
require('scripts/globals/campaign')
require('scripts/globals/teleports')
-----------------------------------
local m = Module:new("campaign_teleport_npcs")

xi.module.ensureTable("xi.campaign.teleports")

xi.teleport.id.CAMPAIGN = 68

m:addOverride("xi.effects.teleport.onEffectLose", function(target, effect)
    local destination = effect:getPower()
    if destination == 68 then
        local campaignDestination = effect:getSubPower()
        xi.teleport.toCampaign(target, campaignDestination)
    else
        super(target, effect)
    end
end)

local baseCampaignCSID = 454
local baseTeleportCSID = 458

-----------------------------------
-- Campaign Zones enum
-- Used to drive Campaign Teleport Menus and logic
-----------------------------------
local campaignZone = {
    [1] =
    {
        zone = xi.zone.XARCABARD_S,
        fee = 60,
    },
    [2] =
    {
        zone = xi.zone.BEAUCEDINE_GLACIER_S,
        fee = 60,
    },
    [3] =
    {
        zone = xi.zone.BATALLIA_DOWNS_S,
        fee = 50,
    },
    [4] =
    {
        zone = xi.zone.ROLANBERRY_FIELDS_S,
        fee = 50,
    },
    [5] =
    {
        zone = xi.zone.SAUROMUGUE_CHAMPAIGN_S,
        fee = 50,
    },
    [6] =
    {
        zone = xi.zone.JUGNER_FOREST_S,
        fee = 40,
    },
    [7] =
    {
        zone = xi.zone.PASHHOW_MARSHLANDS_S,
        fee = 40,
    },
    [8] =
    {
        zone = xi.zone.MERIPHATAUD_MOUNTAINS_S,
        fee = 40,
    },
    [9] =
    {
        zone = xi.zone.VUNKERL_INLET_S,
        fee = 30,
    },
    [10] =
    {
        zone = xi.zone.GRAUBERG_S,
        fee = 30,
    },
    [11] =
    {
        zone = xi.zone.FORT_KARUGO_NARUGO_S,
        fee = 30,
    },
    [12] =
    {
        zone = xi.zone.EAST_RONFAURE_S,
        fee = 20,
    },
    [13] =
    {
        zone = xi.zone.NORTH_GUSTABERG_S,
        fee = 20,
    },
    [14] =
    {
        zone = xi.zone.WEST_SARUTABARUTA_S,
        fee = 20,
    },
    [15] =
    {
        zone = xi.zone.SOUTHERN_SAN_DORIA_S,
        fee = 20,
    },
    [16] =
    {
        zone = xi.zone.BASTOK_MARKETS_S,
        fee = 20,
    },
    [17] =
    {
        zone = xi.zone.WINDURST_WATERS_S,
        fee = 20,
    },
    -- have not found documentation on the cost of these zones other than that they are enabled
    -- going by the "distance from home nation" trend, these zones are one step further than Rolanberry/Batallia/Sauromugue
    [18] =
    {
        zone = xi.zone.GARLAIGE_CITADEL_S,
        fee = 60,
    },
    [19] =
    {
        zone = xi.zone.CRAWLERS_NEST_S,
        fee = 60,
    },
    [20] =
    {
        zone = xi.zone.THE_ELDIEME_NECROPOLIS_S,
        fee = 60,
    },
}

local zoneToAlliedNation = {
    [xi.zone.SOUTHERN_SAN_DORIA_S] = xi.alliedNation.SANDORIA,
    [xi.zone.BASTOK_MARKETS_S] = xi.alliedNation.BASTOK,
    [xi.zone.WINDURST_WATERS_S] = xi.alliedNation.WINDURST,
}

local function deductFeeAndRetrace(player, fee)
    if player:getCurrency('allied_notes') >= fee then
        player:delCurrency('allied_notes', fee)
        player:addStatusEffectEx(xi.effect.TELEPORT, 0, xi.teleport.id.RETRACE, 0, 1)
    end
end

-- -----------------------------------------------------------------------------------------------
-- getAllowedCampaignTeleports()
-- Returns the WotG allowed teleport zones that the player has previously visted in a set of bits
-- -----------------------------------------------------------------------------------------------
local function getAllowedCampaignTeleports(player)
    local allowedTeleports = 0
    for i, v in pairs(campaignZone) do
        if player:hasVisitedZone(v.zone) then
            local teleBit = i
            -- Note: There is a gap here for bits 15, 16, and 17.  The menu supports but does not use these slots for the 3 WotG Towns
            if
                teleBit < 15 or
                teleBit > 17
            then
                allowedTeleports = bit.bor(allowedTeleports, bit.lshift(1, teleBit))
            end
        end
    end

    if allowedTeleports == 0 then
        -- If a player manages to get to a CA in town w/o going through any past zones - dont show an empty menu - show a "no thanks option"
        allowedTeleports = 1
    end

    return allowedTeleports
end

 ----------------------------------------------------------------------------------------------------------
 -- showTeleportOption
 -- returns 32 (the hide teleport option flag value) if the npc is in the zone the player would retrace to
 ----------------------------------------------------------------------------------------------------------
local function showTeleportOption(npc, campaignAllegiance)
    -- Note all CampaignNPCs should reject players with no allegiance
    local menuOptions = 0
    if
        zoneToAlliedNation[npc:getZoneID()] == campaignAllegiance
    then
        menuOptions = 32
    end

    return menuOptions
end

-- ----------------------------------------------------------------------------
-- teleportFee()
-- Determines fee for a player teleporting from the teleport nation to the destination
-- ----------------------------------------------------------------------------
local function getTeleportFee(player, destination)
    -- TODO:Campgain_Control - We dont have control modeled yet - until we do, we cannot calculate cost based on control
    -- Cost Source: https://ffxiclopedia.fandom.com/wiki/Allied_Notes#Teleportation

    local campaignAllegiance = player:getCampaignAllegiance()
    local npcAllegiance = zoneToAlliedNation[player:getZoneID()]

    local fee = 0
    local destinationDict = campaignZone[destination]
    if destinationDict ~= nil then
        fee = destinationDict.fee
    else
        fee = 60
    end

    -- Calculate control cost - Until control is modeled treat all zones as allied controlled
    fee = fee + 20
    -- if not getCampaignZoneControl(destination) == campaignAllegiance then -- controlled by player's allegiance
        -- if getCampaignZoneControl(destination) == 0 then  -- Beastmen Controlled
        --      -- fee = fee + 40
        -- else -- Allied controlled
        --      -- fee = fee + 20
        -- end
    -- end
    if campaignAllegiance ~= npcAllegiance then
        fee = fee * 1.2
    end

    return fee
end

-- ----------------------------------------------------------------------------
-- teleporterOnTrigger()
-- Triggered when a campaign teleport NPC is triggered
-- Will perform initial campaign teleport setup by providing allegiance and AN
-- ----------------------------------------------------------------------------
m:addOverride("xi.campaign.teleports.teleporterOnTrigger", function(player, npc)
    -- Priming event - sends allegiance and current allied notes
    local campaignAllegiance = player:getCampaignAllegiance()  -- 0 = none, 1 = San d'Oria Iron Rams, 2 = Bastok Fighting Fourth, 3 = Windurst Cobras
    if campaignAllegiance == 0 then
        -- player not allied with any WotG nation
        player:startEvent(baseTeleportCSID - 1) -- Civilian
    else
        player:startEvent(baseTeleportCSID, campaignAllegiance, 0, player:getCurrency('allied_notes'))
    end
end)

-- ----------------------------------------------------------------------------
-- teleporterOnEventUpdate()
-- Triggered when a campaign teleport NPC provides an update
-- Will provide allowed zones and their control (control is future) on intiial update (50)
-- Will provide fee required on destination selection
-- ----------------------------------------------------------------------------
m:addOverride("xi.campaign.teleports.teleporterOnEventUpdate", function(player, csid, option)
    if csid == baseTeleportCSID and option == 50 then
        -- TODO:Campgain_Control - We dont have control modeled yet - until we do, do not show control in the teleport menu
        local ownershipFlags1 = 0 -- Xarc to Pashhow
        local ownershipFlags2 = 0 -- Meriphataud to West Saru
        local ownershipFlags3 = 0 -- Garliage, Crawlers, and Eldime

        local campaignAllegiance = player:getCampaignAllegiance()
        local allowedTeleports = getAllowedCampaignTeleports(player)

        -- Update with allowed teleports and ownership
        player:updateEvent(allowedTeleports, ownershipFlags1, ownershipFlags2, ownershipFlags3, campaignAllegiance, 4, 1, 0) -- TODO research on the last 3 params
    elseif csid == baseTeleportCSID and option <= 20 and option >= 1 then
        -- interestingly - the handling of not enough AN is done client side
        -- we could add an anti-cheat mechanism on envent finish to ensure no exploitation
        local fee = getTeleportFee(player, option)
        player:updateEvent(0, fee)
    end
end)

-- ----------------------------------------------------------------------------
-- teleporterOnEventFinish()
-- Triggered when a campaign teleport NPC provides an EventFinish
-- Will deduct AN and teleport player when provided a destination option
-- ----------------------------------------------------------------------------
m:addOverride("xi.campaign.teleports.teleporterOnEventFinish", function(player, csid, option)
    if csid == baseTeleportCSID and option <= 20 and option >= 1 then
        local fee = getTeleportFee(player, option)
        if player:getCurrency('allied_notes') >= fee then
            player:delCurrency('allied_notes', fee)
            -- TODO:Campaign_Control - We dont have control modeled yet - until we do there is only one telepoint per zone
            -- Once control is modeled - 2 telepoint per zone - one for nation controlled one for beastmen controlled
            player:addStatusEffectEx(xi.effect.TELEPORT, 0, xi.teleport.id.CAMPAIGN, 0, 1, 0, option)
        end
    end
end)

local getNPCAllegiance = function(npc)
    local npcSuffix = string.sub(npc:getName(), -3)
    if
        npcSuffix == '_TK' or
        npcSuffix == '_RK'
    then
        return 1
    elseif
        npcSuffix == '_LC' or
        npcSuffix == '_IM'
    then
        return 2
    elseif
        npcSuffix == '_CC' or
        npcSuffix == '_MC'
    then
        return 3
    else
        return 4
    end
end

-- ----------------------------------------------------------------------------
-- campaignArbiterOnTrigger()
-- Triggered when a campaign NPC is interacted with
-- ----------------------------------------------------------------------------
m:addOverride("xi.campaign.teleports.campaignArbiterOnTrigger", function(player, npc)
    -- Priming event - sends allegiance and current allied notes
    -- 32 will hide teleportation
    -- 4 will show New Allied Tags and allow you to get new allied tags
    -- 4 and 8 will show New Allied Tags and not allow you to get new tags (medal has expired)
    -- 16 enables Performance Assessment and Union Registration
    -- 1 will reduce the Retrace cost, signaling ownership
    local campaignAllegiance = player:getCampaignAllegiance()
    local npcAllegiance = getNPCAllegiance(npc)
    local npcBaseCampaignCSID = baseCampaignCSID + npcAllegiance - 1
    if player:getCampaignAllegiance() == 0 then
        player:startEvent(npcBaseCampaignCSID - 4)
    else
        local ownershipOffset = 0
        if npcAllegiance == campaignAllegiance then
            ownershipOffset = 1
        end

        local menuOptions = showTeleportOption(npc, campaignAllegiance) + ownershipOffset

        player:startEvent(npcBaseCampaignCSID, campaignAllegiance, menuOptions, player:getCurrency('allied_notes'))
    end
end)

-- ----------------------------------------------------------------------------
-- campaignArbiterOnEventUpdate()
-- Triggered when a campaign arbiter provides an update
-- ----------------------------------------------------------------------------
m:addOverride("xi.campaign.teleports.campaignArbiterOnEventUpdate", function(player, csid, option)
    --printf("campaignArbiterOnEventUpdate - csid %u  option %u", csid, option)
end)

-- ----------------------------------------------------------------------------
-- campaignArbiterOnEventFinish()
-- Triggered when a campaign arbiter provides an finish
-- ----------------------------------------------------------------------------
m:addOverride("xi.campaign.teleports.campaignArbiterOnEventFinish", function(player, csid, option)
    if option == 1 then
        local fee = 0
        -- csid offsets in order:
        -- Sandorian Retrace NPC (TK/RK)
        -- Bastokan Retrace NPC (LC/IM)
        -- Windurstian Retrace NPC (CC/MC)
        if baseCampaignCSID - csid - 1 == player:getCampaignAllegiance() then
            -- player's allegiance matches npc's
            fee = 10
        elseif baseCampaignCSID - csid >= 3 then
            -- Juenonian Retrace NPC (CA) -- we cannot match - we are paying 50
            fee = 50
        else
            -- npc allegiance is mismatched from player
            fee = 30
        end

        deductFeeAndRetrace(player, fee)
    end
end)

-- list from https://gitlab.com/ffxiwings/wings/-/merge_requests/589
-- Bastok_Markets_[S].Hostarfaux_TK
-- Bastok_Markets_[S].Narkissa_CA
-- Bastok_Markets_[S].Haralka_CC
-- Bastok_Markets_[S].Waldemar_IM
-- Batallia_Downs_[S].Granite_Fist_LC
-- Batallia_Downs_[S].Myllue_RK
-- Batallia_Downs_[S].Rhu_Arlohtt_MC
-- Batallia_Downs_[S].Telford_CA
-- Beadeaux_[S].Detlef_IM
-- Beadeaux_[S].Parna_CC
-- Beadeaux_[S].Rouquillot_TK
-- Beadeaux_[S].Thurlow_CA
-- Beaucedine_Glacier_[S].Disserond_RK
-- Beaucedine_Glacier_[S].Gray_Colossus_LC
-- Beaucedine_Glacier_[S].Luhk_Leotih_MC
-- Beaucedine_Glacier_[S].Moana_CA
-- Castle_Oztroja_[S].Ortwin_IM
-- Castle_Oztroja_[S].Tangu_CC
-- Castle_Oztroja_[S].Ulmer_CA
-- Castle_Oztroja_[S].Yaibroux_TK
-- Castle_Zvahl_Baileys_[S].Aymeric_TK
-- Castle_Zvahl_Baileys_[S].Holger_IM
-- Castle_Zvahl_Baileys_[S].Lekra_CC
-- Castle_Zvahl_Baileys_[S].Saunders_CA
-- Castle_Zvahl_Keep_[S].Barton_CA
-- Castle_Zvahl_Keep_[S].Amajo_CC
-- Castle_Zvahl_Keep_[S].Lothur_IM
-- Castle_Zvahl_Keep_[S].Sidoine_TK
-- Crawlers_Nest_[S].Astrid_IM
-- Crawlers_Nest_[S].Chefroucauld_TK
-- Crawlers_Nest_[S].Landon_CA
-- Crawlers_Nest_[S].Montsu_CC
-- East_Ronfaure_[S].Aged_Stone_LC
-- East_Ronfaure_[S].Arlayse_RK
-- East_Ronfaure_[S].Ihli_Llamhya_MC
-- East_Ronfaure_[S].Waldo_CA
-- Fort_Karugo-Narugo_[S].Caulaise_RK
-- Fort_Karugo-Narugo_[S].Lamurara_CC
-- Fort_Karugo-Narugo_[S].Marianna_IM
-- Fort_Karugo-Narugo_[S].Wren_CA
-- Garlaige_Citadel_[S].Darden_CA
-- Garlaige_Citadel_[S].Lidaise_TK
-- Garlaige_Citadel_[S].Ruediger_IM
-- Garlaige_Citadel_[S].Nomoa_CC
-- Grauberg_[S].Myuhn_Rohli_MC
-- Grauberg_[S].Polished_Fang_LC
-- Grauberg_[S].Ulaciont_RK
-- Grauberg_[S].Xenia_CA
-- Jugner_Forest_[S].Blazing_Lizard_LC
-- Jugner_Forest_[S].Emmah_Laali_MC
-- Jugner_Forest_[S].Larkin_CA
-- Jugner_Forest_[S].Roiloux_RK
-- La_Vaule_[S].Aloysius_IM
-- La_Vaule_[S].Rasco_CC
-- La_Vaule_[S].Framaraix_TK
-- La_Vaule_[S].Radnor_CA
-- Meriphataud_Mountains_[S].Dhen_Kwherri_MC
-- Meriphataud_Mountains_[S].Kearney_CA
-- Meriphataud_Mountains_[S].Pondering_Peak_LC
-- Meriphataud_Mountains_[S].Raurart_RK
-- North_Gustaberg_[S].Dhi_Emwaltih_MC
-- North_Gustaberg_[S].Estineau_RK
-- North_Gustaberg_[S].Jagged_Onyx_LC
-- North_Gustaberg_[S].Uriah_CA
-- Pashhow_Marshlands_[S].Barnett_CA
-- Pashhow_Marshlands_[S].Stray_Boar_LC
-- Pashhow_Marshlands_[S].Yhu_Lyehga_MC
-- Pashhow_Marshlands_[S].Yuvalbaux_RK
-- Rolanberry_Fields_[S].Gisbert_CA
-- Rolanberry_Fields_[S].Hedioste_RK
-- Rolanberry_Fields_[S].Nori_Kharoiro_MC
-- Rolanberry_Fields_[S].Wayward_Echo_LC
-- Sauromugue_Champaign_[S].Alreage_RK
-- Sauromugue_Champaign_[S].Dancing_Thunder_LC
-- Sauromugue_Champaign_[S].Hdya_Mhirako_MC
-- Sauromugue_Champaign_[S].Marius_CA
-- Southern_San_dOria_[S].Norkum_CC
-- Southern_San_dOria_[S].Kilian_IM
-- Southern_San_dOria_[S].Saphiriance_TK
-- Southern_San_dOria_[S].Scarlette_CA
-- The_Eldieme_Necropolis_[S].Amaliya_CA
-- The_Eldieme_Necropolis_[S].Johann_IM
-- The_Eldieme_Necropolis_[S].Renvriche_TK
-- The_Eldieme_Necropolis_[S].Rululu_CC
-- Vunkerl_Inlet_[S].Felicia_CA
-- Vunkerl_Inlet_[S].Feral_Moon_LC
-- Vunkerl_Inlet_[S].Kih_Katteh_MC
-- Vunkerl_Inlet_[S].Toulsard_RK
-- West_Sarutabaruta_[S].Addison_CA
-- West_Sarutabaruta_[S].Madelleon_RK
-- West_Sarutabaruta_[S].Mhik_Liusihlo_MC
-- West_Sarutabaruta_[S].Tenacious_Fool_LC
-- Windurst_Waters_[S].Dynause_TK
-- Windurst_Waters_[S].Luise_IM
-- Windurst_Waters_[S].Wenonah_CA
-- Windurst_Waters_[S].Soimin-Oimin_CC
-- Xarcabard_[S].Estaud_RK
-- Xarcabard_[S].Sleiney_CA
-- Xarcabard_[S].Timid_Scorpion_LC
-- Xarcabard_[S].Yimi_Jomkeh_MC


-- one in each nation to send you to campaign outposts
local teleportNPCs =
{
    'Bastok_Markets_[S].npcs.Narkissa_CA',
    'Southern_San_dOria_[S].npcs.Scarlette_CA',
    'Windurst_Waters_[S].npcs.Wenonah_CA',
}

-- one in each _S zone and nation to retrace you
-- not all will be enabled, sql file will be updated for one per zone
local retraceNPCs = 
{
    'Bastok_Markets_[S].npcs.Waldemar_IM',
    'Batallia_Downs_[S].npcs.Granite_Fist_LC',
    'Batallia_Downs_[S].npcs.Telford_CA',
    'Beadeaux_[S].npcs.Thurlow_CA',
    'Beaucedine_Glacier_[S].npcs.Moana_CA',
    'Castle_Oztroja_[S].npcs.Ulmer_CA',
    'Castle_Zvahl_Baileys_[S].npcs.Saunders_CA',
    'Castle_Zvahl_Keep_[S].npcs.Barton_CA',
    'Crawlers_Nest_[S].npcs.Landon_CA',
    'East_Ronfaure_[S].npcs.Waldo_CA',
    'Fort_Karugo-Na.npcsrugo_[S].Wren_CA',
    'Garlaige_Citadel_[S].npcs.Darden_CA',
    'Grauberg_[S].npcs.Xenia_CA',
    'Jugner_Forest_[S].npcs.Larkin_CA',
    'La_Vaule_[S].npcs.Radnor_CA',
    'Meriphataud_Mountains_[S].npcs.Kearney_CA',
    'North_Gustaberg_[S].npcs.Uriah_CA',
    'Pashhow_Marshlands_[S].npcs.Barnett_CA',
    'Rolanberry_Fields_[S].npcs.Gisbert_CA',
    'Sauromugue_Champaign_[S].npcs.Marius_CA',
    'Southern_San_dOria_[S].npcs.Saphiriance_TK',
    'The_Eldieme_Necropolis_[S].npcs.Amaliya_CA',
    'Vunkerl_Inlet_[S].npcs.Felicia_CA',
    'West_Sarutabaruta_[S].npcs.Addison_CA',
    'Windurst_Waters_[S].npcs.Soimin-Oimin_CC',
    'Xarcabard_[S].npcs.Sleiney_CA',
}

-- Add functions for all teleport NPCs
for k, v in ipairs(teleportNPCs) do
    local npcPath = string.format("xi.zones.%s", v)
    xi.module.ensureTable(npcPath)

    m:addOverride(npcPath .. ".onTrigger", function(player, npc)
        xi.campaign.teleports.teleporterOnTrigger(player, npc)
    end)

    m:addOverride(npcPath .. ".onEventUpdate", function(player, csid, option, npc)
        xi.campaign.teleports.teleporterOnEventUpdate(player, csid, option)
    end)

    m:addOverride(npcPath .. ".onEventFinish", function(player, csid, option, npc)
        xi.campaign.teleports.teleporterOnEventFinish(player, csid, option)
    end)
end

-- Add functions for all Retrace NPCs
for k, v in ipairs(retraceNPCs) do
    local npcPath = string.format("xi.zones.%s", v)
    xi.module.ensureTable(npcPath)

    m:addOverride(npcPath .. ".onTrigger", function(player, npc)
        xi.campaign.teleports.campaignArbiterOnTrigger(player, npc)
    end)

    m:addOverride(npcPath .. ".onEventUpdate", function(player, csid, option, npc)
        xi.campaign.teleports.campaignArbiterOnEventUpdate(player, csid, option)
    end)

    m:addOverride(npcPath .. ".onEventFinish", function(player, csid, option, npc)
        xi.campaign.teleports.campaignArbiterOnEventFinish(player, csid, option)
    end)
end

return m
