-----------------------------------
-- Dark Ixion Skills
-----------------------------------
require("modules/module_utils")
require("scripts/globals/mobskills")
require("scripts/globals/utils")
require("scripts/globals/dark_ixion")
-----------------------------------
local m = Module:new("dark_ixion_skills")

-----------------------------------
--  Acheron Kick
--
--  Description: Physical Cone Attack damage behind user.
--  Type: Physical
--  Utsusemi/Blink absorb: 2-3 shadows
--  Range: Back
--  Dark Ixion CAN turn around to use this move on anyone with hate, regardless of their original position or even distance.
-----------------------------------

-----------------------------------
-- Damsel Memento
-- Recovers 5% (5,000) of his HP and removes all debuffs.
-- If Dark Ixion's horn has been broken in battle, there's a chance that it will regenerate.
-- Type: Magical
-----------------------------------

-----------------------------------
--  Dark Ixion (Glow)
--  Glow before Wrath of Zeus or Lightning Spear
-----------------------------------

-----------------------------------
-- Dark Ixion (Horn Attack)
-- Unnamed attack for use to gore the target
-----------------------------------

-----------------------------------
-- Dark Ixion (Kick Attack)
-- Unnamed attack for use to kick the target
-----------------------------------

-----------------------------------
-- Trample: Charges forward, dealing high damage to,(400-1000) and lowering the MP (10-30%) of, anyone in his path.
-- No message is displayed in the chat log.
--
-- This move is triggered during onMobFight and is only advertised by the fact that DI runs towards random players in range
--
-- When Dark Ixion's HP is low, he can do up to 3 Tramples in succession.
--     Can be avoided easily by moving out of its path.
--     May charge in the opposite, or an entirely random, direction from the one he is currently facing.
--     Will load a set number of targets in his path before ramming forward. Occasionally,
--     a person in his path will not be hit, as well as those wandering in its path after it has begun its charge.
-----------------------------------

-----------------------------------
--  Lightning Spear
--
--  Description: Wide Cone Attack lightning damage (600-1500) and powerful Amnesia.
--  Type: Magical
--  Notes: Will pick a random person on the hate list for this attack.
-----------------------------------

-----------------------------------
-- Rampant Stance
-- Physical area of effect damage that inflicts stun.
-- Utsusemi/Blink absorb: 1-4 shadows, wipes Third Eye
-- Range: 7.0 (add 0.1-4 depending on terrain elevation)
-- Notes: Takes roughly three seconds to charge the TP move up, enough time for anyone within range to easily back out and run back in directly after the animation begins.
-----------------------------------

-----------------------------------
--  Wrath of Zeus
--
--  Description: Area of Effect lightning damage around Ixion (400-1000) and Silence.
--  Type: Magical
-----------------------------------

local function defaultSkillCheck(target, mob, skill)
    return 0
end

local onSkillCheck =
{
    ["acheron_kick"] = defaultSkillCheck,

    ["damsel_memento"] = function(target, mob, skill)
        if
            mob:getAnimationSub() == 1 or
            mob:getLocalVar('charging') == 1
        then
            return 1
        end

        return 0
    end,

    ["di_glow"] = function(target, mob, skill)
        if
            mob:getAnimationSub() == 1 or
            mob:getLocalVar('charging') == 1
        then
            return 1
        else
            return 0
        end
    end,

    ["di_horn_attack"] = function(target, mob, skill)
        -- don't autoattack between TP moves
        -- only initiate attack within 15 yalms, but db has higher range so you can't "outrun" an attack that starts
        if
            mob:getLocalVar('timeSinceWS') + 1 > os.time() or
            mob:checkDistance(target) > 15
        then
            return 1
        else
            return 0
        end
    end,

    ["di_kick_attack"] = function(target, mob, skill)
        -- don't autoattack between TP moves
        -- only initiate attack within 15 yalms, but db has higher range so you can't "outrun" an attack that starts
        if
            mob:getLocalVar('timeSinceWS') + 1 > os.time() or
            mob:checkDistance(target) > 15
        then
            return 1
        else
            return 0
        end
    end,

    ["di_trample"]      = defaultSkillCheck,
    ["lightning_spear"] = defaultSkillCheck,
    ["rampant_stance"]  = defaultSkillCheck,
    ["wrath_of_zeus"]   = defaultSkillCheck,
}

local onSkillUse =
{
    -- modeled after Sable Breath
    ["acheron_kick"] = function(target, mob, skill)
        -- to allow primary player to run out of conal and to respect the real range of the mobskill
        if
            not target:isBehind(mob, 64) or
            target:checkDistance(mob) > 10
        then
            skill:setMsg(xi.msg.basic.SKILL_MISS)
            return
        end

        local numhits = math.random(2, 3)
        local accmod  = 1
        local dmgmod  = 1
        local info    = xi.mobskills.mobPhysicalMove(mob, target, skill, numhits, accmod, dmgmod)
        local dmg     = 0

        if info.hitslanded > 0 then
            dmg = xi.mobskills.mobFinalAdjustments(info.dmg, mob, skill, target, xi.attackType.PHYSICAL, xi.damageType.SLASHING, info.hitslanded)
        else
            skill:setMsg(xi.msg.basic.SKILL_MISS)
            return
        end

        if skill:getMsg() ~= xi.msg.basic.SHADOW_ABSORB then
            target:takeDamage(dmg, mob, xi.attackType.PHYSICAL, xi.damageType.SLASHING)
            skill:setMsg(xi.msg.basic.DAMAGE)
        end

        return dmg
    end,

    ["damsel_memento"] = function(target, mob, skill)
        mob:delStatusEffectsByFlag(xi.effectFlag.WALTZABLE, false)
        mob:delStatusEffectsByFlag(xi.effectFlag.ERASABLE, false)
        skill:setMsg(xi.msg.basic.SELF_HEAL)

        return xi.mobskills.mobHealMove(mob, mob:getMaxHP() * 5 / 100)
    end,

    ["di_glow"] = function(target, mob, skill)
        skill:setMsg(xi.msg.basic.NONE)
        return 0 -- cosmetic move only
    end,

    ["di_horn_attack"] = function(target, mob, skill)
        local numhits    = 1
        local accmod     = 1
        local dmgmod     = 1
        local typeEffect = xi.effect.BIND
        local duration   = math.random(3, 15)
        local info       = xi.mobskills.mobPhysicalMove(mob, target, skill, numhits, accmod, dmgmod)
        local dmg        = 0

        if info.hitslanded > 0 then
            dmg = xi.mobskills.mobFinalAdjustments(info.dmg, mob, skill, target, xi.attackType.PHYSICAL, xi.damageType.SLASHING, info.hitslanded)
        else
            skill:setMsg(xi.msg.basic.EVADES)
            return
        end

        if skill:getMsg() ~= xi.msg.basic.SHADOW_ABSORB then
            if math.random(100) <= 75 then
                xi.mobskills.mobStatusEffectMove(mob, target, typeEffect, 1, 0, duration)
            end

            target:takeDamage(dmg, mob, xi.attackType.PHYSICAL, xi.damageType.SLASHING)
            skill:setMsg(xi.msg.basic.HIT_DMG)
        end

        return dmg
    end,

    ["di_kick_attack"] = function(target, mob, skill)
        local numhits    = 1
        local accmod     = 1
        local dmgmod     = .6
        local typeEffect = xi.effect.WEIGHT
        local duration   = math.random(3, 15)
        local power      = 50
        local info       = xi.mobskills.mobPhysicalMove(mob, target, skill, numhits, accmod, dmgmod)
        local dmg        = 0
        if info.hitslanded > 0 then
            dmg = xi.mobskills.mobFinalAdjustments(info.dmg, mob, skill, target, xi.attackType.PHYSICAL, xi.damageType.SLASHING, info.hitslanded)
        else
            skill:setMsg(xi.msg.basic.EVADES)
            return
        end

        if skill:getMsg() ~= xi.msg.basic.SHADOW_ABSORB then
            if math.random(100) <= 75 then
                xi.mobskills.mobStatusEffectMove(mob, target, typeEffect, power, 0, duration)
            end

            target:takeDamage(dmg, mob, xi.attackType.PHYSICAL, xi.damageType.SLASHING)
            skill:setMsg(xi.msg.basic.HIT_DMG)
        end

        return dmg
    end,

    ["di_trample"] = function(target, mob, skill)
        local numhits      = 1
        local accmod       = 1
        local dmgmod       = 3
        local remainingMPP = 1 - math.random(10, 30) / 100

        local info = xi.mobskills.mobPhysicalMove(mob, target, skill, numhits, accmod, dmgmod)
        local dmg  = 0
        if info.hitslanded > 0 then
            dmg = xi.mobskills.mobFinalAdjustments(info.dmg, mob, skill, target, xi.attackType.PHYSICAL, xi.damageType.BLUNT, xi.mobskills.shadowBehavior.IGNORE_SHADOWS)
            target:takeDamage(dmg, mob, xi.attackType.PHYSICAL, xi.damageType.SLASHING)
            target:setMP(target:getMP() * remainingMPP)
            skill:setMsg(xi.msg.basic.HIT_DMG)
        else
            skill:setMsg(xi.msg.basic.EVADES)
            return
        end

        return dmg
    end,

    ["lightning_spear"] = function(target, mob, skill)
        -- to allow primary player to run out of conal and to respect the real range of the mobskill
        if
            not target:isInfront(mob, 64) or
            target:checkDistance(mob) > 20
        then
            skill:setMsg(xi.msg.basic.NONE)
            return
        end

        local typeEffect = xi.effect.AMNESIA
        local duration   = math.random(30, 120)

        xi.mobskills.mobStatusEffectMove(mob, target, typeEffect, 1, 0, duration)

        local dmgmod = 10 -- unbuffed player hit for ~2k
        local info   = xi.mobskills.mobMagicalMove(mob, target, skill, mob:getWeaponDmg() * 2, xi.element.THUNDER, dmgmod)
        local dmg    = xi.mobskills.mobFinalAdjustments(info, mob, skill, target, xi.attackType.MAGICAL, xi.damageType.THUNDER, xi.mobskills.shadowBehavior.IGNORE_SHADOWS)
        target:takeDamage(dmg, mob, xi.attackType.MAGICAL, xi.damageType.THUNDER)

        return dmg
    end,

    ["rampant_stance"] = function(target, mob, skill)
        -- to allow primary player to run out of range and to respect the real range of the mobskill
        if target:checkDistance(mob) > 8 then
            skill:setMsg(xi.msg.basic.NONE)
            return
        end

        local numhits = 3
        local accmod  = 2
        local dmgmod  = 1

        local typeEffect = xi.effect.STUN
        local duration = math.random(1, 3)

        local info = xi.mobskills.mobPhysicalMove(mob, target, skill, numhits, accmod, dmgmod)
        local dmg  = 0
        if info.hitslanded > 0 then
            dmg = xi.mobskills.mobFinalAdjustments(info.dmg, mob, skill, target, xi.attackType.PHYSICAL, xi.damageType.BLUNT, info.hitslanded)
        else
            skill:setMsg(xi.msg.basic.EVADES)
            return
        end

        if skill:getMsg() ~= xi.msg.basic.SHADOW_ABSORB then
            xi.mobskills.mobStatusEffectMove(mob, target, typeEffect, 1, 0, duration)

            target:takeDamage(dmg, mob, xi.attackType.PHYSICAL, xi.damageType.SLASHING)
            skill:setMsg(xi.msg.basic.HIT_DMG)
        end

        return dmg
    end,

    ["wrath_of_zeus"] = function(target, mob, skill)
        -- to allow primary player to run out of range and to respect the real range of the mobskill
        if target:checkDistance(mob) > 15 then
            skill:setMsg(xi.msg.basic.NONE)
            return
        end

        local typeEffect = xi.effect.SILENCE
        local duration   = math.random(30, 60)

        xi.mobskills.mobStatusEffectMove(mob, target, typeEffect, 1, 0, duration)

        local dmgmod = 4.5 -- unbuffed hit for ~700
        local info   = xi.mobskills.mobMagicalMove(mob, target, skill, mob:getWeaponDmg() * 2, xi.element.THUNDER, dmgmod)
        local dmg    = xi.mobskills.mobFinalAdjustments(info, mob, skill, target, xi.attackType.MAGICAL, xi.damageType.THUNDER, xi.mobskills.shadowBehavior.IGNORE_SHADOWS)
        target:takeDamage(dmg, mob, xi.attackType.MAGICAL, xi.damageType.THUNDER)

        return dmg
    end,
}

for skillUseName, skillUseFunc in pairs(onSkillUse) do
    local skillUsePath = string.format("xi.actions.mobskills.%s", skillUseName)

    xi.module.ensureTable(skillUsePath)
    m:addOverride(skillUsePath .. ".onMobWeaponSkill", skillUseFunc)
end

for skillCheckName, skillCheckFunc in pairs(onSkillCheck) do
    local skillCheckPath = string.format("xi.actions.mobskills.%s", skillCheckName)

    xi.module.ensureTable(skillCheckPath)
    m:addOverride(skillCheckPath .. ".onMobSkillCheck", skillCheckFunc)
end

return m
