-----------------------------------
-- Fomor Kings
-- Area: Crawlers Nest [S]
-- Mob: Lugh
-- Author: GM Hax
-- Features:
-- weak to water, high meva / sdt to all other elements
-- 30 min rage timer
-- idle despawn will set once damage is started, fast depop
-- drawn in alliance on a cooldown, trigger is target greater than melee range
-- enmity decay every 30-45 seconds
-- large enmity vs pet masters on pet atk commands
-- enchanced enmity on physical damage taken / ws used vs, on a 10s recast
-- Sword ws bonus, 50% chance to use a secondary ws
-- level up on a 25% chance when using fire sword skills or blue magic, caps at + 10 levels - not actual level ups, just stats+
-- casting any magic against it will trigger a magic reflect with a fire 4, locked behind a cooldown
-- enfire / plauge on hit
-- mighty strikes unlocked at 35% hp, 90-120s cooldown. multi hit / haste and no magic casting during.
-----------------------------------
require('modules/module_utils')
require('scripts/globals/hunts')
require('scripts/mixins/fomor_hate')
require('scripts/mixins/job_special')
require('scripts/mixins/rage')
-----------------------------------
local m = Module:new('fomor_lugh')

local modulePath = 'xi.zones.Crawlers_Nest_[S].mobs.Lugh'

xi.module.ensureTable('xi.zones.Crawlers_Nest_[S].mobs.Lugh')

local spawnPos = { x = -251.177, y = -0.436, z = 250.864, rotation = 0 }

local petAbilities = set{
    xi.jobAbility.FIGHT,
    xi.jobAbility.ASSAULT,
    xi.jobAbility.DEPLOY,
}

local function canPerformAction(mob)
    local act = mob:getCurrentAction()
    local isBusy = act == xi.act.MOBABILITY_START
        or act == xi.act.MOBABILITY_USING
        or act == xi.act.MOBABILITY_FINISH
        or act == xi.act.MAGIC_CASTING
        or act == xi.act.MAGIC_START
        or act == xi.act.MAGIC_FINISH

    local isActionQueueEmpty = mob:actionQueueEmpty()
    local isAlive = mob:isAlive()
    local canAct = isAlive and isActionQueueEmpty and not isBusy

    return canAct
end

local function drawInController(mob, target)
    local drawInRecast = mob:getLocalVar('drawInRecast')
    mob:setMobMod(xi.mobMod.DRAW_IN, 2)
    mob:resetEnmity(target)
    mob:setLocalVar('drawInRecast', os.time() + math.random(5, 10))
    mob:timer(4000, function(mobArg)
        mobArg:addTP(3000)
        mobArg:setMobMod(xi.mobMod.DRAW_IN, 0)
    end)
end

local function levelUp(mob)
    if math.random(1, 4) ~= 1 then  -- 25% chance to level up
        return
    end

    local totalLevelUps = mob:getLocalVar('totalLevelUps') or 0

    if totalLevelUps >= 5 then
        return
    end

    mob:injectActionPacket(mob:getID(), 4, 5000, 0, 0, 0, 0, 0)
    mob:addMod(xi.mod.ATT, 15)
    mob:addMod(xi.mod.DEF, 15)
    mob:addMod(xi.mod.HP, 5000)
    mob:addMod(xi.mod.REGEN, 2)
    mob:updateHealth()
    mob:setHP(mob:getMaxHP())
    mob:setLocalVar('totalLevelUps', totalLevelUps + 1)
end

local function enmityDecay(mob, target)
    mob:lowerEnmity(target, 75)  -- 75% enmity reduction
end

m:addOverride(modulePath .. '.onMobInitialize', function(mob)
    g_mixins.job_special(mob)
    g_mixins.rage(mob)
end)

m:addOverride(modulePath .. '.onMobSpawn', function(mob)
    mob:setMod(xi.mod.FIRE_MEVA, 300)  -- weak to water. high meva / sdt to all other elements.
    mob:setMod(xi.mod.ICE_MEVA, 300)
    mob:setMod(xi.mod.WIND_MEVA, 300)
    mob:setMod(xi.mod.EARTH_MEVA, 300)
    mob:setMod(xi.mod.THUNDER_MEVA, 300)
    mob:setMod(xi.mod.LIGHT_MEVA, 300)
    mob:setMod(xi.mod.DARK_MEVA, 300)
    mob:setMod(xi.mod.FIRE_SDT, 300)
    mob:setMod(xi.mod.ICE_SDT, 300)
    mob:setMod(xi.mod.WIND_SDT, 300)
    mob:setMod(xi.mod.EARTH_SDT, 300)
    mob:setMod(xi.mod.THUNDER_SDT, 300)
    mob:setMod(xi.mod.LIGHT_SDT, 300)
    mob:setMod(xi.mod.DARK_SDT, 300)
    mob:setMod(xi.mod.REGEN, 15)  -- ini for this
    mob:setMod(xi.mod.FIRE_ABSORB, 100)
    mob:setMod(xi.mod.DOUBLE_ATTACK, 20)
    mob:addImmunity(xi.immunity.GRAVITY)
    mob:addImmunity(xi.immunity.BIND)
    mob:addImmunity(xi.immunity.SILENCE)
    mob:addImmunity(xi.immunity.LIGHT_SLEEP)
    mob:addImmunity(xi.immunity.DARK_SLEEP)
    mob:addImmunity(xi.immunity.PETRIFY)
    mob:addImmunity(xi.immunity.TERROR)
    mob:addMod(xi.mod.ACC, 46)  -- 400 total acc
    mob:addMod(xi.mod.ATT, 95)  -- 500 total atk
    mob:addMod(xi.mod.DEF, 153) -- 500 total def
    mob:addMod(xi.mod.EVA, 35)  -- 375 total eva
    mob:addMod(xi.mod.DEX, 9)   -- 90 total dex
    mob:addMod(xi.mod.VIT, 12)  -- 85 total vit
    mob:addMod(xi.mod.AGI, 5)   -- 90 total agi
    mob:addMod(xi.mod.MND, 19)  -- 90 total mnd
    mob:addMod(xi.mod.INT, -5)  -- 80 total int
    mob:addMod(xi.mod.MDEF, 20)
    mob:setMod(xi.mod.UFASTCAST, 35)
    mob:addMod(xi.mod.REGAIN, 20)
    mob:addStatusEffect(xi.effect.BLAZE_SPIKES, math.random(15,25), 0, 0)
    mob:setMobMod(xi.mobMod.ADD_EFFECT, 1)
    mob:setLocalVar("[rage]timer", 1800) -- 30 minutes
    mob:setLocalVar('enmityDecayTime', os.time() + 30)  -- ini setup
    mob:addListener('ABILITY_TAKE', 'LUGH_PET_ABILITY_TAKE', function(mobArg, player, ability, action)
        if petAbilities[ability:getID()] then
            mobArg:addEnmity(player, 0, 1800)
        end
    end)
    mob:addListener('MAGIC_TAKE', 'LUGH_MAGIC_TAKE', function(target, caster, spell)
        local currentTime = os.time()
        local magicReflectRecast = target:getLocalVar('magicReflectRecast')

        if spell:tookEffect() and (caster:isPC() or caster:isPet()) then
            if currentTime > magicReflectRecast then
                target:setLocalVar('magicReflect', 1)
                target:setLocalVar('magicReflectRecast', currentTime + math.random(15, 30))
            end
        end
    end)
    mob:addListener('TAKE_DAMAGE', 'LUGH_TAKE_DAMAGE', function(mobArg, amount, attacker, attackType, damageType)
        local currentTime = os.time()
        local lastTakeDamageTime = mobArg:getLocalVar('lastTakeDamageTime')

        if currentTime > lastTakeDamageTime + 10 then
            lastTakeDamageTime = currentTime

            if attackType == xi.attackType.PHYSICAL then
                mobArg:addEnmity(attacker, 500, 500)
            end
        end
    end)
    mob:addListener('WEAPONSKILL_TAKE', 'LUGH_WEAPONSKILL_TAKE', function(targetArg, attacker, skillid, tp, action)
        local currentTime = os.time()
        local lastWeaponSkillTakeTime = targetArg:getLocalVar('lastWeaponSkillTakeTime')

        if currentTime > lastWeaponSkillTakeTime + 10 then
            lastWeaponSkillTakeTime = currentTime

            targetArg:addEnmity(attacker, 500, 500)
        end
    end)
    mob:addListener('EFFECT_LOSE', 'LUGH_SP_END', function(mobArg, effect)
        if effect:getEffectType() == xi.effect.MIGHTY_STRIKES then
            mobArg:delMod(xi.mod.HASTE_ABILITY, 2500)
            mobArg:setMagicCastingEnabled(true)
            mobArg:setMobMod(xi.mobMod.MULTI_HIT, 0)
        end
    end)
    mob:addListener('WEAPONSKILL_STATE_EXIT', 'LUGH_WEAPONSKILL_LVL_UP', function(mobArg, skillId)
        if skillId == xi.weaponskill.BURNING_BLADE or skillId == xi.weaponskill.RED_LOTUS_BLADE then
            levelUp(mobArg)
        end

        local burningBlade = xi.weaponskill.BURNING_BLADE
        local redLotusBlade = xi.weaponskill.RED_LOTUS_BLADE
        local savageBlade = xi.weaponskill.SAVAGE_BLADE
        local repeatSkill = math.random() <= 0.5
        local currentTime = os.time()
        local wsBonusRecast = mobArg:getLocalVar('wsBonusRecast') or 0

        if (skillId == burningBlade or skillId == redLotusBlade or skillId == savageBlade) and currentTime > wsBonusRecast then
            if repeatSkill then
                mobArg:setLocalVar('wsBonusRecast', currentTime + 60)
                mobArg:useMobAbility(skillId)
            end
        end
    end)
    xi.mix.jobSpecial.config(mob, {
        specials =
        {
            {
                id = xi.jsa.MIGHTY_STRIKES,
                cooldown = math.random(90, 120),
                hpp = 35,
                endCode = function(mobArg)
                    mobArg:addMod(xi.mod.HASTE_ABILITY, 2500)
                    mobArg:setMagicCastingEnabled(false)
                    mobArg:setMobMod(xi.mobMod.MULTI_HIT, 1)
                end,
            },
        },
    })
end)

m:addOverride(modulePath .. '.onAdditionalEffect', function(mob, target, damage)
    if math.random(0, 1) == 0 then
        return xi.mob.onAddEffect(mob, target, damage, xi.mob.ae.PLAGUE, {chance = 75, duration = math.random(30, 45)})
    else
        return xi.mob.onAddEffect(mob, target, damage, xi.mob.ae.ENFIRE, {power = math.random(16, 35)})
    end
end)

m:addOverride(modulePath .. '.onMobEngage', function(mob, target)
    mob:addListener('MAGIC_STATE_EXIT', 'LUGH_MAGIC_LVL_UP', function(mobArg, spell)
        if (spell:getID() == xi.magic.spell.REFUELING or
            spell:getID() == xi.magic.spell.LOWING or
            spell:getID() == xi.magic.spell.HEAT_BREATH or
            spell:getID() == xi.magic.spell.EXUVIATION)
        then
            levelUp(mobArg)
        end
    end)
end)

m:addOverride(modulePath .. '.onMobDisengage', function(mob)
    mob:removeListener('LUGH_WEAPONSKILL_LVL_UP')  -- explictly remove this while not in combat
end)

m:addOverride(modulePath .. '.onMobFight', function(mob, target)
    local time = os.time()
    local drawInRecast = mob:getLocalVar('drawInRecast')
    local magicReflect = mob:getLocalVar('magicReflect')
    local magicReflectRecast = mob:getLocalVar('magicReflectRecast')
    local enmityDecayTime = mob:getLocalVar('enmityDecayTime')
    local lifePercent = mob:getHPP()
    local idleDespawnTrigger = mob:getLocalVar("idleDespawnTrigger")
    local lastDrawInCheck = mob:getLocalVar('lastDrawInCheck')

    if lifePercent < 99 and idleDespawnTrigger == 0 then
        mob:setMobMod(xi.mobMod.IDLE_DESPAWN, 10)
        mob:setLocalVar("idleDespawnTrigger", 1)
    end

    if time > lastDrawInCheck + 2 then
        mob:setLocalVar('lastDrawInCheck', time)

        if target then
            local distance = mob:checkDistance(target)
            if distance and distance > 5 and time > drawInRecast then
                drawInController(mob, target)
            end
        end
    end

    if magicReflect == 1 then
        if canPerformAction(mob) then
            mob:castSpell(xi.magic.spell.FIRE_IV, target)
            mob:setLocalVar('magicReflect', 0)
        end
    end

    if time > enmityDecayTime then
        enmityDecay(mob, target)
        mob:setLocalVar('enmityDecayTime', time + math.random(30, 45))
    end
end)

m:addOverride(modulePath .. '.onMobDeath', function(mob, player, optParams)
    xi.hunts.checkHunt(mob, player, 516)
    player:setTitle(xi.title.LUGH_EXORCIST)
end)

m:addOverride(modulePath .. '.onMobDespawn', function(mob)
    mob:resetLocalVars()
    mob:removeListener('LUGH_PET_ABILITY_TAKE')
    mob:removeListener('LUGH_MAGIC_TAKE')
    mob:removeListener('LUGH_TAKE_DAMAGE')
    mob:removeListener('LUGH_WEAPONSKILL_TAKE')
    mob:removeListener('LUGH_SP_END')
    mob:removeListener('LUGH_WEAPONSKILL_LVL_UP')
    mob:removeListener('LUGH_MAGIC_LVL_UP')
end)

m:addOverride('xi.actions.spells.blue.firespit.onSpellCast', function(caster, target, spell)
    local mobName = caster:getName()

    if caster:isMob() and mobName == 'Lugh' then
        local params = {}
        params.ecosystem = xi.ecosystem.BEASTMEN
        params.attackType = xi.attackType.MAGICAL
        params.damageType = xi.damageType.FIRE
        params.attribute = xi.mod.INT
        params.multiplier = 3.0
        params.tMultiplier = 1.5
        params.duppercap = 69
        params.str_wsc = 0.0
        params.dex_wsc = 0.0
        params.vit_wsc = 0.0
        params.agi_wsc = 0.0
        params.int_wsc = 0.2
        params.mnd_wsc = 0.2
        params.chr_wsc = 0.0

        if params.tphitslanded == nil then
            params.tphitslanded = 1
        end

        local damage = xi.spells.blue.useMagicalSpell(caster, target, spell, params)
        local randomizedDamage = math.random(125, 225)

        return randomizedDamage
    else
        return super(caster, target, spell)
    end
end)

m:addOverride('xi.actions.spells.blue.lowing.onSpellCast', function(caster, target, spell)
    local mobName = caster:getName()

    if caster:isMob() and mobName == 'Lugh' then
        local params = {}
        params.ecosystem = xi.ecosystem.BEAST
        params.effect = xi.effect.PLAGUE

        local power = 5
        local tick = 0
        local duration = math.random(15, 30)
        local resistThreshold = 0.0
        local isGaze = false
        local isConal = false

        return xi.spells.blue.useEnfeeblingSpell(caster, target, spell, params, power, tick, duration, resistThreshold, isGaze, isConal)
    else
        return super(caster, target, spell)
    end
end)

return m
