-----------------------------------
-- Create Job Traits for CW items

-- NOTES:
-- onItemUnequip will remove the trait ID
-- even if the job passively has the trait
--
-- If a job is to grant the same trait as /sj
-- things will start to break (ie. negative values in modifiers)
-- this is due to removeJobTrait setting the modifier to a negative value on removal
--
-- If a Rank(2+) JobTrait is being used in the table structure,
-- removal of the trait will start to break things
-- ie. WHM will begin to see a -1 HP return
--
-- The fix I have found is Mog House changing the job or using direct modifier edits
-----------------------------------
require("modules/module_utils")
--------------------------------------------

local m = Module:new("cw_armor_job_traits")

local itemTraits =
{
    -- ItemID, TraitID, Rank, Modifier, Value, MeritID(Default 0)
    {
        item      = "sakpatas_cuisses",
        traitName = "Shield Def. Bonus",
        traitID   = 102,  -- shield def. bonus
        rank      = 1,    -- Rank I
        mod       = 905,  -- SHIELD_DEF_BONUS = 905, // Shield Defense Bonus
        modValue  = 2,    -- 2
        job       = xi.job.WAR,
        lvl       = 75,
    },
    {
        item      = "naga_samue",
        traitName = "Tactical Guard",
        traitID   = 101,  -- tactical guard
        rank      = 1,    -- Rank I
        mod       = 899,  -- TACTICAL_GUARD = 899, // Tp increase when guarding
        modValue  = 30,   -- 30
        job       = xi.job.MNK,
        lvl       = 75,
    },
    -- skip Naga Hakama
    -- skip Bunzis Robe
    -- skip Bunzis Pants
    -- skip Vanya Robe
    {
        item      = "vanya_slops",
        traitName = "Shield Def. Bonus",
        traitID   = 109,  -- Occult Acumen
        rank      = 1,    -- Rank I
        mod       = 902,  -- OCCULT_ACUMEN = 902, // Grants bonus TP when dealing damage with elemental or dark magic
        modValue  = 25,   -- 25
        job       = xi.job.BLM,
        lvl       = 75,
    },
    {
        item      = "malignance_tabard",
        traitName = "Shield Mastery",
        traitID   = 25,   -- shield mastery
        rank      = 1,    -- Rank I
        mod       = 485,  -- SHIELD_MASTERY_TP = 485,  // Shield mastery TP bonus when blocking with a shield
        modValue  = 10,
        job       = xi.job.RDM,
        lvl       = 75,
    },
    {
        item      = "malignance_tights",
        traitName = "Fencer",
        traitID   = 107,  -- fencer
        rank      = 1,    -- Rank I
        mod       = 903,  -- FENCER_TP_BONUS = 903, // TP Bonus to weapon skills from Fencer Trait
        modValue  = 200,
        job       = xi.job.RDM,
        lvl       = 75,
    },
    {
        item      = "adhemar_jacket",
        traitName = "Critical Attack Bonus",
        traitID   = 98,   -- crit. atk. bonus
        rank      = 1,    -- Rank I
        mod       = 421,  -- CRIT_DMG_INCREASE = 421, // Raises the damage of critical hit by percent %
        modValue  = 5,
        job       = xi.job.THF,
        lvl       = 75,
    },
    {
        item      = "founders_breastplate",
        traitName = "Smite",
        traitID   = 127,  -- smite
        rank      = 1,    -- Rank I
        mod       = 898,  -- SMITE = 898, // Raises attack when using H2H or 2H weapons (256 scale)
        modValue  = 25,
        job       = xi.job.PLD,
        lvl       = 75,
    },
    {
        item      = "founders_hose", -- ITEM TRAIT DOES NOT MATCH INGAME DIALOGUE
        traitName = "Attack Bonus (Damage Limit+)", -- NOTE: Damage Limit+ is not coded. Replacing with Attack Bonus.
        traitID   = 3,   -- attack
        rank      = 1,   -- Rank I
        mod       = 23,  -- ATT = 23, // Attack
        modValue  = 20,  -- Granting +20 attack in place of DMG Limit+
        job       = xi.job.PLD,
        lvl       = 75,
    },
    -- skip Gleti's Cuirass
    {
        item      = "gletis_breeches",
        traitName = "Tactical Parry",
        traitID   = 100, -- tactical parry
        rank      = 1,   -- Rank I
        mod       = 486, -- TACTICAL_PARRY = 486, // Tactical Parry Tp Bonus
        modValue  = 20,
        job       = xi.job.DRK,
        lvl       = 75,
    },
    {
        item      = "valorous_mail",
        traitName = "Stout Servant",
        traitID   = 103, -- stout servant
        rank      = 1,   -- Rank I
        mod       = 0,   -- see https://github.com/LandSandBoat/server/blob/base/src/map/utils/petutils.cpp#L1043
        modValue  = 5,   -- PPet->addModifier(Mod::DMG, -(trait->getValue() * 100));
        job       = xi.job.BST,
        lvl       = 75,
    },
    {
        item      = "valorous_hose",
        traitName = "Fencer",
        traitID   = 107,  -- fencer
        rank      = 1,    -- Rank I
        mod       = 903,  -- FENCER_TP_BONUS = 903, // TP Bonus to weapon skills from Fencer Trait
        modValue  = 200,
        job       = xi.job.BST,
        lvl       = 75,
    },
    {
        item      = "pursuers_doublet",
        traitName = "Double Attack",
        traitID   = 15,  -- double attack
        rank      = 1,   -- Rank I
        mod       = 288, -- DOUBLE_ATTACK = 288,  // Percent chance to proc
        modValue  = 10,
        job       = xi.job.BRD,
        lvl       = 75,
    },
    {
        item      = "pursuers_pants",
        traitName = "Fencer",
        traitID   = 107,  -- fencer
        rank      = 1,    -- Rank I
        mod       = 903,  -- FENCER_TP_BONUS = 903, // TP Bonus to weapon skills from Fencer Trait
        modValue  = 200,
        job       = xi.job.BRD,
        lvl       = 75,
    },
    {
        item      = "herculean_vest",
        traitName = "Conserve TP",
        traitID   = 108, -- conserve tp
        rank      = 1,   -- Rank I
        mod       = 944, -- CONSERVE_TP = 944, // Conserve TP trait, random chance between 10 and 200 TP
        modValue  = 15,
        job       = xi.job.RNG,
        lvl       = 75,
    },
    -- skip Herculean Trousers (not coded)
    {
        item      = "chironic_doublet",
        traitName = "Stout Servant",
        traitID   = 103, -- stout servant
        rank      = 1,   -- Rank I
        mod       = 0,   -- see https://github.com/LandSandBoat/server/blob/base/src/map/utils/petutils.cpp#L1043
        modValue  = 5,   -- PPet->addModifier(Mod::DMG, -(trait->getValue() * 100));
        job       = xi.job.SMN,
        lvl       = 75,
    },
    -- skip Chironic Hose
    {
        item      = "ryuo_domaru",
        traitName = "Tactical Parry",
        traitID   = 100, -- tactical parry
        rank      = 1,   -- Rank I
        mod       = 486, -- TACTICAL_PARRY = 486, // Tactical Parry Tp Bonus
        modValue  = 20,
        job       = xi.job.SAM,
        lvl       = 75,
    },
    {
        item      = "ryuo_hakama",
        traitName = "Conserve TP",
        traitID   = 108,  -- conserve tp
        rank      = 1,    -- Rank I
        mod       = 944,  -- CONSERVE_TP = 944, // Conserve TP trait, random chance between 10 and 200 TP
        modValue  = 15,
        job       = xi.job.SAM,
        lvl       = 75,
    },
    {
        item      = "mpacas_doublet",
        traitName = "Tactical Parry",
        traitID   = 100, -- tactical parry
        rank      = 1,   -- Rank I
        mod       = 486, -- TACTICAL_PARRY = 486, // Tactical Parry Tp Bonus
        modValue  = 20,
        job       = xi.job.NIN,
        lvl       = 75,
    },
    {
        item      = "mpacas_hose",
        traitName = "Magic Burst Bonus",
        traitID   = 110,  -- mag. burst bonus
        rank      = 1,    -- Rank I
        mod       = 487,  -- MAG_BURST_BONUS = 487, // Magic Burst Bonus Modifier (percent)
        modValue  = 5,    -- 5%
        job       = xi.job.NIN,
        lvl       = 75,
    },
    -- skip Odyssean Chestplate
    -- skip Odyssean Cuisses
--[[
    -- BLU changes JT table often resulting in broken traits
    -- manually adding JT to BLU below and setting modifiers for power
    {
        item      = "agwus_robe",
        traitName = "Weaponskill Damage Boost",
        traitID   = 134, -- ws damage boost
        rank      = 1,   -- Rank I
        mod       = 840, -- ALL_WSDMG_ALL_HITS = 840,  // Generic (all Weaponskills) damage, on all hits.
        modValue  = 7,
        job       = xi.job.BLU,
        lvl       = 75,
    },
]]
    {
        item      = "ikengas_vest",
        traitName = "Magic Attack Bonus",
        traitID   = 5,   -- magic atk. bonus
        rank      = 1,   -- Rank I
        mod       = 28,  -- MATT = 28, // Magic Attack
        modValue  = 20,
        job       = xi.job.COR,
        lvl       = 75,
    },
    -- skip Ikenga's Trousers
    {
        item      = "thurandaut_tabard",
        traitName = "Stout Servant",
        traitID   = 103, -- stout servant
        rank      = 1,   -- Rank I
        mod       = 0,   -- see https://github.com/LandSandBoat/server/blob/base/src/map/utils/petutils.cpp#L1043
        modValue  = 5,   -- PPet->addModifier(Mod::DMG, -(trait->getValue() * 100));
        job       = xi.job.PUP,
        lvl       = 75,
    },
    -- skip Thurandaut Tights
    {
        item      = "rawhide_vest",
        traitName = "Tactical Parry",
        traitID   = 100, -- tactical parry
        rank      = 1,   -- Rank I
        mod       = 486, -- TACTICAL_PARRY = 486, // Tactical Parry Tp Bonus
        modValue  = 20,
        job       = xi.job.THF,
        lvl       = 75,
    },
    {
        item      = "rawhide_trousers",
        traitName = "Conserve TP",
        traitID   = 108,  -- conserve tp
        rank      = 1,    -- Rank I
        mod       = 944,  -- CONSERVE_TP = 944, // Conserve TP trait, random chance between 10 and 200 TP
        modValue  = 15,
        job       = xi.job.THF,
        lvl       = 75,
    },
    {
        item      = "merlinic_jubbah",
        traitName = "Magic Burst Bonus",
        traitID   = 110,  -- mag. burst bonus
        rank      = 1,    -- Rank I
        mod       = 487,  -- MAG_BURST_BONUS = 487, // Magic Burst Bonus Modifier (percent)
        modValue  = 5,    -- 5%
        job       = xi.job.SCH,
        lvl       = 75,
    },
    -- skip Merlinic Shalwar
    -- skip Amalric Doublet
    -- skip Amalric Slops
    -- skip Eschite Breastplate
    -- skip Eschite Cuisses
}

for k, v in pairs(itemTraits) do
    xi.module.ensureTable(string.format("xi.items.%s", v.item))

    m:addOverride(string.format("xi.items.%s.onItemEquip", v.item), function(target)
        if
            target:getMainJob() ~= v.job or
            target:getMainLvl() < v.lvl
        then
            return
        end

        target:addJobTrait(v.traitID, v.rank, v.mod, v.modValue, 0)
        target:printToPlayer(string.format("Job Trait: %s Granted.", v.traitName), xi.msg.channel.SYSTEM_3)
    end)

    m:addOverride(string.format("xi.items.%s.onItemUnequip", v.item), function(target)

        target:removeJobTrait(v.traitID, v.rank, v.mod, v.modValue, 0)
        target:printToPlayer(string.format("Job Trait: %s Removed.", v.traitName), xi.msg.channel.SYSTEM_3)
    end)
end

----------------------------------------------------------------
-- Handle II - V rank traits here
-- Jobs that have a trait that is passive to the level naturally
-- we will be adding to the modifier as removeJobTrait will set negitive values
----------------------------------------------------------------
local ensureTables =
{
    {
        item = "naga_hakama",
    },
    {
        item = "bunzis_robe",
    },
    {
        item = "bunzis_pants",
    },
    {
        item = "vanya_robe",
    },
    {
        item = "gletis_cuirass",
    },
    {
        item = "herculean_trousers",
    },
    {
        item = "chironic_hose",
    },
    {
        item = "odyssean_chestplate",
    },
    {
        item = "odyssean_cuisses",
    },
    {
        item = "ikengas_trousers",
    },
    {
        item = "thurandaut_tights",
    },
    {
        item = "amalric_doublet",
    },
    {
        item = "amalric_slops",
    },
    {
        item = "eschite_breastplate",
    },
    {
        item = "eschite_cuisses",
    },
    {
        item = "agwus_robe",
    },
    {
        item = "agwus_slops",
    },
    {
        item = "adhemar_kecks",
    },
}
----------------------------------------------------------------
-- Handle EnsureTable
for k, v in pairs(ensureTables) do
    xi.module.ensureTable(string.format("xi.items.%s", v.item))
end
----------------------------------------------------------------
local items =
{
    {
        name = "naga_hakama",
        job  = xi.job.MNK,
        lvl  = 75,
        text = "Counter II",
        mods =
        {
            { xi.mod.COUNTER, 4 },
        },
    },
    {
        name = "bunzis_robe",
        job  = xi.job.WHM,
        lvl  = 75,
        text = "Auto Regen II",
        mods =
        {
            { xi.mod.REGEN, 1 },
        },
    },
    {
        name = "bunzis_pants",
        job  = xi.job.WHM,
        lvl  = 75,
        text = "Magic Def. Bonus V",
        mods =
        {
            { xi.mod.MDEF, 2 },
        },
    },
    {
        name = "vanya_robe",
        job  = xi.job.BLM,
        lvl  = 75,
        text = "Mag. Burst Bonus IV",
        mods =
        {
            { xi.mod.MAG_BURST_BONUS, 2 },
        },
    },
    {
        name = "gletis_cuirass",
        job  = xi.job.DRK,
        lvl  = 75,
        text = "Attack Bonus V",
        mods =
        {
            { xi.mod.ATT,  12 },
            { xi.mod.RATT, 12 },
        },
    },
    {
        -- ("True Shot" not coded)
        -- using +15 rather than (* 1.03) as the removal would very based on case
        name = "herculean_trousers",
        job  = xi.job.RNG,
        lvl  = 75,
        text = "Ranged Attack Bonus",
        mods =
        {
            { xi.mod.RATT, 15 },
        },
    },
    {
        name = "chironic_hose",
        job  = xi.job.SMN,
        lvl  = 75,
        text = "Blood Boon III",
        mods =
        {
            { xi.mod.BLOOD_BOON, 3 },
        },
    },
    {
        name = "odyssean_chestplate",
        job  = xi.job.DRG,
        lvl  = 75,
        text = "Strafe IV",
        mods =
        {
            { xi.mod.WYVERN_BREATH_MACC, 5 },
        },
    },
    {
        name = "odyssean_cuisses",
        job  = xi.job.DRG,
        lvl  = 75,
        text = "Accuracy Bonus III",
        mods =
        {
            { xi.mod.ACC,  13 },
            { xi.mod.RACC, 13 },
        },
    },
    {
        name = "ikengas_trousers",
        job  = xi.job.COR,
        lvl  = 75,
        text = "Rapid Shot II",
        mods =
        {
            { xi.mod.RAPID_SHOT, 5 },
        },
    },
    {
        name = "thurandaut_tights",
        job  = xi.job.PUP,
        lvl  = 75,
        text = "Martial Arts IV",
        mods =
        {
            { xi.mod.MARTIAL_ARTS, 20 },
        },
    },
    {
        name = "amalric_doublet",
        job  = xi.job.GEO,
        lvl  = 75,
        text = "Clear Mind IV",
        mods =
        {
            { xi.mod.MPHEAL, 3 },
        },
    },
    {
        name = "amalric_slops",
        job  = xi.job.GEO,
        lvl  = 75,
        text = "Conserve MP IV",
        mods =
        {
            { xi.mod.CONSERVE_MP, 3 },
        },
    },
    {
        name = "eschite_breastplate",
        job  = xi.job.RUN,
        lvl  = 75,
        text = "Max HP Boost IV",
        mods =
        {
            { xi.mod.HP, 60 },
        },
    },
    {
        name = "eschite_cuisses",
        job  = xi.job.RUN,
        lvl  = 75,
        text = "Accuracy Bonus III",
        mods =
        {
            { xi.mod.ACC,  13 },
            { xi.mod.RACC, 13 },
        },
    },
    {
        name = "agwus_robe",
        job  = xi.job.BLU,
        lvl  = 75,
        text = "Weaponskill Damage Boost",
        mods =
        {
            { xi.mod.ALL_WSDMG_ALL_HITS,  7 },
        },
    },
}

for _, itemInfo in pairs(items) do
    m:addOverride(string.format("xi.items.%s.onItemEquip", itemInfo.name), function(target)
        if
            target:getMainJob() ~= itemInfo.job or
            target:getMainLvl() < itemInfo.lvl
        then
            return
        end

        if itemInfo.mods ~= nil then
            for _, mod in pairs(itemInfo.mods) do
                target:setMod(mod[1], target:getMod(mod[1]) + mod[2])
            end

            target:printToPlayer(string.format("Job Trait: %s Granted.", itemInfo.text), xi.msg.channel.SYSTEM_3)
        end
    end)

    m:addOverride(string.format("xi.items.%s.onItemUnequip", itemInfo.name), function(target)
        if itemInfo.mods ~= nil then
            for _, mod in pairs(itemInfo.mods) do
                target:setMod(mod[1], target:getMod(mod[1]) - mod[2])
            end

            target:printToPlayer(string.format("Job Trait: %s Removed.", itemInfo.text), xi.msg.channel.SYSTEM_3)
        end
    end)
end

return m
