-----------------------------------
-- Ventures - Util
-----------------------------------
require("modules/module_utils")
require("scripts/globals/helm")
require("scripts/globals/utils")
require("scripts/globals/npc_util")
-----------------------------------
local m = Module:new("base_ventures_util")

cexi          = cexi or {}
cexi.ventures = cexi.ventures or {}

local helmTypes =
{
    "Harvesting",
    "Excavation",
    "Logging",
    "Mining",
}

cexi.ventures.exp           = cexi.ventures.exp or {}
cexi.ventures.exp.objective = "You defeated a venture target. (Progress: {}/{})"
cexi.ventures.exp.complete  = "You have successfully completed the venture."
cexi.ventures.exp.points    = "{} gains {} venture points."
cexi.ventures.exp.pointsVar = "[VENTURE]EXP_POINTS"
cexi.ventures.exp.lockVar   = "[VENTURE]EXP_CURRENT"
cexi.ventures.exp.progVar   = "[VENTURE]EXP_PROGRESS"
cexi.ventures.exp.missVar   = "[VENTURE]EXP_MISSED_%u"
cexi.ventures.exp.rollVar   = "[VENTURE]EXP_%u"
cexi.ventures.exp.tallyVar  = "[VENTURE]EXP_TALLY"
cexi.ventures.exp.multiply  = 3
cexi.ventures.exp.multpre50 = 4

cexi.ventures.dynamis           = cexi.ventures.dynamis or {}
cexi.ventures.dynamis.objective = "You defeated a dynamis venture target. (Progress: {}/{})"
cexi.ventures.dynamis.complete  = "You have successfully completed the venture."
cexi.ventures.dynamis.points    = "{} gains {} dynamis venture points."
cexi.ventures.dynamis.pointsVar = "[VENTURE]DYNAMIS_POINTS"
cexi.ventures.dynamis.lockVar   = "[VENTURE]DYNAMIS_CURRENT"
cexi.ventures.dynamis.progVar   = "[VENTURE]DYNAMIS_PROGRESS"
cexi.ventures.dynamis.missVar   = "[VENTURE]DYNAMIS_MISSED_%u"
cexi.ventures.dynamis.rollVar   = "[VENTURE]DYNAMIS_%u"
cexi.ventures.dynamis.tallyVar  = "[VENTURE]DYNAMIS_TALLY"
cexi.ventures.dynamis.dynamisMultiply = true

cexi.ventures.fishing           = cexi.ventures.fishing or {}
cexi.ventures.fishing.objective = "You have completed a special venture objective. (Progress: {}/{})"
cexi.ventures.fishing.complete  = "You have successfully completed the venture."
cexi.ventures.fishing.points    = "{} gains {} fishing venture points."
cexi.ventures.fishing.pointsVar = "[VENTURE]FISHING_POINTS"
cexi.ventures.fishing.lockVar   = "[VENTURE]FISHING_CURRENT"
cexi.ventures.fishing.progVar   = "[VENTURE]FISHING_PROGRESS"
cexi.ventures.fishing.missVar   = "[VENTURE]FISHING_MISSED_%u"
cexi.ventures.fishing.rollVar   = "[VENTURE]FISHING_%u"
cexi.ventures.fishing.tallyVar  = "[VENTURE]FISHING_TALLY"

for _, helmType in pairs(helmTypes) do
    local helmName = string.lower(helmType)
    local helmVar  = string.upper(helmType)

    cexi.ventures[helmName]            = cexi.ventures[helmName] or {}
    cexi.ventures[helmName].multiple   = true
    cexi.ventures[helmName].objective  = "You accumulate {} special venture objective point. (Progress: {}/{})"
    cexi.ventures[helmName].objectives = "You accumulate {} special venture objective points. (Progress: {}/{})"
    cexi.ventures[helmName].complete   = "You have successfully completed the venture."
    cexi.ventures[helmName].points     = "{} gains {} " .. string.lower(helmType) ..  " venture points."
    cexi.ventures[helmName].pointsVar  = "[VENTURE]" .. helmVar  .. "_POINTS"
    cexi.ventures[helmName].lockVar    = "[VENTURE]" .. helmVar  .. "_CURRENT"
    cexi.ventures[helmName].progVar    = "[VENTURE]" .. helmVar  .. "_PROGRESS"
    cexi.ventures[helmName].missVar    = "[VENTURE]" .. helmVar  .. "_MISSED_%u"
    cexi.ventures[helmName].rollVar    = "[VENTURE]" .. helmVar  .. "_%u"
    cexi.ventures[helmName].tallyVar   = "[VENTURE]" .. helmVar  .. "_TALLY"
end

local function usingVentureRing(player)
    if player:hasStatusEffect(xi.effect.DEDICATION) then
        local effect = player:getStatusEffect(xi.effect.DEDICATION)
        local power  = effect:getPower()

        if power == 0 then
            return true
        end
    end

    return false
end

local dynamisExtensions =
{
    xi.ki.CRIMSON_GRANULES_OF_TIME,
    xi.ki.AZURE_GRANULES_OF_TIME,
    xi.ki.AMBER_GRANULES_OF_TIME,
    xi.ki.ALABASTER_GRANULES_OF_TIME,
    xi.ki.OBSIDIAN_GRANULES_OF_TIME,
}

cexi.ventures.updateVenture = function(player, index, expired, ventureInfo, amount, qty, ventureType, rewardItem)
    local result = player:getCharVar(ventureType.progVar)

    -- exp ventures can be reset and retain the kill count, which may have a lower requirement next venture
    if result >= qty and ventureType.pointsVar ~= "[VENTURE]EXP_POINTS" then
        return
    end

    if ventureType.multiple then
        if amount > 1 then
            player:fmt(ventureType.objectives, amount, utils.clamp(result + amount, 1, qty), qty)
        else
            player:fmt(ventureType.objective, amount, utils.clamp(result + amount, 1, qty), qty)
        end
    else
        player:fmt(ventureType.objective, result + amount, qty)
    end

    player:incrementCharVar(ventureType.progVar, amount)

    if result + amount >= qty then
        player:incrementCharVar(ventureType.tallyVar, 1)
        player:setCharVar(ventureType.progVar, (result + amount) - qty)
        player:sys(ventureType.complete, "\129\154", index, "\129\154")

        local isLevel75      = player:getJobLevel(player:getMainJob()) == 75
        local extensionBonus = 1

        if ventureType.dynamisMultiply ~= nil then
            local currentTE      = player:getCharVar("[VENTURE]DYNAMIS_TE")
            extensionBonus = 1 + utils.mask.countBits(currentTE) * 0.2
        end

        if ventureInfo.pts ~= nil then
            local balance = player:getCharVar(ventureType.pointsVar)
            local reward  = math.random(ventureInfo.pts[1], ventureInfo.pts[2])

            -----------------------------------
            -- EXP Venture Multipliers
            -----------------------------------
            if ventureType.multiply ~= nil then
                if isLevel75 then
                    reward = reward * 2
                end
            end

            -----------------------------------
            -- Dynamis Venture Multipliers
            -----------------------------------
            if ventureType.dynamisMultiply ~= nil then
                reward = math.floor(reward * extensionBonus)

            -- Non-Dynamis Ventures get bonus Venture Points under Venture Ring
            elseif usingVentureRing(player) then
                reward = reward * 2
            end

            player:setCharVar(ventureType.pointsVar, balance + reward)
            player:sys(ventureType.points, player:getName(), reward)
        end

        if ventureInfo.exp ~= nil then
            local expAmount = ventureInfo.exp

            -- Only award bonus for EXP Ventures
            if ventureType.multiply ~= nil then
                if
                    player:isCrystalWarrior() and
                    player:getCharVar("[CW]MASTER") == 1
                then
                    local jobID = player:getMainJob()
                    local tier  = player:getCharVar(fmt("[CW]PRESTIGE_{}", xi.jobNames[jobID][1]))

                    -- Job has prestige (+20-100% Bonus EXP)
                    if tier > 0 then
                        expAmount = expAmount + (expAmount * (tier / 5))
                    end
                end

                -- Lv75 bonus multipliers
                if isLevel75 then
                    if player:partyHighestLevel() < 50 then
                        expAmount = expAmount * ventureType.multpre50
                    else
                        expAmount = expAmount * ventureType.multiply
                    end
                end
            end

            -- Only award TE-based bonus for Dynamis Ventures
            if ventureType.dynamisMultiply ~= nil then
                expAmount = math.floor(expAmount * extensionBonus)

                -- Non-Lv75s get reduced EXP reward
                if not isLevel75 then
                    expAmount = math.floor(expAmount / 2)
                end

            -- Non-Dynamis Ventures get reduced EXP under Venture Ring
            elseif usingVentureRing(player) then
                expAmount = math.floor(expAmount / 2)
            end

            -- Award EXP to all Ventures that qualify
            player:addExp(expAmount)
        end

        if rewardItem ~= nil then
            local itemQty = 1
            local itemID  = rewardItem

            if type(rewardItem) == "table" then
                itemQty = math.random(rewardItem[2], rewardItem[3])
                itemID  = rewardItem[1]

                if type(itemID) == "table" then
                    itemID = itemID[math.random(1, #itemID)]
                end
            end

            if ventureType.dynamisMultiply ~= nil then
                itemQty = math.floor(itemQty * extensionBonus)
            end

            if not npcUtil.giveItem(player, { { itemID, itemQty } }) then
                player:incrementCharVar(string.format(ventureType.missVar, itemID), itemQty)
            end
        end

        if expired then
            player:printToPlayer("Your venture has expired.", xi.msg.channel.NS_SAY)
            player:setCharVar(ventureType.lockVar, 0)
            player:setCharVar(ventureType.progVar, 0)
        else
            player:printToPlayer("Your venture begins anew.", xi.msg.channel.NS_SAY)
        end
    end
end

local helmHead =
{
    [xi.helmType.HARVESTING] = 25557, -- Harvester's Sun Hat
    [xi.helmType.EXCAVATION] = 25558, -- Excavator's Shades
    [xi.helmType.LOGGING]    = 25559, -- Lumberjack's Beret
    [xi.helmType.MINING]     = 25560, -- Miner's Helmet
}

cexi.ventures.onHelmResult = function(player, helmType, broke, itemID, immediate)
    if itemID == 0 then
        return
    end

    local head = player:getEquippedItem(xi.slot.HEAD)

    if
        head ~= nil and
        head:getID() == helmHead[helmType] and
        math.random(0, 4) < 1 -- 20%
    then
        if immediate then
            npcUtil.giveItem(player, itemID)
        else
            player:timer(4000, function()
                npcUtil.giveItem(player, itemID)
            end)
        end
    end

    local helmName    = string.lower(helmTypes[helmType])
    local ventureID   = player:getCharVar(cexi.ventures[helmName].lockVar)
    local ventureInfo = cexi.ventures[helmName].mapping[ventureID]

    if ventureInfo == nil then
        return
    end

    local info = cexi.ventures[helmName].ventures[ventureInfo[1]].list[ventureInfo[2]]
    local data = cexi.ventures[helmName].ventures[ventureInfo[1]]

    if player:getZoneName() ~= info[1] then
        return
    end

    for _, itemInfo in pairs(info[2]) do
        if itemInfo[1] == itemID then
            local qty     = itemInfo[3]
            local total   = cexi.ventures[helmName].ventures[ventureInfo[1]].qty
            local expired = GetServerVariable(string.format(cexi.ventures[helmName].rollVar, ventureInfo[1])) ~= ventureInfo[2]
            local delay   = 1000

            if not immediate then
                delay = 3000
            end

            player:timer(delay, function()
                local mounted = player:getLocalVar("Mounted")

                if os.time() > mounted then
                    cexi.ventures.updateVenture(player, ventureID, expired, data, qty, total, cexi.ventures[helmName], data.box)
                else
                    player:sys("To receive credit for special ventures, you must wait longer after using a mount. ({} seconds remaining)", mounted - os.time())
                end
            end)
        end
    end
end

return m
