-----------------------------------
-- Custom HELM
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("base_helm")

cexi        = cexi or {}
cexi.helm   = cexi.helm or {}
cexi.helmID =
{
    HARVESTING = 1,
    EXCAVATION = 2,
    LOGGING    = 3,
    MINING     = 4,
}

local settings =
{
    dialog =
    {
        check = "%s is possible here if you have a %s.",
        full  = "You cannot carry any more items. Your inventory is full.",
    },

    vars =
    {
        NEXT_TRADE = "[HELM]Next_Trade",
        LOCAL_USES = "[HELM]Uses",
    },

    spawnRate =
    {
        minimum = 2, -- Minimum number of points to spawn
        rate    = 1, -- Number of additional points to spawn per 10
    },

    [cexi.helmID.HARVESTING] =
    {
        name         = "Harvest. Point",
        type         = "Harvesting",
        tool         = "sickle",
        toolID       = xi.item.SICKLE,
        look         = 2422,
        animation    = xi.emote.HARVESTING,
        mod          = xi.mod.HARVESTING_RESULT,
        settingRate  = xi.settings.main.HARVESTING_RATE,
        settingBreak = xi.settings.main.HARVESTING_BREAK_CHANCE,
        unable       = "You are unable to harvest anything.",
        success      = "You successfully harvest %s!",
        process      = "You harvest %s, but your %s breaks.", -- Intentional
        breaks       = "Your %s breaks!",
    },

    [cexi.helmID.EXCAVATION] =
    {
        name         = "Excav. Point",
        type         = "Excavation",
        tool         = "pickaxe",
        toolID       = xi.item.PICKAXE,
        look         = 2424,
        animation    = xi.emote.EXCAVATION,
        mod          = xi.mod.EXCAVATION_RESULT,
        settingRate  = xi.settings.main.EXCAVATION_RATE,
        settingBreak = xi.settings.main.EXCAVATION_BREAK_CHANCE,
        unable       = "You are unable to mine anything.",
        success      = "You successfully dig up %s!",
        process      = "You dig up %s, but your %s breaks in the process.",
        breaks       = "Your %s breaks!",
    },

    [cexi.helmID.LOGGING] =
    {
        name         = "Logging Point",
        type         = "Logging",
        tool         = "hatchet",
        toolID       = xi.item.HATCHET,
        look         = 2423,
        animation    = xi.emote.LOGGING,
        mod          = xi.mod.LOGGING_RESULT,
        settingRate  = xi.settings.main.LOGGING_RATE,
        settingBreak = xi.settings.main.LOGGING_BREAK_CHANCE,
        unable       = "You are unable to log anything.",
        success      = "You successfully cut off %s!",
        process      = "You cut off %s, but your %s breaks in the process.",
        breaks       = "Your %s breaks!",
    },

    [cexi.helmID.MINING] =
    {
        name         = "Mining Point",
        type         = "Mining",
        tool         = "pickaxe",
        toolID       = xi.item.PICKAXE,
        look         = 2424,
        animation    = xi.emote.EXCAVATION,
        mod          = xi.mod.MINING_RESULT,
        settingRate  = xi.settings.main.MINING_RATE,
        settingBreak = xi.settings.main.MINING_BREAK_CHANCE,
        unable       = "You are unable to mine anything.",
        success      = "You successfully dig up %s!",
        process      = "You dig up %s, but your %s breaks in the process.",
        breaks       = "Your %s breaks!",
    },
}

local conditional = 
{
    [xi.item.RED_ROCK] =
    {
        condition   = VanadielDayElement,
        replacement =
        {
            [xi.element.FIRE   ] = { xi.item.RED_ROCK,         "a red rock",         },
            [xi.element.ICE    ] = { xi.item.TRANSLUCENT_ROCK, "a translucent rock", },
            [xi.element.WIND   ] = { xi.item.GREEN_ROCK,       "a green rock",       },
            [xi.element.EARTH  ] = { xi.item.YELLOW_ROCK,      "a yellow rock",      },
            [xi.element.THUNDER] = { xi.item.PURPLE_ROCK,      "a purple rock",      },
            [xi.element.WATER  ] = { xi.item.BLUE_ROCK,        "a blue rock",        },
            [xi.element.LIGHT  ] = { xi.item.WHITE_ROCK,       "a white rock",       },
            [xi.element.DARK   ] = { xi.item.BLACK_ROCK,       "a black rock",       },
        },
    },

    [xi.item.FIRE_CLUSTER] =
    {
        condition   = VanadielDayElement,
        replacement =
        {
            [xi.element.FIRE   ] = { xi.item.FIRE_CLUSTER,      "a fire cluster",      },
            [xi.element.ICE    ] = { xi.item.ICE_CLUSTER,       "an ice cluster",      },
            [xi.element.WIND   ] = { xi.item.WIND_CLUSTER,      "a wind cluster",      },
            [xi.element.EARTH  ] = { xi.item.EARTH_CLUSTER,     "an earth cluster",    },
            [xi.element.THUNDER] = { xi.item.LIGHTNING_CLUSTER, "a lightning cluster", },
            [xi.element.WATER  ] = { xi.item.WATER_CLUSTER,     "a water cluster",     },
            [xi.element.LIGHT  ] = { xi.item.LIGHT_CLUSTER,     "a light cluster",     },
            [xi.element.DARK   ] = { xi.item.DARK_CLUSTER,      "a dark cluster",      },
        },
    },

    [xi.item.CHUNK_OF_FIRE_ORE] =
    {
        condition   = VanadielDayElement,
        replacement =
        {
            [xi.element.FIRE   ] = { xi.item.CHUNK_OF_FIRE_ORE,      "a chunk of fire ore"      },
            [xi.element.ICE    ] = { xi.item.CHUNK_OF_ICE_ORE,       "a chunk of ice ore"       },
            [xi.element.WIND   ] = { xi.item.CHUNK_OF_WIND_ORE,      "a chunk of wind ore"      },
            [xi.element.EARTH  ] = { xi.item.CHUNK_OF_EARTH_ORE,     "a chunk of earth ore"     },
            [xi.element.THUNDER] = { xi.item.CHUNK_OF_LIGHTNING_ORE, "a chunk of lightning ore" },
            [xi.element.WATER  ] = { xi.item.CHUNK_OF_WATER_ORE,     "a chunk of water ore"     },
            [xi.element.LIGHT  ] = { xi.item.CHUNK_OF_LIGHT_ORE,     "a chunk of light ore"     },
            [xi.element.DARK   ] = { xi.item.CHUNK_OF_DARK_ORE,      "a chunk of dark ore"      },
        },
    },
}

local pickItem = function(player, helmType, items)
    if math.random(100) > (helmType.settingRate + player:getMod(2000)) then
        return { 0, 0, "" }
    end

    local result = cexi.util.pickItem(items)

    for itemID, conditions in pairs(conditional) do
        if result[2] == itemID then
            result = conditions.replacement[conditions.condition()]
            return { 0, result[1], result[2] }
        end
    end

    return result
end

local doesToolBreak = function(player, helmType)
    local roll  = math.random(1, 100)

    if helmType.mod then
        roll = roll + (player:getMod(helmType.mod) / 10)
    end

    if roll <= helmType.settingBreak then
        player:tradeComplete()
        return true
    end

    return false
end

local doMove = function(npc, x, y, z)
    return function(entity)
        entity:setPos(x, y, z, 0)
    end
end

local movePoint = function(npc, points, immediate)
    local point  = points[math.random(1, #points)]

    if immediate then
        npc:hideNPC(3)
        npc:queue(3000, doMove(npc, unpack(point)))
    else
        npc:hideNPC(30)
        npc:queue(3000, doMove(npc, unpack(point)))
    end
end

local isLocked = function(player, tbl)
    if tbl.info.var then
        if player:getCharVar(tbl.info.var) == 0 then
            local message = string.format("You have not unlocked the ability to perform %s in the current area.", settings[tbl.info.type].type)
            player:printToPlayer(message, xi.msg.channel.SYSTEM_3)
            return true
        end
    end

    return false
end

cexi.helm.helmOnTrigger = function(player, npc, helmType, tbl)
    if isLocked(player, tbl) then
        return
    end

    player:printToPlayer(string.format(
        settings.dialog.check,
        helmType.type,
        helmType.tool
    ), xi.msg.channel.NS_SAY)
end

local handleBreak = function(player, item, helmType)
    -- Tool broke, found item
    if item[2] ~= 0 then
        player:printToPlayer(string.format(
            helmType.process,
            item[3],
            helmType.tool
        ), xi.msg.channel.NS_SAY)

    -- Tool broke, found nothing
    else
        player:printToPlayer(string.format(
            helmType.breaks,
            helmType.tool
        ), xi.msg.channel.NS_SAY)
    end
end

local handleResult = function(player, item, helmType, extra)
    local breaks = doesToolBreak(player, helmType)

    if breaks then
        handleBreak(player, item, helmType)
    end

    if item[2] ~= 0 then

        if not breaks then
            player:printToPlayer(string.format(
                helmType.success,
                item[3]
            ), xi.msg.channel.NS_SAY)
        end

        -- Handle ??? item
        if extra ~= nil then
            for itemID, _ in pairs(extra.appraisal) do
                if itemID == item[2] then
                    player:addItem({ id = item[2], appraisal = extra.sourceID })
                    return true
                end
            end
        end

        -- Handle regular item
        player:addItem(item[2])
        cexi.ventures.onHelmResult(player, cexi.helmID[string.upper(helmType.type)], false, item[2], true)

        return true
    end

    if not breaks then
        player:printToPlayer(helmType.unable, xi.msg.channel.NS_SAY)
    end

    return false
end

local handleUses = function(npc, points, immediate)
    local uses = (npc:getLocalVar(settings.vars.LOCAL_USES) - 1) % 4
    npc:setLocalVar(settings.vars.LOCAL_USES, uses)

    if uses == 0 then
        movePoint(npc, points, immediate)
    end
end

cexi.helm.helmOnTrade = function(player, npc, trade, tbl)
    if isLocked(player, tbl) then
        return
    end

    local helmType   = settings[tbl.info.type]
    local zoneId     = player:getZoneID()
    local nextTrade  = player:getLocalVar(settings.vars.NEXT_TRADE)
    local validTrade = trade:hasItemQty(helmType.toolID, 1) and trade:getItemCount() == 1

    -- HELM should remove invisible
    player:delStatusEffect(xi.effect.INVISIBLE)

    if not validTrade then
        cexi.helm.helmOnTrigger(player, npc, helmType, tbl)
        return
    end

    if os.time() < nextTrade then
        player:messageBasic(xi.msg.basic.WAIT_LONGER, 0, 0)
        return
    end

    player:sendEmote(npc, helmType.animation, xi.emoteMode.MOTION)
    player:selfEmote(npc, helmType.animation, xi.emoteMode.MOTION)

    if player:getFreeSlotsCount() == 0 then
        player:printToPlayer(settings.dialog.full, xi.msg.channel.NS_SAY)
        return
    end

    local item = pickItem(player, helmType, tbl.items)

    -- Allow 3 seconds for animation
    player:timer(3000, function(playerArg)
        if handleResult(playerArg, item, helmType, tbl.extra) then
            handleUses(npc, tbl.points, tbl.immediate)
            player:triggerRoeEvent(xi.roeTrigger.HELM_SUCCESS, { ['skillType'] = tbl.info.type })
        end
    end)

    player:setLocalVar(settings.vars.NEXT_TRADE, os.time() + 3)
end

cexi.helm.helmAdd = function(source, tbl)
    if xi.settings.main.ENABLE_CW == 0 then
        return
    end

    source:addOverride(string.format("xi.zones.%s.Zone.onInitialize", tbl.info.zone), function(zone)
        super(zone)

        local zoneId = zone:getID()
        local total = #tbl.points
        local spawn = math.ceil((total / 10) * settings.spawnRate.rate) + settings.spawnRate.minimum

        for i = 1, spawn do
            local dynamicPoint = zone:insertDynamicEntity({
                name      = settings[tbl.info.type].name,
                objtype   = xi.objType.NPC,
                look      = settings[tbl.info.type].look,
                x         = tbl.points[i][1],
                y         = tbl.points[i][2],
                z         = tbl.points[i][3],
                rotation  = 0,
                widescan  = 0,

                onTrigger = function(player, npc)
                    cexi.helm.helmOnTrigger(player, npc, settings[tbl.info.type], tbl)
                end,

                onTrade   = function(player, npc, trade)
                    cexi.helm.helmOnTrade(player, npc, trade, tbl)
                end,
            })

            movePoint(dynamicPoint, tbl.points, true)
        end
    end)

    if
        tbl.extra ~= nil and
        tbl.extra.appraisal ~= nil
    then
        cexi.appraisal = cexi.appraisal or {}

        for itemID, itemInfo in pairs(tbl.extra.appraisal) do
            cexi.appraisal[itemID] = itemInfo
        end
    end
end

return m
