-----------------------------------
-- Incursion
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("base_incursion")

cexi           = cexi or {}
cexi.incursion = cexi.incursion or {}

local info =
{
    -- Player vars
    zone      = "[INC]ZONE",       -- Can probably be safely removed once refactored
    delay     = "[INC]DELAY_{}",   -- Zone lockout
    phase     = "[INC]PHASE",      -- Current phase (eg. 1/4)
    mobs      = "[INC]MOBS",       -- Phase mob counter
    bonus     = "[INC]BONUS",      -- Bonus type
    bonusMobs = "[INC]BONUS_MOBS", -- Bonus mob counter
    buffs     = "[INC]BUFFS",      -- Temporary mods

    -- Local vars
    source    = "[INC]SOURCE",     -- Player is eligible to use a source
    buff1     = "[INC]SOURCE_1",
    buff2     = "[INC]SOURCE_2",
    buff3     = "[INC]SOURCE_3",

    -- Server vars
    res       = "[INC]RESERVED_{}",
}

cexi.incursion.bonusType =
{
    CHEST = 1,
    MOBS  = 2,
    BOSS  = 3,
}

-----------------------------------
-- Temp Items
-----------------------------------
local tempItems =
{
    { "Dusty Reraise",  5436, 150 },
    { "Dusty Elixir",   5433, 100 },
    { "Lucid Potion I", 5824,  50 },
    { "Lucid Ether I",  5827,  50 },
}

local extraTemps =
{
    5440, -- Dusty Wing
    4255, -- Mana Powder
    5390, -- Braver's Drink
    4206, -- Catholicon
    4147, -- Body Boost
    4200, -- Mana Boost
}

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function confirmPurchase(player, npc, item)
    local pts = player:getCharVar("[GARRISON]POINTS")

    if pts < item[3] then
        npc:facePlayer(player, true)
        player:fmt("You don't have enough points.")
        return
    end

    delaySendMenu(player, {
        title   = fmt("Buy {} ({} pts)?", item[1], item[3]),
        options =
        {
            {
                "Yes",
                function()
                    local ID = zones[player:getZoneID()]

                    if player:addTempItem(item[2]) then
                        player:incrementCharVar("[GARRISON]POINTS", item[3])
                        player:messageSpecial(ID.text.ITEM_OBTAINED, item[2])
                    end
                end,
            },
            {
                "No",
                function()
                end,
            },
        },
    })
end

local function updateBuffs(player)
    local currentBuffs = player:getCharVar(info.buffs)
    local effect       = player:getStatusEffect(xi.effect.LEVEL_RESTRICTION)

    if currentBuffs > 0 then
        for index = 1, 32 do
            if utils.mask.getBit(currentBuffs, index - 1) then
                for _, modInfo in pairs(cexi.buffs[index].mods) do
                    player:addMod(modInfo[1], modInfo[2])
                    effect:addMod(modInfo[1], modInfo[2])
                end
            end
        end
    end
end

-----------------------------------
-- Utils
-----------------------------------
local function updateStatus(player, cap, zoneID)
    local lockout   = player:getCharVar(fmt(info.delay, zoneID))
    local remaining = lockout - os.time()

    print(fmt("[INCURSION] {} entered/recovered session at {} cap for {} minutes", player:getName(), cap, (remaining / 60)))

    if player:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
        player:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
    end

    player:addStatusEffectEx(
        xi.effect.LEVEL_RESTRICTION,
        xi.effect.LEVEL_RESTRICTION,
        cap,
        0,
        remaining,
        0,
        0,
        0,
        xi.effectFlag.ON_ZONE
    )

    updateBuffs(player)

    player:sys("You have {} minutes remaining inside this Incursion.", math.floor(remaining / 60))

    local pet = player:getPet()

    if pet ~= nil then
        if pet:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
            pet:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
        end

        pet:addStatusEffectEx(
            xi.effect.LEVEL_RESTRICTION,
            xi.effect.LEVEL_RESTRICTION,
            cap,
            0,
            remaining,
            0,
            0,
            0,
            xi.effectFlag.ON_ZONE
        )
    end
end

cexi.incursion.getMobList = function(phase)
    return fmt("{} enemies ({}, {}, {})",
        phase.boss.reqs,
        string.gsub(phase.mobs[1], "_", " "),
        string.gsub(phase.mobs[2], "_", " "),
        string.gsub(phase.mobs[3], "_", " ")
    )
end

local function inInstance(player, phaseIndex)
    if
        player == nil or
        not player:isPC() or
        not player:hasStatusEffect(xi.effect.LEVEL_RESTRICTION)
    then
        return false
    end

    local zoneID = player:getZoneID()

    if
        player:getCharVar(info.zone) ~= zoneID or
        (phaseIndex ~= nil and player:getCharVar(info.phase) ~= phaseIndex)
    then
        return false
    end

    return true
end

local function giveCurrency(player, currency, minAmount, maxAmount)
    local amount = math.random(minAmount, maxAmount)

    if not npcUtil.giveItem(player, { { currency, amount }}) then
        player:incrementCharVar(fmt("[INC]MISSED_{}", currency), amount)
        player:fmt("{} pieces of missed currency were stored at Garret.", amount)
    end
end

cexi.incursion.wsTrigger = function(mob, currency)
    mob:addListener('WEAPONSKILL_TAKE', 'WS_PROC_CHECK', function(target, user, wsid)
        local triggered = mob:getLocalVar("TRIGGERED")

        if
            triggered > 0 or
            user == nil or
            not user:isPC()
        then
            return
        end

        local triggerType = mob:getLocalVar("TRIGGER")

        if triggerType == 0 then
            triggerType = xi.weaponskillTypes[math.random(1, #xi.weaponskillTypes)]
            mob:setLocalVar("TRIGGER", triggerType)

            local mobName = string.gsub(mob:getName(), "DE_", "")
            user:printToArea(fmt("appears to be susceptible to {} weaponskills.", xi.skillName[triggerType]), xi.msg.channel.EMOTION, xi.msg.area.SAY, mobName)

            return
        end

        if xi.weaponskillType[wsid] == triggerType then
            mob:setLocalVar("TRIGGERED", 1)
            mob:weaknessTrigger(2)
            mob:addStatusEffect(xi.effect.TERROR, 0, 0, 10)

            -- Take 25% more damage
            mob:setMod(xi.mod.UDMGPHYS,   2500)
            mob:setMod(xi.mod.UDMGRANGE,  2500)
            mob:setMod(xi.mod.UDMGMAGIC,  2500)
            mob:setMod(xi.mod.UDMGBREATH, 2500)

            local alliance   = user:getAlliance()
            local zoneID     = user:getZoneID()
            local ID         = zones[zoneID]
            local playerName = user:getName()

            for _, member in pairs(alliance) do
                if
                    member ~= nil and
                    member:isPC() and
                    member:getZoneID() == zoneID
                then
                    local temp = extraTemps[math.random(1, #extraTemps)]

                    member:fmt("{} successfully triggers the weakness!", playerName)
                    member:messageSpecial(ID.text.ITEM_OBTAINED + 9, temp, 1)
                    member:addTempItem(temp)

                    giveCurrency(member, currency, 1, 2)
                end
            end
        end
    end)
end

cexi.incursion.etherealSource = function(player, npc)
    if player:getLocalVar(info.source) == 0 then
        player:fmt("Nothing happens.")
        return
    end

    local current = player:getCharVar(info.buffs)
    local rolls   =
    {
        player:getLocalVar(info.buff1),
        player:getLocalVar(info.buff2),
        player:getLocalVar(info.buff3),
    }

    -- If this is the first time player opened menu, roll the buffs
    if
        rolls[1] == 0 or
        rolls[2] == 0 or
        rolls[3] == 0
    then
        rolls = cexi.util.randomBitNoRepeat(current, 3)

        player:setLocalVar(info.buff1, rolls[1])
        player:setLocalVar(info.buff2, rolls[2])
        player:setLocalVar(info.buff3, rolls[3])
    end

    local options = {}

    for _, roll in pairs(rolls) do
        local buff     = cexi.buffs[roll]
        local fullName = fmt("{} ({})", buff.name, string.rep("\129\154", buff.rank))

        table.insert(options, {
            fullName,
            function()
                player:setLocalVar(info.source, 0)

                player:sys("{} gains the effect of {}: {}", player:getName(), fullName, buff.desc)

                local zoneID  = player:getZoneID()
                local lockout = player:getCharVar(fmt(info.delay, zoneID))
                local result  = utils.mask.setBit(current, roll - 1, true) -- Starts at 0

                -- Set expiry to remaining time
                player:setCharVar(info.buffs, result, lockout)

                -- Clear options
                player:setLocalVar(info.buff1, 0)
                player:setLocalVar(info.buff2, 0)
                player:setLocalVar(info.buff3, 0)


                local effect = player:getStatusEffect(xi.effect.LEVEL_RESTRICTION)

                -- Apply changes
                for _, modInfo in pairs(buff.mods) do
                    player:addMod(modInfo[1], modInfo[2])
                    effect:addMod(modInfo[1], modInfo[2])
                end
            end,
        })
    end

    player:customMenu({
        title   = "Choose a power:",
        options = options,
    })
end

-----------------------------------
-- Entry
-----------------------------------
cexi.incursion.checkInstance = function(player, npc, battle)
    if not player:isCrystalWarrior() then
        player:fmt("You see nothing out of the ordinary.")
        return
    end

    local delayVar = fmt(info.delay, battle.zone)

    if player:getCharVar(delayVar) ~= 0 then
        if player:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
            local balance = player:getCharVar("[GARRISON]POINTS")
            cexi.util.simpleShop(player, npc, tempItems, confirmPurchase, fmt("Temp Items ({}):", balance))
            return
        end

        player:fmt("You have already attempted this area for the current tally period.")
        return
    end

    local alliance = player:getAlliance()

    for _, member in pairs(alliance) do
        if member:getCharVar(delayVar) ~= 0 then
            player:fmt("A member of your alliance is currently locked for this instance.")
            return
        end
    end

    player:fmt("Something may happen if you plant a Beastman Banner here...")
end


cexi.incursion.beginInstance = function(player, npc, trade, battle, required)
    if not player:isCrystalWarrior() then
        player:fmt("This content is not available to your character type.")
        return
    end

    local delayVar = fmt(info.delay, battle.zone)

    if player:getCharVar(delayVar) ~= 0 then
        player:fmt("You have already attempted this area for the current tally period.")
        return
    end

    local alliance = player:getAlliance()

    if #alliance > battle.size then
        player:fmt("Your group is too large to begin this incursion. (Max. {})", battle.size)
        return
    end

    for _, member in pairs(alliance) do
        if member:getCharVar(delayVar) ~= 0 then
            player:fmt("A member of your alliance has already attempted this area in the current tally period.")
            return
        end
    end

    local resVar    = fmt(info.res, player:getZoneID())
    local remaining = GetServerVariable(resVar)

    if remaining > os.time() then
        local minutes = math.ceil((remaining - os.time()) / 60)
        player:fmt("This area is reserved for the next {} minute(s).", minutes)
        return
    end

    local areaName = string.gsub(battle.area, "_", " ")

    if npcUtil.tradeHasExactly(trade, required) then
        local zone     = player:getZone()
        local flagDE   = zone:queryEntitiesByName("DE_Incursion Flag")
        local delayVar = fmt(info.delay, battle.zone)
        local expiry   = os.time() + 5400 -- 90 Minutes

        SetServerVariable(resVar, expiry, expiry)

        for _, member in pairs(alliance) do
            member:setCharVar(info.zone, battle.zone)
            member:setCharVar(info.phase, 1)
            member:setCharVar(info.mobs,  0)
            member:setCharVar(delayVar, expiry, JstMidnight())
            updateStatus(member, battle.phases[1].cap, battle.zone)
            member:fmt("Incursion [{}] Begins!", areaName)
            member:fmt("New Objective: Defeat {}", cexi.incursion.getMobList(battle.phases[1]))

            if flagDE ~= nil then
                flagDE[1]:ceSpawn(member)
            end
        end

        player:tradeComplete()
    end
end

cexi.incursion.recoverSession = function(player, prevZone, super, battle)
    local lockout = player:getCharVar(fmt(info.delay, battle.zone))

    if lockout == 0 then
        return super(player, prevZone)
    end

    if os.time() > lockout then
        player:setCharVar(info.bonus, 0)
        player:setCharVar(info.phase, 0)
        player:setCharVar(info.mobs,  0)
        player:setCharVar(info.zone,  0)

        return super(player, prevZone)
    end

    local phase = player:getCharVar(info.phase)

    if phase == 0 then
        phase = 1
    end

    player:timer(2000, function()
        local areaName = string.gsub(battle.area, "_", " ")
        player:fmt("Incursion [{}] Recovering session...", areaName)
    end)

    player:timer(5000, function()
        updateStatus(player, battle.phases[phase].cap, battle.zone)
    end)
end

-----------------------------------
-- Bonus Objectives
-----------------------------------
cexi.incursion.getBonusVarReset = { info.bonus, 0 }

cexi.incursion.getBonusSpawnerRequirements = function(phase, phaseIndex)
    return {
        condition = { info.phase, phaseIndex },
        dialog    =
        {
            "You do not currently have the required bonus objective.",
        },
    }
end

cexi.incursion.getBonusChecks = function(battle, phaseIndex)
    return {
        vareq = { info.area,  battle.zone                   },
        vareq = { info.phase, phaseIndex                    },
        vareq = { info.bonus, cexi.incursion.bonusType.BOSS },
    }
end

local function spawnBonusChest(zone, nextPhase)
    local result = zone:queryEntitiesByName("DE_Incursion Chest")

    if
        result ~= nil and
        result[1] ~= nil
    then
        local nextPos = nextPhase.bonus.chest[math.random(1, #nextPhase.bonus.chest)]
        result[1]:setPos(unpack(nextPos))
        result[1]:setStatus(xi.status.NORMAL)
        result[1]:setLocalVar("OPENED", 0)
        result[1]:setLocalVar("EXPIRY", os.time() + 600)

        result[1]:timer(605000, function(npcArg)
            if os.time() > npcArg:getLocalVar("EXPIRY") then
                print("[INC] Hiding Incursion Chest (Player did not find after 10 min.)")
                npcArg:setStatus(xi.status.DISAPPEAR)
                npcArg:setLocalVar("OPENED", 0)
            end
        end)
    end
end

local function rollBonusObjective(mob, player, battle, nextPhase)
    if nextPhase.bonus == nil then
        return
    end

    local bonusType = mob:getLocalVar("BONUS_ROLLED")

    if bonusType == 0 then
        bonusType = math.random(1, 3)
        mob:setLocalVar("BONUS_ROLLED", bonusType)

        if bonusType == cexi.incursion.bonusType.CHEST then
            spawnBonusChest(mob:getZone(), nextPhase)
        end
    end

    player:timer(2000, function()
        player:setCharVar(info.bonus, bonusType, os.time() + 600)

        if bonusType == cexi.incursion.bonusType.CHEST then
            player:fmt("Bonus Objective: Find the hidden chest! (Expires in 10 Minutes)")

        elseif bonusType == cexi.incursion.bonusType.MOBS then
            local mobName = string.gsub(nextPhase.bonus.mobs[1], "_", " ")
            player:fmt("Bonus Objective: Defeat {} {}! (Expires in 10 Minutes)", nextPhase.bonus.mobs[2], mobName)

        elseif bonusType == cexi.incursion.bonusType.BOSS then
            local mobName = string.gsub(nextPhase.bonus.boss.name, "_", " ")
            player:fmt("Bonus Objective: Defeat {} at {}! (Expires in 10 Minutes)", mobName, nextPhase.bonus.boss.desc)
        end
    end)

    return 1
end

cexi.incursion.defeatBonusMob = function(mob, player, battle, phase, phaseIndex)
    if not inInstance(player, phaseIndex) then
        return
    end

    if
        player:getCharVar(info.bonus) ~= cexi.incursion.bonusType.MOBS or
        player:getCharVar(info.phase) ~= phaseIndex
    then
        return
    end

    local total = player:getCharVar(info.bonusMobs)

    if total < phase.bonus.mobs[2] then
        local areaName = string.gsub(battle.area, "_", " ")
        local mobName = string.gsub(phase.bonus.mobs[1], "_", " ")
        player:setCharVar(info.bonusMobs, total + 1)
        player:fmt("Incursion [{}] Bonus Objective: {} {}/{}", areaName, mobName, total + 1, phase.bonus.mobs[2])

        if total + 1 >= phase.bonus.mobs[2] then
            player:setCharVar(info.bonus,     0)
            player:setCharVar(info.bonusMobs, 0)
            player:fmt("Incursion [{}] Bonus Objective Complete!", areaName)

            if mob:getLocalVar("BONUS_ROLLED") == 0 then
                cexi.util.treasurePool(player, battle.bonus)
                mob:setLocalVar("BONUS_ROLLED", 1)
            end

            giveCurrency(player, battle.currency, 2, 3)
        end
    end
end

local function rollMount(player, mountName, mountID)
    local alliance = player:getAlliance()
    local zoneID   = player:getZoneID()

    for _, member in pairs(alliance) do
        if
            member ~= nil and
            member:getZoneID() == zoneID
        then
            local current = member:getCharVar("[CW]MOUNT_LIST")

            if not utils.mask.getBit(current, mountID) then
                if current == 0 then
                    member:sys("You have unlocked Riding Thunder's services in Upper Jeuno!")
                end

                local result = utils.mask.setBit(current, mountID, true)

                member:sys("You've learned how to ride the {}!", mountName)
                member:setCharVar("[CW]MOUNT_LIST", result)
            end
        end
    end
end

cexi.incursion.defeatBonusBoss = function(mob, player, battle, phase, phaseIndex)
    if not inInstance(player, phaseIndex) then
        return
    end

    local areaName = string.gsub(battle.area, "_", " ")
    player:fmt("Incursion [{}] Bonus Objective Complete!", areaName)

    if mob:getLocalVar("BONUS_ROLLED") == 0 then
        cexi.util.treasurePool(player, battle.bonus)
        mob:setLocalVar("BONUS_ROLLED", 1)

        if
            phase.bonus.boss.mount ~= nil and
            math.random(0, 100) < 15
        then
            rollMount(player, phase.bonus.boss.mount[1], phase.bonus.boss.mount[2])
        end
    end

    giveCurrency(player, battle.currency, 2, 3)
    updateStatus(player, phase.cap, battle.zone)
end

cexi.incursion.openBonusChest = function(player, npc, battle)
    if not cexi.util.openChest(npc) then
        return
    end

    if player:getCharVar(info.bonus) ~= cexi.incursion.bonusType.CHEST then
        player:fmt("It's locked.")
        return
    end

    cexi.util.treasurePool(player, battle.bonus, npc)

    local alliance   = player:getAlliance()
    local zoneID     = player:getZoneID()
    local playerName = player:getName()

    for _, member in pairs(alliance) do
        if
            member ~= nil and
            member:isPC() and
            member:getZoneID() == zoneID
        then
            if member:getName() == playerName then
                member:fmt("You found the Bonus Chest!")
            else
                member:fmt("{} found the Bonus Chest!", player:getName())
            end

            giveCurrency(member, battle.currency, 2, 3)
            player:setCharVar(info.bonus, 0)
        end
    end
end

-----------------------------------
-- Regular Objectives
-----------------------------------
cexi.incursion.getBossChecks = function(battle, phase, phaseIndex)
    return {
        vareq = { info.area,  battle.zone         },
        vareq = { info.phase, phaseIndex          },
        vargt = { info.mobs,  phase.boss.reqs - 1 },
    }
end

cexi.incursion.getBossSpawnerRequirements = function(phase, phaseIndex)
    return {
        condition = { info.phase, phaseIndex },
        dialog    =
        {
            fmt("To begin this encounter you must defeat {}", cexi.incursion.getMobList(phase)),
        },
    }
end

cexi.incursion.defeatMob = function(player, battle, phase, phaseIndex)
    if not inInstance(player, phaseIndex) then
        return
    end

    local total = player:getCharVar(info.mobs)

    if total < phase.boss.reqs then
        local areaName = string.gsub(battle.area, "_", " ")
        player:setCharVar(info.mobs, total + 1)
        player:fmt("Incursion [{}] Phase #{} {}/{}", areaName, phaseIndex, total + 1, phase.boss.reqs)

        if total + 1 >= phase.boss.reqs then
            player:fmt("New Objective: Defeat {} at {}!", phase.boss.name, phase.boss.desc)
        end
    end
end

cexi.incursion.defeatBoss = function(mob, player, battle, phase, phaseIndex)
    if not inInstance(player) then
        return
    end

    local notFinalBoss = (phaseIndex < #battle.phases)

    -- Spawn source (Once per group)
    if notFinalBoss and mob:getLocalVar("SPAWN_SOURCE") == 0 then
        mob:setLocalVar("SPAWN_SOURCE", 1)
        local zone     = player:getZone()
        local sourceDE = zone:queryEntitiesByName("DE_Ethereal Source")

        if sourceDE ~= nil then
            local pos = mob:getPos()
            sourceDE[1]:setPos(pos.x + 0.2, pos.y, pos.z) -- We offset slightly so it doesn't appear exactly on top of spawner
            sourceDE[1]:setStatus(xi.status.NORMAL)

            -- Disappears after 60 seconds
            sourceDE[1]:timer(60000, function()
                sourceDE[1]:setStatus(xi.status.DISAPPEAR)
            end)
        end
    end

    player:setLocalVar(info.source, 1)

    -- Roll rewards
    giveCurrency(player, battle.currency, phase.boss.currency[1], phase.boss.currency[2])

    if battle.exp ~= nil then
        local amount = battle.exp

        if
            battle.merit ~= nil and
            player:getJobLevel(player:getMainJob()) == 75
        then
            amount = amount * battle.merit
        end

        player:addExp(amount)
        npcUtil.giveCurrency(player, 'gil', amount)
    end

    -- Advance player to next phase
    if notFinalBoss then
        player:incrementCharVar("[GARRISON]POINTS", 150)
        player:sys("{} gains {} garrison points.", player:getName(), 150)

        player:setCharVar(info.mobs,      0)
        player:setCharVar(info.bonus,     0)
        player:setCharVar(info.bonusMobs, 0)
        player:setCharVar(info.phase, phaseIndex + 1)
        local nextPhase = battle.phases[phaseIndex + 1]

        updateStatus(player, nextPhase.cap, battle.zone)
        rollBonusObjective(mob, player, battle, nextPhase)

        player:timer(2000, function()
            player:fmt("New Objective: Defeat {}", cexi.incursion.getMobList(nextPhase))
        end)
    else
        player:incrementCharVar("[GARRISON]POINTS", 300)
        player:sys("{} gains {} garrison points.", player:getName(), 300)

        player:setCharVar(info.mobs,      0)
        player:setCharVar(info.phase,     0)
        player:setCharVar(info.bonus,     0)
        player:setCharVar(info.bonusMobs, 0)
        player:setCharVar(info.buffs,     0)
        player:delStatusEffect(xi.effect.LEVEL_RESTRICTION)

        local pet = player:getPet()

        if pet ~= nil then
            pet:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
        end

        player:timer(2000, function()
            local areaName = string.gsub(battle.area, "_", " ")
            player:fmt("Incursion [{}] Complete!", areaName)
        end)
    end
end

return m
