-----------------------------------
-- Aether Caskets System
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("aether_caskets")

cexi        = cexi or {}
cexi.aether = cexi.aether or {}

local settings =
{
    nqName = "Aether Casket",
    hqName = "Aether Trove",
    nqLook = 966,
    hqLook = 969,

    hqRate = 15,
    hqVar  = "[CHEST]HQ",
}

-- HQ chests have better rates on rare items
-- HQ keys has better rates on rare items
-- (Max increase: +2)
local rate =
{
    VERY_COMMON = { 240, 200, 180 }, --  24% ->  20% ->  18%
    COMMON      = { 150, 120, 100 }, --  15% ->  12% ->  10%
    UNCOMMON    = { 100, 120, 150 }, --  10% ->  12% ->  15%
    RARE        = {  50,  60,  80 }, --   5% ->   6% ->   8%
    VERY_RARE   = {  10,  15,  20 }, --   1% -> 1.5% ->   2%
    SUPER_RARE  = {   5,   6,   8 }, -- 0.5% -> 0.6% -> 0.8%
    ULTRA_RARE  = {   1,   2,   3 }, -- 0.1% -> 0.2% -> 0.3%
}

local areas = {}

-----------------------------------
-- Sky 2.0
-----------------------------------
-- Technically these are not dynamic but they have no base lua
-- and super() cannot be used with ensureTable
-----------------------------------
areas["Escha_RuAun"] =
{
    ["Genbu"] =
    {
        dynamic = true,
        loot    =
        {
            -- Salvage Items
            {
                {       750, 0     }, -- Nothing           (75%)
                { rate.RARE, 14964 }, -- deimoss_gauntlets ( 5%)
                { rate.RARE, 14968 }, -- freyas_gloves     ( 5%)
                { rate.RARE, 14972 }, -- hikazu_gote       ( 5%)
                { rate.RARE, 14976 }, -- enlils_kolluks    ( 5%)
                { rate.RARE, 14980 }, -- machas_cuffs      ( 5%)
            },
            -- Equipment
            {
                {       800, 0     }, -- Nothing       (80%)
                { rate.RARE, 15831 }, -- Fenian Ring   ( 5%)
                { rate.RARE, 17758 }, -- Ruler         ( 5%)
                { rate.RARE, 16158 }, -- Gnadbods Helm ( 5%)
                { rate.RARE, 15918 }, -- Witch's Sash  ( 5%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  908 }, -- Adamantoise Shell (24%)
                { rate.COMMON,       646 }, -- Adaman Ore        (15%)
                { rate.COMMON,       703 }, -- Petrified Log     (15%)
                { rate.COMMON,       722 }, -- Divine Log        (15%)
                { rate.COMMON,      1311 }, -- Oxblood           (15%)
                { rate.RARE,         860 }, -- Behemoth Hide     ( 5%)
                { rate.RARE,         901 }, -- Venomous Claw     ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment   ( 5%)
                { rate.VERY_RARE,  27539 }, -- Genmei Earring    ( 1%)
            },
        },
        rare =
        {
            { rate.UNCOMMON, 1311 }, -- Oxblood           (20%)
            { rate.UNCOMMON,  655 }, -- Adaman Ingot      (20%)
            { rate.UNCOMMON,  722 }, -- Divine Log        (20%)
            { rate.UNCOMMON,  901 }, -- Venomous Claw     (20%)
            { rate.UNCOMMON, 2999 }, -- Aether Fragment   (20%)
        },
    },
    ["Suzaku"] =
    {
        dynamic = true,
        loot    =
        {
            -- Salvage Iterms
            {
                {       750, 0     }, -- Nothing          (75%)
                { rate.RARE, 15712 }, -- enyos_leggings   ( 5%)
                { rate.RARE, 15716 }, -- njords_ledelsens ( 5%)
                { rate.RARE, 15720 }, -- hoshikazu_kyahan ( 5%)
                { rate.RARE, 15724 }, -- anus_gaiters     ( 5%)
                { rate.RARE, 15728 }, -- nemains_sabots   ( 5%)
            },
            -- Equipment
            {
                {       800, 0     }, -- Nothing            (80%)
                { rate.RARE, 16002 }, -- roundel_earring    ( 5%)
                { rate.RARE, 13624 }, -- enhancing_mantle   ( 5%)
                { rate.RARE, 15917 }, -- cuchulains_belt    ( 5%)
                { rate.RARE, 11365 }, -- karasutengu_kogake ( 5%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  844 }, -- Phoenix Feather   (24%)
                { rate.COMMON,      1313 }, -- Siren's Hair      (15%)
                { rate.COMMON,       700 }, -- Mahogany Log      (15%)
                { rate.COMMON,       747 }, -- Orichalcum Ingot  (15%)
                { rate.COMMON,      1311 }, -- Oxblood           (15%)
                { rate.RARE,         722 }, -- Divine Log        ( 5%)
                { rate.RARE,         901 }, -- Venomous Claw     ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment   ( 5%)
                { rate.VERY_RARE,  26161 }, -- Shukuyu Ring      ( 1%)
            },
        },
        rare =
        {
            { rate.UNCOMMON, 1313 }, -- Siren's Hair     (20%)
            { rate.UNCOMMON,  836 }, -- Damascene Cloth  (20%)
            { rate.UNCOMMON,  722 }, -- Divine Log       (20%)
            { rate.UNCOMMON,  747 }, -- Orichalcum Ingot (20%)
            { rate.UNCOMMON, 2999 }, -- Aether Fragment  (20%)
        },
    },
    ["Seiryu"] =
    {
        dynamic = true,
        loot    =
        {
            -- Salvage Items
            {
                {       750, 0     }, -- Nothing            (75%)
                { rate.RARE, 15713 }, -- phoboss_sabatons   ( 5%)
                { rate.RARE, 15717 }, -- freyrs_ledelsens   ( 5%)
                { rate.RARE, 15721 }, -- tsukikazu_sune-ate ( 5%)
                { rate.RARE, 15725 }, -- eas_crackows       ( 5%)
                { rate.RARE, 15729 }, -- bodbs_pigaches     ( 5%)
            },
            -- Equipment
            {
                {       800, 0     }, -- Nothing           (80%)
                { rate.RARE, 11392 }, -- koschei_crackows  ( 5%)
                { rate.RARE, 18598 }, -- prester           ( 5%)
                { rate.RARE, 16159 }, -- zhagos_barbut     ( 5%)
                { rate.RARE, 16001 }, -- crapaud_earring   ( 5%)
            },
            -- Materials
            {
                { rate.VERY_COMMON, 1122 }, -- Wyvern Skin     (24%)
                { rate.COMMON,       867 }, -- Dragon Scales   (15%)
                { rate.COMMON,      1133 }, -- Dragon Blood    (15%)
                { rate.COMMON,      4272 }, -- Dragon Meat     (15%)
                { rate.COMMON,      4486 }, -- Dragon Heart    (15%)
                { rate.RARE,         903 }, -- Dragon Talon    ( 5%)
                { rate.RARE,        1712 }, -- Chasmere Wool   ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment ( 5%)
                { rate.VERY_RARE,  26320 }, -- Kobo Obi        ( 1%)
            },
        },
        rare =
        {
            { rate.UNCOMMON, 1133 }, -- Dragon Blood    (20%)
            { rate.UNCOMMON, 4272 }, -- Dragon Meat     (20%)
            { rate.UNCOMMON, 4486 }, -- Dragon Heart    (20%)
            { rate.UNCOMMON, 1712 }, -- Cashmere Wool   (20%)
            { rate.UNCOMMON, 2999 }, -- Aether Fragment (20%)
        },
    },
    ["Byakko"] =
    {
        dynamic = true,
        loot    =
        {
            -- Salvage Items
            {   
                {       750, 0     }, -- Nothing          (75%)
                { rate.RARE, 15714 }, -- deimoss_leggings ( 5%)
                { rate.RARE, 15718 }, -- freyas_ledelsens ( 5%)
                { rate.RARE, 15722 }, -- hikazu_sune-ate  ( 5%)
                { rate.RARE, 15726 }, -- enlils_crackows  ( 5%)
                { rate.RARE, 15730 }, -- machas_pigaches  ( 5%)
            },
            -- Equipment
            {
                {       800, 0     }, -- Nothing           (80%)
                { rate.RARE, 18498 }, -- balestarius        (5%)
                { rate.RARE, 13403 }, -- assault_earring    (5%)
                { rate.RARE, 16241 }, -- cuchulains_mantle  (5%)
                { rate.RARE, 16356 }, -- nimues_tights      (5%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  884 }, -- Blk. Tiger Fang  (24%)
                { rate.COMMON,       861 }, -- Tiger Hide       (15%)
                { rate.COMMON,       860 }, -- Behemoth Hide    (15%)
                { rate.COMMON,       837 }, -- Malboro Fiber    (15%)
                { rate.COMMON,       722 }, -- Divine Log       (15%)
                { rate.RARE,         658 }, -- Damascus Ingot   ( 5%)
                { rate.RARE,        2169 }, -- Cerberus Hide    ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment  ( 5%)
                { rate.VERY_RARE,  27525 }, -- Jokuchu Chain    ( 1%)
            },
        },
        rare =
        {
            { rate.UNCOMMON,  860 }, -- Behemoth Hide    (20%)
            { rate.UNCOMMON, 2169 }, -- Cerberus Hide    (20%)
            { rate.UNCOMMON,  722 }, -- Divine Log       (20%)
            { rate.UNCOMMON,  658 }, -- Damascus Ingot   (20%)
            { rate.UNCOMMON, 2999 }, -- Aether Fragment  (20%)
        },
    },
    ["Kouryu"] =
    {
        dynamic = true,
        loot    =
        {
            -- Elemental Ores
            {
                { cexi.rate.COMMON, 1255 }, -- Fire Ore
                { cexi.rate.COMMON, 1256 }, -- Ice Ore
                { cexi.rate.COMMON, 1257 }, -- Wind Ore
                { cexi.rate.COMMON, 1258 }, -- Earth Ore
                { cexi.rate.COMMON, 1259 }, -- Lightning Ore
                { cexi.rate.COMMON, 1260 }, -- Water Ore
                { cexi.rate.COMMON, 1261 }, -- Light Ore
                { cexi.rate.COMMON, 1262 }, -- Dark Ore
            },
            -- Equipment
            {
                -- 50%
                { rate.UNCOMMON,  1116 }, -- Manticore Hide        (10%)
                { rate.UNCOMMON,  1123 }, -- Manticore Fang        (10%)
                { rate.UNCOMMON,   867 }, -- Dragon Scales         (10%)
                { rate.UNCOMMON,   831 }, -- Shining Cloth         (10%)
                { rate.UNCOMMON,   836 }, -- Damascene Cloth       (10%)

                --- 30%
                { rate.UNCOMMON, 16160 }, -- Ree Habalo's Headgear (10%)
                { rate.UNCOMMON, 19030 }, -- Wise Strap            (10%)
                { rate.UNCOMMON, 15833 }, -- Diverter's Ring       (10%)

                -- 20%
                { rate.RARE,     11281 }, -- Hachiryu Haramaki     ( 5%)
                { rate.RARE,     18447 }, -- Nanatsusayanotachi    ( 5%)
                { rate.RARE,     18759 }, -- Shenlongs Baghnakhs   ( 5%)
                { rate.RARE,     18594 }, -- Dorje                 ( 5%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  747 }, -- Orichalcum Ingot           (24%)
                { rate.COMMON,      3503 }, -- Chunk of Mulcibars Scoria  (15%)
                { rate.COMMON,      2999 }, -- Aether Fragment            (15%)
                { rate.COMMON,      1110 }, -- Beetle Blood               (15%)
                { rate.COMMON,       722 }, -- Divine Log                 (15%)
                { rate.RARE,         658 }, -- Damascus Ingot             ( 5%)
                { rate.RARE,        2169 }, -- Cerberus Hide              ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment            ( 5%)
                { rate.VERY_RARE,  20842 }, -- Reikiono                   ( 1%)
            },
        },
        rare =
        {
            { rate.UNCOMMON, 3503 }, -- Chunk of Mulcibars Scoria (20%)
            { rate.UNCOMMON,  722 }, -- Divine Log                (20%)
            { rate.UNCOMMON,  658 }, -- Damascus Ingot            (20%)
            { rate.UNCOMMON, 2169 }, -- Cerberus Hide             (20%)
            { rate.UNCOMMON, 2999 }, -- Aether Fragment           (20%)
        },
    },
}

-----------------------------------
-- Sky (Ru'Aun Gardens)
-----------------------------------
areas["The_Shrine_of_RuAvitau"] =
{
    ["Kirin"] =
    {
        loot =
        {
            {
                -- Equipment & Scrolls
                { rate.COMMON,             4748 }, -- Raise III    (15%)
                { rate.UNCOMMON,           4612 }, -- Cure IV      (10%)
                { rate.UNCOMMON,           4620 }, -- Raise        (10%)
                { rate.RARE,               4621 }, -- Raise II     ( 5%)

                { rate.COMMON,             4818 }, -- Quake        (15%)
                { rate.UNCOMMON,           4799 }, -- Stonega III  (10%)
                { rate.UNCOMMON,           4769 }, -- Stone III    (10%)
                { rate.RARE,               4770 }, -- Stone IV     ( 5%)

                { rate.COMMON,            17567 }, -- Kirins Pole  (15%)
                { rate.RARE,              12562 }, -- Kirins Osode ( 5%)
            },
            {
                -- Materials
                { rate.VERY_COMMON,         644 }, -- Mythril Ore      (24%)
                { rate.COMMON,              739 }, -- Orichalcum Ore   (15%)
                { rate.COMMON,             1132 }, -- Raxa             (15%)
                { rate.COMMON,              831 }, -- Shining Cloth    (15%)
                { rate.UNCOMMON,            747 }, -- Orichalcum Ingot (10%)
                { rate.RARE,                658 }, -- Damascus Ingot   ( 5%)
                { rate.RARE,                655 }, -- Adaman Ingot     ( 5%)
                { rate.RARE,               1313 }, -- Siren's Hair     ( 5%)
                { rate.RARE,                836 }, -- Damascene Cloth  ( 5%)
                { rate.VERY_RARE,          2999 }, -- Aether Fragment  ( 1%)
            },
            {
                -- Abjurations
                { rate.UNCOMMON, { 3275, 1, 3 } }, -- Genbu Scrap x1-3  (10%)
                { rate.UNCOMMON, { 3276, 1, 3 } }, -- Suzaku Scrap x1-3 (10%)
                { rate.UNCOMMON, { 3277, 1, 3 } }, -- Seiryu Scrap x1-3 (10%)
                { rate.UNCOMMON, { 3278, 1, 3 } }, -- Byakko Scrap x1-3 (10%)

                { rate.COMMON,   { 3279, 1, 3 } }, -- Neptunal Tatter x1-3      (15%)
                { rate.COMMON,   { 3282, 1, 3 } }, -- Dryadic Tatter x1-3       (15%)
                { rate.COMMON,   { 3284, 1, 3 } }, -- Wyrmal Tatter x1-3        (15%)
                { rate.RARE,               1315 }, -- Dryadic Abjuration: Body  ( 5%)
                { rate.RARE,               1337 }, -- Wyrmal Abjuration: Legs   ( 5%)
                { rate.RARE,               1340 }, -- Neptunal Abjuration: Body ( 5%)
            },
        },
        rare =
        {
            { rate.UNCOMMON,                831 }, -- Shining Cloth     (20%)
            { rate.UNCOMMON,                658 }, -- Damascus Ingot    (20%)
            { rate.UNCOMMON,                836 }, -- Damascene Cloth   (20%)
            { rate.UNCOMMON,               1313 }, -- Siren's Hair      (20%)
            { rate.UNCOMMON,               2999 }, -- Aether Fragment   (20%)
        },
    },
}

areas["RuAun_Gardens"] =
{
    ["Genbu"] =
    {
        loot =
        {
            -- Equipment
            {
                { 800,             18161 }, -- Arctic Wind   (80%)
                { rate.UNCOMMON,   12296 }, -- Genbus Shield (10%)
                { rate.UNCOMMON,   12434 }, -- Genbus Kabuto (10%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  646 }, -- Adaman Ore        (24%)
                { rate.COMMON,       700 }, -- Mahogany Log      (15%)
                { rate.COMMON,       908 }, -- Adamantoise Shell (15%)
                { rate.COMMON,      1311 }, -- Oxblood           (15%)
                { rate.UNCOMMON,     655 }, -- Adaman Ingot      (10%)
                { rate.RARE,         722 }, -- Divine Log        ( 5%)
                { rate.RARE,         901 }, -- Venomous Claw     ( 5%)
                { rate.RARE,        1110 }, -- Beetle Blood      ( 5%)
                { rate.VERY_RARE,   2999 }, -- Aether Fragment   ( 1%)
            },
            -- Abjurations
            {
                { 600, { 3275, 1, 3 }    }, -- Genbu Scrap x1-3           (60%)
                { rate.UNCOMMON,    1338 }, -- Wyrmal Abjuration: Feet    (10%)
                { rate.UNCOMMON,    1324 }, -- Aquarian Abjuration: Head  (10%)
                { rate.UNCOMMON,    1326 }, -- Aquarian Abjuration: Hands (10%)
                { rate.UNCOMMON,    1331 }, -- Martial Abjuration: Hands  (10%)
            },
        },
        rare =
        {
            { rate.UNCOMMON,        1311 }, -- Oxblood           (20%)
            { rate.UNCOMMON,         655 }, -- Adaman Ingot      (20%)
            { rate.UNCOMMON,         722 }, -- Divine Log        (20%)
            { rate.UNCOMMON,         901 }, -- Venomous Claw     (20%)
            { rate.UNCOMMON,        2999 }, -- Aether Fragment   (20%)
        },
    },
    ["Suzaku"] =
    {
        loot =
        {
            -- Equipment
            {
                { 800,             18164 }, -- Antarctic Wind   (80%)
                { rate.UNCOMMON,   12946 }, -- Suzakus Sune-Ate (10%)
                { rate.UNCOMMON,   18043 }, -- Suzakus Scythe   (10%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  816 }, -- Silk Thread      (24%)
                { rate.COMMON,       844 }, -- Phoenix Feather  (15%)
                { rate.COMMON,      1313 }, -- Sirens Hair      (15%)
                { rate.COMMON,       836 }, -- Damascene Cloth  (15%)
                { rate.UNCOMMON,     831 }, -- Shining Cloth    (10%)
                { rate.RARE,        1110 }, -- Beetle Blood     ( 5%)
                { rate.RARE,        1313 }, -- Siren's Hair     ( 5%)
                { rate.RARE,         901 }, -- Venomous Claw    ( 5%)
                { rate.RARE,         747 }, -- Orichalcum Ingot ( 5%)
                { rate.VERY_RARE,   2999 }, -- Aether Fragment  ( 1%)
            },
            -- Abjurations
            {
                { 600, { 3276, 1, 3 }    }, -- Suzaku Scrap x1-3         (60%)
                { rate.UNCOMMON,    1343 }, -- Neptunal Abjuration: Feet (10%)
                { rate.UNCOMMON,    1316 }, -- Dryadic Abjuration: Hands (10%)
                { rate.UNCOMMON,    1319 }, -- Earthen Abjuration: Head  (10%)
                { rate.UNCOMMON,    1327 }, -- Aquarian Abjuration: Legs (10%)
            },
        },
        rare =
        {
            { rate.UNCOMMON,        1313 }, -- Sirens Hair      (20%)
            { rate.UNCOMMON,         836 }, -- Damascene Cloth  (20%)
            { rate.UNCOMMON,         831 }, -- Shining Cloth    (20%)
            { rate.UNCOMMON,         747 }, -- Orichalcum Ingot (20%)
            { rate.UNCOMMON,        2999 }, -- Aether Fragment  (20%)
        },
    },
    ["Seiryu"] =
    {
        loot =
        {
            -- Equipment
            {
                { 800,             18162 }, -- East Wind     (80%)
                { rate.UNCOMMON,   12690 }, -- Seiryus Kote  (10%)
                { rate.UNCOMMON,   17659 }, -- Seiryus Sword (10%)
            },
            -- Materials
            {
                { rate.VERY_COMMON, 1122 }, -- Wyvern Skin     (24%)
                { rate.COMMON,      1124 }, -- Wyvern Wing     (15%)
                { rate.COMMON,      1133 }, -- Dragon Blood    (15%)
                { rate.COMMON,      4272 }, -- Dragon Meat     (15%)
                { rate.UNCOMMON,    4486 }, -- Dragon Heart    (10%)
                { rate.RARE,         903 }, -- Dragon Talon    ( 5%)
                { rate.RARE,        1110 }, -- Beetle Blood    ( 5%)
                { rate.RARE,        1311 }, -- Oxblood         ( 5%)
                { rate.RARE,        1312 }, -- Angel Skin      ( 5%)
                { rate.VERY_RARE,   2999 }, -- Aether Fragment ( 1%)
            },
            -- Abjurations
            {
                { 600, { 3277, 1, 3 }    }, -- Seiryu Scrap x1-3         (60%)
                { rate.UNCOMMON,    1336 }, -- Wyrmal Abjuration: Hands  (10%)
                { rate.UNCOMMON,    1314 }, -- Dryadic Abjuration: Head  (10%)
                { rate.UNCOMMON,    1327 }, -- Aquarian Abjuration: Legs (10%)
                { rate.UNCOMMON,    1329 }, -- Martial Abjuration: Head  (10%)
            },
        },
        rare =
        {
            { rate.UNCOMMON,        1133 }, -- Dragon Blood    (20%)
            { rate.UNCOMMON,        4272 }, -- Dragon Meat     (20%)
            { rate.UNCOMMON,        4486 }, -- Dragon Heart    (20%)
            { rate.UNCOMMON,        1312 }, -- Angel Skin      (20%)
            { rate.UNCOMMON,        2999 }, -- Aether Fragment (20%)
        },
    },
    ["Byakko"] =
    {
        loot =
        {
            -- Equipment
            {
                { 800,             18163 }, -- Zephyr          (80%)
                { rate.UNCOMMON,   18198 }, -- Byakkos Axe     (10%)
                { rate.UNCOMMON,   12818 }, -- Byakkos Haidate (10%)
            },
            -- Materials
            {
                { rate.VERY_COMMON,  884 }, -- Blk. Tiger Fang (24%)
                { rate.COMMON,       861 }, -- Tiger Hide      (15%)
                { rate.COMMON,       860 }, -- Behemoth Hide   (15%)
                { rate.COMMON,       722 }, -- Divine Log      (15%)
                { rate.UNCOMMON,     658 }, -- Damascus Ingot  (10%)
                { rate.RARE,         837 }, -- Malboro Fiber   ( 5%)
                { rate.RARE,         860 }, -- Behemoth Hide   ( 5%)
                { rate.RARE,        1311 }, -- Oxblood         ( 5%)
                { rate.RARE,        1110 }, -- Beetle Blood    ( 5%)
                { rate.VERY_RARE,   2999 }, -- Aether Fragment ( 1%)
            },
            -- Abjurations
            {
                { 600, { 3278, 1, 3 }    }, -- Byakko Scrap x1-3          (60%)
                { rate.UNCOMMON,    1341 }, -- Neptunal Abjuration: Hands (10%)
                { rate.UNCOMMON,    1317 }, -- Dryadic Abjuration: Legs   (10%)
                { rate.UNCOMMON,    1323 }, -- Earthen Abjuration: Feet   (10%)
                { rate.UNCOMMON,    1324 }, -- Aquarian Abjuration: Head  (10%)
            },
        },
        rare =
        {
            { rate.UNCOMMON,         860 }, -- Behemoth Hide   (20%)
            { rate.UNCOMMON,         722 }, -- Divine Log      (20%)
            { rate.UNCOMMON,         658 }, -- Damascus Ingot  (20%)
            { rate.UNCOMMON,        1110 }, -- Beetle Blood    (20%)
            { rate.UNCOMMON,        2999 }, -- Aether Fragment (20%)
        },
    },
}

-----------------------------------
-- Dragonslaying
-----------------------------------
areas["Pashhow_Marshlands"] =
{
    ["Arfarvegr"] =
    {
        dynamic = true,
        loot    =
        {
            -- Wyrm Mats
            {
                { rate.VERY_COMMON,  867 }, -- Dragon Scales    (24%)
                { rate.VERY_COMMON, 1133 }, -- Dragon Blood     (15%)
                { rate.UNCOMMON,    9447 }, -- Wyrm Scale       (10%)
                { rate.UNCOMMON,    9453 }, -- Wyrm Tooth       (10%)
                { rate.RARE,        9454 }, -- Embersteel Ingot ( 5%)
            },
            -- Abjurations
            {
                {       400, 4020 }, -- Titanium Ore              (40%)
                {       400, 9455 }, -- Fable Thread              (40%)
                { rate.RARE, 1342 }, -- Neptunal Abjuration: Legs ( 5%)
                { rate.RARE, 1328 }, -- Aquarian Abjuration: Feet ( 5%)
                { rate.RARE, 1331 }, -- Martial Abjuration: Hands ( 5%)
                { rate.RARE, 1315 }, -- Dryadic Abjuration: Body  ( 5%)
                
            },
            -- Materials
            {
                { rate.VERY_COMMON,  722 }, -- Divine Log       (24%)
                { rate.VERY_COMMON, 1110 }, -- Beetle Blood     (24%)
                { rate.VERY_COMMON,  836 }, -- Damascene Cloth  (24%)
                { rate.VERY_COMMON,  655 }, -- Adaman Ingot     (24%)
                { rate.VERY_COMMON, 1313 }, -- Siren's Hair     (24%)
                { rate.COMMON,       903 }, -- Dragon Talon     (15%)
                { rate.COMMON,      4272 }, -- Dragon Meat      (15%)
                { rate.COMMON,      4486 }, -- Dragon Heart     (15%)
                { rate.COMMON,      1712 }, -- Cashmere Wool    (15%)
                { rate.COMMON,       860 }, -- Behemoth Hide    (15%)
                { rate.UNCOMMON,     734 }, -- Thokcha Ore      (10%)
                { rate.UNCOMMON,     647 }, -- Molybdenum Ore   (10%)
                { rate.UNCOMMON,    1703 }, -- Kunwu Ore        (10%)
                { rate.RARE,        1311 }, -- Oxblood          ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment  ( 5%)
            },
        },
        rare =
        {
            { rate.VERY_COMMON,  9447 }, -- Wyrm Scale        (24%)
            { rate.VERY_COMMON,  9453 }, -- Wyrm Tooth        (24%)
            { rate.VERY_COMMON,  2999 }, -- Aether Fragment   (24%)
            { rate.VERY_COMMON,  9515 }, -- Wyrm Gold         (24%)
            { rate.RARE,        27738 }, -- Assailant's Visor ( 5%)
            { rate.RARE,        28459 }, -- Subtle Sash       ( 5%)
            { rate.RARE,        28062 }, -- Ornate Gloves     ( 5%)
            { rate.VERY_RARE,    9454 }, -- Embersteel Ingot  ( 1%)
            { rate.VERY_RARE,    9515 }, -- Fine Ochre        ( 1%)
        },
    },
}

areas["Jugner_Forest"] =
{
    ["Skrudningr"] =
    {
        dynamic = true,
        loot    =
        {
            -- Wyrm Mats
            {
                { rate.VERY_COMMON,  867 }, -- Dragon Scales    (24%)
                { rate.VERY_COMMON, 1133 }, -- Dragon Blood     (15%)
                { rate.UNCOMMON,    9447 }, -- Wyrm Scale       (10%)
                { rate.UNCOMMON,    9453 }, -- Wyrm Tooth       (10%)
                { rate.RARE,        9454 }, -- Embersteel Ingot ( 5%)
            },
            -- Abjurations
            {
                {       400, 4020 }, -- Titanium Ore               (40%)
                {       400, 9455 }, -- Fable Thread               (40%)
                { rate.RARE, 1333 }, -- Martial Abjuration: Feet   ( 5%)
                { rate.RARE, 1326 }, -- Aquarian Abjuration: Hands ( 5%)
                { rate.RARE, 1322 }, -- Earthen Abjuration: Legs   ( 5%)
                { rate.RARE, 1335 }, -- Wyrmal Abjuration: Body    ( 5%)
                
            },
            -- Materials
            {
                { rate.VERY_COMMON,  722 }, -- Divine Log       (24%)
                { rate.VERY_COMMON, 1110 }, -- Beetle Blood     (24%)
                { rate.VERY_COMMON,  836 }, -- Damascene Cloth  (24%)
                { rate.VERY_COMMON,  655 }, -- Adaman Ingot     (24%)
                { rate.VERY_COMMON, 1313 }, -- Siren's Hair     (24%)
                { rate.COMMON,       903 }, -- Dragon Talon     (15%)
                { rate.COMMON,      4272 }, -- Dragon Meat      (15%)
                { rate.COMMON,      4486 }, -- Dragon Heart     (15%)
                { rate.COMMON,      1712 }, -- Cashmere Wool    (15%)
                { rate.COMMON,       860 }, -- Behemoth Hide    (15%)
                { rate.UNCOMMON,     734 }, -- Thokcha Ore      (10%)
                { rate.UNCOMMON,     647 }, -- Molybdenum Ore   (10%)
                { rate.UNCOMMON,    1703 }, -- Kunwu Ore        (10%)
                { rate.RARE,        1311 }, -- Oxblood          ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment  ( 5%)
            },
        },
        rare =
        {
            { rate.VERY_COMMON,  9447 }, -- Wyrm Scale        (24%)
            { rate.VERY_COMMON,  9453 }, -- Wyrm Tooth        (24%)
            { rate.VERY_COMMON,  2999 }, -- Aether Fragment   (24%)
            { rate.VERY_COMMON,  9515 }, -- Wyrm Gold         (24%)
            { rate.RARE,        27779 }, -- Conqueror's Helm  ( 5%)
            { rate.RARE,        28384 }, -- Loxo Scarf        ( 5%)
            { rate.RARE,        28458 }, -- Covenant Belt     ( 5%)
            { rate.VERY_RARE,    9454 }, -- Embersteel Ingot  ( 1%)
            { rate.VERY_RARE,    9515 }, -- Fine Ochre        ( 1%)
        },
    },
}

areas["Meriphataud_Mountains"] =
{
    ["Midgardsormr"] =
    {
        dynamic = true,
        loot    =
        {
            -- Wyrm Mats
            {
                { rate.VERY_COMMON,  867 }, -- Dragon Scales    (24%)
                { rate.VERY_COMMON, 1133 }, -- Dragon Blood     (15%)
                { rate.UNCOMMON,    9447 }, -- Wyrm Scale       (10%)
                { rate.UNCOMMON,    9453 }, -- Wyrm Tooth       (10%)
                { rate.RARE,        9454 }, -- Embersteel Ingot ( 5%)
            },
            -- Abjurations
            {
                {       400, 4020 }, -- Titanium Ore               (40%)
                {       400, 9455 }, -- Fable Thread               (40%)
                { rate.RARE, 1333 }, -- Martial Abjuration: Feet   ( 5%)
                { rate.RARE, 1326 }, -- Aquarian Abjuration: Hands ( 5%)
                { rate.RARE, 1322 }, -- Earthen Abjuration: Legs   ( 5%)
                { rate.RARE, 1335 }, -- Wyrmal Abjuration: Body    ( 5%)
                
            },
            -- Materials
            {
                { rate.VERY_COMMON,  722 }, -- Divine Log       (24%)
                { rate.VERY_COMMON, 1110 }, -- Beetle Blood     (24%)
                { rate.VERY_COMMON,  836 }, -- Damascene Cloth  (24%)
                { rate.VERY_COMMON,  655 }, -- Adaman Ingot     (24%)
                { rate.VERY_COMMON, 1313 }, -- Siren's Hair     (24%)
                { rate.COMMON,       903 }, -- Dragon Talon     (15%)
                { rate.COMMON,      4272 }, -- Dragon Meat      (15%)
                { rate.COMMON,      4486 }, -- Dragon Heart     (15%)
                { rate.COMMON,      1712 }, -- Cashmere Wool    (15%)
                { rate.COMMON,       860 }, -- Behemoth Hide    (15%)
                { rate.UNCOMMON,     734 }, -- Thokcha Ore      (10%)
                { rate.UNCOMMON,     647 }, -- Molybdenum Ore   (10%)
                { rate.UNCOMMON,    1703 }, -- Kunwu Ore        (10%)
                { rate.RARE,        1311 }, -- Oxblood          ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment  ( 5%)
            },
        },
        rare =
        {
            { rate.VERY_COMMON,  9447 }, -- Wyrm Scale         (24%)
            { rate.VERY_COMMON,  9453 }, -- Wyrm Tooth         (24%)
            { rate.VERY_COMMON,  2999 }, -- Aether Fragment    (24%)
            { rate.VERY_COMMON,  9515 }, -- Wyrm Gold          (24%)
            { rate.RARE,        28577 }, -- Titanium Band      ( 5%)
            { rate.RARE,        27767 }, -- Erudite Cap        ( 5%)
            { rate.RARE,        28201 }, -- Acrobat's Breeches ( 5%)
            { rate.VERY_RARE,    9454 }, -- Embersteel Ingot   ( 1%)
            { rate.VERY_RARE,    9515 }, -- Fine Ochre         ( 1%)
        },
    },
}

areas["Bibiki_Bay"] =
{
    ["Brjota"] =
    {
        dynamic = true,
        loot    =
        {
            -- Wyrm Mats
            {
                { cexi.rate.COMMON,   9447 }, -- Wyrm Scale       (15%)
                { cexi.rate.COMMON,   9453 }, -- Wyrm Tooth       (15%)
                { cexi.rate.UNCOMMON, 9454 }, -- Embersteel Ingot (10%)
            },
            -- Abjurations
            {
                {       400, 4020 }, -- Titanium Ore                (40%)
                {       400, 9455 }, -- Fable Thread                (40%)
                { rate.RARE, 1320 }, -- Earthen Abjuration: Body    ( 5%)
                { rate.RARE, 1325 }, -- Aquarian Abjuration: Body   ( 5%)
                { rate.RARE, 2430 }, -- Phantasmal Abjuration: Body ( 5%)
                { rate.RARE, 2435 }, -- Hadean Abjuration: Body     ( 5%)
                
            },
            -- Materials
            {
                { rate.VERY_COMMON,  722 }, -- Divine Log       (24%)
                { rate.VERY_COMMON, 1110 }, -- Beetle Blood     (24%)
                { rate.VERY_COMMON,  836 }, -- Damascene Cloth  (24%)
                { rate.VERY_COMMON,  655 }, -- Adaman Ingot     (24%)
                { rate.VERY_COMMON, 1313 }, -- Siren's Hair     (24%)
                { rate.COMMON,       903 }, -- Dragon Talon     (15%)
                { rate.COMMON,      4272 }, -- Dragon Meat      (15%)
                { rate.COMMON,      4486 }, -- Dragon Heart     (15%)
                { rate.COMMON,      1712 }, -- Cashmere Wool    (15%)
                { rate.COMMON,       860 }, -- Behemoth Hide    (15%)
                { rate.UNCOMMON,     734 }, -- Thokcha Ore      (10%)
                { rate.UNCOMMON,     647 }, -- Molybdenum Ore   (10%)
                { rate.UNCOMMON,    1703 }, -- Kunwu Ore        (10%)
                { rate.RARE,        1311 }, -- Oxblood          ( 5%)
                { rate.RARE,        2999 }, -- Aether Fragment  ( 5%)
            },
        },
        rare =
        {
            { rate.VERY_COMMON,   9447 }, -- Wyrm Scale         (24%)
            { rate.VERY_COMMON,   9453 }, -- Wyrm Tooth         (24%)
            { rate.VERY_COMMON,   2999 }, -- Aether Fragment    (24%)
            { rate.COMMON,        9515 }, -- Wyrm Gold          (15%)
            { rate.UNCOMMON,      5965 }, -- Isleracea          ( 5%)
            { rate.RARE,          9454 }, -- Embersteel Ingot   ( 5%)
            { rate.VERY_RARE,     3545 }, -- Serica Cloth       ( 1%)
        },
    },

    ["Shen"] =
    {
        loot =
        {
            {
                { cexi.rate.COMMON, 1255 }, -- Fire Ore
                { cexi.rate.COMMON, 1256 }, -- Ice Ore
                { cexi.rate.COMMON, 1257 }, -- Wind Ore
                { cexi.rate.COMMON, 1258 }, -- Earth Ore
                { cexi.rate.COMMON, 1259 }, -- Lightning Ore
                { cexi.rate.COMMON, 1260 }, -- Water Ore
                { cexi.rate.COMMON, 1261 }, -- Light Ore
                { cexi.rate.COMMON, 1262 }, -- Dark Ore
            },
            {
                { rate.VERY_COMMON, 1618 }, -- Uragnite Shell   (24%)
                { rate.COMMON,       888 }, -- Seashell         (15%)
                { rate.COMMON,      4484 }, -- Shall Shell      (15%)
                { rate.COMMON,      5128 }, -- Cone Calamary    (15%)
                { rate.UNCOMMON,    5131 }, -- Vongola Clam     (10%)
                { rate.UNCOMMON,    4318 }, -- Bibiki Urchin    (10%)
                { rate.RARE,        5123 }, -- Jacknife         ( 5%)
                { rate.RARE,        4471 }, -- Bladefish        ( 5%)
                { rate.VERY_RARE,   1586 }, -- Titanictus Shell ( 1%)
            },
            {
                { rate.VERY_COMMON,  887 }, -- Coral Fragment   (24%)
                { rate.COMMON,       888 }, -- Seashell         (15%)
                { rate.COMMON,       881 }, -- Crab Shell       (15%)
                { rate.COMMON,      1193 }, -- H.Q. Crab Shell  (15%)
                { rate.UNCOMMON,     792 }, -- Pearl            (10%)
                { rate.UNCOMMON,     793 }, -- Black Pearl      (10%)
                { rate.RARE,        1311 }, -- Oxblood          ( 5%)
                { rate.RARE,        1312 }, -- Angel Skin       ( 5%)
                { rate.VERY_RARE,   2999 }, -- Aether Fragment  ( 1%)
            },
        },
        rare =
        {
            { rate.COMMON, 1311 }, -- Oxblood         (15%)
            { rate.COMMON, 1312 }, -- Angel Skin      (15%)
            { rate.COMMON,  722 }, -- Divine Log      (15%)
            { rate.COMMON, 1193 }, -- H.Q. Crab Shell (15%)
            { rate.COMMON, 2999 }, -- Aether Fragment (15%)
        },
    },
}

------------------------------------
-- Casket Functions
-----------------------------------
local function receiveItem(player, itemInfo)
    if itemInfo[2] == 0 then
        return
    end

    if type(itemInfo[2]) == "table" then
        local roll = math.random(itemInfo[2][2], itemInfo[2][3])

        if not cexi.util.giveItem(player, { { itemInfo[2][1], roll } }, { multiple = true }) then
            -- TODO: Implement an NPC to pick these up from
            player:incrementCharVar(fmt("[CHEST]LOST_{}", itemInfo[2]), roll)
        end
    else
        if not cexi.util.giveItem(player, itemInfo[2], { multiple = true }) then
            -- TODO: Implement an NPC to pick these up from
            player:incrementCharVar(fmt("[CHEST]LOST_{}", itemInfo[2]), 1)
        end
    end
end

local function openChest(player, npc, trade, mobInfo, chestName, highQuality, chestType)
    player:setLocalVar(chestName, 0)

    npc:ceAnimationPacket(player, "open", npc)

    npc:timer(5000, function()
        npc:ceAnimationPacket(player, "close", npc)
    end)

    for _, pool in pairs(mobInfo.loot) do
        local result = cexi.util.pickItem(pool, 1 + chestType + highQuality)
        receiveItem(player, result)
    end

    if highQuality == 1 then
        local result = cexi.util.pickItem(mobInfo.rare, 1 + chestType)
        receiveItem(player, result)
    end
end

local function onTrigger(player, npc, chestName)
    if player:getLocalVar(chestName) == 1 then
        player:fmt("You need an Aether Key or Prismatic Key to open this casket.")
    else
        player:fmt("It's locked.")
    end
end

local function onTrade(player, npc, trade, mobInfo, chestName)
    if player:getLocalVar(chestName) == 0 then
        return
    end

    local freeSlots = player:getFreeSlotsCount()
    local chestType = npc:getLocalVar(settings.hqVar)

    if cexi.util.tradeHasExactly(trade, xi.item.AETHER_KEY) then
        if freeSlots < 3 then
            player:fmt("You need more inventory space to open that.")
        else
            player:tradeComplete()
            openChest(player, npc, trade, mobInfo, chestName, 0, chestType)
        end

    elseif cexi.util.tradeHasExactly(trade, xi.item.PRISMATIC_KEY) then
        if freeSlots < 4 then
            player:fmt("You need more inventory space to open that.")
        else
            player:tradeComplete()
            openChest(player, npc, trade, mobInfo, chestName, 1, chestType)
        end
    end
end

------------------------------------
-- Entity Setup
-----------------------------------
local entities = {}

cexi.aether.rollCasket = function(mob, player)
    if not player:isPC() then
        return
    end

    local chestName = fmt("{}_Chest", string.gsub(mob:getName(), "DE_", ""))

    player:setLocalVar(chestName, 1)

    if mob:getLocalVar("[CHEST]SPAWNED") == 0 then
        mob:setLocalVar("[CHEST]SPAWNED", 1)

        local zone   = mob:getZone()
        local result = zone:queryEntitiesByName("DE_" .. chestName)

        if
            result ~= nil and
            result[1] ~= nil
        then
            if math.random(0, 100) < settings.hqRate then
                result[1]:setLocalVar(settings.hqVar, 1)
                result[1]:setModelId(settings.hqLook)
                result[1]:renameEntity(settings.hqName)
            end

            local pos = mob:getPos()

            -- Offset position to not overlap regular Caskets
            result[1]:setPos(pos.x + 1, pos.y + 1, pos.z, pos.rot)
            result[1]:setStatus(xi.status.NORMAL)

            result[1]:clearTimerQueue() -- Clear any previous timers
            result[1]:timer(utils.minutes(3) * 1000, function()
                result[1]:setStatus(xi.status.DISAPPEAR)

                if result[1]:getLocalVar(settings.hqVar) > 0 then
                    result[1]:setLocalVar(settings.hqVar, 0)
                    result[1]:setModelId(settings.nqLook)
                    result[1]:renameEntity(settings.nqName)
                end
            end)
        end
    end
end

for zoneName, zoneInfo in pairs(areas) do
    entities[zoneName] = entities[zoneName] or {}

    for mobName, mobInfo in pairs(zoneInfo) do
        local chestName = fmt("{}_Chest", mobName)

        if not mobInfo.dynamic then
            local path = fmt("xi.zones.{}.mobs.{}.onMobDeath", zoneName, mobName)

            cexi.util.ensureMob(zoneName, mobName)

            m:addOverride(path, function(mob, player, optParams)
                super(mob, player, optParams)
                cexi.aether.rollCasket(mob, player)
            end)
        end

        table.insert(entities[zoneName], {
            name       = chestName,
            packetName = settings.nqName,
            type       = xi.objType.NPC,
            look       = 969,
            x          = 1,
            y          = 1,
            z          = 1,
            rotation   = 0,
            hidden     = true,
            hidename   = true,

            onTrigger  = function(player, npc)
                onTrigger(player, npc, chestName)
            end,

            onTrade    = function(player, npc, trade)
                onTrade(player, npc, trade, mobInfo, chestName)
            end
        })
    end
end

cexi.util.liveReload(m, entities)

return m
