-----------------------------------
-- Ventures
-- !pos -54.260 12.000 -31.180 243
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
-----------------------------------
local m = Module:new("ventures_npc_populox")

local settings =
{
    name      = "Populox",
    pos       = { 21.987, -1.000, -49.796, 204 }, -- !pos 21.987 -1.000 -49.796 244
    area      = "Upper_Jeuno",
    look      = 85,

    dialog    =
    {
        DEFAULT = { "Oi, do I know you? I'm very busy gob you know!" },
        DECLINE = { "Woah, relax buddy. You don't have enough points for that one yet!" },
        EXPLAIN =
        {
            "Glad ya asked. Me and my goblin pals are looking for new lands to establish our trading business.",
            " Howevers, first we'll need a lil' clean up, so its safe enough to traverse.",
            { delay = 2000 },
            "Grab your mates, clear out the nastiest beasties and you'll receive some special tender for our store.",
            " Oh, and don't try any of that spooky trust magic either. Us gobs don't like that one bit!",
            { message = "\129\166 Note: Monsters must be at least 3 levels above the highest level party member." },
            { message = " (Parties with trusts may not participate.)" },
        },
        EXPLORING =
        {
            "Hmmm... %s? Today we're exploring %s.",
            " For this one, we need ya to clear out %u of the following eligible monsters:",
        },
        STARTING = { "Knew ya'd come around! Remember it's %u and no less!" },
        REMINDER = { "Jus' as a reminder, here's what we're looking to clean up in %s:" },
        RETIRE   =
        {
            "Hah! Giving up already? Guess adventurin' folk ain't so tough after all.",
            "You'll keep your progress next time ya start a venture.",
        },
        UPGRADE  =
        {

        },
        UPGRADED = { "A deal's a deal." },
        MISSED   = { "Looks like you dropped this!" },
    },
}

local ACE = "ACE"
local WEW = "WEW"
local CW  = "CW"

local rewardList =
{
    {
        "Currency & Materials",
        {
            { "Cotton coin purse", {5735},   150, { ACE, WEW, CW } },
            { "Aether Dust",       {9470},   250, { ACE, WEW, CW } },
            { "Prismatic Cluster", {9471},  3000, { ACE, WEW, CW } },
        },
    },
    {
        "Battlefields",
        {
            { "Goblin Brew",         9435,   400, { ACE, WEW, CW } },
            { "Goblin Brew +1",      9436,   800, { ACE, WEW, CW } },
            { "Goblin Brew +2",      9437,  3000, { ACE, WEW, CW } },

            { "Silver Voucher",      9277,  1500, { ACE, WEW     } },
            { "Cloud Evoker",        1842,  2000, { ACE, WEW, CW } },

            { "Cloudy Orb (20)",     1551,   400, { ACE, WEW, CW } },
            { "Sky Orb (30)",        1552,   600, { ACE, WEW, CW } },
            { "Star Orb (40)",       1131,   800, { ACE, WEW, CW } },
            { "Comet Orb (50)",      1177,  1000, { ACE, WEW, CW } },
            { "Moon Orb (60)",       1130,  1200, { ACE, WEW, CW } },

            { "Atropos Orb (30)",    1180,  1500, { ACE, WEW, CW } },
            { "Clotho Orb (30)",     1175,  1500, { ACE, WEW, CW } },
            { "Lachesis Orb (30)",   1178,  1500, { ACE, WEW, CW } },
            { "Themis Orb (99)",     1553,  4500, { ACE, WEW, CW } },
        },
    },
    {
        "Equipment",
        {
            { "Moogle Masque",      10429,   350, { WEW, CW      } }, -- Available from ACE Login Campaign
            { "Moogle Suit",        10250,   350, { WEW, CW      } }, -- Available from ACE Login Campaign
            { "Venture Ring",       10870,   500, { ACE, WEW, CW } }, -- 1/1 / Dedication: +0% Experience Points / +100% Venture Points
            { "Chocobo Whistle",    15533,   600, { WEW, CW      } }, -- 25 charges / 5 min recast

            { "Echad Ring",         27556,   750, { ACE          } },
            { "Goblin Masque",      27715,   750, { ACE, WEW, CW } },
            { "Goblin Suit",        27866,   750, { ACE, WEW, CW } },

            { "Craftkeeper's Ring", 28585,  1000, { ACE, WEW, CW } },
            { "Artificer's Ring",   28587,  1000, { ACE, WEW, CW } },
            { "Craftmaster's Ring", 28586,  2000, { ACE, WEW, CW } },

            { "Midras's Helm +1",   25637,  3000, { ACE, WEW, CW } },
            { "Wh. Rarab Cap +1",   25679,  5000, { ACE, WEW     } },
        },
    },
    {
        "Misc.",
        {
            { "Mog Wardrobe Slot", "MOGW",   300, { WEW, CW      } }, -- 1 Mog Wardrobe 7/8 Slot
            { "Mog Ward. Slot x5", "MOG5",  1500, { WEW, CW      } }, -- 5 Mog Wardrobe 7/8 Slots
        },
    },
}

local rewards = {}
rewards.ACE   = {}
rewards.WEW   = {}
rewards.CW    = {}

for categoryIndex, category in pairs(rewardList) do
    rewards.ACE[categoryIndex] = { category[1], {} }
    rewards.WEW[categoryIndex] = { category[1], {} }
    rewards.CW[categoryIndex]  = { category[1], {} }

    for _, rewardInfo in pairs(category[2]) do
        if utils.contains(ACE, rewardInfo[4]) then
            table.insert(rewards.ACE[categoryIndex][2], { rewardInfo[1], rewardInfo[2], rewardInfo[3] })
        end

        if utils.contains(WEW, rewardInfo[4]) then
            table.insert(rewards.WEW[categoryIndex][2], { rewardInfo[1], rewardInfo[2], rewardInfo[3] })
        end

        if utils.contains(CW, rewardInfo[4]) then
            table.insert(rewards.CW[categoryIndex][2], { rewardInfo[1], rewardInfo[2], rewardInfo[3] })
        end
    end
end

-----------------------------------
-- Util functions
-----------------------------------
local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

-----------------------------------
-- Menu functions
-----------------------------------
local function ventureStart(player, npc, info)
    if cexi.hasSetting(player, cexi.setting.SKIP_VENTURE_RP) then
        player:printToPlayer(string.format("%s Venture #%u Accepted %s", "\129\153", info.index, "\129\153"), xi.msg.channel.SYSTEM_3)
        player:setCharVar(cexi.ventures.exp.lockVar, info.index)
    else
        local starting = { string.format(settings.dialog.STARTING[1], info.qty) }
        local delay    = cexi.util.dialogDelay(starting)

        cexi.util.dialog(player, starting, settings.name, { npc = npc })

        npc:facePlayer(player, true)

        player:timer(delay, function()
            player:printToPlayer(string.format("%s Venture #%u Accepted %s", "\129\153", info.index, "\129\153"), xi.msg.channel.SYSTEM_3)
            player:setCharVar(cexi.ventures.exp.lockVar, info.index)
        end)
    end
end

local function getMobList(mobs)
    local str = " " .. string.gsub(mobs[1], "_", " ")
    local max = 6

    if #mobs < max then
        max = #mobs
    end

    for i = 2, max do
        str = str .. ", " .. string.gsub(mobs[i], "_", " ")
    end

    if max == 6 then
        str = str .. " and more!"
    end

    return str
end

local function ventureRange(player, npc, category)
    local rangeIndex   = category[2]
    local ventureIndex = GetServerVariable(string.format(cexi.ventures.exp.rollVar, rangeIndex))
    local venture      = cexi.ventures.exp.ventures[rangeIndex][ventureIndex]
    local areaName     = string.gsub(venture.area, "_", " ")

    local tbl =
    {
        string.format(settings.dialog.EXPLORING[1], category[1], areaName),
        string.format(settings.dialog.EXPLORING[2], venture.qty)
    }

    local mobList = getMobList(venture.mobs)
    table.insert(tbl, mobList)

    local delay = cexi.util.dialogDelay(tbl)
    cexi.util.dialog(player, tbl, settings.name, { npc = npc })

    npc:facePlayer(player, true)
    player:setLocalVar("[NPC]BUSY", 1)

    player:timer(delay, function()
        player:setLocalVar("[NPC]BUSY", 0)
        delaySendMenu(player, {
            title   = "Start this venture?",
            options =
            {
                {
                    "Not yet.",
                    function()
                    end,
                },
                {
                    "Let's go!",
                    function()
                        ventureStart(player, npc, venture)
                    end,
                }
            },
        })
    end)
end

local wardrobeName =
{
    [xi.inv.WARDROBE7] = "Mog Wardrobe 7",
    [xi.inv.WARDROBE8] = "Mog Wardrobe 8",
}

local function increaseWardrobeSize(player, wardrobe, amount)
    local oldSize = player:getContainerSize(wardrobe)
    player:changeContainerSize(wardrobe, amount)

    local newSize = player:getContainerSize(wardrobe)
    player:sys("\129\154 Your {} capacity has been increased by 1 from {} to {}! \129\154", wardrobeName[wardrobe], oldSize, newSize)
end

local shopAction =
{
    MOGW = function(player, cost, balance)
        if player:getContainerSize(xi.inv.WARDROBE8) < 80 then
            increaseWardrobeSize(player, xi.inv.WARDROBE8, 1)
            return true

        elseif player:getContainerSize(xi.inv.WARDROBE7) < 80 then
            increaseWardrobeSize(player, xi.inv.WARDROBE7, 1)
            return true

        else
            player:sys("You have already reached maximum capacity with Mog Wardrobe 7/8.")
        end

        return false
    end,

    MOG5 = function(player, cost, balance)
        local w7 = player:getContainerSize(xi.inv.WARDROBE7)
        local w8 = player:getContainerSize(xi.inv.WARDROBE8)

        if (w7 + w8 + 5) > 160 then
            player:sys("You have already reached maximum capacity with Mog Wardrobe 7/8.")
            return false
        end

        local remaining = 5
        local diff      = utils.clamp(80 - w8, 0, 5)

        if diff > 0 then
            increaseWardrobeSize(player, xi.inv.WARDROBE8, diff)
            remaining = remaining - diff
        end

        if remaining > 0 then
            increaseWardrobeSize(player, xi.inv.WARDROBE7, remaining)
        end

        return true
    end,
}

local qtyList  =
{
    {  "x1",  1 },
    {  "x2",  2 },
    {  "x3",  3 },
    {  "x6",  6 },
    { "x12", 12 },
    { "x24", 24 },
    { "x36", 36 },
    { "x99", 99 },
}

local function quantityPage(player, npc, item, varName)
    local tbl     = {}
    local balance = player:getCharVar(varName)

    for _, qtyInfo in pairs(qtyList) do
        local label = qtyInfo[1]
        local qty   = qtyInfo[2]

        if qty * item[3] <= balance then
            table.insert(tbl, {
                fmt("{} ({})", label, item[3] * qty),
                function()
                    player:timer(500, function()
                        if npcUtil.giveItem(player, { { item[2][1], qty } }) then
                            player:incrementCharVar(varName, -(item[3] * qty))

                            if (balance - item[3]) > 0 then
                                quantityPage(player, npc, item, varName)
                            end
                        end
                    end)
                end,
            })
        end
    end

    delaySendMenu(player, {
        title   = fmt("Purchase {} ({}):", item[1], balance),
        options = tbl,
    })
end

-----------------------------------
-- Menu functions
-----------------------------------
local function confirmPurchase(player, npc, item)
    local varName = cexi.ventures.exp.pointsVar

    if type(item[2]) == "table" then
        quantityPage(player, npc, item, varName)
        return
    end

    local balance     = player:getCharVar(varName)
    local itemName    = item[1]
    local itemID      = item[2]
    local itemCost    = item[3]

    npc:facePlayer(player, true)

    delaySendMenu(player, {
        title   = string.format("Spend %u points? (%u available)", itemCost, balance),
        options =
        {
            {
                "No way",
                function()
                end,
            },
            {
                string.format("Purchase: %s", itemName),
                function()
                    if
                        balance >= itemCost and
                        itemID ~= nil
                    then
                        if type(item[2]) == "string" then
                            if shopAction[item[2]](player, cost, balance) then
                                player:setCharVar(cexi.ventures.exp.pointsVar, balance - itemCost)
                            end
                        else
                            if npcUtil.giveItem(player, itemID) then
                                player:setCharVar(cexi.ventures.exp.pointsVar, balance - itemCost)
                            end
                        end
                    else
                        cexi.util.dialog(player, settings.dialog.DECLINE, settings.name, { npc = npc })
                    end
                end,
            },
        },
    })
end

local function ventureRedeem(player, npc)
    local points = player:getCharVar(cexi.ventures.exp.pointsVar)
    local title  = string.format("Venture Points (%u):", points)

    npc:facePlayer(player, true)

    if player:isCrystalWarrior() then
        cexi.util.categoryMenu(player, npc, rewards.CW, confirmPurchase, title)
    elseif player:isClassicMode() then
        cexi.util.categoryMenu(player, npc, rewards.WEW, confirmPurchase, title)
    else
        cexi.util.categoryMenu(player, npc, rewards.ACE, confirmPurchase, title)
    end
end

local function ventureCancel(player, npc)
    npc:facePlayer(player, true)

    local index    = player:getCharVar(cexi.ventures.exp.lockVar)
    local tallyVar = string.format(cexi.ventures.exp.progVar)
    local total    = player:getCharVar(tallyVar)
    local venture  = cexi.ventures.exp.mapping[index]
    venture        = cexi.ventures.exp.ventures[venture[1]][venture[2]]

    delaySendMenu(player, {
        title   = string.format("Give up? (%u/%u kills)", total, venture.qty),
        options =
        {
            {
                "No way!",
                function()
                    local tbl     = { string.format(settings.dialog.REMINDER[1], string.gsub(venture.area, "_", " ")) }
                    local mobList = getMobList(venture.mobs)

                    table.insert(tbl, mobList)
                    cexi.util.dialog(player, tbl, settings.name, { npc = npc })

                    local delay = cexi.util.dialogDelay(tbl)

                    player:setLocalVar("[NPC]BUSY", 1)
                    player:timer(delay, function()
                        player:setLocalVar("[NPC]BUSY", 0)
                    end)
                end,
            },
            {
                "I'm out!",
                function()
                    if cexi.hasSetting(player, cexi.setting.SKIP_VENTURE_RP) then
                        player:printToPlayer(string.format("%s Venture Cancelled! %s", "\129\166", "\129\166"), xi.msg.channel.SYSTEM_3)
                        player:setCharVar(cexi.ventures.exp.lockVar, 0)
                    else
                        local delay = cexi.util.dialogDelay(settings.dialog.RETIRE)
                        cexi.util.dialog(player, settings.dialog.RETIRE, settings.name, { npc = npc })

                        player:setLocalVar("[NPC]BUSY", 1)
                        player:timer(delay, function()
                            player:setLocalVar("[NPC]BUSY", 0)
                            player:printToPlayer(string.format("%s Venture Cancelled! %s", "\129\166", "\129\166"), xi.msg.channel.SYSTEM_3)
                        end)

                        -- Do not reset kill count
                        -- player:setCharVar(cexi.ventures.exp.progVar, 0)
                        player:setCharVar(cexi.ventures.exp.lockVar, 0)
                    end
                end,
            },
        },
    })
end

-- Potential missing boxes
local boxes =
{
    5109,
    5111,
    6264,
}

local function onTrigger(player, npc)
    if player:getLocalVar("[NPC]BUSY") == 1 then
        return
    end

    npc:facePlayer(player, true)

    for _, box in pairs(boxes) do
        local varName     = string.format(cexi.ventures.exp.missVar, box)
        local missedBoxes = player:getCharVar(varName)

        if missedBoxes > 0 then
            cexi.util.dialog(player, settings.dialog.MISSED, settings.name, { npc = npc })

            if npcUtil.giveItem(player, box) then
                player:setCharVar(varName, missedBoxes - 1)
            end

            return
        end
    end

    local options =
    {
        {
            "Nothing",
            function()
            end,
        },
    }

    if player:getCharVar(cexi.ventures.exp.lockVar) > 0 then
        table.insert(options, {
            "Cancel this Venture",
            function()
                ventureCancel(player, npc)
            end,
        })
    else
        table.insert(options, {
            "Embark on a Venture",
            function()
                cexi.util.simpleMenu(player, npc, cexi.ventures.exp.categories, ventureRange, "Select a level range:")
            end,
        })
    end

    table.insert(options, {
        "Redeem Venture Points",
        function()
            ventureRedeem(player, npc)
        end,
    })

    table.insert(options, {
        "What's this all about?",
        function()
            cexi.util.dialog(player, settings.dialog.EXPLAIN, settings.name, { npc = npc })
        end,
    })

    delaySendMenu(player, {
        title   = "Whad'ya want?",
        options = options,
    })
end

-----------------------------------
-- Upgrade Items
-----------------------------------
local upgrades =
{
    {
        name = "Battle Horn",
        nq   = 21402,
        hq   = 21403,
        cost = 10,
    },
    {
        name = "Wilderness Earring",
        nq   = 28490,
        hq   = 28491,
        cost = 20,
    },
    {
        name = "Thorin's Shield",
        nq   = 27633,
        hq   = 27634,
        cost = 30,
    },
}

local cost = { 4, 6, 8 }

local augments =
{
    -----------------------------------
    -- Lv30
    -----------------------------------
    {
        -- Fire res+5, MP+15, Handbell skill+3
        -- GEO
        name = "Zhulong Bell",
        id   = 21425,
        tier =
        {
            { augs = {  25, 0, 133, 0 }, mats = { { 671, 2 } }, desc = "2 silver sheets" }, -- Attack+1 Mag.Atk.Bns.+1
            { augs = {  25, 1, 133, 1 }, mats = { { 671, 3 } }, desc = "3 silver sheets" }, -- Attack+2 Mag.Atk.Bns.+2
            { augs = {  25, 2, 133, 2 }, mats = { { 671, 4 } }, desc = "4 silver sheets" }, -- Attack+3 Mag.Atk.Bns.+3
        },
    },
    {
        -- MP+15 INT+1 MND+1 Healing magic skill+3
        -- SCH
        name = "Enlightenment",
        id   = 21426,
        tier =
        {
            { augs = {  31, 0, 329, 0 }, mats = { { 2550, 2 } }, desc = "2 sheets of vellum" }, -- Evasion+1 Cure potency+1%
            { augs = {  31, 1, 329, 1 }, mats = { { 2550, 3 } }, desc = "3 sheets of vellum" }, -- Evasion+2 Cure potency+2%
            { augs = {  31, 2, 329, 2 }, mats = { { 2550, 4 } }, desc = "4 sheets of vellum" }, -- Evasion+3 Cure potency+3%
        },
    },
    {
        -- DMG:52 Delay 501
        -- Dark res+10, STR+2, Double Attack+2%
        -- BLM
        name = "Sinister Sickle",
        id   = 20829,
        tier =
        {
            { augs = { 740, 1, 516, 1 }, mats = { { 2229, 2 } }, desc = "2 vials of chimera blood" }, -- DMG+2 INT+2
            { augs = { 740, 2, 516, 2 }, mats = { { 2229, 3 } }, desc = "3 vials of chimera blood" }, -- DMG+3 INT+3
            { augs = { 740, 3, 516, 3 }, mats = { { 2229, 4 } }, desc = "4 vials of chimera blood" }, -- DMG+4 INT+4
        },
    },

    -----------------------------------
    -- Lv40
    -----------------------------------
    {
        -- Attack+2 "Store TP"+2
        -- All Jobs
        name = "Tigris Grip",
        id   = 21428,
        tier =
        {
            { augs = { 513, 0, 518, 0 }, mats = { { 855, 2 } }, desc = "2 tiger leather" }, -- DEX+1, CHR+1
            { augs = { 513, 1, 518, 1 }, mats = { { 855, 3 } }, desc = "3 tiger leather" }, -- DEX+2, CHR+2
            { augs = { 513, 2, 518, 2 }, mats = { { 855, 4 } }, desc = "4 tiger leather" }, -- DEX+3, CHR+3
        },
    },
    {
        -- Attack+4 Evasion+4
        -- WAR/THF/RNG/SAM/NIN/DNC
        name = "Slippery Cape",
        id   = 28611,
        tier =
        {
            { augs = { 513, 0, 515, 0 }, mats = { { 1633, 2 } }, desc = "2 clot plasma" }, -- DEX+1, AGI+1
            { augs = { 513, 1, 515, 1 }, mats = { { 1633, 3 } }, desc = "3 clot plasma" }, -- DEX+2, AGI+2
            { augs = { 513, 2, 515, 2 }, mats = { { 1633, 4 } }, desc = "4 clot plasma" }, -- DEX+3, AGI+3
        },
    },
    {
        -- Light res+10, HP+15, Parrying skill+3
        -- WAR/PLD/RUN
        name = "Chivalric Emblem",
        id   = 20965,
        tier =
        {
            { augs = { 517, 0, 23, 0 }, mats = { {  676, 2 } }, desc = "2 steel scales" }, -- MND+1, Accuracy+1
            { augs = { 517, 1, 23, 1 }, mats = { {  676, 3 } }, desc = "3 steel scales" }, -- MND+2, Accuracy+2
            { augs = { 517, 2, 23, 2 }, mats = { {  676, 4 } }, desc = "4 steel scales" }, -- MND+3, Accuracy+3
        },
    },

    -----------------------------------
    -- Lv50
    -----------------------------------
    {
        -- DMG:35 Delay 548
        -- Ranged Accuracy+8
        -- AGI+2 INT+2
        -- THF/RNG/NIN/COR
        name = "Liberty",
        id   = 21257,
        tier =
        {
            { augs = { 29, 0, 31, 0 }, mats = { { 745, 2 } }, desc = "2 gold ingots" }, -- Ranged Attack+1, Evasion+1
            { augs = { 29, 1, 31, 1 }, mats = { { 745, 3 } }, desc = "3 gold ingots" }, -- Ranged Attack+2, Evasion+2
            { augs = { 29, 2, 31, 2 }, mats = { { 745, 4 } }, desc = "4 gold ingots" }, -- Ranged Attack+3, Evasion+3
        },
    },
    {
        -- DEF:4 STR+4, VIT+4
        -- MNK/WHM/RDM/THF/PLD/BST/BRD/DRG/SMN/BLU/COR/PUP/DNC
        name = "Wyrt Gorget",
        id   = 28400,
        tier =
        {
            { augs = { 23, 0, 25, 0 }, mats = { { 923, 2 } }, desc = "2 dryad roots" }, -- Accuracy+1, Attack+1
            { augs = { 23, 1, 25, 1 }, mats = { { 923, 3 } }, desc = "3 dryad roots" }, -- Accuracy+2, Attack+2
            { augs = { 23, 2, 25, 2 }, mats = { { 923, 4 } }, desc = "4 dryad roots" }, -- Accuracy+3, Attack+3
        },
    },
    {
        -- HP+20 VIT+3 Earth res
        -- Phys. dmg taken -2%
        -- All Jobs
        name = "Myrmeleo Ring",
        id   = 28576,
        tier =
        {
            { augs = { 31, 0, 39, 0 }, mats = { { 1616, 2 } }, desc = "2 antlion jaws" }, -- Evasion+1, Enmity+1
            { augs = { 31, 1, 39, 1 }, mats = { { 1616, 3 } }, desc = "3 antlion jaws" }, -- Evasion+2, Enmity+2
            { augs = { 31, 2, 39, 2 }, mats = { { 1616, 4 } }, desc = "4 antlion jaws" }, -- Evasion+3, Enmity+3
        },
    },
    -----------------------------------
    -- Goblin Brew +1 (Lv75)
    -----------------------------------
    {
        -- DMG:84 Delay:480
        -- HP-20 CHR-10
        -- "Treasure Hunter"+2
        -- WAR/DRK/RUN
        name = "Avarice",
        id   = 21697,
        tier =
        {
            { augs = { 512, 2, 25, 5 }, mats = { { 654,  6 } }, desc = "6 darksteel ingots"  }, -- STR+3 Attack+6
            { augs = { 512, 3, 25, 7 }, mats = { { 654,  8 } }, desc = "8 darksteel ingots"  }, -- STR+4 Attack+8
            { augs = { 512, 4, 25, 9 }, mats = { { 654, 12 } }, desc = "12 darksteel ingots" }, -- STR+5 Attack+12
        },
    },
    {
        -- DMG:80 Delay:450
        -- HP+20 AGI+9
        -- Archery Skill+7
        -- SAM
        name = "Kyugutachi",
        id   = 21054,
        tier =
        {
            { augs = { 25,  5, 29,  5 }, mats = { { 1414,  6 } }, desc = "6 wisteria lumber"  }, -- Attack+6 Ranged Attack+6
            { augs = { 25,  7, 29,  7 }, mats = { { 1414,  8 } }, desc = "8 wisteria lumber"  }, -- Attack+8 Ranged Attack+8
            { augs = { 25, 11, 29, 11 }, mats = { { 1414, 12 } }, desc = "12 wisteria lumber" }, -- Attack+12 Ranged Attack+12
        },
    },
    {
        -- MP+10
        -- Staff Skill +4 Scythe Skill+4
        -- All Jobs
        name = "Sprightly Feather",
        id   = 21386,
        tier =
        {
            { augs = { 25, 1, 31, 1 }, mats = { { 1619,  6 } }, desc = "6 hippogryph feathers"  }, -- Attack+2 Evasion+2
            { augs = { 25, 2, 31, 2 }, mats = { { 1619,  8 } }, desc = "8 hippogryph feathers"  }, -- Attack+3 Evasion+3
            { augs = { 25, 3, 31, 3 }, mats = { { 1619, 12 } }, desc = "12 hippogryph feathers" }, -- Attack+4 Evasion+4
        },
    },

    -----------------------------------
    -- Goblin Brew +2 (Lv75)
    -----------------------------------
    -- Ice Lord
    {
        name = "Hibernal Ring",
        id   = xi.item.HIBERNAL_RING,
        tier =
        {
            { augs = { 513, 1, 23, 1         }, mats = { { 1280, 4 }, { 13519, 1 } }, desc = "4 squares of sarcenet cloths and an mythril ring +1" }, -- DEX+2 Accuracy+2
            { augs = { 513, 2, 23, 2         }, mats = { { 1280, 6 }, { 13539, 1 } }, desc = "6 squares of sarcenet cloths and a genius ring"      }, -- DEX+3 Accuracy+3
            { augs = { 513, 3, 23, 3, 133, 1 }, mats = { { 1280, 8 }, { 14640, 1 } }, desc = "8 squares of sarcenet cloths and a snow ring"        }, -- DEX+4 Accuracy+4, MAB+2
        },
    },
    {
        name = "Hailstone Hose",
        id   = 28166,
        tier =
        {
            { augs = { 515, 1, 35, 1        }, mats = { { 1280, 4 }, { 12907, 1 } }, desc = "4 squares of sarcenet cloths and a pair of wool hose +1"   }, -- AGI+2, Magic Accuracy+2
            { augs = { 515, 2, 35, 2        }, mats = { { 1280, 6 }, { 14237, 1 } }, desc = "6 squares of sarcenet cloths and a pair of battle hose +1" }, -- AGI+3, Magic Accuracy+3
            { augs = { 515, 3, 35, 3, 49, 1 }, mats = { { 1280, 8 }, { 12879, 1 } }, desc = "8 squares of sarcenet cloths and a pair of dusk trousers"  }, -- AGI+4, Magic Accuracy+4, Haste+2%
        },
    },

    -- Cape Chelonian
    {
        name = "Rikugame Nodowa",
        id   = 28385,
        tier =
        {
            { augs = { 1,  9, 23, 5         }, mats = { { 908, 4 }, { 13124, 1 } }, desc = "4 adamantoise shells and a nodowa +1"           }, -- HP+10 Accuracy+6
            { augs = { 1, 19, 23, 6         }, mats = { { 908, 6 }, { 13133, 1 } }, desc = "6 adamantoise shells and a darksteel nodowa +1" }, -- HP+20 Accuracy+7
            { augs = { 1, 29, 23, 7, 513, 1 }, mats = { { 908, 8 }, { 15539, 1 } }, desc = "8 adamantoise shells and an orochi nodowa"      }, -- HP+30 Accuracy+8, DEX+2
        },
    },
    {
        name = "Tellus Pendulum",
        id   = 28387,
        tier =
        {
            { augs = { 9,  9, 516, 1         }, mats = { { 908, 4 }, { 16262, 1 } }, desc = "4 adamantoise shells and mohbwa scarf +1 "  }, -- MP+10 INT+2
            { augs = { 9, 19, 516, 2         }, mats = { { 908, 6 }, { 13126, 1 } }, desc = "6 adamantoise shells and a torque +1"       }, -- MP+20 INT+3
            { augs = { 9, 29, 516, 3, 133, 1 }, mats = { { 908, 8 }, { 13087, 1 } }, desc = "8 adamantoise shells and a jeweled collar"  }, -- MP+30 INT+4, MAB+2
        },
    },

    -- Avalerion
    {
        name = "Bagua Ring",
        id   = 28548,
        tier =
        {
            { augs = { 774, 2, 23, 2         }, mats = { { 2228, 4 }, { 15837, 1 } }, desc = "4 chunks of luminium ore and a smilodon ring +1" }, -- Light res+3, Accuracy+3
            { augs = { 774, 4, 23, 3         }, mats = { { 2228, 6 }, { 13544, 1 } }, desc = "6 chunks of luminium ore and an allure ring"     }, -- Light res+6, Accuracy+4
            { augs = { 774, 7, 23, 4, 110, 1 }, mats = { { 2228, 8 }, { 14642, 1 } }, desc = "8 chunks of luminium ore and a light ring"       }, -- Light res+9, Accuracy+5, Pet: Regen+2
        },
    },
    {
        name = "Luminous Earring",
        id   = 28613,
        tier =
        {
            { augs = { 1,  4, 35, 0         }, mats = { { 2228, 4 }, { 13363, 1 } }, desc = "4 chunks of luminium ore and tortoise earring +1" }, -- HP+5,  Magic Accuracy+1
            { augs = { 1,  9, 35, 1         }, mats = { { 2228, 6 }, { 13396, 1 } }, desc = "6 chunks of luminium ore and an allure earring"   }, -- HP+10, Magic Accuracy+2
            { augs = { 1, 14, 35, 2, 142, 1 }, mats = { { 2228, 8 }, { 13357, 1 } }, desc = "8 chunks of luminium ore and an angel earring"    }, -- HP+15, Magic Accuracy+3, "Store TP"+2
        },
    },

    -- Pyrolynx
    {
        name = "Emberpearl Earring",
        id   = 28612,
        tier =
        {
            { augs = { 1,  4, 25, 0         }, mats = { { 1634, 4 }, { 13326, 1 } }, desc = "4 rhodonites and a beetle earring +1" }, -- HP+5,  Attack+1
            { augs = { 1,  9, 25, 1         }, mats = { { 1634, 6 }, { 13369, 1 } }, desc = "6 rhodonites and a spike earring"     }, -- HP+10, Attack+2
            { augs = { 1, 14, 25, 2, 326, 1 }, mats = { { 1634, 8 }, { 13352, 1 } }, desc = "8 rhodonites and a ruby earring"      }, -- HP+15, Attack+3, WS Acc+2
        },
    },
    {
        name = "Flamedancer Glaive",
        id   = 21026,
        tier =
        {
            { augs = { 512, 1, 513, 1        }, mats = { { 1634, 4 }, { 17289, 1 } }, desc = "4 rhodonites and a chakram +1"    }, -- STR+2, DEX+2
            { augs = { 512, 2, 513, 2        }, mats = { { 1634, 6 }, { 17295, 1 } }, desc = "6 rhodonites and a rising sun +1" }, -- STR+3, DEX+3
            { augs = { 512, 3, 513, 3, 23, 1 }, mats = { { 1634, 8 }, { 18708, 1 } }, desc = "8 rhodonites and a snakeeye"      }, -- STR+4, DEX+4, Accuracy+2
        },
    },

    -- Yasuo Hasaki
    {
        name = "Kennan's Longbow",
        id   = xi.item.KENNANS_LONGBOW,
        tier =
        {
            { augs = { 40,  2, 27, 0          }, mats = { { xi.item.ANIMATED_LUMBER, 2 }, { xi.item.DRAGON_ESSENCE, 2 }, { 17181, 1 } }, desc = "2 animated lumber, 2 dragon essences and a battle bow +1"    }, -- Enmity-3, Ranged Accuracy+3
            { augs = { 40,  3, 27, 1          }, mats = { { xi.item.ANIMATED_LUMBER, 3 }, { xi.item.DRAGON_ESSENCE, 3 }, { 17241, 1 } }, desc = "3 animated lumber, 3 dragon essences and a lightning bow +1" }, -- Enmity-4, Ranged Accuracy+6
            { augs = { 40,  4, 27, 2, 1264, 1 }, mats = { { xi.item.ANIMATED_LUMBER, 4 }, { xi.item.DRAGON_ESSENCE, 4 }, { 18695, 1 } }, desc = "4 animated lumber, 4 dragon essences and a cerberus bow"     }, -- Enmity-5, Ranged Accuracy+9, Meditate Duration+2
        },
    },
    {
        name = "Oathbreaker",
        id   = xi.item.OATHBREAKER,
        tier =
        {
            { augs = { 775,  4, 514, 1        }, mats = { { xi.item.TITANIUM_INGOT, 2 }, { xi.item.DRAGON_ESSENCE, 2 }, { 17613, 1 } }, desc = "2 titanium ingots, 2 dragon essences and a beetle knife +1"   }, -- Dark res+5,  VIT+2
            { augs = { 775,  9, 514, 2        }, mats = { { xi.item.TITANIUM_INGOT, 3 }, { xi.item.DRAGON_ESSENCE, 3 }, { 16510, 1 } }, desc = "3 titanium ingots, 3 dragon essences and a venom baselard +1" }, -- Dark res+10, VIT+3
            { augs = { 775, 14, 514, 3, 39, 1 }, mats = { { xi.item.TITANIUM_INGOT, 4 }, { xi.item.DRAGON_ESSENCE, 4 }, { 18030, 1 } }, desc = "4 titanium ingots, 4 dragon essences and a khimaira jambiya"  }, -- Dark res+15, VIT+4, Enmity+2
        },
    },
    {
        name = "Manaflow Sash",
        id   = xi.item.MANAFLOW_SASH,
        tier =
        {
            { augs = { 40,  2, 49, 2         }, mats = { { xi.item.FABLE_CLOTH, 2 }, { xi.item.DRAGON_ESSENCE, 2 }, { 13274, 1 } }, desc = "2 fable cloths, 2 dragon essences and a twinthread obi +1" }, -- Enmity-3, Haste+3%
            { augs = { 40,  3, 49, 3         }, mats = { { xi.item.FABLE_CLOTH, 3 }, { xi.item.DRAGON_ESSENCE, 3 }, { 13276, 1 } }, desc = "3 fable cloths, 3 dragon essences and an arachne obi +1"   }, -- Enmity-4, Haste+4%
            { augs = { 40,  4, 49, 4, 371, 1 }, mats = { { xi.item.FABLE_CLOTH, 4 }, { xi.item.DRAGON_ESSENCE, 4 }, { 11765, 1 } }, desc = "4 fable cloths, 4 dragon essences and a visionary obi"     }, -- Enmity-5, Haste+5%, Regen Potency+2
        },
    },
    {
        name = "Beastly Girdle",
        id   = xi.item.BEASTLY_GIRDLE,
        tier =
        {
            { augs = {  49, 2, 96, 2         }, mats = { { xi.item.EMBERSTEEL_RIVET, 2 }, { xi.item.DRAGON_ESSENCE, 2 }, { 15914, 1 } }, desc = "2 embersteel rivets, 2 dragon essences and a peiste belt +1" }, -- Haste+3%, Pet: Accuracy+3 Rng.Acc+3
            { augs = {  49, 3, 96, 3         }, mats = { { xi.item.EMBERSTEEL_RIVET, 3 }, { xi.item.DRAGON_ESSENCE, 3 }, { 13279, 1 } }, desc = "3 embersteel rivets, 3 dragon essences and a muscle belt +1" }, -- Haste+4%, Pet: Accuracy+4 Rng.Acc+4
            { augs = {  49, 4, 96, 4, 110, 1 }, mats = { { xi.item.EMBERSTEEL_RIVET, 4 }, { xi.item.DRAGON_ESSENCE, 4 }, { 11763, 1 } }, desc = "4 embersteel rivets, 4 dragon essences and a companion belt" }, -- Haste+5%, Pet: Accuracy+5 Rng.Acc+5, Pet: Regen+2
        },
    },
}

for augmentIndex, augmentInfo in pairs(augments) do
    for tierIndex, tierInfo in pairs(augmentInfo.tier) do
        table.insert(augments[augmentIndex].tier[tierIndex].mats, { augmentInfo.id, 1 })
        table.insert(augments[augmentIndex].tier[tierIndex].mats, { xi.item.GOBLIN_COMMENDATION, cost[tierIndex] })
    end
end

local function getAugTier(itemInfo, itemAug)
    local tier = 0

    for augTier = 1, #itemInfo.tier do
        if
            itemInfo.tier[augTier].augs[1] == itemAug[1][1] and
            itemInfo.tier[augTier].augs[2] == itemAug[1][2] and
            itemInfo.tier[augTier].augs[3] == itemAug[2][1] and
            itemInfo.tier[augTier].augs[4] == itemAug[2][2]
        then
            tier = augTier
        end
    end

    return tier
end

local function onTrade(player, npc, trade)
    for _, item in pairs(upgrades) do
        if cexi.util.tradeHasExactly(trade, item.nq) then
            local tbl =
            {
                fmt("Woah! Is that a {}? Ya know, I could make that one even better... for a price.", item.name),
                fmt(" Bring that thing back with {} goblin commendations and ya got yourself a deal.", item.cost),
            }

            npc:facePlayer(player, true)
            cexi.util.dialog(player, tbl, settings.name, { npc = npc })

            return
        end

        if cexi.util.tradeHasExactly(trade, { { item.nq, 1 }, { xi.item.GOBLIN_COMMENDATION, item.cost } }) then
            npc:facePlayer(player, true)
            cexi.util.dialog(player, settings.dialog.UPGRADED, settings.name, { npc = npc })

            if npcUtil.giveItem(player, item.hq) then
                player:tradeComplete()
            end
        end
    end

    for _, item in pairs(augments) do
        if npcUtil.tradeHas(trade, item.id, false) then
            local obj  = cexi.util.augment.getTradedItem(trade, item.id)
            local augs = cexi.util.augment.getAugments(player, obj)
            local tier = 0

            if #augs > 0 then
                tier = getAugTier(item, augs)
            end

            if tier == #cost then
                cexi.util.dialog(player, { fmt("No more! That's as good as that {} is gonna get!", item.name) }, settings.name, { npc = npc })
                return
            end

            local tierInfo = item.tier[tier + 1]

            if cexi.util.tradeHasExactly(trade, item.id) then
                local tbl =
                {
                    fmt("Woah! Is that a {}? Ya know, I could make that one even better... for a price.", item.name),
                    fmt(" Bring that thing back with a few materials and {} goblin commendations and ya got yourself a deal.", cost[tier + 1]),
                    fmt(" This one's going to need {}.", tierInfo.desc)
                }

                cexi.util.dialog(player, tbl, settings.name, { npc = npc })
                return
            end

            if cexi.util.tradeHasExactly(trade, tierInfo.mats) then
                player:tradeComplete()
                cexi.util.dialog(player, settings.dialog.UPGRADED, settings.name, { npc = npc })

                player:addItem(item.id, 1, tierInfo.augs[1], tierInfo.augs[2], tierInfo.augs[3], tierInfo.augs[4], tierInfo.augs[5], tierInfo.augs[6])
                player:timer(1000, function()
                    npc:independentAnimation(player, 12, 2)
                    player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, item.id)
                end)
            end
        end
    end
end

cexi.util.liveReload(m, {
    [settings.area] =
    {
        {
            name      = settings.name,
            objtype   = xi.objType.NPC,
            look      = settings.look,
            x         = settings.pos[1],
            y         = settings.pos[2],
            z         = settings.pos[3],
            rotation  = settings.pos[4],
            onTrigger = onTrigger,
            onTrade   = onTrade,
        },
    },
})

return m
