-----------------------------------
-- Ventures
-- !pos -54.260 12.000 -31.180 243
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("ventures_npc_alternix")

local settings =
{
    name      = "Alternix",
    pos       = { 23.563, -1.000, -50.282, 208 }, -- !pos 23.563 -1.000 -50.282 244
    area      = "Upper_Jeuno",
    look      = 488,

    dialog    =
    {
        DEFAULT      = { "Oi, do I know you? I'm very busy gob you know!" },
        DECLINE      = { "Woah, relax buddy. You don't have enough points for that one yet!" },
        STARTING     = { "Knew ya'd come around! Remember it's %u and no less!" },
        REMINDER     = { "Jus' as a reminder, the current objective is %s!" },
        INSUFFICIENT = { "Hah! Giving up already? Guess adventurin' folk ain't so tough after all." },
        RETIRE       =
        {
            "Hah! Giving up already? Guess adventurin' folk ain't so tough after all.",
            " This is all I can give ya based on your current progress.",
            "Better luck next time buddy!",
        },
        UPGRADE      =
        {
            "Woah! Is that a %s? Ya know, I could make that one even better... for a price.",
            " Bring that thing back with %u %s and %s, then ya got yourself a deal.",
        },
        UPGRADED     = { "A deal's a deal." },
        MISSED       = { "Looks like you dropped this!" },
    },
}

local rewards =
{
    FISHING =
    {
        { "M. Carp Creel",              { 5810 },    45 },
        { "Angl. Commendation",             4033,  1000 },
        { "Sand Charm",                    13095,  1500 },
        { "Tlahtlamah Glasses",            25608,  1800 },
        { "Mariner's Tunica",              26535,  3000 },
        { "Mariner's Gloves",              25986,  3000 },
        { "Mariner's Hose",                25899,  3000 },
        { "Mariner's Boots",               25966,  3000 },
    },

    HARVESTING =
    {
        { "Hay Bale",             6590,   300 },
        { "Special Brew",         9434,   800 },
        { "Hoe",                 20909,  1200 },
        { "Plain Gloves",        25984,  3000 },
        { "Harv. Sun Hat",       25557,  5000 },
    },

    EXCAVATION =
    {
        { "Museum Case",          6591,   300 },
        { "Special Brew",         9434,   800 },
        { "Caver's Shovel",      18600,  1200 },
        { "Plain Tunica",        26533,  3000 },
        { "Excavator's Shades",  25558,  5000 },
    },

    LOGGING =
    {
        { "Scrap Lumber",         6592,   300 },
        { "Special Brew",         9434,   800 },
        { "Dullahan Axe",        21745,  1200 },
        { "Plain Boots",         25964,  3000 },
        { "Lumberjack's Beret",  25559,  5000 },
    },

    MINING =
    {
        { "Box of Rocks",         6593,   300 },
        { "Special Brew",         9434,   800 },
        { "Worm Feelers",        20532,  1200 },
        { "Plain Hose",          25897,  3000 },
        { "Miner's Helmet",      25560,  5000 },
    },
}

local cwRewards =
{
    DYNAMIS =
    {
        { "Infinity Core",             { 1474 },    50 },
        { "100 Byne Bill",             { 1456 },   500 },
        { "M. Silverpiece",            { 1453 },   500 },
        { "L. Jadeshell",              { 1450 },   500 },
        { "Ark Tachi",                    18464,  1800 },
        { "Ark Tabar",                    18545,  1800 },
        { "Ark Scythe",                   18563,  1800 },
        { "Ark Saber",                    18912,  1800 },
        { "Ark Sword",                    18913,  1800 },
    },

    FISHING =
    {
        { "M. Carp Creel",              { 5810 },    45 },
        { "Angl. Commendation",             4033,  1000 },
        { "Sand Charm",                    13095,  1500 },
        { "Tlahtlamah Glasses",            25608,  1800 },
        { "Mariner's Tunica",              26535,  3000 },
        { "Mariner's Gloves",              25986,  3000 },
        { "Mariner's Hose",                25899,  3000 },
        { "Mariner's Boots",               25966,  3000 },
        { "Ephemeral Expansion", "[EB]FISHING",  1500, "You have unlocked the Fishing expansion for the Ephemeral Box!" },
    },

    HARVESTING =
    {
        { "Hay Bale",             6590,   300 },
        { "Special Brew",         9434,   800 },
        { "Hoe",                 20909,  1800 },
        { "Plain Gloves",        25984,  3000 },
        { "Harv. Sun Hat",       25557,  5000 },
    },

    EXCAVATION =
    {
        { "Museum Case",          6591,   300 },
        { "Special Brew",         9434,   800 },
        { "Caver's Shovel",      18600,  1800 },
        { "Plain Tunica",        26533,  3000 },
        { "Excavator's Shades",  25558,  5000 },
    },

    LOGGING =
    {
        { "Scrap Lumber",         6592,   300 },
        { "Special Brew",         9434,   800 },
        { "Dullahan Axe",        21745,  1800 },
        { "Plain Boots",         25964,  3000 },
        { "Lumberjack's Beret",  25559,  5000 },
    },

    MINING =
    {
        { "Box of Rocks",         6593,   300 },
        { "Special Brew",         9434,   800 },
        { "Worm Feelers",        20532,  1800 },
        { "Plain Hose",          25897,  3000 },
        { "Miner's Helmet",      25560,  5000 },
    },
}

-----------------------------------
-- Util functions
-----------------------------------
local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local qtyList  =
{
    {  "x1",  1 },
    {  "x2",  2 },
    {  "x3",  3 },
    {  "x6",  6 },
    { "x12", 12 },
    { "x24", 24 },
    { "x36", 36 },
    { "x99", 99 },
}

local function quantityPage(player, npc, item, varName)
    local tbl     = {}
    local balance = player:getCharVar(varName)

    for _, qtyInfo in pairs(qtyList) do
        local label = qtyInfo[1]
        local qty   = qtyInfo[2]

        if qty * item[3] <= balance then
            table.insert(tbl, {
                fmt("{} ({})", label, item[3] * qty),
                function()
                    player:timer(500, function()
                        if npcUtil.giveItem(player, { { item[2][1], qty } }) then
                            player:incrementCharVar(varName, -(item[3] * qty))

                            if (balance - item[3]) > 0 then
                                quantityPage(player, npc, item, varName)
                            end
                        end
                    end)
                end,
            })
        end
    end

    delaySendMenu(player, {
        title   = fmt("Purchase {} ({}):", item[1], balance),
        options = tbl,
    })
end

-----------------------------------
-- Menu functions
-----------------------------------
local function confirmPurchase(player, npc, item, param)
    local varName = cexi.ventures[param.ventureType].pointsVar

    if type(item[2]) == "table" then
        quantityPage(player, npc, item, varName)
        return
    end

    local balance     = player:getCharVar(varName)
    local itemName    = item[1]
    local itemID      = item[2]
    local itemCost    = item[3]

    npc:facePlayer(player, true)

    delaySendMenu(player, {
        title   = string.format("Spend %u points? (%u available)", itemCost, balance),
        options =
        {
            {
                "No way",
                function()
                end,
            },
            {
                string.format("Purchase: %s", itemName),
                function()
                    if balance >= itemCost and itemID ~= nil then
                        if type(itemID) == "number" then
                            if npcUtil.giveItem(player, itemID) then
                                player:setCharVar(cexi.ventures[param.ventureType].pointsVar, balance - itemCost)
                            end
                        else
                            player:setCharVar(itemID, 1)
                            player:setCharVar(cexi.ventures[param.ventureType].pointsVar, balance - itemCost)
                            player:sys(item[4])
                        end
                    else
                        cexi.util.dialog(player, settings.dialog.DECLINE, settings.name, { npc = npc })
                    end
                end,
            },
        },
    })
end

local function getHelmList(info)
    local result = {}
    local name   = string.gsub(info[1], "_", " ")

    table.insert(result, name)

    for _, itemInfo in pairs(info[2]) do
        table.insert(result, itemInfo[2])
    end

    return result
end

local function ventureRedeem(player, npc, typeSelected)
    local ventureName = string.lower(typeSelected)
    local rewardName  = string.upper(typeSelected)
    local balance     = player:getCharVar(cexi.ventures[ventureName].pointsVar)
    local title       = string.format("%s Venture Points (%u):", typeSelected, balance)

    if player:isCrystalWarrior() then
        cexi.util.simpleShop(player, npc, cwRewards[rewardName], confirmPurchase, title, 0, { ventureType = ventureName })
    else
        cexi.util.simpleShop(player, npc, rewards[rewardName], confirmPurchase, title, 0, { ventureType = ventureName })
    end
end

local function getVentureObjective(typeSelected, rangeIndex)
    local ventureType  = string.lower(typeSelected)
    local ventureIndex = GetServerVariable(string.format(cexi.ventures[ventureType].rollVar, rangeIndex))

    if ventureType == "dynamis" then
        local target = cexi.ventures.dynamis.ventures[rangeIndex][ventureIndex]
        return
        {
            index   = target.index,
            qty     = target.qty,
            message =
            {
                fmt("Hmm... For Dynamis Cities, we need {} eligible mobs in {}.", target.qty, string.gsub(target.area, "_", " ")),
            },
        }

    elseif ventureType == "fishing" then
        local target = cexi.ventures.fishing.ventures[rangeIndex].list[ventureIndex]
        return
        {
            index   = target[4],
            qty     = target[3],
            message =
            {
                fmt("Hmm... for Fishing Venture (#{}), we're fishin' for {} ({}) x{} (Level {}+).", target[4], target[2], target[5], target[3], cexi.ventures.fishing.ventures[rangeIndex].lvl),
            },
        }

    elseif
        ventureType == "harvesting" or
        ventureType == "excavation" or
        ventureType == "logging" or
        ventureType == "mining"
    then
        local target = cexi.ventures[ventureType].ventures[rangeIndex]
        local list   = getHelmList(target.list[ventureIndex])
        return
        {
            index   = target.list[ventureIndex][3],
            qty     = target.qty,
            message =
            {
                fmt("Hmm... for {} Venture (#{}), we're in {} lookin' for {}, {},", typeSelected, target.list[ventureIndex][3], list[1], list[2], list[3]),
                fmt(" {}, {} and {}.", list[4], list[5], list[6]),
            },
        }
    end
end

local function getVentureInfo(ventureType, index)
    if ventureType == "dynamis" then
        local venture = cexi.ventures.dynamis.mapping[index]
        local info    = cexi.ventures.dynamis.ventures[venture[1]][venture[2]]

        return {
            name = info.area,
            qty  = info.qty,
            pts  = info.pts,
            exp  = info.exp,
        }

    elseif ventureType == "fishing" then
        local venture = cexi.ventures.fishing.mapping[index]
        local info    = cexi.ventures.fishing.ventures[venture[1]].list[venture[2]]

        return {
            name = info[2],
            qty  = info[3],
            pts  = cexi.ventures.fishing.ventures[venture[1]].pts,
            exp  = cexi.ventures.fishing.ventures[venture[1]].exp,
        }

    elseif
        ventureType == "harvesting" or
        ventureType == "excavation" or
        ventureType == "logging" or
        ventureType == "mining"
    then
        local venture = cexi.ventures[ventureType].mapping[index]

        if venture == nil then
            return nil
        end

        local info    = cexi.ventures[ventureType].ventures[venture[1]]
        local list    = getHelmList(info.list[venture[2]])

        return {
            name = info.list[venture[2]][1],
            qty  = info.qty,
            pts  = info.pts,
            exp  = info.exp,
            list = list,
        }
    end
end

local function ventureCancel(player, npc, typeSelected)
    npc:facePlayer(player, true)

    local ventureName = string.lower(typeSelected)
    local index       = player:getCharVar(cexi.ventures[ventureName].lockVar)
    local total       = player:getCharVar(cexi.ventures[ventureName].progVar)
    local venture     = getVentureInfo(ventureName, index)

    if venture == nil then
        player:setCharVar(cexi.ventures[ventureName].lockVar, 0)
        player:setCharVar(cexi.ventures[ventureName].progVar, 0)
        player:sys("That venture no longer exists and has been removed.")
        return
    end

    delaySendMenu(player, {
        title   = string.format("Give up? (%u/%u)", total, venture.qty),
        options =
        {
            {
                "No way!",
                function()
                    local delay = 0

                    if
                        ventureName == "harvesting" or
                        ventureName == "excavation" or
                        ventureName == "logging" or
                        ventureName == "mining"
                    then
                        local message =
                        {
                            fmt("Jus' as a reminder, today we're in {} and we're looking for {}, {},", venture.list[1], venture.list[2], venture.list[3]),
                            fmt(" {}, {} and {}.", venture.list[4], venture.list[5], venture.list[6]),
                        }

                        cexi.util.dialog(player, message, settings.name, { npc = npc })
                        delay = cexi.util.dialogDelay(message)
                    else
                        cexi.util.dialog(player, settings.dialog.REMINDER, settings.name, { npc = npc, [1] = string.gsub(venture.name, "_", " ") })
                        delay = cexi.util.dialogDelay(settings.dialog.REMINDER)
                    end

                    player:setLocalVar("[NPC]BUSY", 1)
                    player:timer(delay, function()
                        player:setLocalVar("[NPC]BUSY", 0)
                    end)
                end,
            },
            {
                "I'm out!",
                function()
                    if total < 2 then
                        if cexi.hasSetting(player, cexi.setting.SKIP_VENTURE_RP) then
                            player:printToPlayer(string.format("%s Venture Cancelled! %s", "\129\166", "\129\166"), xi.msg.channel.SYSTEM_3)
                        else
                            local delay = cexi.util.dialogDelay(settings.dialog.INSUFFICIENT)
                            cexi.util.dialog(player, settings.dialog.INSUFFICIENT, settings.name, { npc = npc })

                            player:setLocalVar("[NPC]BUSY", 1)
                            player:timer(delay, function()
                                player:setLocalVar("[NPC]BUSY", 0)
                                player:printToPlayer(string.format("%s Venture Cancelled! %s", "\129\166", "\129\166"), xi.msg.channel.SYSTEM_3)
                            end)
                        end
                    else
                        local ptsRange  = math.random(venture.pts[1], venture.pts[2])
                        local rewardPts = math.floor(((total / venture.qty) * ptsRange) * 0.75 + 1) -- (Half of % progress value)
                        local rewardExp = math.floor(((total / venture.qty) * venture.exp) * 0.75 + 50)

                        player:incrementCharVar(cexi.ventures[ventureName].pointsVar, rewardPts)

                        if cexi.hasSetting(player, cexi.setting.SKIP_VENTURE_RP) then
                            player:printToPlayer(string.format("%s Venture Cancelled! %s", "\129\166", "\129\166"), xi.msg.channel.SYSTEM_3)
                            player:printToPlayer(string.format("%s gains %u %s venture points.", player:getName(), rewardPts, string.lower(ventureName)), xi.msg.channel.SYSTEM_3)
                            player:addExp(rewardExp)
                        else
                            local delay = cexi.util.dialogDelay(settings.dialog.RETIRE)
                            cexi.util.dialog(player, settings.dialog.RETIRE, settings.name, { npc = npc })

                            player:setLocalVar("[NPC]BUSY", 1)
                            player:timer(delay, function()
                                player:setLocalVar("[NPC]BUSY", 0)
                                player:printToPlayer(string.format("%s Venture Cancelled! %s", "\129\166", "\129\166"), xi.msg.channel.SYSTEM_3)
                                player:printToPlayer(string.format("%s gains %u %s venture points.", player:getName(), rewardPts, string.lower(ventureName)), xi.msg.channel.SYSTEM_3)
                                player:addExp(rewardExp)
                            end)
                        end
                    end

                    player:setCharVar(cexi.ventures[ventureName].progVar, 0)
                    player:setCharVar(cexi.ventures[ventureName].lockVar, 0)
                end,
            },
        },
    })
end

local function ventureRange(player, npc, selected, params)
    local ventureName  = string.lower(params.typeSelected)
    local objective    = getVentureObjective(params.typeSelected, selected[2])
    local delay        = cexi.util.dialogDelay(objective.message)

    cexi.util.dialog(player, objective.message, settings.name, { npc = npc })

    npc:facePlayer(player, true)
    player:setLocalVar("[NPC]BUSY", 1)

    player:timer(delay, function()
        player:setLocalVar("[NPC]BUSY", 0)
        delaySendMenu(player, {
            title   = "Start this venture?",
            options =
            {
                {
                    "Not yet.",
                    function()
                    end,
                },
                {
                    "Let's go!",
                    function()
                        if cexi.hasSetting(player, cexi.setting.SKIP_VENTURE_RP) then
                            player:sys("\129\153 {} Venture #{} Accepted \129\153", params.typeSelected, objective.index)
                            player:setCharVar(cexi.ventures[ventureName].lockVar, objective.index)
                        else
                            local starting = { string.format(settings.dialog.STARTING[1], objective.qty) }
                            local delay    = cexi.util.dialogDelay(starting)

                            cexi.util.dialog(player, starting, settings.name, { npc = npc })

                            npc:facePlayer(player, true)

                            player:timer(delay, function()
                                player:sys("\129\153 {} Venture #{} Accepted \129\153", params.typeSelected, objective.index)
                                player:setCharVar(cexi.ventures[ventureName].lockVar, objective.index)
                            end)
                        end
                    end,
                }
            },
        })
    end)
end

local function specialVentureMenu(player, npc, typeSelected)
    local ventureName = string.lower(typeSelected)
    cexi.util.simpleMenu(player, npc, cexi.ventures[ventureName].categories, ventureRange, "Select a tier:", nil, { typeSelected = typeSelected })
end

local missedItems =
{
    [cexi.ventures.dynamis.missVar] =
    {
        1452,
        1455,
        1449,
    },

    [cexi.ventures.fishing.missVar] =
    {
        5110, -- tiny_tacklebox
        5112, -- titanic_tacklebox
    },

    [cexi.ventures.harvesting.missVar] = { 6345 },-- goblin_gatherbox
    [cexi.ventures.excavation.missVar] = { 6345 },-- goblin_gatherbox
    [cexi.ventures.logging.missVar]    = { 6345 },-- goblin_gatherbox
    [cexi.ventures.mining.missVar]     = { 6345 },-- goblin_gatherbox
}

local types = { "Fishing", "Harvesting", "Excavation", "Logging", "Mining" }

local function ventureRedeemSelect(player, npc)
    local options =
    {
        {
            "Nothing",
            function()
            end,
        },
    }

    if player:isCrystalWarrior() then
        table.insert(options, {
            "Dynamis Points",
            function()
                ventureRedeem(player, npc, "Dynamis")
            end,
        })
    end

    for _, ventureType in pairs(types) do
        table.insert(options, {
            string.format("%s Points", ventureType),
            function()
                ventureRedeem(player, npc, ventureType)
            end,
        })
    end

    delaySendMenu(player, {
        title   = "Select Type:",
        options = options,
    })
end

local function getVentureMenu(player, npc, ventureType)
    local ventureName = string.lower(ventureType)

    if player:getCharVar(cexi.ventures[ventureName].lockVar) > 0 then
        return {
            string.format("Cancel %s Venture", ventureType),
            function()
                ventureCancel(player, npc, ventureType)
            end,
        }
    else
        return {
            string.format("Begin %s Venture", ventureType),
            function()
                specialVentureMenu(player, npc, ventureType)
            end,
        }
    end
end

local function mainMenu(player, npc, page)
    local options = {}

    if page == 0 then
        table.insert(options, {
            "Nothing",
            function()
            end,
        })

        table.insert(options, {
            string.format("Spend Venture Points"),
            function()
                ventureRedeemSelect(player, npc)
            end,
        })

        if player:isCrystalWarrior() then
            table.insert(options, getVentureMenu(player, npc, "Dynamis"))
        end

        table.insert(options, getVentureMenu(player, npc, "Fishing"))

        table.insert(options, {
            "(Next)",
            function()
                mainMenu(player, npc, 1)
            end,
        })
    else
        table.insert(options, {
            "(Prev)",
            function()
                mainMenu(player, npc, 0)
            end,
        })

        table.insert(options, getVentureMenu(player, npc, "Harvesting"))
        table.insert(options, getVentureMenu(player, npc, "Excavation"))
        table.insert(options, getVentureMenu(player, npc, "Logging"))
        table.insert(options, getVentureMenu(player, npc, "Mining"))
    end

    delaySendMenu(player, {
        title   = "Here for a Special Venture?",
        options = options,
    })
end

local function onTrigger(player, npc)
    if player:getLocalVar("[NPC]BUSY") == 1 then
        return
    end

    npc:facePlayer(player, true)

    for missedType, itemBoxes in pairs(missedItems) do
        for _, itemID in pairs(itemBoxes) do
            local varName = string.format(missedType, itemID)
            local missed  = player:getCharVar(varName)

            if missed > 0 then
                cexi.util.dialog(player, settings.dialog.MISSED, settings.name, { npc = npc })

                if npcUtil.giveItem(player, { { itemID, missed } }) then
                    player:setCharVar(varName, 0)
                end

                return
            end
        end
    end

    mainMenu(player, npc, 0)
end

local function onTrade(player, npc, trade)
    for _, item in pairs(cexi.ventures.upgrades) do
        if cexi.util.tradeHasExactly(trade, item.nq) then
            local tbl =
            {
                string.format(settings.dialog.UPGRADE[1], item.name),
                string.format(settings.dialog.UPGRADE[2], item.cost.qty, item.cost.name, item.item.name),
            }

            npc:facePlayer(player, true)
            cexi.util.dialog(player, tbl, settings.name, { npc = npc })

            return
        end

        if npcUtil.tradeHasExactly(trade, {
            { item.nq, 1 },
            { item.item.id, 1 },
            { item.cost.id, item.cost.qty },
        }) then
            npc:facePlayer(player, true)
            cexi.util.dialog(player, settings.dialog.UPGRADED, settings.name, { npc = npc })

            if npcUtil.giveItem(player, item.hq) then
                player:tradeComplete()
            end
        end
    end
end

cexi.util.liveReload(m, {
    [settings.area] =
    {
        {
            name      = settings.name,
            objtype   = xi.objType.NPC,
            look      = settings.look,
            x         = settings.pos[1],
            y         = settings.pos[2],
            z         = settings.pos[3],
            rotation  = settings.pos[4],
            onTrigger = onTrigger,
            onTrade   = onTrade,
        },
    },
})

return m
