-----------------------------------
-- Ventures - Core
-----------------------------------
require("modules/module_utils")
require("scripts/globals/helm")
-----------------------------------
local m = Module:new("ventures_core")

local settings =
{
    resetVar = "[VENTURE]RESET",
    respawn  = 120, -- 2 minutes
}

m:addOverride("xi.zones.RuLude_Gardens.Zone.onGameHour", function(zone)
    super(zone)

    local nextReset = GetServerVariable(settings.resetVar)

    if
        nextReset == 0 or -- If var doesn't exist yet
        os.time() > nextReset
    then
        for rangeIndex, rangeVentures in pairs(cexi.ventures.exp.ventures) do
            local varName = string.format(cexi.ventures.exp.rollVar, rangeIndex)
            local current = GetServerVariable(varName)
            local roll    = cexi.util.randomNoRepeat(#rangeVentures, current)
            local index   = rangeVentures[roll].index

            SetServerVariable(varName, roll)
            print(string.format("%s %s #%u (%s)", varName, cexi.ventures.exp.categories[rangeIndex][1], index, rangeVentures[roll].area))
        end

        for rangeIndex, rangeVentures in pairs(cexi.ventures.dynamis.ventures) do
            local varName = string.format(cexi.ventures.dynamis.rollVar, rangeIndex)
            local current = GetServerVariable(varName)
            local roll    = cexi.util.randomNoRepeat(#rangeVentures, current)
            local index   = rangeVentures[roll].index

            SetServerVariable(varName, roll)
            print(string.format("%s #%u (%s)", varName, index, rangeVentures[roll].area))
        end

        for _, helmType in pairs(cexi.ventures.helm) do
            for rangeIndex, rangeVentures in pairs(cexi.ventures[helmType].ventures) do
                local varName = string.format(cexi.ventures[helmType].rollVar, rangeIndex)
                local current = GetServerVariable(varName)
                local roll    = cexi.util.randomNoRepeat(#rangeVentures.list, current)
                local index   = rangeVentures.list[roll][3]

                SetServerVariable(varName, roll)
                print(string.format("%s #%u (%s)", varName, index, rangeVentures.list[roll][1]))
            end
        end

        for rangeIndex, rangeVentures in pairs(cexi.ventures.fishing.ventures) do
            local varName = string.format(cexi.ventures.fishing.rollVar, rangeIndex)
            local current = GetServerVariable(varName)
            local roll    = cexi.util.randomNoRepeat(#rangeVentures.list, current)
            local index   = rangeVentures.list[roll][4]

            SetServerVariable(varName, roll)
            print(string.format("%s %s #%u (%s)", varName, rangeVentures.name, index, rangeVentures.list[roll][2]))
        end

        SetServerVariable(settings.resetVar, JstMidnight())
        print(string.format("%s: New Ventures have been selected.", settings.resetVar))
    end
end)

-----------------------------------
-- EXP Ventures
-----------------------------------
for rangeIndex, rangeInfo in pairs(cexi.ventures.exp.ventures) do
    for locIndex, locInfo in pairs(rangeInfo) do
        for _, mobName in pairs(locInfo.mobs) do
            cexi.util.ensureMob(locInfo.area, mobName)

            m:addOverride(string.format("xi.zones.%s.mobs.%s.onMobDeath", locInfo.area, mobName), function(mob, player, optParams)
                super(mob, player, optParams)

                -- Store the default respawn value to preserve it
                -- Restore default when killed for non-venture purposes
                if xi.zones[locInfo.area].mobs[mobName].respawn == nil then
                    xi.zones[locInfo.area].mobs[mobName].respawn = math.floor(mob:getRespawnTime() / 1000)
                end

                if
                    player and
                    player:isPC() and
                    player:getCharVar(cexi.ventures.exp.lockVar) == locInfo.index
                then
                    -- Mob was a venture target, use reduced respawn time
                    if settings.respawn < xi.zones[locInfo.area].mobs[mobName].respawn then
                        mob:setRespawnTime(settings.respawn)
                    end

                    local levelDiff    = 3
                    local highestLevel = player:partyHighestLevel()

                    if highestLevel < 30 then
                        levelDiff = 2
                    end

                    -- player is on this venture, check if they are eligible
                    if player:partyHasTrusts() then
                        player:printToPlayer(fmt("Trusts are not allowed to get venture credit."), xi.msg.channel.SYSTEM_3)
                    elseif player:partyHighestLevel() > mob:getMainLvl() - levelDiff then
                        player:printToPlayer(fmt("Venture mobs must be at least {} levels above your party level to get venture credit.", levelDiff), xi.msg.channel.SYSTEM_3)
                    elseif player:checkSoloPartyAlliance() == 2 then
                        player:printToPlayer(fmt("Alliances are not allowed for venture credit."), xi.msg.channel.SYSTEM_3)
                    else
                        local expired = GetServerVariable(string.format(cexi.ventures.exp.rollVar, rangeIndex)) ~= locIndex

                        player:timer(500, function()
                            cexi.ventures.updateVenture(player, locInfo.index, expired, locInfo, 1, locInfo.qty, cexi.ventures.exp, locInfo.box)
                        end)
                    end
                else
                    -- Mob was not a venture target, default respawn time
                    mob:setRespawnTime(xi.zones[locInfo.area].mobs[mobName].respawn)
                end
            end)
        end
    end
end

-----------------------------------
-- Dynamis Ventures
-----------------------------------
for categoryIndex, categoryInfo in pairs(cexi.ventures.dynamis.ventures) do
    for locIndex, locInfo in pairs(categoryInfo) do
        for _, mobName in pairs(locInfo.mobs) do
            cexi.util.ensureMob(locInfo.area, mobName)

            m:addOverride(string.format("xi.zones.%s.mobs.%s.onMobDeath", locInfo.area, mobName), function(mob, player, optParams)
                if
                    player == nil or
                    not player:isPC()
                then
                    super(mob, player, optParams)
                    return
                end

                if player:getCharVar(cexi.ventures.dynamis.lockVar) == locInfo.index then
                    local expired = GetServerVariable(string.format(cexi.ventures.dynamis.rollVar, categoryIndex)) ~= locIndex
                    cexi.ventures.updateVenture(player, locInfo.index, expired, locInfo, 1, locInfo.qty, cexi.ventures.dynamis, locInfo.box)
                end

                super(mob, player, optParams)
            end)
        end
    end
end

-----------------------------------
-- Fishing Ventures
-----------------------------------
m:addOverride("xi.player.onPlayerFishingResult", function(player, catchType, itemID, itemQty, weight)
    super(player, catchType, itemID, itemQty, weight)

    if itemID == 0 then
        return
    end

    for tierIndex, tierInfo in pairs(cexi.ventures.fishing.ventures) do
        for itemIndex, itemInfo in pairs(tierInfo.list) do
            if
                itemID == itemInfo[1] and
                player:getCharVar(cexi.ventures.fishing.lockVar) == itemInfo[4]
            then
                local level = player:getMainLvl()

                if level < tierInfo.lvl then
                    player:timer(1000, function()
                        player:sys("Your level ({}) is too low to complete the current Venture (Level {}+).", level, tierInfo.lvl)
                    end)

                    return
                end

                local qty     = itemInfo[3]
                local expired = GetServerVariable(string.format(cexi.ventures.fishing.rollVar, tierIndex)) ~= itemIndex

                player:timer(1000, function()
                    cexi.ventures.updateVenture(player, itemInfo[4], expired, tierInfo, 1, qty, cexi.ventures.fishing, tierInfo.box)
                end)

                return
            end
        end
    end
end)

-----------------------------------
-- HELM Ventures
-----------------------------------
m:addOverride("xi.helm.result", function(player, helmType, broke, itemID)
    super(player, helmType, broke, itemID)
    cexi.ventures.onHelmResult(player, helmType, broke, itemID)
end)

m:addOverride("xi.player.onPlayerMount", function(player)
    super(player)
    player:setLocalVar("Mounted", os.time() + 120)
end)

return m
