-----------------------------------
-- Vision - AF+1 Augment System
-----------------------------------
-- Developed by Carver (2022)
-- Rewritten by Loxley (2024)
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("catseyexi_custom_af_augments")

local cost =
{
    [xi.slot.HEAD]  = { xi.item.HIGH_QUALITY_AERN_ORGAN,   xi.item.HIGH_QUALITY_XZOMIT_ORGAN },
    [xi.slot.BODY]  = { xi.item.HIGH_QUALITY_HPEMDE_ORGAN, xi.item.HIGH_QUALITY_PHUABO_ORGAN },
    [xi.slot.HANDS] = { xi.item.HIGH_QUALITY_HPEMDE_ORGAN, xi.item.HIGH_QUALITY_XZOMIT_ORGAN },
    [xi.slot.LEGS]  = { xi.item.HIGH_QUALITY_PHUABO_ORGAN, xi.item.HIGH_QUALITY_XZOMIT_ORGAN },
    [xi.slot.FEET]  = { xi.item.HIGH_QUALITY_AERN_ORGAN,   xi.item.HIGH_QUALITY_HPEMDE_ORGAN },
}

for slotID, slotItems in pairs(cost) do
    local result = {}

    for tier = 1, 3 do
        local row = {}
        table.insert(row, { xi.item.AERN_ORGAN,         1 * tier })
        table.insert(row, { xi.item.HPEMDE_ORGAN,       1 * tier })
        table.insert(row, { xi.item.PHUABO_ORGAN,       1 * tier })
        table.insert(row, { xi.item.XZOMIT_ORGAN,       1 * tier })
        table.insert(row, { xi.item.LUMINIAN_TISSUE,    2 * tier })

        table.insert(row, { slotItems[1],               1 * tier })
        table.insert(row, { slotItems[2],               1 * tier })
        table.insert(row, { xi.item.ANCIENT_BEASTCOIN, 10 * tier })

        table.insert(result, row)
    end

    cost[slotID] = result
end

-- Make this very long name shorter for our table
xi.item.DRACHEN_GAUNTLETS_P1 = xi.item.DRACHEN_FINGER_GAUNTLETS_P1

local augments =
{
     -- WAR
    [xi.item.FIGHTERS_MASK_P1]        = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.FIGHTERS_LORICA_P1]      = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.ENMITY]             = {   3,   4,   5 }, [cexi.augment_id.PHYS_DMG_TAKEN]      = {   0,   1,   2 } } },
    [xi.item.FIGHTERS_MUFFLERS_P1]    = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.ENMITY]             = {   3,   4,   5 }, [cexi.augment_id.PHYS_DMG_TAKEN]      = {   0,   1,   2 } } },
    [xi.item.FIGHTERS_CUISSES_P1]     = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.ENMITY]             = {   3,   4,   5 }, [cexi.augment_id.PHYS_DMG_TAKEN]      = {   0,   1,   2 } } },
    [xi.item.FIGHTERS_CALLIGAE_P1]    = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.STORE_TP]            = {   3,   4,   5 } } },

     -- MNK
    [xi.item.TEMPLE_CROWN_P1]         = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.TEMPLE_CYCLAS_P1]        = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MARTIAL_ARTS]       = { nil,   0,   1 }, [cexi.augment_id.STORE_TP]            = {   2,   3,   4 } } },
    [xi.item.TEMPLE_GLOVES_P1]        = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.TEMPLE_HOSE_P1]          = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.VIT]                = {   8,   9,  10 }, [cexi.augment_id.STR]                 = {   8,   9,  10 } } },
    [xi.item.TEMPLE_GAITERS_P1]       = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.KICK_ATTACK_RATE]   = {   2,   3,   4 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

     -- WHM
    [xi.item.HEALERS_CAP_P1]          = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },
    [xi.item.HEALERS_BRIAULT_P1]      = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MP]                 = {  11,  13,  17 }, [cexi.augment_id.DAMAGE_TAKEN]        = {   2,   3,   4 } } },
    [xi.item.HEALERS_MITTS_P1]        = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.CURE_POTENCY]       = {   0,   1,   2 }, [cexi.augment_id.CONSERVE_MP]         = {   2,   3,   4 } } },
    [xi.item.HEALERS_PANTALOONS_P1]   = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.MAGIC_ACCURACY]     = {   2,   4,   5 }, [cexi.augment_id.CURE_CAST_TIME]      = {   0,   1,   2 } } },
    [xi.item.HEALERS_DUCKBILLS_P1]    = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MP_33]              = {   0,   3,   6 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },

     -- BLM
    [xi.item.WIZARDS_PETASOS_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },
    [xi.item.WIZARDS_COAT_P1]         = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MP]                 = {  11,  14,  17 }, [cexi.augment_id.MAGIC_ACCURACY]      = {   2,   3,   4 } } },
    [xi.item.WIZARDS_GLOVES_P1]       = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.MAGIC_ATK_BONUS]    = {   2,   3,   4 }, [cexi.augment_id.CONSERVE_MP]         = {   2,   3,   4 } } },
    [xi.item.WIZARDS_TONBAN_P1]       = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.MAGIC_ACCURACY]     = {   2,   3,   4 }, [cexi.augment_id.DRAIN_ASPIR_POTENCY] = {   2,   3,   4 } } },
    [xi.item.WIZARDS_SABOTS_P1]       = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MP_33]              = {   0,   3,   6 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },

     -- RDM
    [xi.item.WARLOCKS_CHAPEAU_P1]     = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.MAGIC_ATK_BONUS]    = {   3,   4,   5 }, [cexi.augment_id.CONSERVE_MP]         = {   2,   3,   4 } } },
    [xi.item.WARLOCKS_TABARD_P1]      = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MP]                 = {  14,  17,  20 }, [cexi.augment_id.MAGIC_ACCURACY]      = {   2,   3,   4 } } },
    [xi.item.WARLOCKS_GLOVES_P1]      = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.WARLOCKS_TIGHTS_P1]      = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.MP_33]              = {   1,   2,   3 }, [cexi.augment_id.CURE_POTENCY]        = {   3,   4,   5 } } },
    [xi.item.WARLOCKS_BOOTS_P1]       = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MP_33]              = {   0,   3,   6 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },

     -- THF
    [xi.item.ROGUES_BONNET_P1]        = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.ROGUES_VEST_P1]          = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.REGEN]              = { nil,   0,   1 }, [cexi.augment_id.DUAL_WIELD]          = {   0,   1,   2 } } },
    [xi.item.ROGUES_ARMLETS_P1]       = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.TREASURE_HUNTER]    = { nil, nil,   1 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.ROGUES_CULOTTES_P1]      = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.RANGED_ACCURACY]    = {   5,   6,   7 }, [cexi.augment_id.RANGED_ATTACK]       = {   5,   6,   7 } } },
    [xi.item.ROGUES_POULAINES_P1]     = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

     -- PLD
    [xi.item.GALLANT_CORONET_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.DAMAGE_TAKEN]       = {   0,   1,   2 }, [cexi.augment_id.CURE_POTENCY_RCVD]   = {   2,   3,   4 } } },
    [xi.item.GALLANT_SURCOAT_P1]      = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.BLOCK_CHANCE]        = {   0,   1,   2 } } },
    [xi.item.GALLANT_GAUNTLETS_P1]    = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.MAGIC_DMG_TAKEN]    = {   2,   3,   4 }, [cexi.augment_id.ENEMY_CRIT_RATE]     = {   0,   1,   2 } } },
    [xi.item.GALLANT_BREECHES_P1]     = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.SHIELD_MASTERY]     = { nil,   0,   1 }, [cexi.augment_id.SHIELD_SKILL]        = {   2,   3,   4 } } },
    [xi.item.GALLANT_LEGGINGS_P1]     = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.BREATH_DMG_TAKEN]   = {   2,   3,   4 }, [cexi.augment_id.MAGIC_DEF_BONUS]     = {   0,   1,   2 } } },

     -- DRK
    [xi.item.CHAOS_BURGEONET_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.CHAOS_CUIRASS_P1]        = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.REGEN]               = {   0,   1,   2 } } },
    [xi.item.CHAOS_GAUNTLETS_P1]      = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.STORE_TP]           = {   2,   3,   4 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.CHAOS_FLANCHARD_P1]      = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.INT]                = {   2,   3,   4 }, [cexi.augment_id.DRAIN_ASPIR_POTENCY] = {   2,   3,   4 } } },
    [xi.item.CHAOS_SOLLERETS_P1]      = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

     -- BST
    [xi.item.BEAST_HELM_P1]           = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.PET_HASTE]          = {   2,   3,   4 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.BEAST_JACKCOAT_P1]       = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.PET_DAMAGE_TAKEN]   = {   2,   3,   4 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.BEAST_GLOVES_P1]         = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.BEAST_TROUSERS_P1]       = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.PET_DA_CRIT_RATE]   = {   1,   2,   3 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.BEAST_GAITERS_P1]        = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.PET_REGEN]          = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },

    -- BRD
    [xi.item.CHORAL_ROUNDLET_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.REGEN]               = {   0,   1,   2 } } },
    [xi.item.CHORAL_JUSTAUCORPS_P1]   = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.SINGING_SKILL]      = {   2,   3,   4 }, [cexi.augment_id.WIND_SKILL]          = {   3,   4,   5 } } },
    [xi.item.CHORAL_CUFFS_P1]         = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.CHR]                = {   0,   1,   2 }, [cexi.augment_id.MAGIC_ACCURACY]      = {   0,   1,   2 } } },
    [xi.item.CHORAL_CANNIONS_P1]      = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.SONG_RECAST]        = {   0,   1,   2 }, [cexi.augment_id.EARTH_MAG_ACCURACY]  = {   2,   3,   4 } } },
    [xi.item.CHORAL_SLIPPERS_P1]      = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.HHP]                = {   0,   1,   2 }, [cexi.augment_id.HMP]                 = {   2,   3,   4 } } },

    -- RNG
    [xi.item.HUNTERS_BERET_P1]        = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.RAPID_SHOT]         = {   0,   1,   2 }, [cexi.augment_id.RANGED_ACCURACY]     = {   2,   3,   4 } } },
    [xi.item.HUNTERS_JERKIN_P1]       = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.ACCURACY]           = {   7,   8,   9 }, [cexi.augment_id.STORE_TP]            = {   5,   6,   7 } } },
    [xi.item.HUNTERS_BRACERS_P1]      = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.RANGED_ACCURACY]    = {   3,   4,   5 }, [cexi.augment_id.BARRAGE]             = { nil,   0,   1 } } },
    [xi.item.HUNTERS_BRACCAE_P1]      = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.DEX]                = {   8,   9,  10 }, [cexi.augment_id.AGI]                 = {   8,   9,  10 } } },
    [xi.item.HUNTERS_SOCKS_P1]        = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },

    -- SAM
    [xi.item.MYOCHIN_KABUTO_P1]       = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.MYOCHIN_DOMARU_P1]       = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MEDITATE_DURATION]  = { nil, nil,   0 }, [cexi.augment_id.REGEN]               = {   0,   1,   2 } } },
    [xi.item.MYOCHIN_KOTE_P1]         = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.ZANSHIN]            = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.MYOCHIN_HAIDATE_P1]      = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.STORE_TP]           = {   2,   3,   4 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.MYOCHIN_SUNE_ATE_P1]     = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

    -- NIN
    [xi.item.NINJA_HATSUBURI_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.NINJA_CHAINMAIL_P1]      = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.DUAL_WIELD]         = {   0,   1,   2 }, [cexi.augment_id.ACCURACY]            = {   5,   6,   7 } } },
    [xi.item.NINJA_TEKKO_P1]          = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.NINJA_HAKAMA_P1]         = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.EVASION]            = {   2,   3,   4 }, [cexi.augment_id.NIN_TOOL_EXPERTISE]  = {   0,   1,   2 } } },
    [xi.item.NINJA_KYAHAN_P1]         = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MAGIC_ATK_BONUS]    = {   7,   8,   9 }, [cexi.augment_id.NINJUTSU_SKILL]      = {   2,   3,   4 } } },

    -- DRG
    [xi.item.DRACHEN_ARMET_P1]        = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.DRACHEN_MAIL_P1]         = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.ACCURACY]           = {   7,   8,   9 }, [cexi.augment_id.STORE_TP]            = {   5,   6,   7 } } },
    [xi.item.DRACHEN_GAUNTLETS_P1]    = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.DRACHEN_BRAIS_P1]        = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.PET_ACC_RNG_ACC]    = {  12,  13,  14 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.DRACHEN_GREAVES_P1]      = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.PET_REGEN]          = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

    -- SMN
    [xi.item.EVOKERS_HORN_P1]         = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.HMP]                 = {   1,   2,   3 } } },
    [xi.item.EVOKERS_DOUBLET_P1]      = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.PET_ACC_RNG_ACC]    = {   5,   6,   7 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },
    [xi.item.EVOKERS_BRACERS_P1]      = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.PET_DAMAGE_TAKEN]   = {   2,   3,   4 }, [cexi.augment_id.PET_HASTE]           = {   2,   3,   4 } } },
    [xi.item.EVOKERS_SPATS_P1]        = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.PET_ATK_RNG_ATK]    = {   0,   1,   2 }, [cexi.augment_id.PET_DOUBLE_ATTACK]   = {   0,   1,   2 } } },
    [xi.item.EVOKERS_PIGACHES_P1]     = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.PET_REGEN]          = {   0,   1,   2 }, [cexi.augment_id.ELEMENTAL_SIPHON]    = { nil, nil,   0 } } },

    -- BLU
    [xi.item.MAGUS_KEFFIYEH_P1]       = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.MAGUS_JUBBAH_P1]         = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.ACCURACY]           = {   7,   8,   9 }, [cexi.augment_id.DUAL_WIELD]          = {   0,   1,   2 } } },
    [xi.item.MAGUS_BAZUBANDS_P1]      = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.MAGUS_SHALWAR_P1]        = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.REGEN]               = {   0,   1,   2 } } },
    [xi.item.MAGUS_CHARUQS_P1]        = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MAGIC_ATK_BONUS]    = {   7,   8,   9 }, [cexi.augment_id.BLUE_MAGIC_SKILL]    = {   2,   3,   4 } } },

    -- COR
    [xi.item.CORSAIRS_TRICORNE_P1]    = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.RANGED_ACCURACY]    = {   3,   4,   5 }, [cexi.augment_id.REGEN]               = { nil,   0,   1 } } },
    [xi.item.CORSAIRS_FRAC_P1]        = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MARKSMANSHIP_SKILL] = {   0,   1,   2 }, [cexi.augment_id.AGI]                 = {   8,   9,  10 } } },
    [xi.item.CORSAIRS_GANTS_P1]       = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.CORSAIRS_CULOTTES_P1]    = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.ACCURACY]           = {   3,   4,   5 }, [cexi.augment_id.DUAL_WIELD]          = {   0,   1,   2 } } },
    [xi.item.CORSAIRS_BOTTES_P1]      = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.RAPID_SHOT]         = {   2,   3,   4 }, [cexi.augment_id.SNAPSHOT]            = {   0,   1,   2 } } },

    -- PUP
    [xi.item.PUPPETRY_TAJ_P1]         = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.PET_HASTE]          = {   2,   3,   4 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.PUPPETRY_TOBE_P1]        = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MARTIAL_ARTS]       = { nil,   0,   1 }, [cexi.augment_id.STORE_TP]            = {   2,   3,   4 } } },
    [xi.item.PUPPETRY_DASTANAS_P1]    = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.PUPPETRY_CHURIDARS_P1]   = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.PET_DA_CRIT_RATE]   = {   1,   2,   3 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.PUPPETRY_BABOUCHES_P1]   = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.PET_REGEN]          = {   0,   1,   2 }, [cexi.augment_id.REPAIR_POTENCY]      = {   0,   1,   2 } } },

    -- DNC M
    [xi.item.DANCERS_TIARA_M_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.DANCERS_CASAQUE_M_P1]    = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.WALTZ_DELAY]        = {   2,   3,   4 }, [cexi.augment_id.WALTZ_POTENCY]       = {   2,   3,   4 } } },
    [xi.item.DANCERS_BANGLES_M_P1]    = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DUAL_WIELD]         = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.DANCERS_TIGHTS_M_P1]     = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.STR]                = {   8,   9,  10 }, [cexi.augment_id.DEX]                 = {   8,   9,  10 } } },
    [xi.item.DANCERS_TOE_SHOES_M_P1]  = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

    -- DNC F
    [xi.item.DANCERS_TIARA_F_P1]      = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.CRIT_HIT_RATE]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   2,   3,   4 } } },
    [xi.item.DANCERS_CASAQUE_F_P1]    = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.WALTZ_DELAY]        = {   2,   3,   4 }, [cexi.augment_id.WALTZ_POTENCY]       = {   2,   3,   4 } } },
    [xi.item.DANCERS_BANGLES_F_P1]    = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.DUAL_WIELD]         = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },
    [xi.item.DANCERS_TIGHTS_F_P1]     = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.STR]                = {   8,   9,  10 }, [cexi.augment_id.DEX]                 = {   8,   9,  10 } } },
    [xi.item.DANCERS_TOE_SHOES_F_P1]  = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.DOUBLE_ATTACK]      = {   0,   1,   2 }, [cexi.augment_id.HASTE]               = {   0,   1,   2 } } },

    -- SCH
    [xi.item.SCHOLARS_MORTARBOARD_P1] = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.MAGIC_BURST_DMG]    = {   0,   1,   2 }, [cexi.augment_id.CONSERVE_MP]         = {   2,   3,   4 } } },
    [xi.item.SCHOLARS_GOWN_P1]        = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MAGIC_ACCURACY]     = {   2,   3,   4 }, [cexi.augment_id.CONSERVE_MP]         = {   2,   3,   4 } } },
    [xi.item.SCHOLARS_BRACERS_P1]     = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.HMP]                 = {   0,   1,   2 } } },
    [xi.item.SCHOLARS_PANTS_P1]       = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.CONSERVE_MP]        = {   2,   3,   4 }, [cexi.augment_id.CURE_POTENCY]        = {   3,   4,   5 } } },
    [xi.item.SCHOLARS_LOAFERS_P1]     = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MAGIC_ACCURACY]     = {   2,   3,   4 }, [cexi.augment_id.DRAIN_ASPIR_POTENCY] = {   2,   3,   4 } } },

    -- GEO
    [xi.item.GEOMANCY_GALERO]         = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.REFRESH]            = { nil, nil,   1 }, [cexi.augment_id.CONSERVE_MP]         = {   2,   3,   4 } } },
    [xi.item.GEOMANCY_TUNIC]          = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.DAMAGE_TAKEN]       = {   2,   3,   4 }, [cexi.augment_id.MAGIC_ACCURACY]      = {   2,   3,   4 } } },
    [xi.item.GEOMANCY_MITAINES]       = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.PET_DAMAGE_TAKEN]   = {   2,   3,   4 }, [cexi.augment_id.CONSERVE_MP]         = {   0,   1,   2 } } },
    [xi.item.GEOMANCY_PANTS]          = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.INT]                = {   8,   9,  10 }, [cexi.augment_id.MND]                 = {   8,   9,  10 } } },
    [xi.item.GEOMANCY_SANDALS]        = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.MP_33]              = {   0,   3,   6 }, [cexi.augment_id.HMP]                 = {   2,   3,   4 } } },

    -- RUN
    [xi.item.RUNEIST_BANDEAU]         = { slot = xi.slot.HEAD,  augs = { [cexi.augment_id.ENMITY]             = {   2,   3,   4 }, [cexi.augment_id.REGEN]               = { nil,   0,   1 } } },
    [xi.item.RUNEIST_COAT]            = { slot = xi.slot.BODY,  augs = { [cexi.augment_id.MAGIC_DMG_TAKEN]    = {   0,   1,   2 }, [cexi.augment_id.PARRYING_RATE]       = {   2,   3,   4 } } },
    [xi.item.RUNEIST_MITONS]          = { slot = xi.slot.HANDS, augs = { [cexi.augment_id.GREAT_SWORD_SKILL]  = {   2,   3,   4 }, [cexi.augment_id.ENMITY]              = {   0,   1,   2 } } },
    [xi.item.RUNEIST_TROUSERS]        = { slot = xi.slot.LEGS,  augs = { [cexi.augment_id.VIT]                = {   8,   9,  10 }, [cexi.augment_id.MND]                 = {   8,   9,  10 } } },
    [xi.item.RUNEIST_BOTTES]          = { slot = xi.slot.FEET,  augs = { [cexi.augment_id.BREATH_DMG_TAKEN]   = {   0,   1,   2 }, [cexi.augment_id.ENMITY]              = {   0,   1,   2 } } },
}

local categories =
{
    "Sea",
    "Apollyon",
    "Temenos",
}

local storage =
{

    -- Sea Organs
    {
        xi.item.PHUABO_ORGAN,
        xi.item.XZOMIT_ORGAN,
        xi.item.AERN_ORGAN,
        xi.item.HPEMDE_ORGAN,
        xi.item.YOVRA_ORGAN,
        xi.item.EUVHI_ORGAN,
        xi.item.HIGH_QUALITY_PHUABO_ORGAN,
        xi.item.HIGH_QUALITY_XZOMIT_ORGAN,
        xi.item.HIGH_QUALITY_AERN_ORGAN,
        xi.item.HIGH_QUALITY_HPEMDE_ORGAN,
        xi.item.HIGH_QUALITY_EUVHI_ORGAN,
        xi.item.LUMINIAN_TISSUE,
        xi.item.GHRAH_M_CHIP,
    },

    -- Apollyon Items
    {
        xi.item.ARGYRO_RIVET,
        xi.item.ANCIENT_BRASS_INGOT,
        xi.item.SPOOL_OF_BENEDICT_YARN,
        xi.item.SPOOL_OF_DIABOLIC_YARN,
        xi.item.SQUARE_OF_CARDINAL_CLOTH,
        xi.item.SPOOL_OF_LIGHT_FILAMENT,
        xi.item.WHITE_RIVET,
        xi.item.BLACK_RIVET,
        xi.item.FETID_LANOLIN_CUBE,
        xi.item.SQUARE_OF_BROWN_DOESKIN,
        xi.item.SQUARE_OF_CHARCOAL_COTTON,
        xi.item.SHEET_OF_KUROGANE,
        xi.item.POT_OF_EBONY_LACQUER,
        xi.item.BLUE_RIVET,
        xi.item.SQUARE_OF_ASTRAL_LEATHER,
        xi.item.SQUARE_OF_FLAMESHUN_CLOTH,
        xi.item.SQUARE_OF_CANVAS_TOILE,
        xi.item.SQUARE_OF_CORDUROY_CLOTH,
        xi.item.GOLD_STUD,
        xi.item.ELECTRUM_STUD,
    },

    -- Temenos Items
    {
        xi.item.SQUARE_OF_ECARLATE_CLOTH,
        xi.item.UTOPIAN_GOLD_THREAD,
        xi.item.SPOOL_OF_BENEDICT_YARN,
        xi.item.SQUARE_OF_DIABOLIC_SILK,
        xi.item.SPOOL_OF_RUBY_SILK_THREAD,
        xi.item.SQUARE_OF_SUPPLE_SKIN,
        xi.item.CHUNK_OF_SNOWY_CERMET,
        xi.item.DARK_ORICHALCUM_INGOT,
        xi.item.SQUARE_OF_SMALT_LEATHER,
        xi.item.SPOOL_OF_COILED_YARN,
        xi.item.SPOOL_OF_CHAMELEON_YARN,
        xi.item.SPOOL_OF_SCARLET_ODOSHI,
        xi.item.PLAITED_CORD,
        xi.item.SHEET_OF_COBALT_MYTHRIL,
        xi.item.SPOOL_OF_GLITTERING_YARN,
        xi.item.SPOOL_OF_LUMINIAN_THREAD,
        xi.item.SPOOL_OF_SILKWORM_THREAD,
        xi.item.PANTIN_WIRE,
        xi.item.SQUARE_OF_FILET_LACE,
        xi.item.SQUARE_OF_BRILLIANTINE,
    },
}

local storageList  = {}

for categoryIndex, category in pairs(storage) do
    for itemIndex, itemID in pairs(category) do
        storage[categoryIndex][itemIndex] = { cexi.itemName[itemID], itemID }
        storageList[itemID] = categories[categoryIndex]
    end
end

-- TODO:
local jobNames = utils.map(xi.jobNames, function(_, jobName) return { jobName[2], jobName[1] } end)

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

-----------------------------------
-- Upgrade Utilities
-----------------------------------
local function getRequirementList(player, requirements)
    local result1 = "Requirements: "
    local result2 = ""

    for index = 1, 4 do
        local itemInfo = requirements[index]
        local qty      = player:getCharVar(fmt("[Sea]{}", itemInfo[1]))

        result1 = result1 .. fmt("{} x{} ({})", cexi.itemName[itemInfo[1]], itemInfo[2], qty)
        result1 = result1 .. ", "
    end

    for index = 5, #requirements do
        local itemInfo = requirements[index]
        local qty      = 0

        if itemInfo[1] == xi.item.ANCIENT_BEASTCOIN then
            qty = player:getCurrency("ancient_beastcoin")
        else
            qty = player:getCharVar(fmt("[Sea]{}", itemInfo[1]))
        end

        result2 = result2 .. fmt("{} x{} ({})", cexi.itemName[itemInfo[1]], itemInfo[2], qty)

        if index < #requirements then
            result2 = result2 .. ", "
        end
    end

    return { result1, result2 }
end

local function getAugmentTier(player, itemID, augs)
    local tier = 0
    local curr = cexi.util.augment.getAugments(player, itemID)

    for t = 1, 3 do
        -- Provide extra leniency for existing augments
        -- Only one augment must match to be considered equal tier
        if
            (curr[1] ~= nil and (curr[1][2] >= augs[curr[1][1]][t])) or
            (curr[2] ~= nil and (curr[2][2] >= augs[curr[2][1]][t]))
        then
            tier = t
        end
    end

    return tier
end

local function getTierAugment(player, augs, tier)
    local result = {}

    for augmentID, augments in pairs(augs) do
        if augments[tier] ~= nil then
            table.insert(result, augmentID)
            table.insert(result, augments[tier])
        end
    end

    return result
end

local function hasRequiredMaterials(player, itemID, requirements)
    for _, itemInfo in pairs(requirements) do
        if itemInfo[1] == xi.item.ANCIENT_BEASTCOIN then
            if player:getCurrency("ancient_beastcoin") < itemInfo[2] then
                return false
            end
        else
            local stored = player:getCharVar(fmt("[Sea]{}", itemInfo[1]))

            if stored < itemInfo[2] then
                return false
            end
        end
    end

    return true
end

local function delRequiredMaterials(player, itemID, requirements)
    for _, itemInfo in pairs(requirements) do
        if itemInfo[1] == xi.item.ANCIENT_BEASTCOIN then
            player:delCurrency("ancient_beastcoin", itemInfo[2])
        else
            player:incrementCharVar(fmt("[Sea]{}", itemInfo[1]), -itemInfo[2])
        end
    end
end

-----------------------------------
-- Storage Utilities
-----------------------------------
local qtyList  =
{
    {  "x1",  1 },
    {  "x2",  2 },
    {  "x3",  3 },
    {  "x6",  6 },
    { "x12", 12 },
    { "x24", 24 },
    { "x36", 36 },
    { "x99", 99 },
}

local function quantityPage(player, npc, itemID)
    local tbl = {}
    local varName = fmt("[Sea]{}", itemID)
    local curr = player:getCharVar(varName)

    for _, qtyInfo in pairs(qtyList) do
        local label = qtyInfo[1]
        local qty   = qtyInfo[2]

        if qty <= curr then
            table.insert(tbl, {
                label,
                function()
                    npc:ceAnimate(player, player, 75, 4)

                    player:timer(500, function()
                        if npcUtil.giveItem(player, { { itemID, qty } }) then
                            player:incrementCharVar(varName, -qty)

                            if (curr - qty) > 0 then
                                quantityPage(player, npc, itemID)
                            end
                        end
                    end)
                end,
            })
        end
    end

    delaySendMenu(player, {
        title   = "Select a quantity:",
        options = tbl,
    })
end

local function retrieveItem(player, npc, item)
    local varName = fmt("[Sea]{}", item[2])
    local balance = player:getCharVar(varName)

    player:fmt("You have {} {}. (Vision \129\168 {})", balance, item[1], storageList[item[2]])

    if balance > 0 then
        quantityPage(player, npc, item[2])
    end
end

local function onTradeMaterials(player, npc, trade)
    -- Item Storage
    local lastSlot = trade:getSlotCount() - 1
    local complete = {}

    -- Check if tradeable
    for i = 0, lastSlot do
        local itemID = trade:getItemId(i)

        if storageList[itemID] == nil then
            player:sys("This item cannot be processed here.")
            return
        end
    end

    for i = 0, lastSlot do
        local itemID  = trade:getItemId(i)
        local itemQty = trade:getSlotQty(i)
        local varName = fmt("[Sea]{}", itemID)
        local balance = player:getCharVar(varName)

        player:fmt("You store {} x{} (Total: {}) {} \129\168 {}", cexi.itemName[itemID], itemQty, balance + itemQty, "Vision", storageList[itemID])
        player:incrementCharVar(varName, itemQty)
    end

    player:ceEmote(player, xi.emote.TOSS, xi.emoteMode.MOTION)
    npc:ceAnimate(player, player, 37, 4)

    player:tradeComplete()
end

-----------------------------------
-- Upgrade Functions
-----------------------------------
local function onTrade(player, npc, trade)
    -- AF+1 Upgrade Process
    for itemID, itemInfo in pairs(augments) do
        if cexi.util.tradeHasExactly(trade, itemID) then
            local tier = getAugmentTier(player, itemID, itemInfo.augs)

            if tier < 3 then
                local reqs = getRequirementList(player, cost[itemInfo.slot][tier + 1])

                if hasRequiredMaterials(player, itemID, cost[itemInfo.slot][tier + 1]) then
                    cexi.util.dialog(player, {
                        reqs[1],
                        reqs[2],
                    })

                    delaySendMenu(player, {
                        title   = "Perform upgrade and consume materials?",
                        options =
                        {
                            {
                                "No",
                                function()
                                end,
                            },
                            {
                                fmt("Upgrade to Tier {}", cexi.numeral[tier + 1]),
                                function()
                                    local augs = getTierAugment(player, itemInfo.augs, tier + 1)

                                    npc:ceAnimate(player, player, 4, 2)

                                    player:tradeComplete()
                                    player:addItem(itemID, 1, augs[1], augs[2], augs[3], augs[4])

                                    delRequiredMaterials(player, itemID, cost[itemInfo.slot][tier + 1])

                                    player:timer(1000, function(playerArg)
                                        local ID = zones[playerArg:getZoneID()]
                                        playerArg:messageSpecial(ID.text.ITEM_OBTAINED, itemID)
                                    end)
                                end,
                            }
                        },
                    })
                else
                    cexi.util.dialog(player, {
                        "You do not have enough stored materials to perform this upgrade.",
                        reqs[1],
                        reqs[2],
                    })
                end
            else
                -- If player obtained a T3 roll through the old system but did not max all stats
                -- We give them an opportunity to max this augment free of cost
                if tier == 3 then
                    local augs = cexi.util.augment.getAugments(player, itemID)

                    if
                        augs[1][2] < itemInfo.augs[augs[1][1]][3] or
                        augs[2][2] < itemInfo.augs[augs[2][1]][3]
                    then
                        delaySendMenu(player, {
                            title   = "Resolve max tier augment?",
                            options =
                            {
                                {
                                    "Not yet",
                                    function()
                                    end,
                                },
                                {
                                    "Finish augment",
                                    function()
                                        local augs = getTierAugment(player, itemInfo.augs, 3)

                                        npc:ceAnimate(player, player, 4, 2)

                                        player:tradeComplete()
                                        player:addItem(itemID, 1, augs[1], augs[2], augs[3], augs[4])

                                        player:timer(1000, function(playerArg)
                                            local ID = zones[playerArg:getZoneID()]
                                            playerArg:messageSpecial(ID.text.ITEM_OBTAINED, itemID)
                                        end)
                                    end,
                                },
                            },
                        })

                        return
                    end
                end

                player:sys("That item can no longer be upgraded.")
            end

            return
        end
    end

    onTradeMaterials(player, npc, trade)
end

local function retrieveItems(player, npc)
    delaySendMenu(player, {
        title   = "Select a category",
        options =
        {
            {
                "Sea Organs",
                function()
                    cexi.util.simpleMenu(player, npc, storage[1], retrieveItem, "Select an item:")
                end,
            },
            {
                "Apollyon Materials",
                function()
                    cexi.util.simpleMenu(player, npc, storage[2], retrieveItem, "Select an item:")
                end,
            },
            {
                "Temenos Materials",
                function()
                    cexi.util.simpleMenu(player, npc, storage[3], retrieveItem, "Select an item:")
                end,
            },
        },
    })
end

-----------------------------------
-- Material Conversion
-----------------------------------
local organs =
{
    { "Phuabo Organ", xi.item.PHUABO_ORGAN },
    { "Hpemde Organ", xi.item.HPEMDE_ORGAN },
    { "Euvhi Organ",  xi.item.EUVHI_ORGAN  },
    { "Xzomit Organ", xi.item.XZOMIT_ORGAN },
    { "Aern Organ",   xi.item.AERN_ORGAN   },
}

local function confirmMaterial(player, npc, itemInfo)
    local varName = fmt("[Sea]{}", xi.item.YOVRA_ORGAN)
    local qty     = player:getCharVar(varName)
    local req     = 1

    if npcUtil.giveItem(player, { { itemInfo[2], 1 }, { xi.item.LUMINIAN_TISSUE, math.random(1, 5) } }) then
        player:incrementCharVar(varName, -req)
        player:fmt("You now have {} stored Yovra Organs.", qty - req)
    end
end

local function convertMaterial(player, npc, itemInfo)
    local varName = fmt("[Sea]{}", itemInfo[3])
    local qty     = player:getCharVar(varName)
    local req     = 10

    if itemInfo[2] == "Yovra" then
        req = 1
    end

    if qty < req then
        player:fmt("This process requires {} stored {} but you only have {}.", req, cexi.itemName[itemInfo[3]], qty)
        return
    end

    if req == 1 then
        cexi.util.simpleMenu(player, npc, organs, confirmMaterial, "Select a result:")
        return
    end

    if npcUtil.giveItem(player, { { itemInfo[3], 1 }, { xi.item.LUMINIAN_TISSUE, math.random(1, 5) } }) then
        player:incrementCharVar(varName, -req)
        player:fmt("You now have {} stored {}.", qty - req, cexi.itemName[itemInfo[3]])
    end
end

local conversion =
{
    { "H.Q. \129\168 Phuabo Organ ({})", xi.item.PHUABO_ORGAN, xi.item.HIGH_QUALITY_PHUABO_ORGAN },
    { "H.Q. \129\168 Hpemde Organ ({})", xi.item.HPEMDE_ORGAN, xi.item.HIGH_QUALITY_HPEMDE_ORGAN },
    { "H.Q. \129\168 Euvhi Organ ({})",  xi.item.EUVHI_ORGAN,  xi.item.HIGH_QUALITY_EUVHI_ORGAN  },
    { "H.Q. \129\168 Xzomit Organ ({})", xi.item.XZOMIT_ORGAN, xi.item.HIGH_QUALITY_XZOMIT_ORGAN },
    { "H.Q. \129\168 Aern Organ ({})",   xi.item.AERN_ORGAN,   xi.item.HIGH_QUALITY_AERN_ORGAN   },
    { "Yovra \129\168 Any Organ ({})",   "Yovra",              xi.item.YOVRA_ORGAN               },
}

-----------------------------------
-- NPC Functions
-----------------------------------
local function onTrigger(player, npc)
    delaySendMenu(player, {
        title   = "What do you seek?",
        options =
        {
            {
                "Retrieve items",
                function()
                    retrieveItems(player, npc)
                end,
            },
            {
                "Learn about upgrades",
                function()
                    cexi.util.dialog(player, {
                        "Collect a piece of Artifact +1 and bring it to me...",
                    }, npc:getPacketName(), { npc = npc })
                end,
            },
            {
                "Convert materials",
                function()
                    local tbl = {}

                    for _, itemInfo in pairs(conversion) do
                        local qty = player:getCharVar(fmt("[Sea]{}",  itemInfo[3]))
                        local req = 10

                        if itemInfo[2] == "Yovra" then
                            req = 1
                        end

                        table.insert(tbl, {
                            fmt(itemInfo[1], math.floor(qty / req)),
                            itemInfo[2],
                            itemInfo[3]
                        })
                    end

                    cexi.util.simpleMenu(player, npc, tbl, convertMaterial, "Select a process:")
                end,
            },
        },
    })
end

local function onTriggerGleam(player, npc)
    player:fmt("You glimpse another plane of existence. Though you are unable to travel through, you may be able to stow away some items here.")
end

cexi.util.liveReload(m, {
    ["Port_Jeuno"] =
    {
        {
            name      = "Vision",
            objtype   = xi.objType.NPC,
            look      = 2833,
            x         =  -4.217,
            y         =   0.001,
            z         = -10.612,
            rotation  = 0,
            widescan  = 1,
            onTrigger = onTrigger,
            onTrade   = onTrade,
        },
        {
            name      = "Lumorian Gleam",
            objtype   = xi.objType.NPC,
            look      = 1382,
            hideName  = true,
            x         = -144.256,
            y         = -1.000,
            z         = 6.792,
            rotation  = 95,
            widescan  = 1,
            onTrade   = onTradeMaterials,
            onTrigger = onTriggerGleam,
        },
    },
    ["AlTaieu"] =
    {
        {
            name      = "Lumorian Gleam",
            objtype   = xi.objType.NPC,
            look      = 1382,
            hideName  = true,
            x         = 29.702,
            y         = -2.094,
            z         = -623.868,
            rotation  = 133,
            widescan  = 1,
            onTrade   = onTradeMaterials,
            onTrigger = onTriggerGleam,
        },
    },
    ["Temenos"] =
    {
        {
            name      = "Lumorian Gleam",
            objtype   = xi.objType.NPC,
            look      = 1382,
            hideName  = true,
            x         = 602.690,
            y         = 0.500,
            z         = 40.088,
            rotation  = 126,
            widescan  = 1,
            onTrade   = onTradeMaterials,
            onTrigger = onTriggerGleam,
        },
    },
    ["Apollyon"] =
    {
        {
            name      = "Lumorian Gleam",
            objtype   = xi.objType.NPC,
            look      = 1382,
            hideName  = true,
            x         = 577.055,
            y         = 0.000,
            z         = -626.538,
            rotation  = 225,
            widescan  = 1,
            onTrade   = onTradeMaterials,
            onTrigger = onTriggerGleam,
        },
        {
            name      = "Lumorian Gleam",
            objtype   = xi.objType.NPC,
            look      = 1382,
            hideName  = true,
            x         = -575.001,
            y         = 0.000,
            z         = -627.253,
            rotation  = 159,
            widescan  = 1,
            onTrade   = onTradeMaterials,
            onTrigger = onTriggerGleam,
        },
    },
})

return m
