-----------------------------------
-- Crystal Warrior - Provenance System
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
require('scripts/globals/player')
require('scripts/globals/npc_util')
local scrying  = require("modules/catseyexi/lua/additive_overrides/skills/scrying")
local lapidary = require("modules/catseyexi/lua/additive_overrides/skills/lapidary")
-----------------------------------
local m = Module:new("provenance_battlefields")

local RADIANT_BOSS = { 186.639, 16.885, -218.959, 147 }
local ASTRAL_BOSS  = { 656.065, 20.715, -748.799,  83 }

local battles =
{
    -- Tier I (Level 30)
    [xi.item.ASTRAL_ORB] =
    {
        {
            name = "Tier I (A)",
            mobs =
            {
                {
                    name        = "Panzer Percival",
                    groupId     = 44,
                    groupZoneId = 104,
                    level       = 30,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.COSMIC_ORB   }, -- 24%
                    { cexi.rate.VERY_COMMON, 846, "an insect wing" }, -- 24%
                    { cexi.rate.COMMON,      889, "a beetle shell" }, -- 15%
                },
                {
                    { cexi.rate.VERY_COMMON, 846, "an insect wing" }, -- 24%
                    { cexi.rate.COMMON,      894, "a beetle jaw"   }, -- 15%
                },
            },
        },
        {
            name = "Tier I (B)",
            mobs =
            {
                {
                    name        = "Rampaging Ram",
                    groupId     = 29,
                    groupZoneId = 108,
                    level       = 30,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.COSMIC_ORB }, -- 24%
                    { cexi.rate.VERY_COMMON, 859, "a ram skin"   }, -- 24%
                    { cexi.rate.COMMON,      895, "a ram horn"   }, -- 15%
                },
                {
                    { cexi.rate.VERY_COMMON, 859, "a ram skin" }, -- 24%
                    { cexi.rate.COMMON,      895, "a ram horn" }, -- 15%
                },
            },
        },
        {
            name = "Tier I (C)",
            mobs =
            {
                {
                    name        = "Geyser Lizard",
                    groupId     = 5,
                    groupZoneId = 191,
                    level       = 30,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.COSMIC_ORB  }, -- 24%
                    { cexi.rate.VERY_COMMON, 852, "a lizard skin" }, -- 24%
                    { cexi.rate.COMMON,      926, "a lizard tail" }, -- 15%
                },
                {
                    { cexi.rate.VERY_COMMON, 852, "a lizard skin" }, -- 24%
                },
            },
        },
    },

    -- Tier II (Level 40)
    [xi.item.COSMIC_ORB] =
    {
        {
            name = "Tier II (A)",
            mobs =
            {
                {
                    name        = "Chonchon",
                    groupId     = 19,
                    groupZoneId = 119,
                    level       = 40,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.RADIANT_ORB      }, -- 24%
                    { cexi.rate.VERY_COMMON,  854, "a cockatrice skin" }, -- 24%
                    { cexi.rate.COMMON,      4435, "a cockatrice meat" }, -- 15%
                },
            },
        },
        {
            name = "Tier II (B)",
            mobs =
            {
                {
                    name        = "Kirata",
                    groupId     = 30,
                    groupZoneId = 111,
                    level       = 40,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.RADIANT_ORB    }, -- 24%
                    { cexi.rate.VERY_COMMON, 863, "a coeurl hide"    }, -- 24%
                    { cexi.rate.COMMON,      927, "a coeurl whisker" }, -- 15%
                },
            },
        },
        {
            name = "Tier II (C)",
            mobs =
            {
                {
                    name        = "Argus",
                    groupId     = 38,
                    groupZoneId = 198,
                    level       = 40,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.RADIANT_ORB     }, -- 24%
                    { cexi.rate.VERY_COMMON, 939, "a hecteyes eye"    }, -- 24%
                    { cexi.rate.COMMON,      914, "a vial of mercury" }, -- 15%
                },
            },
        },
    },

    -- Tier III (Level 50)
    [xi.item.RADIANT_ORB] =
    {
        {
            name = "Tier III (A)",
            mobs =
            {
                {
                    name        = "Lumber Jack",
                    groupId     = 8,
                    groupZoneId = 105,
                    level       = 50,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.CRYSTALLINE_ORB  }, -- 24%
                    { cexi.rate.VERY_COMMON, 846, "an insect wing"     }, -- 24%
                    { cexi.rate.COMMON,      889, "a beetle shell"     }, -- 15%
                },
                {
                    { cexi.rate.VERY_COMMON, 846, "an insect wing" }, -- 24%
                    { cexi.rate.COMMON,      894, "a beetle jaw"   }, -- 15%
                },
            },
        },
        {
            name = "Tier III (B)",
            mobs =
            {
                {
                    name        = "King Arthro",
                    groupId     = 7,
                    groupZoneId = 104,
                    level       = 50,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.CRYSTALLINE_ORB          }, -- 24%
                    { cexi.rate.VERY_COMMON,  881, "a crab shell"              }, -- 24%
                    { cexi.rate.COMMON,      4400, "a slice of land crab meat" }, -- 15%
                },
                {
                    { cexi.rate.VERY_COMMON,  936, "a chunk of rock salt"      }, -- 24%
                    { cexi.rate.COMMON,      1193, "a high-quality crab shell" }, -- 15%
                },
            },
        },
        {
            name = "Tier III (C)",
            mobs =
            {
                {
                    name        = "Roc",
                    groupId     = 41,
                    groupZoneId = 120,
                    level       = 50,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, xi.item.CRYSTALLINE_ORB    }, -- 24%
                    { cexi.rate.VERY_COMMON, 842, "a giant bird feather" }, -- 24%
                },
            },
        },
    },

    -- Tier III (Level 60)
    [xi.item.CRYSTALLINE_ORB] =
    {
        {
            name = "Tier III (A)",
            mobs =
            {
                {
                    name        = "Simurgh",
                    groupId     = 41,
                    groupZoneId = 110,
                    level       = 60,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, 842, "a giant bird feather" }, -- 24%
                },
            },
        },
        {
            name = "Tier III (B)",
            mobs =
            {
                {
                    name        = "Cassie",
                    groupId     = 39,
                    groupZoneId = 204,
                    level       = 60,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, 920, "a malboro vine" }, -- 24%
                },
            },
        },
        {
            name = "Tier III (C)",
            mobs =
            {
                {
                    name        = "Serket",
                    groupId     = 41,
                    groupZoneId = 200,
                    level       = 60,
                    pos         = RADIANT_BOSS,
                },
            },
            loot =
            {
                {
                    { cexi.rate.VERY_COMMON, 897, "a scorpion claw" }, -- 24%
                },
            },
        },
    },
}

local other =
{
    {
        name = "Wings of Forever",
        mobs =
        {
            {
                name        = "Primal Test",
                groupId     = 58,
                groupZoneId = 123,
                look        = 2110,
                level       = 90, 
                pos         = ASTRAL_BOSS,
            },
        },
        loot =
        {
            {
                { cexi.rate.VERY_COMMON, xi.item.FIRE_CLUSTER },
                { cexi.rate.VERY_COMMON, xi.item.WIND_CLUSTER },
            },
            {
                { cexi.rate.VERY_COMMON, xi.item.LIGHT_CLUSTER },
            },
        },
    },
}

local entry =
{
    ORB_ITEM = 1,
    KEY_ITEM = 2,
}

local battlefields =
{
    {
        look      = 2492,
        boss      = RADIANT_BOSS,
        entry     = entry.ORB_ITEM,
        entrance  =
        {
            name = "Radiant Anomaly",
            pos  = { -655.874, -20.777, -506.113,  0 },
            to   = {  145.086,  15.500, -183.376, 34 },
        },
        exit     =
        {
            name = "Radiant Egress",
            pos  = {  142.469,  15.500, -179.253, 40 },
            to   = { -650.947, -20.319, -506.961,  0 },
        },
        default = "The anomaly shines radiantly...",
    },
    {
        look      = 2491,
        boss      = ASTRAL_BOSS,
        entry     = entry.KEY_ITEM,
        entrance  =
        {
            name = "Astral Anomaly",
            pos  = { -623.629, -20.763, -493.879, 132 },
            to   = {  623.785,  19.500, -814.550, 217 },
        },
        exit     =
        {
            name = "Astral Egress",
            pos  = {  618.921,  19.500, -817.808, 233 },
            to   = { -628.376, -20.451, -492.900, 135 },
        },
        default = "Nothing happens.",
    },
}

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function removeNPC(npc)
    -- This will send the packet to ensure NPC disappears for player
    npc:setStatus(xi.status.INVISIBLE)

    npc:timer(500, function(npcArg)
        if npcArg ~= nil then
            -- This will delete DE server side on zone tick
            npcArg:setStatus(xi.status.DISAPPEAR)
        end
    end)
end

local function spawnChest(zone, battlefield, battle)
    local dynamicEntity =
    {
        name        = "Radiant Relic",
        objtype     = xi.objType.NPC,
        look        = 969,
        x           = battlefield.boss[1],
        y           = battlefield.boss[2],
        z           = battlefield.boss[3],
        rotation    = battlefield.boss[4],
        widescan    = 1,

        onTrigger   = function(player, npc)
            if npc:getLocalVar("open") == 1 then
                return
            end

            npc:entityAnimationPacket("open")
            npc:setLocalVar("open", 1)

            npc:timer(3000, function(npcArg)
                npcArg:entityAnimationPacket("clos")
            end)

            for _, itemPool in pairs(battle.loot) do
                local result = cexi.util.pickItem(itemPool)

                if
                    result[2] ~= nil and
                    result[2] > 0
                then
                    player:addTreasure(result[2])
                end
            end

            player:timer(5000, function(playerArg)
                cexi.util.leaveEncounter(player)
                player:injectActionPacket(player:getID(), 6, 602, 0, 0, 0, 0, 0)
                player:setPos(unpack(battlefield.exit.to))

                removeNPC(npc)
            end)
        end,

        releaseIdOnDisappear = true,
    }

    local npc = zone:insertDynamicEntity(dynamicEntity)

    npc:hideName(true)
    npc:hideHP(true)

    npc:timer(120000, function(npcArg)
        if npcArg ~= nil then
            removeNPC(npcArg)
        end
    end)
end


local function createBattle(zone, battlefield, battle)
    for _, mobInfo in pairs(battle.mobs) do
        local dynamicEntity =
        {
            name        = mobInfo.name,
            objtype     = xi.objType.MOB,
            groupId     = mobInfo.groupId,
            groupZoneId = mobInfo.groupZoneId,
            look        = mobInfo.look,
            x           = mobInfo.pos[1],
            y           = mobInfo.pos[2],
            z           = mobInfo.pos[3],
            rotation    = mobInfo.pos[4],
            widescan    = 1,
            releaseIdOnDisappear = true,

            onMobDeath  = function(mob, player)
                spawnChest(zone, battlefield, battle)
            end,
        }

        local mob = zone:insertDynamicEntity(dynamicEntity)

        mob:setSpawn(unpack(mobInfo.pos))
        mob:spawn()
        mob:setDropID(0)

        cexi.util.mobEncounter(mob, { level = battle.level })
    end
end

local function enterBattlefield(player, npc, battlefield, battle)
    delaySendMenu(player, {
        title   = "Enter the battlefield?",
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                    cexi.util.allianceEncounter(player, { retainOnDeath = true, level = battle.level })
                    createBattle(player:getZone(), battlefield, battle)

                    player:timer(1250, function(playerArg)
                        player:injectActionPacket(player:getID(), 6, 602, 0, 0, 0, 0, 0)
                        player:setPos(unpack(battlefield.entrance.to))
                    end)
                end,
            },
        },
    })
end

local function leaveBattlefield(player, npc, pos)
    delaySendMenu(player, {
        title   = "Leave the battlefield?",
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                    cexi.util.leaveEncounter(player)
                    -- TODO: Check if we need to delete the mobs

                    player:timer(1250, function(playerArg)
                        player:injectActionPacket(player:getID(), 6, 602, 0, 0, 0, 0, 0)
                        player:setPos(unpack(pos))
                    end)
                end,
            },
        },
    })
end

local function selectBattle(player, npc, trade, battlefield)
    if player:hasStatusEffect(xi.effect.BATTLEFIELD) then
        player:fmt("A battle is already in progress.")
        return
    end

    if battlefield.entry == entry.ORB_ITEM then
        for itemID, battleInfo in pairs(battles) do
            if npcUtil.tradeHasExactly(trade, itemID) then
                local options = {}

                table.insert(options, {
                    "None",
                    function()
                    end,
                })

                for _, battle in pairs(battleInfo) do
                    table.insert(options, {
                        battle.name,
                        function()
                            player:tradeComplete()
                            enterBattlefield(player, npc, battlefield, battle)
                        end,
                    })
                end

                delaySendMenu(player, {
                    title   = "Select a battlefield:",
                    options = options,
                })

                -- TODO:
                -- Need to take orb regardless of option, and set local var or status so they can reselect if cancelled
            end
        end
    end
end

local battlefieldList = {}

for _, battlefield in pairs(battlefields) do
    table.insert(battlefieldList, {
        objtype  = xi.objType.NPC,
        name     = battlefield.entrance.name,
        look     = battlefield.look,
        x        = battlefield.entrance.pos[1],
        y        = battlefield.entrance.pos[2],
        z        = battlefield.entrance.pos[3],
        rotation = battlefield.entrance.pos[4],
        widescan = 1,

        onTrigger = function(player, npc)
            if not xi.settings.main.ENABLE_AOP then
                player:fmt("Nothing happens.")
                return
            end

            if player:hasStatusEffect(xi.effect.BATTLEFIELD) then
                enterBattlefield(player, npc, battlefield.entrance.to)
            else
                player:fmt(battlefield.default)
            end
        end,

        onTrade = function(player, npc, trade)
            if not xi.settings.main.ENABLE_AOP then
                player:fmt("Nothing happens.")
                return
            end

            selectBattle(player, npc, trade, battlefield)
        end,
    })

    table.insert(battlefieldList, {
        objtype  = xi.objType.NPC,
        name     = battlefield.exit.name,
        look     = battlefield.look,
        x        = battlefield.exit.pos[1],
        y        = battlefield.exit.pos[2],
        z        = battlefield.exit.pos[3],
        rotation = battlefield.exit.pos[4],
        widescan = 1,

        onTrigger = function(player, npc)
            if not xi.settings.main.ENABLE_AOP then
                player:fmt("Nothing happens.")
                return
            end

            leaveBattlefield(player, npc, battlefield.exit.to)
        end,
    })
end

cexi.util.liveReload(m, { ["Provenance"] = battlefieldList })

return m
