-----------------------------------
-- Crystal Warrior (Petros) - CW Rewards and Milestones
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
require('scripts/globals/player')
require("scripts/globals/quests")
require('scripts/globals/npc_util')
require('scripts/zones/Toraimarai_Canal/npcs/_4pc')
require('scripts/zones/Toraimarai_Canal/npcs/Transporter')
-----------------------------------

local m = Module:new("npc_petros")

local vars =
{
    CW_INTRO          = "[CW]INTRO",
    CW_SPENT          = "[CW]SPENT",
    CW_EXP_LOST       = "[CW]EXP_LOST",
    CW_EXP_LOST_ITEM  = "[CW]EXP_LOST_ITEM",
    INTRO_CS          = "[CW]INTRO_CS",
    REWARD_POINTS     = "REWARD_POINTS",
    REWARD_PAGE_JOB   = "REWARD_PAGE_JOB",
    REWARD_PAGE_ARMOR = "REWARD_PAGE_ARMOR",
    REWARD_PAGE_PIECE = "REWARD_PAGE_PIECE",
    REWARD_PAGE_MISC  = "REWARD_PAGE_MISC",
}

local settings =
{
    name     = "Petros",
    boxName  = "Mysterious Box",
    currency = "points",
    area     = "Provenance",
    pos      = { 497.592, 27.500, -464.382, 75 }, -- !pos 497.592 27.500 -464.382 222
}

-----------------------------------
-- Crystal Warrior Introduction
-- Intro cutscenes and events
-----------------------------------

local CRYSTAL_COSTUME = 2351

local cutscene =
{
    { music = 153 },
    { costume = CRYSTAL_COSTUME },

    { delay = 3000 },
    "This day marks the beginning of a tale that transcends time itself...",
    { delay = 5000 },
    " ",
    "It all began with a stone, a shimmering beacon of power that fell from the heavens...",
    " ",
    "An ancient story tells of powerful warriors who wielded these very crystals.",
    { delay = 5000 },
    "They were the chosen ones, blessed with extraordinary abilities...",
    "Entrusted with the fate of Vana'diel.",
    " ",
    { delay = 5000 },

    -- Music and animations start
    { music = 103 },
    "The time has come for a new generation of heroes to rise.",
    " ",
    { animate   = 22, mode = 4 },
    { delay     = 300 },
    { animate   = 22, mode = 4 },
    { delay     = 300 },
    { animate   = 22, mode = 4 },
    { animate   = 14, mode = 4 },
    { delay     = 1100 },
    { animate   = 14, mode = 4 },
    { delay     = 1500 },
    { animate   = 22, mode = 4 },
    { delay     = 1000 },
    { animate   = 22, mode = 4 },
    { delay     = 1500 },
    { animate   = 21, mode = 4 },
    { delay     = 10000 },
    { costume   = 0 },
    { charvar = vars.INTRO_CS, value = 1 },
}

local provStart = { 501.448, 27.500, -497.632, 252, 222 }

local entities =
{
    npc =
    {
        name   = settings.name,
        look   = cexi.util.look({
            race = xi.race.HUME_M,
            face = 1,
            head = 395, -- Vanya Hood
            body = 441, -- Delegate's Garb
            hand = 441, -- Delegate's Cuffs
            legs = 230, -- Dinner Hose
            feet = 230, -- Dinner Hose
        })
    },
    box =
    {
        name   = settings.boxName,
        look   = 969,                 -- Treasure Chest
        offset = { -1.2, 0, -0.35 }    -- Offset to Petros
    },
}

local pointType =
{
    LEVEL     =  1,
    MAAT      =  2,
    RANK      =  3,
    ZILART    =  4,
    PROMATHIA =  5,
    TREASURES =  6,
    QUEST     =  7,
    CUSTOM    =  8,
    SKILL     =  9,
    NM        = 10,
    PRESTIGE  = 11,
}

local milestones =
{
    { "Crafting Skills",        pointType.SKILL,     },
    { "Rank Missions",          pointType.RANK       },
    { "Rise of the Zilart",     pointType.ZILART,    },
    { "Chains of Promathia",    pointType.PROMATHIA, },
    { "Treasure of Aht Urhgan", pointType.TREASURES, },
    { "Era Quests",             pointType.QUEST,     },
    { "New Quests",             pointType.CUSTOM,    },
    { "NMs Defeated",           pointType.NM         },
}

local menu =
{
    MAIN =
    {
        TITLE           = "You have %u %s.",
        NOTHING         = "Nothing",
        MILESTONES      = "Check my milestones",
        ARMOR           = "Crystal Armor Sets",
        MISC            = "Crystal Weapons",
        RETURN_VANADIEL = "Return to Vana'diel",
    },

    SELECT_SET   = "Select a set:",
    SELECT_PIECE = "Select a piece:",
    SELECT_ITEM  = "Select an item:",
    RETURN_CONFIRM = "Return to Vana'diel?",
    BUY_FOR      = "Buy %s for %i %s? (%i)",
    AFFORD       = "You can't afford that yet.",
    START_TALLY  = "Let's see here...",
    TOTAL        = "You've accumlated a total of %u %s and your current balance is %i.",

    UPGRADE =
    {
        CURRENT = "Your %s is currently Tier %s and you possess %d Milestone %s.",
        LIMIT   = "Your %s is Tier %s and it can no longer be upgraded at this time.",
        COST_1  = "Your %s is Tier %s and you possess %d Milestone points.\n The next upgrade will cost %d Milestone points.",
        COST_2  = "Unfortunately, you do not have enough.",
        CONFIRM = "Will you upgrade the %s to Tier %s for %d %s?",
        ASK     = "Upgrade %s for %d?",
        DECLINE = "Very well.",
        REJECT  = "I am unable to perform any upgrades, as your balance is currently negative.",
    },
}

local armorCost =
{
    1500,  -- Head
    3500,  -- Body
    2500,  -- Hands
    1500,  -- Legs
    1000   -- Feet
}

local armorSets =
{
    -- Page 1
    {
        {
            title = "Sakpata's Attire",
            job   = "WAR",
            set   =
            {
                { 23757,       "Sakpata's Helm"        },
                { 23764,       "Sakpata's Breastplate" },
                { 23771,       "Sakpata's Gauntlets"   },
                { 23778,       "Sakpata's Cuisses"     },
                { 23785,       "Sakpata's Leggings"    },
            },
        },
        {
            title = "Naga Attire",
            job   = "MNK",
            set   =
            {
                { 26793,       "Naga Somen"            },
                { 26949,       "Naga Samune"           },
                { 27099,       "Naga Tekko"            },
                { 27284,       "Naga Hakama"           },
                { 27459,       "Naga Kyahan"           },
            },
        },
        {
            title = "Bunzi's Attire",
            job   = "WHM",
            set   =
            {
                { 23760,       "Bunzi's Hat"           },
                { 23767,       "Bunzi's Robe"          },
                { 23774,       "Bunzi's Gloves"        },
                { 23781,       "Bunzi's Pants"         },
                { 23788,       "Bunzi's Sabots"        },
            },
        },
        {
            title = "Vanya's Attire",
            job   = "BLM",
            set   =
            {
                { 26797,       "Vanya's Hood"          },
                { 26953,       "Vanya's Robe"          },
                { 27103,       "Vanya's Cuffs"         },
                { 27288,       "Vanya's Slops"         },
                { 27463,       "Vanya's Clogs"         },
            },
        },
    },
    -- Page 2
    {
        {
            title = "Malignance Attire",
            job   = "RDM",
            set   =
            {
                { 23732,      "Malignance Chapeau"     },
                { 23733,      "Malignance Tabard"      },
                { 23734,      "Malignance Gloves"      },
                { 23735,      "Malignance Tights"      },
                { 23736,      "Malignance Boots"       },
            },
        },
        {
            title = "Adhemar Attire",
            job   = "THF",
            set   =
            {
                { 25613,      "Adhemar Bonnet"         },
                { 25686,      "Adhemar Jacket"         },
                { 27117,      "Adhemar Wristbands"     },
                { 27302,      "Adhemar Kecks"          },
                { 27473,      "Adhemar Gamashes"       },
            },
        },
        {
            title = "Founder's Attire",
            job   = "PLD",
            set   =
            {
                { 27764,      "Founder's Corona"       },
                { 27910,      "Founder's Breastplate"  },
                { 28049,      "Founder's Gauntlets"    },
                { 28191,      "Founder's Hose"         },
                { 28330,      "Founder's Greaves"      },
            },
        },
        {
            title = "Gleti's Attire",
            job   = "DRK",
            set   =
            {
                { 23756,      "Gleti's Mask"           },
                { 23763,      "Gleti's Cuirass"        },
                { 23770,      "Gleti's Gauntlets"      },
                { 23777,      "Gleti's Breeches"       },
                { 23784,      "Gleti's Boots"          },
            },
        },
    },
    -- Page 3
    {
        {
            title = "Valorous Attire",
            job   = "BST",
            set   =
            {
                { 25641,      "Valorous Mask"          },
                { 25717,      "Valorous Mail"          },
                { 27139,      "Valorous Mitts"         },
                { 25841,      "Valorous Hose"          },
                { 27495,      "Valorous Greaves"       },
            },
        },
        {
            title = "Pursuer's Attire",
            job   = "BRD",
            set   =
            {
                { 26795,      "Pursuer's Beret"        },
                { 26951,      "Pursuer's Doublet"      },
                { 27101,      "Pursuer's Cuffs"        },
                { 27286,      "Pursuer's Pants"        },
                { 27461,      "Pursuer's Gaiters"      },
            },
        },
        {
            title = "Herculean Attire",
            job   = "RNG",
            set   =
            {
                { 25642,      "Herculean Helm"         },
                { 25718,      "Herculean Vest"         },
                { 27140,      "Herculean Gloves"       },
                { 25842,      "Herculean Trousers"     },
                { 27496,      "Herculean Boots"        },
            },
        },
        {
            title = "Ryuo's Attire",
            job   = "SAM",
            set   =
            {
                { 25611,      "Ryuo Somen"             },
                { 25684,      "Ryuo Domaru"            },
                { 27115,      "Ryuo Tekko"             },
                { 27300,      "Ryuo Hakama"            },
                { 27471,      "Ryuo Sune-Ate"          },
            },
        },
    },
    -- Page 4
    {
        {
            title = "Mpaca's Attire",
            job   = "NIN",
            set   =
            {
                { 23758,      "Mpaca's Cap"            },
                { 23765,      "Mpaca's Doublet"        },
                { 23772,      "Mpaca's Gloves"         },
                { 23779,      "Mpaca's Hose"           },
                { 23786,      "Mpaca's Boots"          },
            },
        },
        {
            title = "Odyssean Attire",
            job   = "DRG",
            set   = 
            {
                { 25640,      "Odyssean Helm"          },
                { 25716,      "Odyssean Breastplate"   },
                { 27138,      "Odyssean Gauntlets"     },
                { 25840,      "Odyssean Cuisses"       },
                { 27494,      "Odyssean Greaves"       },
            },
        },
        {
            title = "Chironic Attire",
            job   = "SMN",
            set   =
            {
                { 25644,      "Chironic Hat"           },
                { 25720,      "Chironic Doublet"       },
                { 27142,      "Chironic Gloves"        },
                { 25844,      "Chironic Hose"          },
                { 27498,      "Chironic Slippers"      },
            },
        },
        {
            title = "Agwu's Attire",
            job   = "BLU",
            set   =
            {
                { 23759,      "Agwu's Cap"             },
                { 23766,      "Agwu's Robe"            },
                { 23773,      "Agwu's Gages"           },
                { 23780,      "Agwu's Slops"           },
                { 23787,      "Agwu's Pigaches"        },
            },
        },
    },
    -- Page 5
    {
        {
            title = "Ikenga's Attire",
            job   = "COR",
            set   =
            {
                { 23755,      "Ikenga's Hat"           },
                { 23762,      "Ikenga's Vest"          },
                { 23769,      "Ikenga's Gloves"        },
                { 23776,      "Ikenga's Trousers"      },
                { 23783,      "Ikenga's Clogs"         },
            },
        },
        {
            title = "Thurandaut Attire",
            job   = "PUP",
            set   =
            {
                { 27784,      "Thurandaut Chapeau"     },
                { 27924,      "Thurandaut Tabard"      },
                { 28064,      "Thurandaut Gloves"      },
                { 28204,      "Thurandaut Tights"      },
                { 28344,      "Thurandaut Boots"       },
            },
        },
        {
            title = "Rawhide Attire",
            job   = "DNC",
            set   =
            {
                { 26794,      "Rawhide Mask"           },
                { 26950,      "Rawhide Vest"           },
                { 27100,      "Rawhide Gloves"         },
                { 27285,      "Rawhide Trousers"       },
                { 27460,      "Rawhide Boots"          },
            },
        },
        {
            title = "Merlinic Attire",
            job   = "SCH",
            set   =
            {
                { 25643,      "Merlinic Hood"          },
                { 25719,      "Merlinic Jubbah"        },
                { 27141,      "Merlinic Dastanas"      },
                { 25843,      "Merlinic Shalwar"       },
                { 27497,      "Merlinic Crackows"      },
            },
        },
    },
    -- Page 6
    {
        {
            title = "Amalric Attire",
            job   = "GEO",
            set   =
            {
                { 25615,      "Amalric Coif"           },
                { 25688,      "Amalric Doublet"        },
                { 27119,      "Amalric Gages"          },
                { 27304,      "Amalric Slops"          },
                { 27475,      "Amalric Nails"          },
            },
        },
        {
            title = "Eschite Attire",
            job   = "RUN",
            set   =
            {
                { 26791,      "Eschite Helm"           },
                { 26947,      "Eschite Breastplate"    },
                { 27097,      "Eschite Gauntlets"      },
                { 27282,      "Eschite Cuisses"        },
                { 27457,      "Eschite Greaves"        },
            },
        },
    },
}

local miscItems =
{
    -- Total Cost: 30,000
    {
        { 27627, "Svalinn",  20000 },
        { 10806, "Adamas",   20000 },
        { 18909, "Claritas", 20000 },
        { 18627, "Xingzhe",  20000 },
        { 22281, "Verutum",  20000 },
    },
}

local upgradeCost =
{
    1000,
    2000,
    3000,
    4000,
}

local upgrades =
{
    {
        27627,
        "Svalinn",
        {
            { 55, 2, 39, 2, 286, 2 }, -- Magic dmg. taken -3%, Enmity +3, Shield Skill +3
            { 55, 3, 39, 3, 286, 3 }, -- Magic dmg. taken -4%, Enmity +4, Shield Skill +4
            { 55, 4, 39, 4, 286, 4 }, -- Magic dmg. taken -5%, Enmity +5, Shield Skill +5
            { 55, 7, 39, 7, 286, 7 }, -- Magic dmg. taken -8%, Enmity +8, Shield Skill +8
        },
    },
    {
        10806,
        "Adamas",
        {
            { 514, 2, 517, 2, 323, 2 }, -- VIT +3, MND +3, Cure Cast Time -3%
            { 514, 3, 517, 3, 323, 3 }, -- VIT +4, MND +4, Cure Cast Time -4%
            { 514, 4, 517, 4, 323, 4 }, -- VIT +5, MND +5, Cure Cast Time -5%
            { 514, 7, 517, 7, 323, 7 }, -- VIT +8, MND +8, Cure Cast Time -8%
        },
    },
    {
        18909,
        "Claritas",
        {
            { 512, 2, 517, 2, 35, 2 }, -- STR +3, MND +3, Magic Accuracy +3
            { 512, 3, 517, 3, 35, 3 }, -- STR +4, MND +4, Magic Accuracy +4
            { 512, 4, 517, 4, 35, 4 }, -- STR +5, MND +5, Magic Accuracy +5
            { 512, 7, 517, 7, 35, 7 }, -- STR +8, MND +8, Magic Accuracy +8
        },
    },
    {
        18627,
        "Xingzhe",
        {
            { 145, 2, 39, 2, 71, 2 }, -- Counter +3, Enmity +3, Damage taken -3%
            { 145, 3, 39, 3, 71, 3 }, -- Counter +4, Enmity +4, Damage taken -4%
            { 145, 4, 39, 4, 71, 4 }, -- Counter +5, Enmity +5, Damage taken -5%
            { 145, 7, 39, 7, 71, 7 }, -- Counter +8, Enmity +8, Damage taken -8%
        },
    },
    {
        22281,
        "Verutum",
        {
            { 514, 2, 25, 2, 112, 2 }, -- VIT +3, Attack +3, Pet: Damage taken -3%
            { 514, 3, 25, 3, 112, 3 }, -- VIT +4, Attack +4, Pet: Damage taken -4%
            { 514, 4, 25, 4, 112, 4 }, -- VIT +5, Attack +5, Pet: Damage taken -5%
            { 514, 7, 25, 7, 112, 7 }, -- VIT +8, Attack +8, Pet: Damage taken -8%
        },
    },
}

local points =
{
    [pointType.LEVEL] =
    {
        { 10, 100 },
        { 20, 200 },
        { 30, 300 },
        { 40, 400 },
        { 50, 500 },
        { 55, 550 },
        { 60, 600 },
        { 65, 650 },
        { 70, 700 },
        { 75, 750 },
        -- Total (Job):   4,750   (5,000)
        -- Total (All): 104,500 (110,000)
    },

    [pointType.PRESTIGE] =
    {
        { 1, 4950 }, -- 4750 from Lv75 + 200
        { 2,  400 },
        { 3,  600 },
        { 4,  800 },
        { 5, 1000 },
    },

    [pointType.MAAT] =
    {
        { 1, 250 },
    },

    [pointType.RANK] =
    {
        {  1,    0 },
        {  2,   25 },
        {  3,   75 },
        {  4,  175 },
        {  5,  350 },
        {  6,  500 },
        {  7,  650 },
        {  8,  725 },
        {  9, 1000 },
        { 10, 1500 },
        -- Total: 5,000
    },

    [pointType.ZILART] =
    {
        { xi.mission.id.zilart.THE_TEMPLE_OF_UGGALEPIH,     200 }, -- ZM 4
        { xi.mission.id.zilart.HEADSTONE_PILGRIMAGE,        500 }, -- ZM 5
        { xi.mission.id.zilart.THROUGH_THE_QUICKSAND_CAVES, 500 }, -- ZM 6
        { xi.mission.id.zilart.RETURN_TO_DELKFUTTS_TOWER,   500 }, -- ZM 8
        { xi.mission.id.zilart.THE_MITHRA_AND_THE_CRYSTAL,  300 }, -- ZM 12
        { xi.mission.id.zilart.ARK_ANGELS,                 1000 }, -- ZM 14
        { xi.mission.id.zilart.THE_CELESTIAL_NEXUS,         500 }, -- ZM 16
        -- Total: 3,500 (0.5 sets)
    },

    [pointType.PROMATHIA] =
    {
        { xi.mission.id.cop.THE_MOTHERCRYSTALS,    1500 }, -- PM 1-3
        { xi.mission.id.cop.DISTANT_BELIEFS,        500 }, -- PM 2-3
        { xi.mission.id.cop.ANCIENT_VOWS,          1000 }, -- PM 2-5
        { xi.mission.id.cop.DARKNESS_NAMED,        1000 }, -- PM 3-5
        { xi.mission.id.cop.THE_SAVAGE,            2000 }, -- PM 4-2
        { xi.mission.id.cop.DESIRES_OF_EMPTINESS,  2000 }, -- PM 5-2
        { xi.mission.id.cop.THREE_PATHS,           3000 }, -- PM 5-3
        { xi.mission.id.cop.ONE_TO_BE_FEARED,      1000 }, -- PM 6-4
        { xi.mission.id.cop.WHEN_ANGELS_FALL,      1000 }, -- PM 8-3
        { xi.mission.id.cop.DAWN,                  1000 }, -- PM 8-4
        -- Total: 15,000 (1.5 sets)
    },

    [pointType.TREASURES] =
    {
        { xi.mission.id.toau.THE_BLACK_COFFIN,     500 }, -- AU 15
        { xi.mission.id.toau.SHIELD_OF_DIPLOMACY,  500 }, -- AU 22
        { xi.mission.id.toau.PUPPET_IN_PERIL,     1000 }, -- AU 29
        { xi.mission.id.toau.SHADES_OF_VENGEANCE,  500 }, -- AU 31
        { xi.mission.id.toau.LEGACY_OF_THE_LOST,   500 }, -- AU 35
        { xi.mission.id.toau.PATH_OF_DARKNESS,    1000 }, -- AU 42
        { xi.mission.id.toau.NASHMEIRAS_PLEA,     1000 }, -- AU 44
        -- Total: 5,000
    },

    [pointType.QUEST] =
    {
        { { xi.questLog.JEUNO,    xi.quest.id.jeuno.CHOCOBOS_WOUNDS },    50 },
        { { xi.questLog.OUTLANDS, xi.quest.id.outlands.DIVINE_MIGHT },  1000 },
        { { xi.questLog.JEUNO,    xi.quest.id.jeuno.STORMS_OF_FATE  },  1000 },
        { { xi.questLog.JEUNO,    xi.quest.id.jeuno.APOCALYPSE_NIGH },  2000 },

        -- Lu Shang's Fishing Rod
        { { xi.questLog.SANDORIA, xi.quest.id.sandoria.THE_RIVALRY     },  2000 },
        { { xi.questLog.SANDORIA, xi.quest.id.sandoria.THE_COMPETITION },  2000 },

        -- Ebisu Fishing Rod
        { { xi.questLog.OUTLANDS, xi.quest.id.outlands.INDOMITABLE_SPIRIT }, 1000 },
    },

    [pointType.CUSTOM] =
    {
        -- A Matter of Trust (Bastok)
        { { "[CW]TRUST_BASTOK",         4 },  50 }, -- [15] A Matter of Trust
        { { "[CW]TRUST_BASTOK",         8 }, 100 }, -- [25] A Matter of Trust II
        { { "[CW]TRUST_BASTOK",        13 }, 200 }, -- [40] A Matter of Trust III
        { { "[CW]MINING_MY_BUSINESS",   2 },  50 }, -- [ 5] Mining My Business
        { { "[CW]RUNNING_RINGS_AROUND", 9 },  50 }, -- [ 1] Running Rings Around

        -- A Matter of Trust (San d'Oria)
        { { "[CW]TRUST_SANDORIA",        4 },  50 }, -- [15] A Matter of Trust
        { { "[CW]TRUST_SANDORIA",       10 }, 100 }, -- [25] A Matter of Trust II
        { { "[CW]TRUST_SANDORIA",       15 }, 200 }, -- [40] A Matter of Trust III
        { { "[CW]HATCHET_JOB",           2 },  50 }, -- [ 5] Hatchet Job
        { { "[CW]RING_AROUND_THE_ROSES", 4 },  50 }, -- [ 1] Ring Around the Roses

        -- A Matter of Trust (Windurst)
        { { "[CW]TRUST_WINDURST",  5 },  50 }, -- [15] A Matter of Trust
        { { "[CW]TRUST_WINDURST", 11 }, 100 }, -- [25] A Matter of Trust II
        { { "[CW]TRUST_WINDURST", 16 }, 200 }, -- [40] A Matter of Trust III
        { { "[CW]REAPING_REWARDS", 2 },  50 }, -- [ 5] Reaping Rewards
        { { "[CW]GIVE_ME_A_RING",  4 },  50 }, -- [ 1] Give Me a Ring

        -- Yagudo Arena
        { { "[CW]ARENA_WINS",     10 }, 100 }, -- 10 Arena wins
        { { "[CW]ARENA_WINS",     25 }, 250 }, -- 25 Arena wins
        { { "[CW]ARENA_WINS",     50 }, 500 }, -- 50 Arena wins

        -- Quest counters (San d'Oria)
        { { "[CW]HARE_MEAT",     300 }, 100 }, -- 300 Hare Meat
        { { "[CW]ACORNS",        300 }, 100 }, -- 300 Acorns
        { { "[CW]SHEEPSKIN",     300 }, 100 }, -- 300 Sheepskin

        -- Quest counters (Bastok)
        { { "[CW]INSECT_WINGS",  300 }, 100 }, -- 300 Insect Wings
        { { "[CW]BIRD_FEATHERS", 300 }, 100 }, -- 300 Bird Feathers
        { { "[CW]TAILS",         300 }, 100 }, -- 300 Lizard Tails

        -- Quest counters (Windurst)
        { { "[CW]YAGUDO_FEATHERS", 300 }, 100 }, -- 300 Yagudo Feathers
        { { "[CW]PAPAKA",          200 }, 100 }, -- 200 Papaka Grass
        { { "[CW]GIANT_STINGERS",  100 }, 100 }, -- 100 Giant Stingers
    },

    [pointType.SKILL] =
    {
        { { xi.skill.FISHING,       250 }, 1000 }, -- Fishing ( 25)
        { { xi.skill.FISHING,       500 }, 1500 }, -- Fishing ( 50)
        { { xi.skill.FISHING,       750 }, 2000 }, -- Fishing ( 75)
        { { xi.skill.FISHING,      1000 }, 2500 }, -- Fishing (100)

        { { xi.skill.WOODWORKING,   250 },  500 }, -- Woodworking ( 25)
        { { xi.skill.WOODWORKING,   500 }, 1000 }, -- Woodworking ( 50)
        { { xi.skill.WOODWORKING,   750 }, 1500 }, -- Woodworking ( 75)
        { { xi.skill.WOODWORKING,  1000 }, 2000 }, -- Woodworking (100)

        { { xi.skill.SMITHING,      250 },  500 }, -- Smithing ( 25)
        { { xi.skill.SMITHING,      500 }, 1000 }, -- Smithing ( 50)
        { { xi.skill.SMITHING,      750 }, 1500 }, -- Smithing ( 75)
        { { xi.skill.SMITHING,     1000 }, 2000 }, -- Smithing (100)

        { { xi.skill.GOLDSMITHING,  250 },  500 }, -- Goldsmithing ( 25)
        { { xi.skill.GOLDSMITHING,  500 }, 1000 }, -- Goldsmithing ( 50)
        { { xi.skill.GOLDSMITHING,  750 }, 1500 }, -- Goldsmithing ( 75)
        { { xi.skill.GOLDSMITHING, 1000 }, 2000 }, -- Goldsmithing (100)

        { { xi.skill.CLOTHCRAFT,    250 },  500 }, -- Clothcraft ( 25)
        { { xi.skill.CLOTHCRAFT,    500 }, 1000 }, -- Clothcraft ( 50)
        { { xi.skill.CLOTHCRAFT,    750 }, 1500 }, -- Clothcraft ( 75)
        { { xi.skill.CLOTHCRAFT,   1000 }, 2000 }, -- Clothcraft (100)

        { { xi.skill.LEATHERCRAFT,  250 },  500 }, -- Leathercraft ( 25)
        { { xi.skill.LEATHERCRAFT,  500 }, 1000 }, -- Leathercraft ( 50)
        { { xi.skill.LEATHERCRAFT,  750 }, 1500 }, -- Leathercraft ( 75)
        { { xi.skill.LEATHERCRAFT, 1000 }, 2000 }, -- Leathercraft (100)

        { { xi.skill.BONECRAFT,     250 },  500 }, -- Bonecraft ( 25)
        { { xi.skill.BONECRAFT,     500 }, 1000 }, -- Bonecraft ( 50)
        { { xi.skill.BONECRAFT,     750 }, 1500 }, -- Bonecraft ( 75)
        { { xi.skill.BONECRAFT,    1000 }, 2000 }, -- Bonecraft (100)

        { { xi.skill.ALCHEMY,       250 },  500 }, -- Alchemy ( 25)
        { { xi.skill.ALCHEMY,       500 }, 1000 }, -- Alchemy ( 50)
        { { xi.skill.ALCHEMY,       750 }, 1500 }, -- Alchemy ( 75)
        { { xi.skill.ALCHEMY,      1000 }, 2000 }, -- Alchemy (100)

        { { xi.skill.COOKING,       250 },  250 }, -- Cooking ( 25)
        { { xi.skill.COOKING,       500 },  500 }, -- Cooking ( 50)
        { { xi.skill.COOKING,       750 },  750 }, -- Cooking ( 75)
        { { xi.skill.COOKING,      1000 }, 1000 }, -- Cooking (100)

        { { xi.skill.DIG,           250 },  250 }, -- Digging ( 25)
        { { xi.skill.DIG,           500 },  500 }, -- Digging ( 50)
        { { xi.skill.DIG,           750 },  750 }, -- Digging ( 75)
        { { xi.skill.DIG,          1000 }, 1000 }, -- Digging (100)
    },

    [pointType.NM] =
    {
        { "Leaping_Lizzy",        100 },
        { "Jaggedy-Eared_Jack",   100 },
        { "Spiny_Spipi",          100 },
        { "Valkurm_Emperor",      200 },
        { "Argus",                300 },

        { "Genbu",               1000 },
        { "Seiryu",              1000 },
        { "Suzaku",              1000 },
        { "Byakko",              1000 },
        { "Kirin",               2000 },

        { "Proto-Omega",         2000 },
        { "Proto-Ultima",        2000 },

        { "Jailer_of_Prudence",  1000 },
        { "Jailer_of_Hope",      1000 },
        { "Jailer_of_Justice",   1000 },
        { "Jailer_of_Love",      3000 },

        { "Overlords_Tombstone",  500 },
        { "GuDha_Effigy",         500 },
        { "Tzee_Xicu_Idol",       500 },
        { "Goblin_Golem",         500 },
        { "Angra_Mainyu",        1000 },
        { "Dynamis_Lord",        2000 },

        { "Cirrate_Christelle",  1000 },
        { "Apocalyptic_Beast",   1000 },
        { "Antaeus",             1000 },
        { "Diabolos_Diamond",    2000 },
    },
}

for i = 1, #points do
    local category = points[i]
    local t = 0

    for j = 1, #category do
        t = t + category[j][2]
        category[j][3] = t
    end
end

local getTierAmount = function(value, pType)
    local highest = 0

    for i = 1, #points[pType] do
        if value >= points[pType][i][1] then
            highest = points[pType][i][3]
        else
            return highest
        end
    end

    return highest
end

local getMissionAmount = function(current, pType, logID)
    local highest = 0

    if current == 65535 then
        return highest
    end

    for i = 1, #points[pType] do
        if current > points[pType][i][1] then
            highest = points[pType][i][3]
        else
            return highest
        end
    end

    return highest
end

local lookup =
{
    [pointType.LEVEL] = function(player)
        local total = 0

        for k, v in pairs(xi.job) do
            local level = player:getJobLevel(v)

            if level < 75 and player:getCharVar(fmt("[CW]PRESTIGE_{}", xi.jobNames[v])) > 0 then
                level = 75
            end

            local tier  = getTierAmount(level, pointType.LEVEL)
            total = total + tier
        end

        return total
    end,

    [pointType.PRESTIGE] = function(player)
        local total = 0

        if player:getCharVar("[CW]MASTER") == 0 then
            return 0
        end

        for jobID = 1, 22 do
            local result = player:getCharVar(fmt("[CW]PRESTIGE_{}", xi.jobNames[jobID][1]))

            if result > 0 then
                total = total + points[pointType.PRESTIGE][result][3]
            end
        end

        return total
    end,

    [pointType.MAAT] = function(player)
        local value = player:getCharVar("maatsCap")
        local total = utils.mask.countBits(value) * points[pointType.MAAT][1][2]

        return total
    end,

    [pointType.RANK] = function(player)
        local sandoria = player:getRank(xi.nation.SANDORIA)
        local windurst = player:getRank(xi.nation.WINDURST)
        local bastok = player:getRank(xi.nation.BASTOK)

        local total =
            points[pointType.RANK][sandoria][3] +
            points[pointType.RANK][windurst][3] +
            points[pointType.RANK][bastok][3]

        return total
    end,

    [pointType.ZILART] = function(player)
        local current = player:getCurrentMission(xi.mission.log_id.ZILART)
        local total   = getMissionAmount(current, pointType.ZILART, xi.mission.log_id.ZILART)

        return total
    end,

    [pointType.PROMATHIA] = function(player)
        local current = player:getCurrentMission(xi.mission.log_id.COP)
        local total   = getMissionAmount(current, pointType.PROMATHIA, xi.mission.log_id.COP)

        return total
    end,

    [pointType.TREASURES] = function(player)
        local current = player:getCurrentMission(xi.mission.log_id.TOAU)
        local total   = getMissionAmount(current, pointType.TREASURES, xi.mission.log_id.TOAU)

        return total
    end,

    [pointType.QUEST] = function(player)
        local total = 0

        for i = 1, #points[pointType.QUEST] do
            local status = player:getQuestStatus(points[pointType.QUEST][i][1][1], points[pointType.QUEST][i][1][2])

            if status == xi.questStatus.QUEST_COMPLETED then
                total = total + points[pointType.QUEST][i][2]
            end
        end

        return total
    end,

    [pointType.CUSTOM] = function(player)
        local total = 0

        for i = 1, #points[pointType.CUSTOM] do
            local status = player:getCharVar(points[pointType.CUSTOM][i][1][1])

            if status >= points[pointType.CUSTOM][i][1][2] then
                total = total + points[pointType.CUSTOM][i][2]
            end
        end

        return total
    end,

    [pointType.SKILL] = function(player)
        local total = 0

        for i = 1, #points[pointType.SKILL] do
            local skillLevel = player:getCharSkillLevel(points[pointType.SKILL][i][1][1])

            if skillLevel >= points[pointType.SKILL][i][1][2] then
                total = total + points[pointType.SKILL][i][2]
            end
        end

        return total
    end,

    [pointType.NM] = function(player)
        local current = player:getCharVar("[CW]NM_UNLOCK")
        local total   = 0

        for i = 1, #points[pointType.NM] do
            if utils.mask.getBit(current, i) then
                total = total + points[pointType.NM][i][2]
            end
        end

        return total
    end,
}

local getPoints = function(player)
    local total = 0

    for k, v in pairs(points) do
        total = total + lookup[k](player)
    end

    return total
end

local getPointsCategory = function(player)
    local total = 0

    for k, v in pairs(points) do
        total = total + lookup[k](player)
    end

    return total
end

local function delaySendMenu(player, menuNext)
    local menu = menuNext

    player:timer(50, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local checkMilestones = function(player)
    local spent   = player:getCharVar(vars.CW_SPENT)
    local point   = player:getLocalVar(vars.REWARD_POINTS)
    local balance = point - spent
    local tbl = {}

    table.insert(tbl, menu.START_TALLY)

    local jobs      = points[pointType.LEVEL][#points[pointType.LEVEL]][3]
    local maat      = points[pointType.MAAT][#points[pointType.MAAT]][3]
    local prestige  = points[pointType.PRESTIGE][#points[pointType.PRESTIGE]][3]
    local jobsTotal = (jobs + maat + prestige) * 22
    local currTotal = lookup[pointType.LEVEL](player) + lookup[pointType.MAAT](player) + lookup[pointType.PRESTIGE](player)

    table.insert(tbl, string.format("[%u / %u] Job Levels", currTotal, jobsTotal))

    for k, v in pairs(milestones) do
        local msText  = v[1]
        local msType  = v[2]
        local current = lookup[msType](player)
        local last    = points[msType][#points[msType]]
        local total   = last[3]

        table.insert(tbl, string.format("[%u / %u] %s", current, total, msText))
    end

    table.insert(tbl, string.format(menu.TOTAL, point, settings.currency, balance))
    cexi.util.dialog(player, tbl, entities.npc.name)
end

local function openBox(zone)
    local result = zone:queryEntitiesByName("DE_" .. settings.boxName)

    if result == nil then
        return
    end

    for _, de in pairs(result) do
        de:entityAnimationPacket("open")

        de:timer(3000, function(npcArg)
            npcArg:entityAnimationPacket("clos")
        end)
    end
end

local purchaseItem = function(player, itemID, itemName, cost)
    local spent   = player:getCharVar(vars.CW_SPENT)
    local point   = player:getLocalVar(vars.REWARD_POINTS)
    local balance = point - spent

    if balance < 0 then
        balance = 0
    end

    if cost > balance then
        cexi.util.dialog(player, { menu.AFFORD }, entities.npc.name)
        return
    end

    openBox(player:getZone())

    -- giveItem should give message if you can't obtain
    -- Only deduct points if item received
    if npcUtil.giveItem(player, itemID) then
        printf("[petros] %s bought %s (%u) for %u, balance: %u -> %u", player:getName(), itemName, itemID, cost, balance, point - (spent + cost))
        player:incrementCharVar(vars.CW_SPENT, cost)
    end
end

local function confirmPage(player)
    local options  = {}
    local page     = player:getLocalVar(vars.REWARD_PAGE_JOB)
    local armor    = player:getLocalVar(vars.REWARD_PAGE_ARMOR)
    local piece    = player:getLocalVar(vars.REWARD_PAGE_PIECE)
    local selected = armorSets[page][armor].set[piece]
    local spent    = player:getCharVar(vars.CW_SPENT)
    local point    = player:getLocalVar(vars.REWARD_POINTS)
    local balance  = point - spent
    local cost     = armorCost[piece]

    if balance < 0 then
        balance = 0
    end

    delaySendMenu(player, {
        title   = string.format(menu.BUY_FOR, selected[2], cost, settings.currency, balance),
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function(playerArg)
                    purchaseItem(playerArg, selected[1], selected[2], cost)
                end,
            },
        },
    })
end

-----------------------------------
-- Armor Page
-- View list pieces on a set
-----------------------------------

local function armorPage(player)
    local options = {}
    local page    = player:getLocalVar(vars.REWARD_PAGE_JOB)
    local armor   = player:getLocalVar(vars.REWARD_PAGE_ARMOR)

    for i = 1, 5 do
        local armorObj = armorSets[page][armor].set[i]
        local cost     = armorCost[i]

        table.insert(options, {
            string.format("(%u) %s", cost, armorObj[2]), 
            function(playerArg)
                player:setLocalVar(vars.REWARD_PAGE_PIECE, i)
                confirmPage(player)
                -- TODO: Purchase menu
            end,
        })
    end

    return options
end

-----------------------------------
-- Job Page
-- View list of all sets
-----------------------------------

local function jobPage(player)
    local options = {}
    local page    = player:getLocalVar(vars.REWARD_PAGE_JOB)

    for i = 1, #armorSets[page] do
        table.insert(options, {
            string.format("(%s) %s", armorSets[page][i].job, armorSets[page][i].title),
            function(playerArg)
                player:setLocalVar(vars.REWARD_PAGE_ARMOR, i)

                delaySendMenu(player, {
                    title   = string.format(menu.SELECT_PIECE),
                    options = armorPage(player),
                })
            end
        })
    end

    if page ~= #armorSets then
        table.insert(options, {
            "Next Page",
            function(player)
                local page  = player:getLocalVar(vars.REWARD_PAGE_JOB)
                player:setLocalVar(vars.REWARD_PAGE_JOB, page + 1)

                delaySendMenu(player, {
                    title   = string.format(menu.SELECT_SET),
                    options = jobPage(player),
                })
            end,
        })
    end

    return options
end

-----------------------------------
-- Misc Page
-- View list of all misc
-----------------------------------

local function miscPage(player)
    local options = {}
    local page    = player:getLocalVar(vars.REWARD_PAGE_MISC)
    local spent   = player:getCharVar(vars.CW_SPENT)
    local point   = player:getLocalVar(vars.REWARD_POINTS)
    local balance = point - spent

    for i = 1, #miscItems[page] do
        table.insert(options, {
            string.format("(%u) %s", miscItems[page][i][3], miscItems[page][i][2]),
            function(playerArg)
                local item = miscItems[page][i]
                local cost = item[3]

                delaySendMenu(player, {
                    title   = string.format(menu.BUY_FOR, item[2], item[3], settings.currency, balance),
                    options =
                    {
                        {
                            "No",
                            function()
                            end,
                        },
                        {
                            "Yes",
                            function(playerArg)
                                purchaseItem(playerArg, item[1], item[2], item[3])
                            end,
                        },
                    },
                })
            end,
        })
    end

    if page ~= #miscItems then
        table.insert(options, {
            "Next Page",
            function(player)
                local page  = player:getLocalVar(vars.REWARD_PAGE_MISC)
                player:setLocalVar(vars.REWARD_PAGE_MISC, page + 1)

                delaySendMenu(player, {
                    title   = string.format(menu.SELECT_ITEM),
                    options = miscPage(player),
                })
            end,
        })
    end

    return options
end

local getCraftTotal = function(player)
    local total = 0

    for i = xi.skill.FISHING, xi.skill.COOKING do
        total = total + player:getCharSkillLevel(i)
    end

    return total
end

-----------------------------------
-- Crystal Warrior Introduction
-- Choose Unbreakable / Receive Provenance Ring
-----------------------------------

local dialog =
{
    INTRO =
    {
        " ",
        { emote = xi.emote.WELCOME },
        "Very well. I will send you to Vana'diel.",
        "But first you must choose your path.",
        " ",
        { message = "=== Unbreakable is an optional hardcore challenge ===" },
        { message = "Unbreakable Crystal Warriors reset to level 1 on death." },
        { message = "Make sure you understand the implications." },
        " ",
    },

    CONFIRM_UNBREAKABLE =
    {
        " ",
        { message = "=== You will become an Unbreakable Crystal Warrior ===" },
        { message = "Unbreakable Crystal Warriors reset to level 1 on death." },
        "This journey is not for the faint hearted. Please make sure you understand.",
    },

    ACCEPT_UNBREAKABLE =
    {
        " ",
        { message = "=== You have become Unbreakable ===" },
        { message = "You will reset to level 1 on death." },
        "It is done.",
        { emote = xi.emote.YES },
    },

    CONFIRM_STANDARD =
    {
        " ",
        { message = "=== You will become a standard Crystal Warrior ===" },
        { message = "You will no longer be able to become \"Unbreakable\"." },
        "This is your only chance to choose. Are you certain?",
    },

    ACCEPT_STANDARD =
    {
        " ",
        { message = "=== You are a standard Crystal Warrior ===" },
        { message = "You will not reset to level 1 on death." },
        "It is done.",
        { emote = xi.emote.YES },
    },

    RING =
    {
        " ",
        "I shall send you to Vana'diel. Your journey begins now.",
        "Take this ring, should you wish to return here.",
        { emote = xi.emote.BOW },
        { delay = 3000 },
    }
}

local function confirmUnbreakable(player, npc)
    local delay = cexi.util.dialogDelay(dialog.CONFIRM_UNBREAKABLE)
    cexi.util.dialog(player, dialog.CONFIRM_UNBREAKABLE, "Petros", { npc = npc })
    player:timer(delay, function(npcArg)
        delaySendMenu(player,
        {
            title   = "Become an Unbreakable Crystal Warrior?",
            options =
            {
                {
                    "Let me think about it.",
                    function()
                    end,
                },
                {
                    "Yes.",
                    function()
                        player:setUnbreakable(true)
                        player:setCharVar(vars.CW_INTRO, 1)
                        cexi.util.dialog(player, dialog.ACCEPT_UNBREAKABLE, "Petros", { npc = npc })
                    end,
                },
            }
        })
    end)
end

local function confirmStandard(player, npc)
    local delay = cexi.util.dialogDelay(dialog.CONFIRM_STANDARD)
    cexi.util.dialog(player, dialog.CONFIRM_STANDARD, "Petros", { npc = npc })
    player:timer(delay, function(playerArg)
        delaySendMenu(player,
        {
            title   = "Remain a Standard Crystal Warrior?",
            options =
            {
                {
                    "Let me think about it.",
                    function()
                    end,
                },
                {
                    "Yes.",
                    function()
                        player:setCharVar(vars.CW_INTRO, 1)
                        cexi.util.dialog(player, dialog.ACCEPT_STANDARD, "Petros", { npc = npc })
                    end,
                },
            }
        })
    end)
end

local function handleIntro(player, npc)
    local delay = cexi.util.dialogDelay(dialog.INTRO)
    cexi.util.dialog(player, dialog.INTRO, "Petros", { npc = npc })

    npc:timer(delay, function(npcArg)
        delaySendMenu(player,
        {
            title = "Which path will you choose?",
            options =
            {
                {
                    "Let me think about it.",
                    function()
                    end,
                },
                {
                    "Standard Crystal Warrior",
                    function(playerArg)
                        confirmStandard(player, npc)
                    end,
                },
                {
                    "Unbreakable Crystal Warrior",
                    function(playerArg)
                        confirmUnbreakable(player, npc)
                    end,
                },
            }
        })
    end)
end

local startAreas =
{
    [xi.nation.SANDORIA] = { 152.145, 0.000, 195.839, 100,    xi.zone.KING_RANPERRES_TOMB }, -- !pos 152.145 0.000 195.839 190
    [xi.nation.BASTOK]   = { -452.215, 3.987, 133.918, 150,   xi.zone.DANGRUF_WADI        }, -- !pos -452.215 3.987 133.918 191
    [xi.nation.WINDURST] = { 180.019, 8.500, 22.654, 64,      xi.zone.TORAIMARAI_CANAL    }, -- !pos 180.019 8.500 22.654 169
}

local finishAreas =
{
    [xi.nation.SANDORIA] = { 658.330, -19.601, -575.852, 170, xi.zone.EAST_RONFAURE       }, -- !pos 658.330 -19.601 -575.852
    [xi.nation.BASTOK]   = { -492.033, 30.919, -489.012, 240, xi.zone.SOUTH_GUSTABERG     }, -- !pos -492.033 30.919 -489.012
    [xi.nation.WINDURST] = { 141.004, -15.049, -405.378, 132, xi.zone.EAST_SARUTABARUTA   }, -- !pos 141.004 -15.049 -405.378
}

local function handleRing(player, npc)
    local delay = cexi.util.dialogDelay(dialog.RING)
    cexi.util.dialog(player, dialog.RING, "Petros", { npc = npc })

    npc:timer(delay, function(npcArg)
        if npcUtil.giveItem(player, 28470) then -- Provenance Ring
            player:setCharVar(vars.CW_INTRO, 2)
            player:addStatusEffectEx(xi.effect.TERROR, xi.effect.TERROR, 0, 0, 13)
            player:independentAnimation(player, 47, 4)
            npcArg:timer(5000, function(npcArg2)
                player:independentAnimation(player, 42, 4)
            end)

            npcArg:timer(10500, function(npcArg2)
                -- Finish animation
                local area = startAreas[player:getNation()]
                player:setHomePointAt(area[1], area[2], area[3], area[4], area[5])
                player:setPos(area[1], area[2], area[3], area[4], area[5])
            end)
        end
    end)
end

local lostMessage =
{
    "It says here, you've lost over %u Experience points! My condolences.",
    "Please take this small gesture as a consolation.",
    { emote = xi.emote.BOW },
}

local function handleLostEXP(player, npc)
    local obtainedLostExpItem = player:getCharVar(vars.CW_EXP_LOST_ITEM)
    local lostEXP = player:getCharVar(vars.CW_EXP_LOST)

    for itemIndex, itemInfo in pairs(cexi.crystal_warrior.milestones.lost_exp) do
        if
            lostEXP >= itemInfo[1] and
            not player:hasItem(itemInfo[2]) and
            not utils.mask.getBit(obtainedLostExpItem, itemIndex)
        then
            local delay = cexi.util.dialogDelay(lostMessage)
            cexi.util.dialog(player, lostMessage, "Petros", { [1] = itemInfo[1], npc = npc })

            player:timer(delay, function(playerArg)
                if npcUtil.giveItem(player, itemInfo[2]) then
                    -- Prevent giving player on repeat visits
                    local result = utils.mask.setBit(obtainedLostExpItem, itemIndex, true)
                    player:setCharVar(vars.CW_EXP_LOST_ITEM, result)
                end
            end)

            return true
        end
    end

    return false
end

-----------------------------------
-- Reward NPC Trigger
-- View all options
-----------------------------------

local npcOnTrigger = function(player, npc)
    npc:independentAnimation(npc, 38, 4)

    if player:getCharVar(vars.INTRO_CS) == 0 then
        print(string.format("DEBUG: Kicking off intro cutscene from Petros for %s.", player:getName()))
        player:setPos(provStart[1], provStart[2], provStart[3], provStart[4])
        player:addStatusEffectEx(xi.effect.TERROR, xi.effect.TERROR, 0, 0, 53)
        player:timer(100, function(playerArg)
            cexi.util.dialog(player, cutscene)
        end)

        return
    end

    if player:getCharVar(vars.CW_INTRO) == 0 then

        handleIntro(player, npc)
        return
    elseif player:getCharVar(vars.CW_INTRO) == 1 then
        handleRing(player, npc)
        return
    end

    if handleLostEXP(player, npc) then
        return
    end

    local spent = player:getCharVar(vars.CW_SPENT)
    local point = getPoints(player)
    local total = point - spent

    if total < 0 then
        total = 0
    end

    local totalLevels = player:getTotalLevel()

    if player:getCharVar("[CW]MASTER") > 0 then
        for jobID = 1, 22 do
            local result = player:getCharVar(fmt("[CW]PRESTIGE_{}", xi.jobNames[jobID][1]))

            if result > 0 then
                totalLevels = totalLevels + (result * 75)
            end
        end
    end

    player:setLocalVar(vars.REWARD_POINTS, point)
    player:setCharVar("[CW]MILESTONES", point)
    player:setCharVar("[CW]GIL", player:getGil())
    player:setCharVar("[CW]TOTAL_LEVEL", totalLevels)
    player:setCharVar("[CW]TOTAL_CRAFT", getCraftTotal(player))

    local options =
    {
        {
            menu.MAIN.NOTHING,
            function()
            end,
        },
        {
            menu.MAIN.MILESTONES,
            checkMilestones,
        },
    }

    if GetServerVariable("[CW]DISABLE_REWARDS") == 0 then
        table.insert(options, {
            menu.MAIN.ARMOR,
            function(playerArg)
                player:setLocalVar(vars.REWARD_PAGE_JOB, 1)
                delaySendMenu(playerArg, {
                    title   = menu.SELECT_SET,
                    options = jobPage(playerArg),
                })
            end,
        })

        table.insert(options, {
            menu.MAIN.MISC,
            function(playerArg)
                player:setLocalVar(vars.REWARD_PAGE_MISC, 1)
                delaySendMenu(playerArg, {
                    title   = menu.SELECT_ITEM,
                    options = miscPage(playerArg),
                })
            end,
        })
    end

    if
        player:getCharVar("[CW]MASTER") == 1 and
        player:getMainLvl() == 75
    then
        local jobID   = player:getMainJob()
        local varName = fmt("[CW]PRESTIGE_{}", xi.jobNames[jobID][1])

        if player:getCharVar(varName) < 5 then
            table.insert(options, {
                fmt("Prestige ({})", xi.jobNames[jobID][2]),
                function(playerArg)
                    local tier  = player:getCharVar(varName) + 1
                    local bonus = cexi.prestigeDesc(jobID, tier)

                    delaySendMenu(player, {
                        title = fmt("Reset {} to Level 1?", xi.jobNames[jobID][1]),
                        options =
                        {
                            {
                                "Not yet",
                                function()
                                end,
                            },
                            {
                                fmt("Gain Bonus: {}", bonus),
                                function()
                                    player:incrementCharVar(varName, 1)
                                    player:setLevel(1)
                                    player:delExp(499)

                                    player:sys(
                                        "You obtained a prestige upgrade! {}: {}{} ({})",
                                        xi.jobNames[jobID][2],
                                        string.rep("\129\154", tier),
                                        string.rep("\129\153", 5 - tier),
                                        bonus
                                    )

                                    player:independentAnimation(player, 113, 4)
                                end,
                            },
                        },
                    })
                end
            })
        end
    end

    table.insert(options, {
        menu.MAIN.RETURN_VANADIEL,
        function(playerArg)
            delaySendMenu(playerArg, {
                title   = menu.RETURN_CONFIRM,
                options =
                {
                    {
                        "Not yet.",
                        function()
                        end,
                    },
                    {
                        "Return me to Vana'diel.",
                        function()
                            local finishPos = finishAreas[player:getNation()]
                            player:setPos(unpack(finishPos))
                        end,
                    },
                }
            })
        end,
    })

    delaySendMenu(player, {
        title   = string.format(menu.MAIN.TITLE, total, settings.currency),
        options = options,
    })
end

local tiersText =
{
    "I",
    "II",
    "III",
    "IV",
    "V",
}

local function getNextTier(item, augs)
    local aug0 = item:getAugment(0)
    local aug1 = item:getAugment(1)

    for i = 1, #augs do
        if
            augs[i][1] == aug0[1] and
            augs[i][2] == aug0[2] and
            augs[i][3] == aug1[1] and
            augs[i][4] == aug1[2]
        then
            return i + 1
        end
    end

    return 1
end

local function getAugTier(trade, itemID, augs)
    for i = 0, trade:getSlotCount() - 1 do
        if trade:getItemId(i) == itemID then
            return getNextTier(trade:getItem(i), augs)
        end
    end

    return 1
end

local npcOnTrade = function(player, npc, trade)
    local spent = player:getCharVar(vars.CW_SPENT)
    local point = getPoints(player)
    local total = point - spent

    if total < 0 then
        cexi.util.dialog(player, { menu.UPGRADE.REJECT }, entities.npc.name)
        return
    end

    for i = 1, #upgrades do
        if npcUtil.tradeHasExactly(trade, upgrades[i][1]) then
            local tier = getAugTier(trade, upgrades[i][1], upgrades[i][3])
            local tbl   = {}

            if tier == #tiersText then
                cexi.util.dialog(player, { string.format(menu.UPGRADE.LIMIT, upgrades[i][2], tiersText[tier]) }, entities.npc.name)
                return
            end

            table.insert(tbl, string.format(menu.UPGRADE.CURRENT, upgrades[i][2], tiersText[tier], total, settings.currency))

            local cost = upgradeCost[tier]

            if cost > total then
                cexi.util.dialog(player, {
                    string.format(menu.UPGRADE.COST_1, upgrades[i][2], tiersText[tier], total, cost),
                    menu.UPGRADE.COST_2
                }, entities.npc.name)

                return
            end

            table.insert(tbl, string.format(menu.UPGRADE.CONFIRM, upgrades[i][2], tiersText[tier + 1], upgradeCost[tier], settings.currency))
            cexi.util.dialog(player, tbl, entities.npc.name)

            local delay = cexi.util.dialogDelay(tbl)

            npc:timer(delay, function()
                delaySendMenu(player, {
                    title = string.format(menu.UPGRADE.ASK, upgrades[i][2], upgradeCost[tier]),
                    options =
                    {
                        {
                            "No",
                            function(playerArg)
                                cexi.util.dialog(player, { menu.UPGRADE.DECLINE }, entities.npc.name)
                            end,
                        },

                        {
                            "Yes",
                            function(playerArg)
                                playerArg:confirmTrade()

                                local ID = zones[player:getZoneID()]

                                if playerArg:getFreeSlotsCount() > 0 then
                                    playerArg:tradeComplete()
                                    playerArg:addItem(upgrades[i][1], 1, unpack(upgrades[i][3][tier]))
                                    playerArg:incrementCharVar(vars.CW_SPENT, upgradeCost[tier])

                                    npc:independentAnimation(playerArg, 4, 2)

                                    npc:timer(2000, function()
                                        playerArg:messageSpecial(ID.text.ITEM_OBTAINED, upgrades[i][1])
                                    end)

                                    return true
                                else
                                    playerArg:messageSpecial(ID.text.ITEM_CANNOT_BE_OBTAINED, upgrades[i][1])
                                    return false
                                end
                            end,
                        },
                    }
                })
            end)
        end
    end
end

-- Door opens during intro
m:addOverride("xi.zones.Toraimarai_Canal.npcs._4pc.onTrigger", function(player, npc)
    if player:isCrystalWarrior() then
        local introStatus = player:getCharVar(vars.CW_INTRO)

        if introStatus > 0 and introStatus < 8 then
            player:startEvent(70, 0, 0, 0, 2)
            return
        end
    end

    super(player, npc)
end)

-- Disable portal to Heaven's Tower during intro
m:addOverride("xi.zones.Toraimarai_Canal.npcs.Transporter.onTrigger", function(player, npc)
    if player:isCrystalWarrior() then
        local introStatus = player:getCharVar(vars.CW_INTRO)

        if introStatus > 0 and introStatus < 8 then
            player:printToPlayer("You can't use that yet.", xi.msg.channel.NS_SAY)
            return
        end
    end

    super(player, npc)
end)

m:addOverride("xi.zones.Provenance.Zone.onZoneIn", function(player, prevZone)
    if
        player:isCrystalWarrior() and
        player:getCharVar(vars.CW_INTRO) == 0
    then
        print(string.format("DEBUG: Kicking off intro cutscene for %s.", player:getName()))
        player:setPos(provStart[1], provStart[2], provStart[3], provStart[4])
        player:addStatusEffectEx(xi.effect.TERROR, xi.effect.TERROR, 0, 0, 53)
        player:timer(100, function(playerArg)
            cexi.util.dialog(player, cutscene)
        end)
    else
        super(player, prevZone)
    end
end)

cexi.util.liveReload(m, {
    [settings.area] =
    {
        {
            name         = entities.box.name,
            objtype      = xi.objType.NPC,
            look         = entities.box.look,
            x            = settings.pos[1] + entities.box.offset[1],
            y            = settings.pos[2] + entities.box.offset[2],
            z            = settings.pos[3] + entities.box.offset[3],
            rotation     = settings.pos[4],
            hideName     = true,
            untargetable = true,
        },
        {
            name      = entities.npc.name,
            objtype   = xi.objType.NPC,
            look      = entities.npc.look,
            x         = settings.pos[1],
            y         = settings.pos[2],
            z         = settings.pos[3],
            rotation  = settings.pos[4],
            onTrigger = npcOnTrigger,
            onTrade   = npcOnTrade,
        },
    },
})

return m
