-----------------------------------
-- Crystal Warrior (Carnelian) - Restores broken/missing CW stuff
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
require('scripts/globals/player')
require('scripts/globals/npc_util')

local cq      = require("modules/catseyexi/lua/additive_overrides/utils/custom_quest")
local unlocks = require("modules/catseyexi/lua/additive_overrides/utils/custom_unlocks")
-----------------------------------
local m = Module:new("npc_carnelian")

local settings =
{
    name    = "Carnelian",
    area    = "Provenance",
    objtype = xi.objType.NPC,
    look    = cexi.util.look({
        race = xi.race.HUME_M,
        face = 5,
        body = 441, -- Delegate's Garb
        hand = 441, -- Delegate's Cuffs
        legs = 230, -- Dinner Hose
        feet = 230, -- Dinner Hose
    }),

    pos    = { 466.972, 28.525, -469.121, 12 }, -- !pos 466.972 28.525 -469.121 222

    dialog =
    {
        NAME    = true,
        DEFAULT = { "I am currently unable to assist you." },
    },

    disabled = false,
}

local titles =
{
    [xi.title.WARRIOR_OF_THE_CRYSTAL] = { "Warrior of the Crystal", xi.mission.log_id.COP, xi.mission.id.cop.WHEN_ANGELS_FALL },
}

local function delayMenu(player, menuNext)
    local menu = menuNext

    player:timer(200, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function setTitle(player, title)
    delayMenu(player, {
        title   = "Restore this title?",
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    player:setTitle(title)
                    player:printToPlayer("You receive a title.", xi.msg.channel.SYSTEM_3)
                end,
            },
        },
    })
end

local function restoreTitle(player)
    local options = {}

    table.insert(options, {
        "None",
        function()
        end,
    })

    for title, info in pairs(titles) do
        if player:getCurrentMission(info[2]) >= info[3] then
            table.insert(options, {
                info[1],
                function()
                    setTitle(player, title)
                end,
            })
        end
    end

    delayMenu(player, {
        title   = "Select a title:",
        options = options,
    })
end

local function getTotalSlots(player)
    local slots =
    {
        [xi.inv.WARDROBE]  = 0,
        [xi.inv.WARDROBE2] = 0,
        [xi.inv.WARDROBE3] = player:getContainerSize(xi.inv.WARDROBE3),
        [xi.inv.WARDROBE4] = 0,
    }

    for logID, questList in pairs(unlocks.wardrobeQuests) do
        for questID, receive in pairs(questList) do
            if player:hasCompletedQuest(logID, questID) then
                slots[receive[1]] = slots[receive[1]] + receive[2]
            end
        end
    end

    for _, row in pairs(unlocks.customQuests) do
        local questVar     = row[1][1]
        local questValue   = row[1][2]
        local wardrobeType = row[2][1]
        local slotsGranted = row[2][2]

        if player:getCharVar(questVar) >= questValue then
            slots[wardrobeType] = slots[wardrobeType] + slotsGranted
        end
    end

    for logID, missionList in pairs(unlocks.wardrobeMissions) do
        for missionID, receive in pairs(missionList) do
            if player:hasCompletedMission(logID, missionID) then
                slots[receive[1]] = slots[receive[1]] + receive[2]
            end
        end
    end

    return slots
end

local wardrobeList =
{
    { name = "Mog Wardrobe 1", id = xi.inv.WARDROBE  },
    { name = "Mog Wardrobe 2", id = xi.inv.WARDROBE2 },
    { name = "Mog Wardrobe 3", id = xi.inv.WARDROBE3 },
    { name = "Mog Wardrobe 4", id = xi.inv.WARDROBE4 },
}

local skillNames =
{
    [xi.skill.WOODWORKING]  = "WOODWORKING",
    [xi.skill.SMITHING]     = "SMITHING",
    [xi.skill.GOLDSMITHING] = "GOLDSMITHING",
    [xi.skill.CLOTHCRAFT]   = "CLOTHCRAFT",
    [xi.skill.LEATHERCRAFT] = "LEATHERCRAFT",
    [xi.skill.BONECRAFT]    = "BONECRAFT",
    [xi.skill.ALCHEMY]      = "ALCHEMY",
    [xi.skill.COOKING]      = "COOKING",
}

local function restoreWardrobe(player, npc)
    local result  = getTotalSlots(player)
    local message = { "I will now check your inventory allocation." }

    for _, wardrobe in pairs(wardrobeList) do
        local currSize = player:getContainerSize(wardrobe.id)
        local text     = string.format(" %s: [%u/%u]", wardrobe.name, currSize, result[wardrobe.id])
        

        if currSize < result[wardrobe.id] then
            local diff = result[wardrobe.id] - currSize
            text       = string.format("%s +%u missing slots restored!", text, diff)
            player:changeContainerSize(wardrobe.id, diff)
        end

        table.insert(message, text)
    end

    local caseSize    = player:getContainerSize(xi.inv.MOGCASE)
    local caseMissing = 0

    for skillID, skillName in pairs(skillNames) do
        local level    = player:getCharSkillLevel(skillID)
        local progVar  = string.format("[CW]SKILL_%s", skillName)
        local rank     = level / 100

        if rank >= 1 then
            local received = player:getCharVar(progVar)

            for i = 1, rank do
                if not utils.mask.getBit(received, i) then
                    received = utils.mask.setBit(received, i, true)
                    caseMissing = caseMissing + 1
                end
            end

            player:setCharVar(progVar, received)
        end
    end

    local caseText = string.format(" Mog Case: [%u/%u]", caseSize, caseSize + caseMissing)

    if caseMissing > 0 then
        if caseSize < 80 then
            caseText = string.format("%s +%u missing slots restored!", caseText, caseMissing)
            player:changeContainerSize(xi.inv.MOGCASE, caseMissing)
        end
    end

    table.insert(message, caseText)
    table.insert(message, "That concludes.")

    cexi.util.dialog(player, message, settings.name, { npc = npc })
end

local function processReset(player, npc)
    if not player:removeItemsCW() then
        player:printToPlayer("Something went wrong.", xi.msg.channel.NS_SAY)
        return
    end

    player:printToPlayer("Reset is being processed. Please wait.", xi.msg.channel.NS_SAY)

    player:setCharVar("[CW]RESET", 1)
    player:setCharVar("[CW]SPENT", 0)

    for jobID = 1, #xi.jobNames do
        player:setCharVar(fmt("[SQUIRE]CW_{}", xi.jobNames[jobID][1]), 0)
    end

    player:timer(5000, function()
        -- Refresh player
        player:setPos(player:getXPos(), player:getYPos(), player:getZPos(), player:getRotPos(), player:getZoneID())
    end)
end

local function resetMilestones(player, npc)
    player:printToPlayer("=== All Crystal Warrior items will be deleted ===", xi.msg.channel.SYSTEM_3)
    player:printToPlayer("=== Your Milestone Points spent will be reset ===", xi.msg.channel.SYSTEM_3)

    delayMenu(player, {
        title   = "Proceed with the reset?",
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    processReset(player, npc)
                end,
            },
        },
    })
end

local trusts =
{
    ["[CW]TRUST_SANDORIA"] =
    {
        [4]  = { 902, "Curilla",    1 },
        [15] = { 899, "Excenmille", 2, { xi.ki.SAN_DORIA_TRUST_PERMIT, xi.ki.RED_INSTITUTE_CARD } },
    },

    ["[CW]TRUST_BASTOK"] =
    {
        [4]  = { 903, "Volker", 1 },
        [13] = { 897, "Naji",   2, { xi.ki.BASTOK_TRUST_PERMIT, xi.ki.BLUE_INSTITUTE_CARD }, { xi.questLog.BASTOK, xi.quest.id.bastok.TRUST_BASTOK } },
    },

    ["[CW]TRUST_WINDURST"] =
    {
        [5]  = { 904, "Ajido-Marujido", 1 },
        [16] = { 898, "Kupipi",         2, { xi.ki.WINDURST_TRUST_PERMIT, xi.ki.GREEN_INSTITUTE_CARD } },
    },
}

local function restoreTrusts(player, npc)
    local message      = { "I will now check your trust allocation." }
    local currentSlots = player:getCharVar("[CW]TRUST")
    local grantedSlots = 0
    local changed      = false

    for varName, questInfo in pairs(trusts) do
        for varVal, trustInfo in pairs(questInfo) do
            if player:getCharVar(varName) >= varVal then
                if not player:hasSpell(trustInfo[1]) then
                    table.insert(message, { message = string.format("You learned Trust: %s!", trustInfo[2]) })
                    player:addSpell(trustInfo[1], true, true)
                    changed = true
                end

                if
                    trustInfo[3] > currentSlots and
                    trustInfo[3] > grantedSlots
                then
                    grantedSlots = trustInfo[3]
                    changed = true
                end

                -- Fixes missing key items
                if trustInfo[4] ~= nil then
                    for _, keyItem in pairs(trustInfo[4]) do
                        if not player:hasKeyItem(keyItem) then
                            npcUtil.giveKeyItem(player, keyItem)
                        end
                    end
                end

                -- Complete required quest
                if trustInfo[5] ~= nil then
                    player:completeQuest(trustInfo[5][1], trustInfo[5][2])
                end
            end
        end
    end

    if grantedSlots > currentSlots then
        table.insert(message, { message = string.format("Your trust capacity has increased to %i.", grantedSlots) })
        player:setCharVar("[CW]TRUST", grantedSlots)
    end

    if changed then
        table.insert(message, "That concludes.")
    else
        table.insert(message, "...")
        table.insert(message, "Everything seems in order.")
    end

    cexi.util.dialog(player, message, settings.name, { npc = npc })
end

local function restoreExpLostItems(player, npc)
    local lostEXP = player:getCharVar("[CW]EXP_LOST")
    local options =
    {
        {
            "None",
            function()
            end,
        },
    }

    for _, itemInfo in pairs(cexi.crystal_warrior.milestones.lost_exp) do
        if
            lostEXP >= itemInfo[1] and
            not player:hasItem(itemInfo[2])
        then
            table.insert(options, {
                itemInfo[3],
                function()
                    player:timer(500, function(playerArg)
                        cexi.util.dialog(player, {
                            "Be careful not to lose this again, adventurer."
                        }, settings.name, { npc = npc })

                        npcUtil.giveItem(player, itemInfo[2])
                    end)
                end,
            })
        end
    end

    delayMenu(player, {
        title   = "Reobtain reward:",
        options = options,
    })
end

local ammoStorage =
{
    { "Cmb.Cst. Arrow",  18740 },
    { "T.K. Arrow",       8738 },
    { "Ptr.Prt. Arrow",  17329 },
    { "Gnd.Kgt. Arrow",  17327 },

    { "Irn.Msk. Bolt",   18739 },
    { "Gld.Msk. Bolt",   17328 },
    { "Hightail Bullet", 19236 },
    { "Animikii Bullet", 21334 },

    { "Heavy Shell",     18255 },
    { "Nok. Shuriken",   18173 },
    { "Mad. Shuriken",   21214 },
}

local function confirmAmmo(player, npc, itemInfo)
    if player:hasItem(itemInfo[2]) then
        cexi.util.dialog(player, { "I can't restore that item if you're already holding it!" }, settings.name, { npc = npc })
        return
    end

    if player:getGil() < 10000 then
        cexi.util.dialog(player, { "This service costs 10,000 gil and you don't have enough." }, settings.name, { npc = npc })
        return
    end

    if player:getFreeSlotsCount() == 0 then
        cexi.util.dialog(player, { "You don't have enough inventory space to receive this." }, settings.name, { npc = npc })
        return
    end

    delayMenu(player, {
        title   = fmt("Restore {}? (10,000 gil)", itemInfo[1]),
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    if npcUtil.giveItem(player, itemInfo[2]) then
                        cexi.util.dialog(player, { "As promised, here is your restored item." }, settings.name, { npc = npc })
                        player:delGil(10000)
                    end
                end,
            },
        },
    })
end


local function restoreAmmo(player, npc)
    local stored = player:getCharVar("[CW]AMMO")

    if stored == 0 then
        cexi.util.dialog(player, {
            "I can restore your lost Ammo and Thown weapons for a fee.",
            " You don't curretly have any items registered with me.",
            " Trade me an eligible item and I will make a record, allowing this restoration service when required.",
        }, settings.name, { npc = npc })
        return
    end

    local ammoTbl = {}

    for itemIndex, itemInfo in pairs(ammoStorage) do
        if utils.mask.getBit(stored, itemIndex - 1) then
            table.insert(ammoTbl, itemInfo)
        end
    end

    cexi.util.simpleMenu(player, npc, ammoTbl, confirmAmmo, "Select an item to restore:")
end

local function onTrade(player, npc, trade)
    local stored = player:getCharVar("[CW]AMMO")

    for itemIndex, itemInfo in pairs(ammoStorage) do
        if
            cexi.util.tradeHasExactly(trade, itemInfo[2]) and
            (stored == 0 or not utils.mask.getBit(stored, itemIndex - 1))
        then
            local result = utils.mask.setBit(stored, itemIndex - 1, 1)
            player:setCharVar("[CW]AMMO", result)
            player:tradeComplete()

            cexi.util.dialog(player, { fmt("Your {} is now registered to restore when needed.", itemInfo[1]) }, settings.name, { npc = npc })
            return
        end
    end
end

local function onTrigger(player, npc)
    if settings.disabled then
        cexi.util.dialog(player, settings.dialog.DEFAULT, settings.name, { npc = npc })
    end

    local options =
    {
        {
            "Nothing",
            function()
            end,
        },
        {
            "Missing inventory slots",
            function()
                restoreWardrobe(player, npc)
            end,
        },
        {
            "Missing trusts",
            function()
                restoreTrusts(player, npc)
            end,
        },
        {
            "Restore Title",
            function()
                restoreTitle(player, npc)
            end,
        },
        {
            "Restore Ammo",
            function()
                restoreAmmo(player, npc)
            end,
        },
    }

    local lostEXP = player:getCharVar("[CW]EXP_LOST")

    if lostEXP > 0 then
        table.insert(options, {
            "EXP Lost",
            function()
                restoreExpLostItems(player, npc)
            end,
        })
    end

    if
        player:getCharVar("[CW]RESET") == 0 and
        player:getCharVar("[CW]SPENT") > 0
    then
        table.insert(options, {
            "Reset my Milestone Points",
            function()
                resetMilestones(player, npc)
            end,
        })
    end

    delayMenu(player, {
        title   = "What ails you, adventurer?",
        options = options,
    })
end

cexi.util.liveReload(m, {
    [settings.area] =
    {
        {
            name      = settings.name,
            objtype   = xi.objType.NPC,
            look      = settings.look,
            x         = settings.pos[1],
            y         = settings.pos[2],
            z         = settings.pos[3],
            rotation  = settings.pos[4],
            onTrigger = onTrigger,
            onTrade   = onTrade,
        }
    }
})

return m
