------------------------------------
-- HNM
------------------------------------
-- Claim shield mixin by Tonzoffun
-- Module rewrite by Loxley
-----------------------------------
require("modules/module_utils")
require("scripts/globals/utils")
require("scripts/mixins/job_special")
-----------------------------------
local m = Module:new("catseyexi_hnm")

m:addOverride("xi.zones.King_Ranperres_Tomb.mobs.Vrtra.onMobEngage", function(mob, target)
    -- full override due to not wanting the base lsb call to resetLocalVars()
    -- Reset the onMobFight variables manually
    mob:setLocalVar("spawnTime", 0)
    mob:setLocalVar("twohourTime", 0)
end)

-- Basic resist for all HNM, applied if:
-- resist = true
local basicResist =
{
    [xi.mod.SILENCERES]  =    50,
    [xi.mod.STUNRES]     =    50,
    [xi.mod.BINDRES]     =    50,
    [xi.mod.GRAVITYRES]  =    50,
    [xi.mod.SLEEPRES]    = 10000,
    [xi.mod.POISONRES]   =   100,
    [xi.mod.PARALYZERES] =   100,
    [xi.mod.LULLABYRES]  = 10000,
}

local areas =
{
    ["Valley_of_Sorrows"] =
    {
        ["Adamantoise"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
        ["Aspidochelone"] =
        {
            rage   = utils.minutes(30),
            resist = true,
        },
    },
    ["Behemoths_Dominion"] =
    {
        ["Behemoth"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
            mods   =
            {
                [xi.mod.TRIPLE_ATTACK] =  5,
                [xi.mod.MDEF]          = 20,
                [xi.mod.EVA]           = 50,
            },
        },
        ["King_Behemoth"] =
        {
            rage   = utils.minutes(30),
            resist = true,
            mods   =
            {
                [xi.mod.MDEF]          =  20,
                [xi.mod.ATT]           = 150,
                [xi.mod.DEF]           = 200,
                [xi.mod.EVA]           = 110,
                [xi.mod.TRIPLE_ATTACK] =   5,
            },
            mobMods =
            {
                [xi.mobMod.ADD_EFFECT] =  1,
                [xi.mobMod.MAGIC_COOL] = 60,
            },
        },
    },
    ["Dragons_Aery"] =
    {
        ["Fafnir"] =
        {
            shield  = true,
            rage    = utils.minutes(30),
            resist  = true,
            mobMods =
            {
                [xi.mobMod.WEAPON_BONUS] = 50, -- Level 90 + 50 = 140 Base Weapon Damage
                [xi.mobMod.DRAW_IN]      =  1,
            },
        },
        ["Nidhogg"] =
        {
            rage    = utils.minutes(30),
            resist  = true,
            mobMods =
            {
                [xi.mobMod.WEAPON_BONUS] = 50, -- Level 90 + 50 = 140 Base Weapon Damage
                [xi.mobMod.DRAW_IN]      =  1,
            },
        },
    },
    ["Garlaige_Citadel"] =
    {
        ["Serket"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,
        },
    },
    ["Rolanberry_Fields"] =
    {
        ["Simurgh"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,
        },
    },
    ["Sauromugue_Champaign"] =
    {
        ["Roc"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,
        },
    },
    ["Attohwa_Chasm"] =
    {
        ["Tiamat"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,

            onMobInitialize = function(mob)
                mob:setCarefulPathing(true)
            end,

            onMobSpawn = function(mob)
                -- Reset animation so it starts grounded.
                mob:setMobSkillAttack(0)
                mob:setAnimationSub(0)
            end,
        },
    },
    ["King_Ranperres_Tomb"] =
    {
        ["Vrtra"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,

            onMobEngage = function(mob, target)
                -- an additive option to individual mobs
                -- We have to fully overwrite this one, though (done at top of file)
            end,
        },
    },
    ["Uleguerand_Range"] =
    {
        ["Jormungand"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,

            onMobSpawn = function(mob)
                -- Reset animation so it starts grounded.
                mob:setMobSkillAttack(0)
                mob:setAnimationSub(0)
            end,
        },
    },
    ["Jugner_Forest"] =
    {
        ["King_Arthro"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Western_Altepa_Desert"] =
    {
        ["King_Vinegarroon"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Mount_Zhayolm"] =
    {
        ["Cerberus"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Wajaom_Woodlands"] =
    {
        ["Hydra"] =
        {
            shield = true,
            rage   = utils.minutes(30),
            resist = true,
        },
    },
    ["Caedarva_Mire"] =
    {
        ["Khimaira"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Halvung"] =
    {
        ["Gurfurlur_the_Menacing"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Mamook"] =
    {
        ["Gulool_Ja_Ja"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
            mods    =
            {
                [xi.mod.DOUBLE_ATTACK] = 20,
            },
            mobMods =
            {
                [xi.mobMod.DRAW_IN] = 2,
            },
        },
        ["Hundredfaced_Hapool_Ja"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Arrapago_Reef"] =
    {
        ["Medusa"] =
        {
            ensure     = false, -- Needed if mob doesn't exist to override
            shield     = true,
            rage       = utils.minutes(20),
            resist     = true,
            jobSpecial = true,

            onMobEngage = function(mob, target)
                -- timer to reset pets to current target
                -- ideally tempVisibleTime + cheatCatchTime + claimshieldTime, but variables aren't available to the sub function
                mob:timer(10000, function(mobArg)
                    local mobTarget = mobArg:getTarget()
                    for i = mobArg:getID() + 1, mobArg:getID() + 4 do
                        local mobPet = GetMobByID(i)
                        if mobTarget and mobPet:isSpawned() then
                            mobPet:resetEnmity(mobPet:getTarget())
                            mobPet:updateEnmity(mobTarget)
                        end
                    end
                end)
            end,

            onMobSpawn = function(mob)
                xi.mix.jobSpecial.config(mob, {
                    chance = 75, -- "Is possible that she will not use Eagle Eye Shot at all." (guessing 75 percent)
                    specials =
                    {
                        {id = xi.jsa.EES_LAMIA, hpp = math.random(5, 99)},
                    },
                })
            end,
        },
    },
    ["Labyrinth_of_Onzozo"] =
    {
        ["Lord_of_Onzozo"] =
        {
            shield  = true,
            rage    = utils.minutes(20),
            mobMods =
            {
                [xi.mobMod.DRAW_IN] =  1,
            },
        },
    },
    ["FeiYin"] =
    {
        ["Capricious_Cassie"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
    },
    ["Rolanberry_Fields"] =
    {
        ["Simurgh"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = true,
        },
    },
    ["Maze_of_Shakhrami"] =
    {
        ["Argus"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
    },
    ["Gustav_Tunnel"] =
    {
        ["Bune"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
    },
    ["Sea_Serpent_Grotto"] =
    {
        ["Zuug_the_Shoreleaper"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
        ["Novv_the_Whitehearted"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
        ["Ocean_Sahagin"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
        ["Charybdis"] =
        {
            shield = true,
            rage   = utils.minutes(20),
            resist = false,
        },
    },
    ["RoMaeve"] =
    {
        ["Shikigami_Weapon"] =
        {
            shield  = true,
            rage    = utils.minutes(20), -- only run claimshield on original aggro, give zero time to attempt extra claims
            csTime  = 0,
        },
    },
}

local tempVisibleTime = 500  -- time mob will be visible before being hidden again
local cheatCatchTime  = 2500 -- hide mob and flag anyone who claims during this time
local claimshieldTime = 7000 -- Miliseconds

local hnmRename = function(mob)
    local function shuffle(str)
        local letters = {}
        for letter in str:gmatch'.[\128-\191]*' do
            table.insert(letters, {letter = letter, rnd = math.random()})
        end
        table.sort(letters, function(a, b) return a.rnd < b.rnd end)
        for i, v in ipairs(letters) do
            letters[i] = v.letter
        end
        return table.concat(letters)
    end

    -- Randomize name to combat shorthand claims
    local origName = string.gsub(mob:getName(), '_', ' ')
    if string.len(origName) > 3 then
        -- shuffle name then strip spaces from start and end
        rngName = shuffle(string.sub(origName,1,-3)) .. string.sub(origName, -2)
        rngName = rngName:gsub("^%s*(.-)%s*$", "%1")
        -- mob:hideName(true)
        mob:renameEntity(rngName)
    end
end

local function setClaimable(mob, value)
    mob:setClaimable(value)
    mob:setUnkillable(not value)
    mob:setCallForHelpBlocked(not value)
    mob:setAutoAttackEnabled(value)
    mob:setMobAbilityEnabled(value)

    if value then
        mob:delStatusEffectSilent(xi.effect.PHYSICAL_SHIELD)
        mob:delStatusEffectSilent(xi.effect.MAGIC_SHIELD)
        mob:delStatusEffectSilent(xi.effect.ARROW_SHIELD)
        mob:delStatusEffectsByFlag(0xFFFF)
        mob:setHP(mob:getMaxHP())
    else
        mob:addStatusEffect(xi.effect.PHYSICAL_SHIELD, 999, 3, 9999)
        mob:addStatusEffect(xi.effect.MAGIC_SHIELD, 999, 3, 9999)
        mob:addStatusEffect(xi.effect.ARROW_SHIELD, 999, 3, 9999)
    end
end

local function logListToFile(fileName, textToAdd)
    local filePath = io.popen("cd"):read'*all'
    local logFile  = filePath:sub(1, -2) .. "\\log\\claim_shield_logs\\" .. fileName .. ".html"
    local textHTML = ""
    local logRead  = io.open(logFile, "r")

    -- Handle the text for the HTML
    if logRead ~= nil then
        local oldFile = logRead:read("*all")
        textHTML = oldFile:sub(1, -15) .. textToAdd .. "</body></html>"
        logRead:close()
    else
        textHTML = "<html><title>Claim Shield Log [" .. os.date("%x", os.time()) .. "]</title><body style=\"background-color:DarkSlateGray; color:LimeGreen; font-family:Helvetica, Sans-Serif;\"><h2>Claim Shield Log [" .. os.date("%x", os.time()) .. "]</h2>" .. textToAdd .. "</body></html>"
    end

    -- Write the HTML file out
    local logWrite = io.open(logFile, "w")
    logWrite:write(textHTML)
    logWrite:close()
end

local function getEntityList(mob)
    local enmityList = mob:getEnmityList()

    -- Filter so that pets will only count as a single entry along with their masters
    local entries = {}

    for _, v in pairs(enmityList) do
        local entity = v["entity"]
        local master = entity:getMaster()

        if entity:getObjType() ~= xi.objType.TRUST then
            local winner = entity

            if
                not entity:isPC() and
                master and
                master:isPC()
            then
                winner = master
            end

            local name = winner:getName()
            entries[name] = {['entity'] = winner}
            entries[name .. "_DUPE1"] = {['entity'] = winner}

            -- Give CW and WEW players twice claimshield credit
            -- this extra key is just that, a key. The real entity is stored and used in the claim win logic
            if winner:isCrystalWarrior() or winner:isClassicMode() then
                entries[name .. "_DUPE2"] = {['entity'] = winner}
            end
        end
    end

    local entities = {}
    for k, v in pairs(entries) do
        table.insert(entities, {['name'] = k, ['entity'] = v.entity})
    end

    return entities
end

local function groupPlayers(mob, claimer, entries)
    local result =
    {
        winners = {},
        losers  = {},
        total   = #entries,
        allies  = 0,
    }

    local alliance = claimer:getAlliance()
    if alliance == nil then
        alliance[1] = claimer
    end
    result.allies = #alliance

    for _, member in pairs(alliance) do
        result.winners[member:getName()] = member
    end

    for _, member in pairs(entries) do
        local memberName = member.entity:getName()

        -- If player wasn't already defined in winners list
        if result.winners[memberName] == nil then
            result.losers[memberName] = member.entity
        end
    end

    return result
end

local function clearTrusts(mob, player)
    local party = player:getPartyWithTrusts()

    if party then
        for _, member in pairs(party) do
            if member:getObjType() == xi.objType.TRUST then
                member:disengage()
                member:stun(5000)

                mob:clearEnmityForEntity(member)
            end
        end
    end
end

local function clearPets(mob, player)
    local pet = player:getPet()
    if pet then
        pet:disengage()
        pet:stun(5000)

        mob:clearEnmityForEntity(pet)
    end
end

local function applyClaim(mob, claimer, players)
    for playerName, player in pairs(players.losers) do
        if player ~= nil then
            if not player:isPet() then
                clearTrusts(mob, player)
            end

            player:disengage()
            player:stun(5000)

            mob:clearEnmityForEntity(player)
        end
    end

    setClaimable(mob, true)
    mob:updateClaim(claimer)
    mob:addEnmity(claimer, 1, 1000)
end

local function logClaim(mob, claimer, players)
    -- Setup Log File
    local fileName  = mob:getName() .. "_" .. os.date("%x", os.time()):gsub("/", "-")
    local textToAdd = "<h3>[" .. os.date("%X", os.time()) .. "] - { " .. mob:getName() .. " }</h3><br><b>&emsp;Winners:</b><br>&emsp;<ul>"

    local header    = string.format("==================================================================")

    local loseOne   = string.format("You were not successful in the lottery for %s.", mob:getName())
    local loseTwo   = string.format("Disengage from the mob immediately if it hasn't happened automatically")

    local winGroup  = string.format("Your group has won the lottery for %s! (out of %i total player weights)", mob:getName(), players.total)
    local winSolo   = string.format("You have won the lottery for %s!", mob:getName())

    if claimer:isPC() then
        claimer:incrementCharVar("[LB]CLAIMS", 1)
    end

    -- Message winner and their party/alliance that they've won
    for _, player in pairs(players.winners) do
        if player:getObjType() == xi.objType.PC then
            local playerWeight = not player:isCrystalWarrior() and not player:isClassicMode() and 2 or 3
            if players.allies == 1 then
                player:printToPlayer(string.format(winSolo .. " (You had a %i out of %i chance)", playerWeight, players.total), xi.msg.channel.SYSTEM_3)
            else
                player:printToPlayer(winGroup, xi.msg.channel.SYSTEM_3)
            end

            -- Add name to Log
            textToAdd = textToAdd .. "<li>" .. player:getName() .. "</li><br>"
        end
    end

    -- Finalize winners log
    textToAdd = textToAdd .. "</ul><br><b>&emsp;Losers:</b><br>&emsp;<ul>"

    -- Message losers and tell them to get good
    for _, player in pairs(players.losers) do
        if player:getObjType() == xi.objType.PC then
            local playerWeight = not player:isCrystalWarrior() and not player:isClassicMode() and 2 or 3
            player:printToPlayer(header, xi.msg.channel.SYSTEM_3)
            player:printToPlayer(string.format(loseOne .. " (You had a %i out of %i chance)", playerWeight, players.total), xi.msg.channel.SYSTEM_3)
            player:printToPlayer(loseTwo, xi.msg.channel.SYSTEM_3)
            player:printToPlayer(header, xi.msg.channel.SYSTEM_3)

            -- Add name to Log
            textToAdd = textToAdd .. "<li>" .. player:getName() .. "</li><br>"
        end
    end

    -- Finalize log and store
    textToAdd = textToAdd .. "</ul><br><br>"
    logListToFile(fileName, textToAdd)
end

local function flagCheaters(mob)
    mob:setStatus(xi.status.UPDATE)

    local enmityList = mob:getEnmityList()

    for _, v in pairs(enmityList) do
        local entity = v["entity"]
        local master = entity:getMaster()

        if
            master and master:isPC()
        then
            entity = master
        end

        if entity:isPC() then
            print(fmt("[CLAIMSHIELD]Flagging cheater tagged NM before it was visible: {} (CE: {} VE: {})", entity:getName(), mob:getCE(entity), mob:getVE(entity)))
        end
    end
end

local function rollClaim(mob)
    -- reset mob name for fight
    -- mob:hideName(false)
    mob:renameEntity("")

    local entries    = getEntityList(mob)
    local winningNum = math.random(1, #entries)
    local claimer    = entries[winningNum]

    if
        claimer == nil or
        claimer.entity == nil
    then
        setClaimable(mob, true)
        return
    end

    print("[CLAIMSHIELD]Explicit winner from EntityList was " .. claimer.name)
    local player  = claimer.entity
    local players = groupPlayers(mob, player, entries)

    for _, results in pairs(mob:getEnmityList()) do
        local entity = results.entity
        if entity ~= nil then
            if not entity:isPet() then
                clearTrusts(mob, player)
            elseif entity:isEngaged() then
                -- is a pet
                entity:disengage()
                entity:stun(5000)
            end

            mob:clearEnmityForEntity(entity)
        end
    end

    applyClaim(mob, player, players)
    logClaim(mob, player, players)
end

local rageBuffs =
{
    xi.mod.ATT,
    xi.mod.RATT,
    xi.mod.ACC,
    xi.mod.RACC,
    xi.mod.MATT,
    xi.mod.MDEF,
    xi.mod.MACC,
    xi.mod.MEVA,
    xi.mod.WSACC,
    xi.mod.EVA,
    xi.mod.RDEF,
    xi.mod.REVA,
}

local function addRage(mob)
    print(string.format("[rage] %s is enraged", mob:getName()))

    -- Play dust cloud animation
    mob:independentAnimation(mob, 319, 2)

    -- Boost attributes
    for modID = xi.mod.STR, xi.mod.CHR do
        local amount = math.ceil(mob:getStat(modID) * 9)
        mob:setLocalVar("[rage]mod_" .. modID, amount)
        mob:addMod(modID, amount)
    end

    -- Boost stats
    for _, modID in pairs(rageBuffs) do
        local amount = math.ceil(mob:getMainLvl() * 9)
        mob:setLocalVar("[rage]mod_" .. modID, amount)
        mob:addMod(modID, amount)
    end
end

local function delRage(mob)
    print(string.format("[rage] %s is no longer enraged", mob:getName()))

    -- Unboost attributes
    for modID = xi.mod.STR, xi.mod.CHR do
        local amount = mob:getLocalVar("[rage]mod_" .. modID)
        mob:setLocalVar("[rage]mod_" .. modID, 0)
        mob:delMod(modID, amount)
    end

    -- Unboost stats
    for _, modID in pairs(rageBuffs) do
        local amount = mob:getLocalVar("[rage]mod_" .. modID)
        mob:setLocalVar("[rage]mod_" .. modID, 0)
        mob:delMod(modID, amount)
    end
end

local function addRageTimer(mobArg, rageTimer)
    mobArg:addListener("ENGAGE", "RAGE_ENGAGE", function(mob)
        if mob:getLocalVar("[rage]active") == 0 then
            mob:setLocalVar("[rage]at", os.time() + rageTimer)
        end
    end)

    mobArg:addListener("COMBAT_TICK", "RAGE_CTICK", function(mob)
        if
            mob:getLocalVar("[rage]active") == 0 and
            os.time() > mob:getLocalVar("[rage]at")
        then
            mob:setLocalVar("[rage]active", 1)
            addRage(mob)
        end
    end)

    mobArg:addListener("ROAM_TICK", "RAGE_RTICK", function(mob)
        if
            mob:getLocalVar("[rage]active") == 1 and
            mob:getHPP() == 100
        then
            mob:setLocalVar("[rage]active", 0)
            mob:setLocalVar("[rage]at", 0)
            delRage(mob)
        end
    end)
end

for area, mobs in pairs(areas) do
    for mobName, mobTable in pairs(mobs) do
        local mobPath = string.format("xi.zones.%s.mobs.%s", area, mobName)

        if mobTable.ensure then
            xi.module.ensureTable(mobPath)
        end

        m:addOverride(mobPath .. ".onMobInitialize", function(mob)
            super(mob)

            if mobTable.jobSpecial ~= nil then
                g_mixins.job_special(mob)
            end

            if mobTable.shield then
                mob:addListener("SPAWN", "CS_SPAWN", function(mobArg)
                    hnmRename(mobArg)
                    setClaimable(mobArg, false)

                    -- to catch shorthand cheaters
                    local locClaimshieldTime = mobTable.csTime and mobTable.csTime or claimshieldTime
                    if locClaimshieldTime > 0 then
                        mobArg:stun(tempVisibleTime + cheatCatchTime + locClaimshieldTime)

                        mobArg:timer(tempVisibleTime, function(mobArg2)
                            -- re-hide to now catch anyone attempting to shorthand cheat
                            mobArg2:setStatus(xi.status.DISAPPEAR)
                        end)

                        -- report people who are caught and unhide
                        mobArg:timer(cheatCatchTime, flagCheaters)
                    end

                    -- finally, choose claimshield winner, tempVisibleTime ms longer than (cheatCatchTime + locClaimshieldTime)
                    mobArg:timer(tempVisibleTime + cheatCatchTime + locClaimshieldTime, rollClaim)
                end)
            end

            if mobTable.resist then
                for mod, value in pairs(basicResist) do
                    mob:setMod(mod, value)
                end
            end

            if
                mobTable.mods ~= nil and
                #mobTable.mods > 0
            then
                for mod, value in pairs(mobTable.mods) do
                    mob:setMod(mod, value)
                end
            end
        end)

        m:addOverride(mobPath .. ".onMobSpawn", function(mob)
            super(mob)

            if mobTable.rage ~= nil then
                addRageTimer(mob, mobTable.rage)
            end

            if
                mobTable.mobMods ~= nil and
                #mobTable.mobMods > 0
            then
                for mod, value in pairs(mobTable.mobMods) do
                    mob:setMod(mod, value)
                end
            end
        end)

        m:addOverride(mobPath .. ".onMobEngage", function(mob, target)
            super(mob, target)

            if mobTable.onMobEngage ~= nil then
                mobTable.onMobEngage(mob, target)
            end
        end)
    end
end

return m
