-----------------------------------
-- Pirate's Chart
-----------------------------------
require("modules/module_utils")
require('scripts/globals/npc_util')
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("fising_pirates_chart")

local settings = 
{
    itemID   = xi.item.PIRATES_CHART,
    itemName = "pirate's chart",
    zoneName = "Valkurm_Dunes",
    eventID  = 14,
    qmName   = "qm4",
    boxName  = "Barnacled_Box",
    mobNames =
    {
        "Houu_the_Shoalwader",
        "Beach_Monk",
        "Heike_Crab",
    },
    message =
    {
        TOO_MANY_IN_PARTY              = -2,  -- Nothing happens. Your party exceeds the maximum number of <number> members.
        ALLIANCE_NOT_ALLOWED           = -1,  -- Nothing happens. You must dissolve your alliance.
    },
    loot =
    {
        {
            { cexi.rate.VERY_COMMON,  4484 }, -- Shall Shell      (24%)
            { cexi.rate.COMMON,        624 }, -- Pamtam Kelp      (15%)
            { cexi.rate.COMMON,      17006 }, -- Drill Calamary   (15%)
            { cexi.rate.UNCOMMON,    17007 }, -- Dwarf Pugil      (10%)
            { cexi.rate.UNCOMMON,     1893 }, -- Salinator        (10%)
            { cexi.rate.UNCOMMON,     4288 }, -- Zebra Eel        (10%)
            { cexi.rate.RARE,         1587 }, -- H.Q. Pugil Scls. ( 5%)
            { cexi.rate.RARE,          887 }, -- Coral Fragment   ( 5%)
        },
        {
            { cexi.rate.VERY_COMMON,  4484 }, -- Shall Shell      (24%)
            { cexi.rate.VERY_COMMON,   688 }, -- Arrowwood Log    (24%)
            { cexi.rate.COMMON,       4580 }, -- Coral Butterfly  (15%)
            { cexi.rate.COMMON,      17006 }, -- Drill Calamary   (15%)
            { cexi.rate.COMMON,      17007 }, -- Dwarf Pugil      (15%)
            { cexi.rate.UNCOMMON,     4361 }, -- Nebimonite       (10%)
            { cexi.rate.RARE,          887 }, -- Coral Fragment   ( 5%)
        },
        {
            { cexi.rate.VERY_COMMON,   18104 }, -- Fuscina        (24%)
            { cexi.rate.RARE,           1311 }, -- Oxblood        ( 5%)
            { cexi.rate.VERY_RARE,     18020 }, -- Mercurial Kris ( 1%) (Equivalent 3.3%)
        },
        {
            { cexi.rate.GUARANTEED,   15555 }, -- Albatross Ring  (24%)
        }
    },
}

-----------------------------------
-- Utilities
-----------------------------------
local function setMusic(player, id)
    if type(id) == "table" then
        player:changeMusic(0, id[1])
        player:changeMusic(1, id[2])
        player:changeMusic(2, id[3])
        player:changeMusic(3, id[4])
    else
        player:changeMusic(0, id)
        player:changeMusic(1, id)
        player:changeMusic(2, id)
        player:changeMusic(3, id)
    end
end

local function getMobs(zone, mobNames)
    local mobs = {}

    for _, mobName in pairs(mobNames) do
        local res = zone:queryEntitiesByName(mobName)

        if res ~= nil then
            for _, mob in pairs(res) do
                table.insert(mobs, mob:getID())
            end
        end
    end

    return mobs
end

-----------------------------------
-- QM Events
-----------------------------------
local qmPath = fmt("xi.zones.{}.npcs.{}", settings.zoneName, settings.qmName)

m:addOverride(qmPath .. ".onTrade", function(player, npc, trade)
    local ID  = zones[player:getZoneID()]
    local msg = ID.text.MONSTERS_KILLED_ADVENTURERS

    if player:getPartySize() > 3 then
        player:messageSpecial(msg + settings.message.TOO_MANY_IN_PARTY, 3)

    elseif player:checkSoloPartyAlliance() == 2 then
        player:messageSpecial(msg + settings.message.ALLIANCE_NOT_ALLOWED, 0)

    elseif npcUtil.tradeHasExactly(trade, settings.itemID) then
        player:fmt("You return the {} to the sea.", settings.itemName)
        player:startEvent(settings.eventID, 0, 0, 0, 3)
    end
end)

m:addOverride(qmPath .. ".onEventUpdate", function(player, csid, option)
    if csid ~= settings.eventID or option ~= 0 then
        return
    end

    player:confirmTrade()

    local party = player:getParty()

    for _, member in pairs(party) do
        setMusic(member, 136)

        member:delStatusEffectsByFlag(xi.effectFlag.DISPELABLE)
        member:addStatusEffectEx(
            xi.effect.LEVEL_RESTRICTION,
            xi.effect.LEVEL_RESTRICTION,
            20,
            0,
            0,
            0,
            0,
            0,
            xi.effectFlag.ON_ZONE
        )
    end
end)

m:addOverride(qmPath .. ".onEventFinish", function(player, csid, option, npc)
    if csid ~= settings.eventID or option ~= 0 then
        return
    end

    local zone = player:getZone()
    local mobs = getMobs(zone, settings.mobNames)

    xi.confrontation.start(player, npc, mobs,
        function(member)
            if member:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
                member:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
            end

            setMusic(member, { 0, 0, 101, 102 })
            member:setLocalVar("[CHART]Finished", 1)
        end,

        -- Lose condition
        function(member)
            member:fmt("Too much time passed.")

            if member:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
                member:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
            end

            setMusic(member, { 0, 0, 101, 102 })
        end,

        {
            allRegPlayerEnmity = true,
            timeLimit          = 600, -- 10 minutes

            validPlayerFunc    = function(member)
                if member:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
                    return (member:getStatusEffect(xi.effect.LEVEL_RESTRICTION):getPower() == 20)
                else
                    return false
                end
            end,
        }
    )
end)

-----------------------------------
-- Mob Events
-----------------------------------
for _, mobName in pairs(settings.mobNames) do
    local mobPath = fmt("xi.zones.{}.mobs.{}", settings.zoneName, mobName)
    cexi.util.ensureMob(settings.zoneName, mobName)

    m:addOverride(mobPath .. ".onMobSpawn", function(mob)
        mob:setMobMod(xi.mobMod.IDLE_DESPAWN, 60)
    end)

    m:addOverride(mobPath .. ".onMobInitialize", function(mob)
        g_mixins.job_special(mob)
        mob:setLocalVar("NO_CASKET", 1)
    end)

    m:addOverride(mobPath .. ".onMobDeath", function(mob, player, optParams)
        -- Skip if other mobs are alive
        local zone = mob:getZone()
        local mobs = getMobs(zone, settings.mobNames)

        for _, mobID in pairs(mobs) do
            local other = GetMobByID(mobID)

            if other ~= nil and other:isAlive() then
                return
            end
        end

        -- Battle complete, spawn box
        local box = zone:queryEntitiesByName(settings.boxName)

        if
            box ~= nil and
            box[1] ~= nil
        then
            local npc = box[1]

            npc:setPos(mob:getXPos(), mob:getYPos(), mob:getZPos())
            npc:setStatus(xi.status.NORMAL)
            npc:setLocalVar("opened", 0)
        end
    end)
end

-----------------------------------
-- Box Events
-----------------------------------
local boxPath = fmt("xi.zones.{}.npcs.{}", settings.zoneName, settings.boxName)
cexi.util.ensureNPC(settings.zoneName, settings.boxName)

m:addOverride(boxPath .. ".onTrigger", function(player, npc)
    if player:getLocalVar("[CHART]Finished") == 0 then
        player:fmt("It's locked.")
        return
    end

    if npc:getLocalVar("opened") == 0 then
        npc:setLocalVar("opened", 1)
        npc:entityAnimationPacket('openH')

        local party = player:getParty()

        for _, member in pairs(party) do
            member:setLocalVar("[CHART]Finished", 0)
        end

        for _, itemInfo in pairs(settings.loot) do
            local pick = cexi.util.pickItem(itemInfo)
            player:addTreasure(pick[2], npc)
        end

        npc:timer(16000, function(npcArg)
            cexi.util.removeNPC(npcArg)
        end)
    end
end)

return m
