-----------------------------------
-- Crooked Jones - Fishing quests
-----------------------------------
require("modules/module_utils")
require('scripts/globals/npc_util')
require("scripts/globals/shop")
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("npc_crooked_jones")

local settings =
{
    name   = "Crooked Jones",
    vars   =
    {
        PROGRESS      = "[FISHING]JONES",
        TARGET_1      = "[FISHING]STASH_TARGET_1",
        TARGET_2      = "[FISHING]STASH_TARGET_2",
        TARGET_3      = "[FISHING]STASH_TARGET_3",
        TARGET_1_QTY  = "[FISHING]STASH_TARGET_1_QTY",
        TARGET_2_QTY  = "[FISHING]STASH_TARGET_2_QTY",
        TARGET_3_QTY  = "[FISHING]STASH_TARGET_3_QTY",
        DOUBLOONS     = "[FISHING]DOUBLOONS",
        TREASURE_KEY  = "[FISHING]TREASURE_KEY",
        TREASURE_BOX  = "[FISHING]TREASURE_BOX",
        TREASURE_LOC  = "[FISHING]TREASURE_LOC",
        TREASURE_HUNT = "[FISHING]TREASURE_HUNT",
    },

    animation = 48,

    limits =
    {
        15,
        60,
        40,
    },

    values =
    {
        { 10, 20 },
        {  1,  2 },
        {  2,  3 },
    },

    dialog =
    {
        DEFAULT =
        {
            "Do I know you? Better get a move on buddy..."
        },

        START =
        {
            "Do I know you? Wait... why if it ain't the stench of the sea! Yer a fisherr like me!",
            { emote = xi.emote.SHOCKED },
            { delay = 1000 },
            " Listen up old seadog. I run a very lucrative side hussle smugglin' rare beauts of the sea.",
            " I'm also shippin' out regular run of the mill fish, ya know fer appearances sake.",
            { emote = xi.emote.THINK },
            " Bring me the fish I need an' we both get paid. Got it?",
            { message = "You have unlocked Crooked Jones' Fishing Quests in Norg!" },
        },

        FISH =
        {
            "Today I'm collectin' fer an order of the legendary %s. While yer at it I could use some %s and %s.",
            { emote = xi.emote.YES },
            " Once you caught em', just toss all in the box right 'ere.",
        },

        GAME =
        {
            "Ya know, you've been quite the handy crew mate. Let's play a game.",
        },

        DECLINE = { "Not so fast! You ain't got enough for that yet..." },

        TREASURE_1 =
        {
            "Avast, ye' scurvy dog! Me an' me mates be lookin' for buried treasure.",
            { emote = xi.emote.THINK },
            "The sea speaks, she says the keys be found in %s somewhere in the water.",
        },

        REMINDER_1 =
        {
            "The sea speaks, she says the keys be found in %s somewhere in the water.",
        },

        TREASURE_2 =
        {
            "Well blow me ov'r! Yarrg did it. Not bad for but a deckswabber like yerself.",
            " Arrg, now finish the deed, will ye?",
            { emote = xi.emote.YES },
            " Dead men tell no tales, but me mates say the booty is buried somewhere in %s.",
        },

        REMINDER_2 =
        {
            " Dead men tell no tales, but me mates say the booty is buried somewhere in %s.",
        },

        TREASURE_3 =
        {
            "Ye did it an' a fine job too! If ay git any more leads, w'll let ye know~",
        },

        BONUS =
        {
            { noturn = true },
            "Crooked Jones : W'll look at that! Ye got all them fish t'day.",
            { entity = "Crooked_Jones", emote = xi.emote.CLAP },
            " Ere's a bonus on the house, fer all yer 'ard work!",
        },
    },
}

local rewards =
{
    { "Brigand's Shovel",     xi.item.BRIGANDS_SHOVEL,  5000 },
    { "Crab Cap +1",                            25669,  8000 },
    { "Buccaneer's Chart",   xi.item.BUCCANEERS_CHART, 10000 },
    { "Brig. Eyepatch",     xi.item.BRIGANDS_EYEPATCH, 12000 },
    { "Red Crab Mount",                         10052, 15000 },
    { "Rusty Fsh. Hook",   xi.item.RUSTY_FISHING_HOOK, 20000 },
}

local rewardsCW =
{
    { "Brigand's Shovel",     xi.item.BRIGANDS_SHOVEL,  5000 },
    { "Crab Cap +1",                            25669,  8000 },
    { "Buccaneer's Chart",   xi.item.BUCCANEERS_CHART, 10000 },
    { "Brig. Eyepatch",     xi.item.BRIGANDS_EYEPATCH, 12000 },
    { "Shaper's Shawl",          xi.item.SHAPERS_SHAWL,15000 },
    { "Rusty Fsh. Hook",   xi.item.RUSTY_FISHING_HOOK, 20000 },
}

local boxRewards =
{
    { cexi.rate.UNCOMMON,                         17006 }, -- Drill Calamary        (10%)
    { cexi.rate.UNCOMMON,                         17007 }, -- Dwarf Pugil           (10%)
    { cexi.rate.RARE,                              1225 }, -- Gold Nugget           (10%)
    { cexi.rate.RARE,                              1452 }, -- O.Bronzepiece         ( 5%)
    { cexi.rate.RARE,                              1455 }, -- One Byne Bill         ( 5%)
    { cexi.rate.RARE,                              1449 }, -- T.Whiteshell          ( 5%)
    { cexi.rate.RARE,      xi.item.ANGLERS_COMMENDATION }, -- Angler's Commendation ( 5%)
    { cexi.rate.RARE,                              5954 }, -- Barnacle              ( 5%)
    { cexi.rate.RARE,                              4484 }, -- Shall Shell           ( 5%)
    { cexi.rate.RARE,                               947 }, -- Firesand              ( 5%)
    { cexi.rate.RARE,                             17278 }, -- Gold Bullet           ( 5%)
    { cexi.rate.RARE,                               568 }, -- Goblin Die            ( 5%)
    { cexi.rate.RARE,                              4155 }, -- Remedy                ( 5%)
    { cexi.rate.RARE,                             17313 }, -- Grenade               ( 5%)
    { cexi.rate.VERY_RARE,                          900 }, -- Fish Bones            ( 1%)
    { cexi.rate.VERY_RARE,                          900 }, -- Fish Bones            ( 1%)
    { cexi.rate.VERY_RARE,                          900 }, -- Fish Bones            ( 1%)
    { cexi.rate.VERY_RARE,                          888 }, -- Seashell              ( 1%)
    { cexi.rate.VERY_RARE,                         1875 }, -- Ancient Beastcoin     ( 1%)
    { cexi.rate.VERY_RARE,                         4156 }, -- Mulsum                ( 1%)
    { cexi.rate.VERY_RARE,                         1889 }, -- White Sand            ( 1%)
    { cexi.rate.VERY_RARE,                          792 }, -- Pearl                 ( 1%)
    { cexi.rate.VERY_RARE,                          793 }, -- Black Pearl           ( 1%)
    { cexi.rate.VERY_RARE,                        17257 }, -- Bandit's Gun          ( 1%)
    { cexi.rate.VERY_RARE,                        17259 }, -- Pirate's Gun          ( 1%)
    { cexi.rate.VERY_RARE,                          624 }, -- Pamtam Kelp           ( 1%)
    { cexi.rate.VERY_RARE,                        17342 }, -- Cannon Shell          ( 1%)
    { cexi.rate.VERY_RARE,     xi.item.BUCCANEERS_CHART }, -- Buccaneer's Chart     ( 1%)
    { cexi.rate.SUPER_RARE,                        1874 }, -- Pirate's Chart        (.5%)
    { cexi.rate.ULTRA_RARE,                        5998 }, -- Adoulin Soup          (.1%)
    { cexi.rate.ULTRA_RARE,                       17760 }, -- Buccaneer's Scimitar  (.1%)
    { cexi.rate.ULTRA_RARE,                       17722 }, -- Phantom Fleuret       (.1%)
    { cexi.rate.ULTRA_RARE,                       16757 }, -- Corsair's Knife       (.1%)
    { cexi.rate.ULTRA_RARE,                        1274 }, -- Southern Pearl        (.1%)
}

local fishedMobs =
{
    -- { "Phanauet_Channel",             "Snipper", "Big_Jaw", "Fishtrap", "Protozoan", "Aipaloovik" }, 
    { "Carpenters_Landing",           "Snipper", "Triangle_Crab", "Clipper", "Greater_Pugil", "Fishtrap"                                       },
    -- { "Manaclipper",                  "Ghost_Crab", "Ghost_Crab", "Greater_Pugil", "Kraken", "Harajnite", "Cyclopean_Conch" },
    { "Bibiki_Bay",                   "Ghost_Crab", "Grindylow", "Greater_Pugil", "Apsaras", "Kraken", "Lancet_Jagil"                          },
    { "Oldton_Movalpolos",            "Snipper", "Blind_Crab", "Cutter", "Ghost_Crab", "Amoebic_Nodule"                                        },
    { "Lufaise_Meadows",              "Clipper", "Grindylow", "Greater_Pugil", "Apsaras"                                                       },
    { "Misareaux_Coast",              "Clipper", "Grindylow", "Greater_Pugil"                                                                  },
    { "Open_sea_route_to_Al_Zahbi",   "Passage_Crab", "Gugru_Jagil", "Blanched_Kraken", "Gugru_Orobon", "Northern_Piranu"                      },
    { "Open_sea_route_to_Mhaura",     "Passage_Crab", "Gugru_Jagil", "Blanched_Kraken", "Gugru_Orobon", "Southern_Piranu"                      },
    { "Wajaom_Woodlands",             "Red_Smoocher", "Azoth_Apsaras", "Red_Osculator", "Kissing_Leech", "Mercurial_Makara"                    },
    { "Bhaflau_Thickets",             "Red_Smoocher", "Azoth_Apsaras", "Red_Osculator", "Kissing_Leech", "Mercurial_Makara"                    },
    -- { "Arrapago_Reef",                "Wootzshell", "Arrapago_Leech", "Chimera_Clot", "Lahama", "Llamhigyn_Y_Dwr", "Giant_Orobon"              },
    { "Silver_Sea_route_to_Nashmau",  "Cyan_Deep_Crab", "Submarine_Nipper", "Thalassic_Pugil", "Abyssal_Pugil", "Bathybic_Kulshedra", "Orobon" },
    { "Silver_Sea_route_to_Al_Zahbi", "Cyan_Deep_Crab", "Submarine_Nipper", "Thalassic_Pugil", "Abyssal_Pugil", "Bathybic_Kulshedra", "Orobon" },
    { "Mount_Zhayolm",                "Sicklemoon_Crab", "Zazalda_Clot", "Vozold_Clot", "Vozold_Jagil", "Giant_Orobon"                         },
    { "Mamook",                       "Suhur_Mas", "Mamool_Ja_Bloodsucker", "Mamook_Mush", "Mamook_Crab"                                       },
    { "Aydeewa_Subterrane",           "Cave_Mold", "Anautogenous_Slug", "Cave_Pugil", "Aydeewa_Crab"                                           },
    { "Caedarva_Mire",                "Caedarva_Pondscum", "Caedarva_Marshscum", "Suhur_Mas", "Llamhigyn_Y_Dwr"                                },
    { "West_Ronfaure",                "Tree_Crab", "Limicoline_Crab", "Land_Crab", "Vermivorous_Crab", "Passage_Crab"                          },
    { "East_Ronfaure",                "Pugil", "Cheval_Pugil", "Mud_Pugil", "Pug_Pugil", "Fighting_Pugil"                                      },
    { "La_Theine_Plateau",            "Coral_Crab", "Pug_Pugil", "Thickshell", "Giant_Pugil", "Puffer_Pugil"                                   },
    { "Valkurm_Dunes",                "Stag_Crab", "Puffer_Pugil", "Cutter"                                                                    },
    { "Jugner_Forest",                "Stag_Crab", "Spring_Pugil", "Thread_Leech", "Ferocious_Pugil", "Huge_Leech"                             },
    { "Batallia_Downs",               "Snipper", "Land_Pugil", "Cutter", "Greater_Pugil", "Kraken"                                             },
    { "North_Gustaberg",              "Stone_Crab", "Sand_Crab", "Land_Crab", "Pug_Pugil", "Fighting_Pugil"                                    },
    { "South_Gustaberg",              "Sand_Crab", "Land_Crab", "Mole_Crab", "Passage_Crab"                                                    },
    { "Pashhow_Marshlands",           "Swamp_Leech", "Stag_Crab", "Swamp_Pugil", "Thread_Leech"                                                },
    { "Rolanberry_Fields",            "Snipper", "Big_Jaw", "Horrid_Fluke", "Greater_Pugil", "Big_Leech"                                       },
    { "Beaucedine_Glacier",           "Greater_Pugil", "Vepar", "Kraken", "Apsaras", "Morgawr"                                                 },
    { "Cape_Teriggan",                "Razorjaw_Pugil", "Rock_Crab", "Stygian_Pugil", "Devil_Manta"                                            },
    { "Eastern_Altepa_Desert",        "Greater_Pugil", "Cutter", "Ironshell", "Makara", "Bigclaw"                                              },
    { "West_Sarutabaruta",            "Palm_Crab", "Savanna_Crab", "Land_Crab", "Mugger_Crab", "Passage_Crab"                                  },
    { "East_Sarutabaruta",            "Palm_Crab", "Savanna_Crab", "Mud_Pugil", "Pug_Pugil", "Fighting_Pugil"                                  },
    { "Buburimu_Peninsula",           "Stag_Crab", "Puffer_Pugil", "Snipper", "Shoal_Pugil", "Clipper"                                         },
    { "Sauromugue_Champaign",         "Big_Jaw", "Snipper", "Greater_Pugil", "Cutter", "Kraken"                                                },
    { "The_Sanctuary_of_ZiTah",       "Greater_Pugil", "Clipper", "Bigclaw", "Apsaras"                                                         },
    { "Yuhtunga_Jungle",              "Ironshell", "Bigclaw", "Bloodsucker"                                                                    },
    { "Yhoator_Jungle",               "Greater_Pugil", "Clipper", "Vepar", "Makara"                                                            },
    { "Western_Altepa_Desert",        "Ironshell", "Apsaras", "Bigclaw", "Razorjaw_Pugil"                                                      },
    { "Qufim_Island",                 "Qufim_Pugil", "Sea_Bishop", "Vepar"                                                                     },
    { "Ghelsba_Outpost",              "Pugil", "Pug_Pugil", "Giant_Pugil", "Puffer_Pugil"                                                      },
    -- { "Fort_Ghelsba",                 "Pug_Pugil", "Giant_Pugil", "Land_Pugil", "Orcish_Fodder"                                                },
    { "Palborough_Mines",             "Mine_Crab", "Coral_Crab", "Stag_Crab", "Snipper"                                                        },
    { "Giddeus",                      "Pugil", "Pug_Pugil", "Puffer_Pugil", "Land_Pugil"                                                       },
    { "Davoi",                        "Ferocious_Pugil", "Greater_Pugil", "Gloop", "Oil_Spill"                                                 },
    -- { "Castle_Oztroja", "Odontotyrannus" },
    { "The_Boyahda_Tree",             "Scavenger_Crab", "Stygian_Pugil", "Bouncing_Ball", "Demonic_Pugil"                                      },
    { "Dragons_Aery",                 "Demonic_Pugil"                                                                                          },
    { "Temple_of_Uggalepih",          "Uggalepih_Leech", "Bloodsucker", "Bouncing_Ball"                                                        },
    { "Den_of_Rancor",                "Rock_Crab", "Razorjaw_Pugil", "Bloodsucker"                                                             },
    { "Bostaunieux_Oubliette",        "Acid_Grease"                                                                                            },
    { "Toraimarai_Canal",             "Bigclaw", "Canal_Leech", "Rock_Crab", "Mousse"                                                          },
    { "Korroloka_Tunnel",             "Snipper", "Greater_Pugil", "Kraken"                                                                     },
    { "Kuftal_Tunnel",                "Scavenger_Crab", "Stygian_Pugil", "Devil_Manta"                                                         },
    { "Sea_Serpent_Grotto",           "Big_Jaw", "Stygian_Pugil"                                                                               },
    { "Dangruf_Wadi",                 "Land_Crab", "Coral_Crab", "Wadi_Leech", "Thread_Leech"                                                  },
    { "Ordelles_Caves",               "Stag_Crab", "Thread_Leech", "Poison_Leech", "Rancid_Ooze"                                               },
    { "Gusgen_Mines",                 "Pirate_Pugil", "Greater_Pugil", "Ooze", "Mush"                                                          },
    { "Ship_bound_for_Selbina",       "Sea_Pugil", "Ocean_Crab", "Ocean_Pugil", "Pirate_Pugil"                                                 },
    { "Ship_bound_for_Mhaura",        "Sea_Pugil", "Ocean_Crab", "Ocean_Pugil", "Pirate_Pugil"                                                 },
}

local fishedMobsLookup = {}

for areaIndex, areaInfo in pairs(fishedMobs) do
    local zoneName = string.upper(areaInfo[1])
    local zoneID   = xi.zone[zoneName]
    fishedMobsLookup[zoneID] = areaIndex
end

for areaIndex, areaInfo in pairs(fishedMobs) do
    local areaName = areaInfo[1]

    for i = 2, #areaInfo do
        local mobName  = areaInfo[i]
        local path = fmt("xi.zones.{}.mobs.{}.onMobDeath", areaName, mobName)

        cexi.util.ensureMob(areaName, mobName)

        m:addOverride(path, function(mob, player, optParams)
            super(mob, player, optParams)

            if
                player ~= nil and
                player:isPC() and
                mob:getMobMod(xi.mobMod.IDLE_DESPAWN) > 0 and -- ensure the mob was really fished up (some mobs can spawn or be fished up in a zone)
                player:getCharVar(settings.vars.TREASURE_HUNT) > 0 and
                not player:hasItem(xi.item.PIRATES_KEY) and
                player:getZoneID() == player:getCharVar(settings.vars.TREASURE_KEY)
            then
                player:addTreasure(xi.item.PIRATES_KEY, mob) --, cexi.rate.RARE)
            end
        end)
    end
end

local buriedAreas =
{
    {
        area = "Valkurm_Dunes",
        zone = xi.zone.VALKURM_DUNES,
        list =
        {
            { 29.387, -2.121, -87.314, 225 }, -- 29.387 -2.121 -87.314 103
            { 121.8037, -1.4065, -51.4815, 183 },
            { 248.6933, -0.8052, 48.3103, 84 },
            { 187.9113, -2.6014, -127.6478, 221 },
            { 238.7540, 4.0000, -153.1625, 39 },
            { 255.4684, 0.1250, -101.5687, 61 },
            { 369.5623, -0.9526, -129.9578, 118 },
            { 432.9333, 0.0000, -189.4295, 133 },
            { 408.7423, -3.0205, -92.5854, 192 },
            { 449.0481, -1.0795, -9.1802, 149 },
            { 528.5175, -9.2488, 67.0858, 158 },
            { 557.1399, -8.4991, -30.8149, 105 },
            { 506.4407, -10.9907, -91.9533, 238 },
            { 452.5287, -11.1566, -93.8691, 168 },
            { 427.9700, -10.5743, -97.1819, 205 },
            { 628.4445, -2.9085, 11.8624, 40 },
            { 690.2877, -1.8722, 6.8819, 104 },
            { 732.8358, -2.8911, -28.2824, 100 },
            { 692.9338, -3.2529, -134.3931, 166 },
            { 660.1567, 0.5649, -53.2482, 245 },
            { 656.7083, -0.8832, -137.3484, 35 },
            { 625.2387, -1.1290, -180.0950, 237 },
            { 629.6876, -1.8745, -206.8262, 225 },
            { 742.6857, -10.3339, -181.3839, 100 },
            { 742.5614, -9.2922, -223.4575, 126 },
            { 741.7056, -10.3241, -263.8137, 211 },
            { 779.2772, -10.4505, -263.4321, 161 },
            { 787.6263, -3.1854, -186.7189, 34 },
            { 820.1929, -0.7638, -216.8253, 248 },
            { 830.0760, -1.5263, -245.6445, 233 },
            { 833.1191, -1.8608, -149.7608, 38 },
            { 897.3079, -0.9470, -183.0233, 231 },
            { 862.6951, -0.3369, -179.9513, 252 },
            { 941.3226, -1.3589, -222.1363, 96 },
            { 866.2806, -3.0901, -292.3040, 227 },
            { 919.0262, -0.8402, -330.8714, 153 },
            { 975.8727, -1.7141, -299.5871, 113 },
            { 975.3420, -1.4337, -379.1898, 117 },
            { 913.6105, -1.6711, -390.0327, 28 },
            { 410.7704, -14.1525, 111.2016, 121 },
            { 445.7847, -15.8408, 165.4407, 87 },
            { 528.0460, -7.8926, 92.7696, 209 },
            { 527.8620, -14.7963, 154.0255, 77 },
            { 609.7196, -9.2585, 105.7414, 246 },
            { 637.9632, -8.0000, 122.3436, 126 },
            { 653.0571, -11.3422, 106.0719, 164 },
            { 668.0865, -10.3690, 185.3924, 207 },
            { 726.5125, -9.7601, 189.8477, 159 },
            { 780.4692, -9.3056, 224.8560, 180 },
            { 810.6303, -10.7461, 228.3208, 162 },
            { 816.6099, -8.8241, 298.5501, 134 },
            { 848.4379, -10.2225, 309.6458, 164 },
            { 828.7530, -8.8951, 366.1507, 52 },
            { 760.3802, -8.8836, 348.8907, 195 },
            { 788.7489, -8.9872, 295.6196, 27 },
            { 718.9821, -8.8872, 251.1136, 51 },
            { 681.4117, -15.5879, 234.2778, 100 },
            { 678.7344, -16.6685, 250.3774, 91 },
            { 649.2975, -16.5471, 278.0221, 74 },
            { 628.9127, -17.0051, 286.3532, 45 },
            { 595.9670, -15.6150, 252.0396, 226 },
            { 563.3005, -16.0000, 319.6269, 18 },
            { 572.6694, -19.1699, 372.3692, 91 },
            { 525.6733, -15.8070, 326.2282, 100 },
            { 479.0924, -15.7383, 350.7427, 69 },
            { 473.5972, -16.1482, 313.2460, 87 },
            { 464.7993, -17.2919, 299.7334, 241 },
            { 462.5345, -17.1180, 256.5049, 130 },
            { 419.4619, -17.1027, 254.3747, 57 },
            { 400.3418, -16.6822, 250.7252, 59 },
            { 402.9097, -16.0000, 203.3501, 80 },
            { 404.7182, -15.2482, 134.9637, 115 },
            { 313.4092, -9.5450, 249.2229, 40 },
            { 328.7292, -11.0155, 251.1282, 64 },
            { 291.1464, -10.6223, 214.8242, 68 },
            { 268.7568, -10.8577, 213.3076, 40 },
            { 286.5468, -7.8109, 165.2762, 55 },
            { 242.0605, -8.0120, 156.2194, 108 },
            { 209.1147, -8.7417, 167.2580, 55 },
            { 149.8071, -10.4676, 170.6460, 1 },
            { 161.7961, -8.2616, 167.4550, 30 },
            { 165.6779, -7.8205, 125.9544, 40 },
            { 181.9689, -7.1734, 79.0454, 48 },
            { 147.2791, -7.9880, 87.8382, 34 },
            { 125.1315, -7.7154, 110.5691, 159 },
            { 128.4843, -8.1816, 125.8378, 52 },
            { 112.5616, -9.9424, 132.3029, 141 },
            { 77.8165, -8.0000, 118.4366, 51 },
            { 48.5254, -9.2029, 166.0167, 91 },
            { 34.7268, -7.9375, 157.1831, 106 },
            { -27.4620, -10.8771, 211.6807, 112 },
            { -100.6679, -9.3530, 215.1817, 57 },
            { -106.8263, -7.9609, 189.2649, 171 },
            { -126.6767, -9.9101, 210.5808, 33 },
            { -116.2375, -7.8669, 153.3151, 55 },
            { -131.3546, -7.6837, 118.5091, 95 },
            { -137.5580, 0.8769, 82.6621, 57 },
            { -109.9282, -0.1941, 77.7992, 255 },
            { -72.1446, -0.9093, 89.4289, 63 },
            { 7.0392, -0.4943, 45.2288, 86 },
            { 23.4850, -0.9314, -16.6395, 107 },
            { 97.8688, -0.6350, -55.8419, 160 },
            { 88.3258, -0.4152, -75.2193, 151 },
            { 52.8147, -0.3441, -89.9678, 1 },
            { 88.6925, -0.4806, -82.1453, 93 },
            { 89.9423, -1.5963, -110.9289, 118 },
            { 86.4168, -1.6166, -129.7425, 158 },
            { 93.7191, 0.5000, -21.8038, 0 },
            { -51.2263, -1.5521, -12.7391, 191 },
            { -80.5115, 0.7955, -23.0436, 214 },
            { -77.1316, -0.7932, -48.8950, 142 },
            { -144.8395, 0.6280, -41.4065, 150 },
            { -163.8687, 0.1032, -24.0696, 210 },
            { -195.9162, 0.0086, 0.8831, 251 },
            { -173.2275, 0.4488, 41.7652, 84 },
            { -218.3030, 0.8925, 43.7117, 128 },
            { -109.5867, -1.3009, -91.7861, 154 },
            { -113.9426, 4.0000, -117.3052, 119 },
            { -185.7312, 3.8223, -108.5444, 80 },
            { -241.1845, 4.0000, -117.7171, 19 },
            { -263.3344, 2.8804, -152.2850, 38 },
            { -311.7220, 0.1670, -145.0287, 36 },
            { -311.0417, 0.1643, -149.9980, 20 },
            { -287.9237, -1.0375, -85.9153, 214 },
            { -274.9396, 0.1202, -48.7557, 192 },
            { -337.1265, 0.7813, -38.7566, 133 },
            { -360.6436, -0.2739, -47.4715, 179 },
            { -372.8597, -0.5246, -20.4637, 2 },
            { -331.6559, -2.6371, 22.1053, 49 },
            { -289.2963, -0.8477, 47.8350, 56 },
            { -300.4575, -8.3443, 62.8724, 122 },
            { -322.9735, -8.0000, 41.3519, 219 },
            { -370.2036, -7.9074, 66.9146, 230 },
            { -353.1609, -7.7106, 87.2457, 220 },
            { -338.5035, -8.3750, 99.1299, 3 },
            { -347.3756, -9.6260, 132.9721, 50 },
            { -371.6396, -10.7085, 131.1399, 42 },
            { -376.3306, -8.7232, 98.3379, 148 },
            { -428.9536, -9.1085, 65.3033, 191 },
            { -448.2720, -9.3832, 70.0228, 186 },
            { -488.5129, -9.2347, 73.9575, 219 },
            { -474.3097, -7.8213, 85.9326, 222 },
            { -530.2741, -10.4242, 110.3853, 213 },
            { -526.1174, -9.1728, 168.1810, 36 },
            { -458.0550, -8.9608, 177.2162, 154 },
            { -426.9374, -10.1429, 152.1743, 143 },
            { -448.4422, -15.0328, 218.6917, 191 },
            { -481.4136, -16.6638, 269.6841, 214 },
            { -424.9193, -18.1963, 293.7210, 18 },
            { -405.6749, -17.1216, 328.4330, 42 },
            { -380.2990, -15.1689, 328.5793, 173 },
            { -350.8344, -9.9747, 328.6741, 84 },
            { -362.7380, -7.0035, 299.9517, 72 },
            { -366.8092, -7.7780, 245.1321, 62 },
            { -307.0610, -10.8839, 251.5731, 103 },
            { -337.6496, -8.7595, 183.7203, 97 },
            { -470.9616, -17.2060, 365.2273, 100 },
            { -495.8898, -16.6401, 373.7025, 37 },
            { -552.6848, -16.9662, 368.4435, 83 },
            { -552.4860, -16.0449, 331.3376, 141 },
            { -572.2709, -18.9898, 307.9301, 227 },
            { -539.2214, -21.7523, 300.7791, 104 },
            { -541.3638, -26.3233, 257.7131, 169 },
            { -582.3676, -16.2643, 238.3960, 200 },
            { -576.5153, -18.2943, 181.8442, 142 },
            { -627.0177, -10.9088, 211.8422, 101 },
            { -671.8592, -9.8908, 212.4219, 67 },
            { -668.2948, -10.9055, 148.1510, 150 },
            { -732.3407, -11.1492, 252.3793, 12 },
            { -710.5626, -9.3575, 245.5423, 137 },
            { -770.0189, -4.0000, 210.1038, 24 },
            { -708.5440, -9.5060, 112.1039, 133 },
            { -713.7353, -9.7718, 69.6887, 179 },
            { -759.4222, -4.0000, 73.3995, 207 },
            { -789.1661, -5.0765, 56.7524, 153 },
            { -812.1581, -8.2055, 27.7806, 252 },
            { -782.3595, -9.1053, 17.4428, 137 },
            { -779.9153, -7.5968, 24.7302, 60 },
        },
    },
}

local buriedAreaLookup = {}

for areaIndex, areaInfo in pairs(buriedAreas) do
    buriedAreaLookup[areaInfo.zone] = areaIndex
end

local function removeNPC(npc)
    -- This will send the packet to ensure NPC disappears for player
    npc:setStatus(xi.status.INVISIBLE)

    npc:timer(500, function(npcArg)
        if npcArg ~= nil then
            -- This will delete DE server side on zone tick
            npcArg:setStatus(xi.status.DISAPPEAR)
        end
    end)
end

local function awardDoubloons(player, amount)
    if type(amount) == "table" then
        amount = math.random(amount[1], amount[2])
    end

    player:incrementCharVar(settings.vars.DOUBLOONS, amount)
    player:sys("{} gains {} doubloons.", player:getName(), amount)
end

local function spawnChest(player, loc)
    local zone   = player:getZone()
    local pos    = player:getPos()
    local entity =
    {
        name        = "Pirate's Hoard",
        objtype     = xi.objType.NPC,
        look        = 961,
        x           = loc[1],
        y           = loc[2],
        z           = loc[3],
        rotation    = loc[4],
        widescan    = 1,

        onTrigger   = function(playerArg, npcArg)
            playerArg:fmt("This elusive chest can only be opened with a Pirate's Key.")
        end,

        onTrade     = function(player, npc, trade)
            if player:getFreeSlotsCount() < 5 then
                player:fmt("You lack the inventory space to collect these spoils.")
                return
            end

            if npcUtil.tradeHasExactly(trade, xi.item.PIRATES_KEY) then
                if npc:getLocalVar('opened') == 0 then
                    npc:setLocalVar('opened', 1)
                    npc:entityAnimationPacket('openH')

                    awardDoubloons(player, { 200, 300 })

                    for i = 1, 5 do
                        player:addTreasure(cexi.util.pickItem(boxRewards)[2], npc)
                    end

                    player:tradeComplete()
                end
            end
        end,

        releaseIdOnDisappear = true,
    }

    local npc = zone:insertDynamicEntity(entity)

    npc:hideName(true)
    npc:hideHP(true)

    npc:timer(60000, function(npcArg)
        if npcArg ~= nil then
            removeNPC(npcArg)
        end
    end)
end

-- Shovel dig
local shovelPath = "xi.items.brigands_shovel"
xi.module.ensureTable(shovelPath)

m:addOverride(shovelPath .. ".onItemCheck", function(player)
    local zoneID = player:getCharVar(settings.vars.TREASURE_BOX)

    if zoneID == player:getZoneID() then
        return 0
    end

    return xi.msg.basic.CANT_BE_USED_IN_AREA
end)

m:addOverride(shovelPath .. ".onItemUse", function(player)
    local loc  = player:getCharVar(settings.vars.TREASURE_LOC)
    local area = buriedAreaLookup[player:getZoneID()]

    if cexi.skill.digTreasure(player, buriedAreas[area].list[loc]) then
        player:timer(2000, function()
            player:fmt("You dig up a buried chest!")
            spawnChest(player, buriedAreas[area].list[loc])

            player:setCharVar(settings.vars.TREASURE_HUNT, 3, JstMidnight())
            player:setCharVar(settings.vars.TREASURE_LOC, 0)
            player:setCharVar(settings.vars.TREASURE_BOX, 0)
            player:setCharVar(settings.vars.TREASURE_KEY, 0)

            player:timer(28000, function()
                player:sys("Pirate's Hoard will despawn in 30 seconds.")
            end)
        end)
    end
end)

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function getTodaysFish(player)
    local todaysFish = {}

    todaysFish[1] = player:getCharVar(settings.vars.TARGET_1)
    todaysFish[2] = player:getCharVar(settings.vars.TARGET_2)
    todaysFish[3] = player:getCharVar(settings.vars.TARGET_3)

    if
        todaysFish[1] == 0 or
        todaysFish[2] == 0 or
        todaysFish[3] == 0
    then
        local totalFish = #cexi.ventures.fishing.list - #cexi.ventures.fishing.ventures[6].list
        local halfFish  = math.floor(totalFish / 2)

        todaysFish[1] = cexi.ventures.fishing.ventures[6].list[math.random(1, #cexi.ventures.fishing.ventures[6].list)][1]
        todaysFish[2] = cexi.ventures.fishing.list[math.random(1, halfFish)]
        todaysFish[3] = cexi.ventures.fishing.list[math.random(halfFish + 1, totalFish)]

        player:setCharVar(settings.vars.TARGET_1, todaysFish[1], JstMidnight())
        player:setCharVar(settings.vars.TARGET_2, todaysFish[2], JstMidnight())
        player:setCharVar(settings.vars.TARGET_3, todaysFish[3], JstMidnight())

        -- Reset amount collected for new day
        player:setCharVar(settings.vars.TARGET_1_QTY, 0, JstMidnight())
        player:setCharVar(settings.vars.TARGET_2_QTY, 0, JstMidnight())
        player:setCharVar(settings.vars.TARGET_3_QTY, 0, JstMidnight())
    end

    return todaysFish
end

local qtyVars =
{
    settings.vars.TARGET_1_QTY,
    settings.vars.TARGET_2_QTY,
    settings.vars.TARGET_3_QTY,
}

local function getTodaysInfo(todaysFish, player)
    local info = {}

    for i = 1, 3 do
        info[todaysFish[i]] =
        {
            name      = cexi.ventures.fishing.fish[todaysFish[i]],
            qtyVar    = qtyVars[i],
            traded    = player:getCharVar(qtyVars[i]),
            required  = settings.limits[i],
            value     = settings.values[i],
        }
    end

    return info
end

local function getFishValues(todaysFish, player)
    local qty = {}

    qty[todaysFish[1]] = 1
    qty[todaysFish[2]] = settings.limits[2] - player:getCharVar(settings.vars.TARGET_2_QTY)
    qty[todaysFish[3]] = settings.limits[3] - player:getCharVar(settings.vars.TARGET_3_QTY)

    return qty
end

local function getKeyZoneID(player)
    local keyZoneID = player:getCharVar(settings.vars.TREASURE_KEY)

    if keyZoneID == 0 then
        local areaIndex = math.random(1, #fishedMobs)
        keyZoneID = xi.zone[string.upper(fishedMobs[areaIndex][1])]
        player:setCharVar(settings.vars.TREASURE_KEY, keyZoneID)
    end

    return keyZoneID
end

local function getBoxZoneID(player)
    local boxZoneID = player:getCharVar(settings.vars.TREASURE_BOX)

    if boxZoneID == 0 then
        local areaIndex = math.random(1, #buriedAreas)
        local areaInfo  = buriedAreas[areaIndex]
        local location = math.random(1, #areaInfo.list)
        player:setCharVar(settings.vars.TREASURE_BOX, areaInfo.zone)
        player:setCharVar(settings.vars.TREASURE_LOC, location)
    end

    return boxZoneID
end

local function buriedTreasure(player, npc)
    local progress  = player:getCharVar(settings.vars.TREASURE_HUNT)
    local keyZoneID = getKeyZoneID(player)
    local boxZoneID = getBoxZoneID(player)

    if progress == 0 then
        -- Tell player to fish key
        local area = fishedMobs[fishedMobsLookup[keyZoneID]]
        cexi.util.dialog(player, settings.dialog.TREASURE_1, settings.name, { npc = npc, [1] = string.gsub(area[1], "_", " ") })
        player:setCharVar(settings.vars.TREASURE_HUNT, 1)
    elseif progress == 1 then
        -- Tell player to find box
        if player:hasItem(xi.item.PIRATES_KEY) then
            local area = buriedAreas[buriedAreaLookup[boxZoneID]]
            cexi.util.dialog(player, settings.dialog.TREASURE_2, settings.name, { npc = npc, [1] = string.gsub(area.area, "_", " ") })
            player:setCharVar(settings.vars.TREASURE_HUNT, 2)
        else
            -- Fishing key reminder
            local area = fishedMobs[fishedMobsLookup[keyZoneID]]
            cexi.util.dialog(player, settings.dialog.REMINDER_1, settings.name, { npc = npc, [1] = string.gsub(area[1], "_", " ") })
        end
    elseif progress == 2 then
        -- Finding box reminder
        local area = buriedAreas[buriedAreaLookup[boxZoneID]]
        cexi.util.dialog(player, settings.dialog.REMINDER_2, settings.name, { npc = npc, [1] = string.gsub(area.area, "_", " ") })
    else
        cexi.util.dialog(player, settings.dialog.TREASURE_3, settings.name, { npc = npc })
    end
end

local function purchaseItem(player, npc, item)
    delaySendMenu(player, {
        title   = string.format("Buy %s for %u dbl.?", item[1], item[3] ),
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    local points = player:getCharVar(settings.vars.DOUBLOONS)
                    npc:facePlayer(player, true)

                    if item[3] > points then
                        player:printToPlayer(settings.name .. " : " .. settings.dialog.DECLINE[1], xi.msg.channel.NS_SAY)
                        return
                    end

                    if npcUtil.giveItem(player, item[2]) then
                        player:setCharVar(settings.vars.DOUBLOONS, points - item[3])
                    end
                end,
            },
        },
    })
end

local function onTriggerJones(player, npc)
    npc:facePlayer(player, true)

    if player:getCharSkillLevel(xi.skill.FISHING) < 200 then
        cexi.util.dialog(player, settings.dialog.DEFAULT, settings.name, { npc = npc })
        return
    end

    local progress = player:getCharVar(settings.vars.PROGRESS)

    if progress == 0 then
        cexi.util.dialog(player, settings.dialog.START, settings.name, { npc = npc })
        player:setCharVar(settings.vars.PROGRESS, 1)
        return
    end

    local options =
    {
        {
            "Nuthin'",
            function()
            end,
        },
        {
            "Today's fish?",
            function()
                local todaysFish = getTodaysFish(player)
                cexi.util.dialog(player, settings.dialog.FISH, settings.name, {
                    npc = npc,
                    [1] = cexi.ventures.fishing.fish[todaysFish[1]],
                    [2] = cexi.ventures.fishing.fish[todaysFish[2]],
                    [3] = cexi.ventures.fishing.fish[todaysFish[3]],
                })
            end,
        },
        {
            "Tradin' doubloons",
            function()
                local points = player:getCharVar(settings.vars.DOUBLOONS)
                local title  = string.format("Tradin' doubloons (%u):", points)
                npc:facePlayer(player, true)

                if player:isCrystalWarrior() then
                    cexi.util.simpleShop(player, npc, rewardsCW, purchaseItem, title)
                else
                    cexi.util.simpleShop(player, npc, rewards, purchaseItem, title)
                end
            end,
        },
    }

    if player:hasItem(xi.item.BRIGANDS_SHOVEL) then
        table.insert(options, {
            "Buried Treasure",
            function()
                buriedTreasure(player, npc)
            end,
        })
    end

    delaySendMenu(player, {
        title   = "What'll it be mate?",
        options = options,
    })
end

local function onTriggerStash(player, npc)
    local progress = player:getCharVar(settings.vars.PROGRESS)

    if progress == 0 then
        player:fmt("It's locked.")
        return
    end

    local todaysFish = getTodaysFish(player)
    local todaysInfo = getTodaysInfo(todaysFish, player)

    player:fmt(" ")
    player:fmt("= Today's Fish =")
    player:fmt("{} ({}/{})", todaysInfo[todaysFish[1]].name, todaysInfo[todaysFish[1]].traded, todaysInfo[todaysFish[1]].required)
    player:fmt("{} ({}/{})", todaysInfo[todaysFish[2]].name, todaysInfo[todaysFish[2]].traded, todaysInfo[todaysFish[2]].required)
    player:fmt("{} ({}/{})", todaysInfo[todaysFish[3]].name, todaysInfo[todaysFish[3]].traded, todaysInfo[todaysFish[3]].required)
end

local function onTradeStash(player, npc, trade)
    local todaysFish = getTodaysFish(player)
    local todaysInfo = getTodaysInfo(todaysFish, player)

    if player:getCharVar(settings.vars.PROGRESS) == 0 then
        player:printToPlayer("It's locked.", xi.msg.channel.NS_SAY)
        return
    end

    local points   = 0
    local storing  = {}
    local lastSlot = trade:getSlotCount() - 1

    for i = 0, lastSlot do
        local itemID  = trade:getItemId(i)
        local itemQty = trade:getSlotQty(i)
        local stored  = false

        if not utils.contains(itemID, todaysFish) then
            player:fmt("{} : Oi, what arr you doin over there? That ain't the fish I asked for!", settings.name)
            return
        end

        if itemQty > (todaysInfo[itemID].required - todaysInfo[itemID].traded) then
            player:fmt("{} : That's more than I can handle for t'day!", settings.name)
            return
        end

        todaysInfo[itemID].updated = true
        todaysInfo[itemID].traded  = todaysInfo[itemID].traded + itemQty
        points = points + (math.random(todaysInfo[itemID].value[1], todaysInfo[itemID].value[2]) * itemQty)
    end

    for _, itemInfo in pairs(todaysInfo) do
        player:setCharVar(itemInfo.qtyVar, itemInfo.traded)
    end

    player:tradeComplete()
    player:incrementCharVar(settings.vars.DOUBLOONS, points)
    npc:ceAnimate(player, player, 46, 4)

    player:timer(500, function()
        for _, itemInfo in pairs(todaysInfo) do
            if itemInfo.updated then
                player:fmt("You stow away some {}. ({}/{})", itemInfo.name, itemInfo.traded, itemInfo.required)
            end
        end

        player:sys("{} obtains {} doubloons.", player:getName(), points)

        if
            player:getCharVar(settings.vars.TARGET_1_QTY) == settings.limits[1] and
            player:getCharVar(settings.vars.TARGET_2_QTY) == settings.limits[2] and
            player:getCharVar(settings.vars.TARGET_3_QTY) == settings.limits[3]
        then
            cexi.util.dialog(player, settings.dialog.BONUS, settings.name, { npc = npc })

            local delay = cexi.util.dialogDelay(settings.dialog.BONUS)

            player:timer(delay, function()
                player:incrementCharVar(settings.vars.DOUBLOONS, 100)
                player:sys("{} obtains {} doubloons.", player:getName(), 100)
            end)
        end
    end)
end

cexi.util.liveReload(m, {
    ["Norg"] =
    {
        {
            objtype   = xi.objType.NPC,
            name      = settings.name,
            look      = cexi.util.look({
                race = xi.race.GALKA,
                face = cexi.face.A8,
                main = 262, -- Falchion
                head = 298,
                body = cexi.model.CARDINAL_VEST,
                hand = 62, -- Crimson Finger Gauntlets
                legs = 98,
                feet = 62, -- Crimson Greaves
            }),
            x         = -15.391,
            y         = -0.164,
            z         = 5.567,
            rotation  = 224,
            widescan  = 1,
            onTrigger = onTriggerJones,
        },
        {
            objtype   = xi.objType.NPC,
            name      = "Sinister Stash",
            look      = 2331,
            x         = -16.578,
            y         = -0.045,
            z         = 7.584,
            rotation  = 237,
            widescan  = 1,
            onTrigger = onTriggerStash,
            onTrade   = onTradeStash,
        },
    },
    ["Lower_Jeuno"] =
    {
        {
            objtype   = xi.objType.NPC,
            name      = "Sinister Stash",
            hideName  = true, -- So we don't clutter up Jeuno
            look      = 2331,
            x         = -25.048,
            y         = -0.100,
            z         = -59.156,
            rotation  = 84,
            widescan  = 1,
            onTrigger = onTriggerStash,
            onTrade   = onTradeStash,
        },
    },
})

return m
