----------------------------------------------------
-- Crystal Warrior Exp
-- Job Level/Death Announcements
----------------------------------------------------
require("modules/module_utils")
require("scripts/globals/mobs")
require("scripts/globals/player")
require("scripts/effects/level_restriction")
-----------------------------------
local m = Module:new("cw_announcements")

local openingDecoration = "\129\154"
local closingDecoration = "\129\154"

-- World First Announcement Function
local checkWorldFirstServerVar = function(player, varName, worldMessage)
    local worldFirst = string.format("WF_%s", varName)
    local worldTime  = string.format("WT_%s", varName)

    if
        -- Record hasn't been set yet and player is not GM
        GetVolatileServerVariable(worldFirst) == 0 and
        player:getGMLevel() == 0
    then
        local decoratedMessage = string.format("%s %s %s", openingDecoration, worldMessage, closingDecoration)
        player:printToArea(decoratedMessage, xi.msg.channel.SYSTEM_3, 0, "") -- Sends announcement via ZMQ to all processes and zones

        -- Write out World First (WF) and World First Time (WT) to server vars)
        SetVolatileServerVariable(worldFirst, player:getID())
        SetVolatileServerVariable(worldTime, os.time())

        -- Summon big swirly starry animation which lingers on the players client in the location
        -- where this event happened. It will linger in that area for anyone that saw it until
        -- they zone.
        player:independentAnimation(player, 12, 3)
    end
end

m:addOverride("xi.effects.level_restriction.onEffectGain", function(target, effect)
    super(target, effect)

    if target:getObjType() == xi.objType.PC then
        local playerLvl = target:getMainLvl()

        if
            playerLvl < 75 and
            target:getCharVar("CHAR_TYPE") == 2 and
            not utils.mask.getBit(target:getCharVar("[CW]75"), target:getMainJob())
        then
            target:setLocalVar("[CW]No_Delevel", 1)
        end
    end
end)

m:addOverride("xi.effects.level_restriction.onEffectLose", function(target, effect)
    super(target, effect)

    if target:getObjType() == xi.objType.PC then
        local playerLvl = target:getMainLvl()

        if
            playerLvl < 75 and
            target:getCharVar("CHAR_TYPE") == 2 and
            not utils.mask.getBit(target:getCharVar("[CW]75"), target:getMainJob()) and
            target:getHP() > 0
        then
            target:setLocalVar("[CW]No_Delevel", 0)
        end
    end
end)

local function getMobFullName(str)
    local newName, dePrefix = string.gsub(str, "DE_", "")
    local realName, _ = string.gsub(newName, "_", " ")
    local initial = string.lower(string.sub(realName, 1, 1))
    local prefix  = 'a '

    if
        initial == 'a' or
        initial == 'e' or
        initial == 'i' or
        initial == 'o' or
        initial == 'u'
    then
        prefix = 'an '
    end

    return realName, prefix .. realName
end

local function getZoneFullName(player)
    local zoneName = player:getZoneName()

    if zoneName == nil then
        return ""
    end

    local realName, _ = string.gsub(zoneName, "_", " ")

    return " in " .. realName
end

-- Set main job to lv1 on death, and issue announcement
m:addOverride("xi.player.onPlayerDeath", function(player)
    super(player)

    local playerLvl = player:getJobLevel(player:getMainJob())

    if
        playerLvl < 75 and
        player:isUnbreakable() and
        not utils.mask.getBit(player:getCharVar("[CW]75"), player:getMainJob()) and
        player:getLocalVar("[CW]No_Delevel") == 0
    then
        local playerName     = player:getName()
        local playerJob      = player:getMainJob()
        local lostEXP        = player:resetCrystalWarrior()
        local deathAnnounced = player:getLocalVar("[CW]Death_Announced")

        -- Make sure death hasn't been announced already
        if deathAnnounced ~= 1 and playerLvl > 4 then
            player:recordDeath(playerLvl, lostEXP)

            local lastAttacker = player:getLastAttacker()

            if lastAttacker ~= "" then
                local mobName, mobFullName = getMobFullName(lastAttacker)

                player:printToArea(
                    string.format(
                        "An Unbreakable Crystal Warrior has shattered! %s fell to %s%s as a Lv%u %s (%s EXP lost).",
                        playerName,
                        mobFullName,
                        getZoneFullName(player),
                        playerLvl,
                        xi.jobNames[playerJob][2],
                        cexi.util.numWithCommas(lostEXP)
                    ),
                    xi.msg.channel.SYSTEM_3
                )
            else
                player:printToArea(
                    string.format(
                        "An Unbreakable Crystal Warrior has shattered! %s was a Level %u %s (%s EXP lost).",
                        playerName,
                        playerLvl,
                        xi.jobNames[playerJob][2],
                        cexi.util.numWithCommas(lostEXP)
                    ),
                    xi.msg.channel.SYSTEM_3
                )
            end

            player:incrementCharVar("[CW]EXP_LOST", lostEXP)
        end

        player:setLocalVar("[CW]Death_Announced", 1)
        player:setLocalVar("[CW]No_Delevel", 0)
    end
end)

local CW_TIME_TO_75 = "[CW]TIME_TO_75"
local CW_FIRST_75   = "[CW]FIRST_75"
local RE_MILESTONE  = "[RE]MILESTONE"

local referralLevels =
{
    cw =
    {
        [10] = { 0 },
        [20] = { 2 },
        [30] = { 4 },
    },

    qol =
    {
        [25] = { 0 },
        [50] = { 2 },
        [75] = { 4 },
    },
}

-- Issue announcement when players reach a leveling milestone
m:addOverride("xi.player.onPlayerLevelUp", function(player)
    super(player)

    local playerLvl = player:getJobLevel(player:getMainJob())

    -- Cleanup new player flag if player hits 25.
    if
        playerLvl >= 25 and
        player:getNewPlayer()
    then
        player:setNewPlayer(false)
    end

    if player:getReferral() ~= "" then
        local reMilestone = player:getCharVar(RE_MILESTONE)
        local result      = referralLevels.qol[playerLvl]

        if player:isCrystalWarrior() then
            result = referralLevels.cw[playerLvl]
        end

        if
            result ~= nil and
            not utils.mask.getBit(reMilestone, result[1])
        then
            player:setCharVar(RE_MILESTONE, utils.mask.setBit(reMilestone, result[1], true))
        end
    end

    if player:isCrystalWarrior() and playerLvl == 75 then
        -- When Crystal Warrior hits 75 for the first time
        -- Update leaderboards for time to 75
        if player:getCharVar(CW_TIME_TO_75) == 0 then
            player:setCharVar(CW_TIME_TO_75, player:getPlaytime())
            player:setCharVar(CW_FIRST_75, player:getMainJob())
        end

        -- "Lock" UCW job after 75, so they can't devel to 1
        if player:isUnbreakable() then
            local currentLocked = player:getCharVar("[CW]75")
            player:setCharVar("[CW]75", utils.mask.setBit(currentLocked, player:getMainJob(), true))
        end
    end

    local levelMilestones = { 10, 20, 30, 40, 50, 60, 70, 75 }
    for _, level in pairs(levelMilestones) do
        if playerLvl == level then
            if player:isUnbreakable() then
                checkWorldFirstServerVar(player,
                    string.format("UCW_JOB_%u_%s", level, xi.jobNames[player:getMainJob()][1]),
                    string.format("%s is the first Unbreakable Crystal Warrior to reach level %u on %s!", player:getName(), level, xi.jobNames[player:getMainJob()][2]))
            elseif player:isCrystalWarrior() then
                checkWorldFirstServerVar(player,
                    string.format("CW_JOB_%u_%s", level, xi.jobNames[player:getMainJob()][1]),
                    string.format("%s is the first Crystal Warrior to reach level %u on %s!", player:getName(), level, xi.jobNames[player:getMainJob()][2]))
            end
        end
    end

    if
        player:isCrystalWarrior() and
        player:getTotalLevel() == 1650 and
        player:getCharVar("[CW]MASTER") == 0 -- Prevent repeat messages same player
    then
        local decoratedMessage = string.format("%s %s has ascended and become a Crystal Warrior Master! %s", openingDecoration, player:getName(), closingDecoration)

        if player:isUnbreakable() then
           decoratedMessage = string.format("%s %s has ascended and become a Crystal Warrior Master (Unbreakable)! %s", openingDecoration, player:getName(), closingDecoration)
        end

        player:printToArea(decoratedMessage, xi.msg.channel.SYSTEM_3, 0, "") -- Sends announcement via ZMQ to all processes and zones
        player:setCharVar("[CW]MASTER", 1)
        player:independentAnimation(player, 12, 3)
    end

end)

m:addOverride("xi.mob.onMobDeathEx", function(mob, player, isKiller, isWeaponSkillKill)
    super(mob, player, isKiller, isWeaponSkillKill)

    local count   = player:getPartySize()
    local str     = ""
    local varName = ""

    if
        mob:isNM() and
        player:isCrystalWarrior() and
        isKiller
    then
        if player:isUnbreakable() then
            varName = "UCW_NM_KILL"

            if count > 1 then
                str = "%s's group are the first Crystal Warriors to defeat %s!"
            else
                str = "%s is the first Unbreakable Crystal Warrior to defeat %s!"
            end
        else
            varName = "CW_NM_KILL"
            if count > 1 then
                str = "%s's group are the first Crystal Warriors to defeat %s!"
            else
                str = "%s is the first Crystal Warrior to defeat %s!"
            end
        end

        if mob:isNM() and isKiller then
            local realName = mob:getPacketName()
            local safeName = string.gsub(mob:getName(), "'", "")

            if realName == "" then
                realName = string.gsub(mob:getName(), "DE_", "")
                realName = string.gsub(realName, "_", " ")
            end

            checkWorldFirstServerVar(player, varName .. string.upper(safeName), string.format(str, player:getName(), realName))
        end
    end
end)

-- Set Master stars on players
m:addOverride("xi.player.onGameIn", function(player, firstLogin, zoning)
    super(player, firstLogin, zoning)

    -- Check if Level Sync needs to be applied to player
    player:updateLevelSync()

    if
        player:getCharVar("[CW]MASTER") == 1 and
        player:isUnbreakable()
    then
        player:setMod(997, 5)
    end
end)

return m
