-----------------------------------
-- Colossal Clamming Competition
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("clamming_system")

local settings =
{
    MODEL     = 2424, -- Clamming Point model id
    DELAY     = 15,   -- Seconds before Clamming Point is available again
    REPEAT    = 3,    -- Seconds before player can trigger Clamming Point again
    ANIMATION = 48,   -- Animation during clamming
    DURATION  = 1000, -- Duration of animation
    cost      = 3000,
    cwCost    = 1000,
}

local vars =
{
    -- Character Variables
    ITEM      = "[CLAMMING]Item_{}",
    WEIGHT    = "[CLAMMING]Weight",
    SIZE      = "[CLAMMING]Size",
    BROKEN    = "[CLAMMING]Broken",
    NEXT      = "[CLAMMING]Next",     -- When the player can next perform clamming
    DELAY     = "[CLAMMING]Delay_{}", -- When the current Clamming Point can be reused
    CREDITS   = "[CLAMMING]Credits",

    -- Server Variables
    AREA      = "[CLAMMING]Area",
    NEXT      = "[CLAMMING]Next",
    EMPEROR   = "[CLAMMING]Emperor_Clam",
}

-- Bucket upgrade increases rate of Uncommon or rarer
-- CLAMMING_IMPROVED_RESULTS increases the tier by 1
local rate =
{
    VERY_COMMON = { 2400, 2400, 2400, 2400, 2400, },
    COMMON      = { 1500, 1500, 1500, 1500, 1500, },
    UNCOMMON    = { 1000, 1200, 1500, 1650, 1800, },
    RARE        = {  500,  600,  700,  750,  800, },
    VERY_RARE   = {  100,  150,  200,  225,  250, },
    SUPER_RARE  = {   50,   75,  100,  120,  140, },
    ULTRA_RARE  = {   10,   20,   30,   35,   40, },
}

-- Chance the bucket will randomly break
-- CLAMMING_REDUCED_INCIDENTS lowers the tier by 1
local incidents =
{
     0, -- 50 pz
     3, -- 100 pz
     8, -- 150 pz
    12, -- 200 pz
}

local weight =
{
    VERY_LIGHT  = 3,
    LIGHT       = 6,
    MODERATE    = 7,
    HEAVY       = 11,
    VERY_HEAVY  = 20,
    SUPER_HEAVY = 35,
}

local areas =
{
    [xi.zone.VALKURM_DUNES] =
    {
        name = "Valkurm_Dunes",
        next = xi.zone.BUBURIMU_PENINSULA,
        loot =
        {
            { rate.VERY_COMMON,   888, "a seashell",                 weight.LIGHT       }, -- 24% 6pz
            { rate.COMMON,      17296, "a pebble",                   weight.LIGHT       }, -- 15% 6pz
            { rate.COMMON,        881, "a crab shell",               weight.MODERATE    }, -- 15% 7pz
            { rate.UNCOMMON,      864, "a handful of fish scales",   weight.VERY_LIGHT  }, -- 10% 3pz
            { rate.UNCOMMON,      885, "a turtle shell",             weight.HEAVY       }, -- 10% 11pz
            { rate.UNCOMMON,      887, "a coral fragment",           weight.LIGHT       }, -- 10% 6pz
            { rate.UNCOMMON,     4484, "a shall shell",              weight.LIGHT       }, -- 10% 6pz
            { rate.RARE,         4361, "a nebimonite",               weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          750, "a silver beastcoin",         weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          936, "a chunk of rock salt",       weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          880, "a bone chip",                weight.LIGHT       }, --  5% 6pz
            { rate.VERY_RARE,     739, "a chunk of orichalcum ore",  weight.VERY_HEAVY  }, --  1% 20pz
            { rate.VERY_RARE,     882, "a sheep tooth",              weight.VERY_LIGHT  }, --  1% 3pz
            { rate.VERY_RARE,    1311, "a piece of oxblood",         weight.LIGHT       }, --  1% 6pz
            { rate.VERY_RARE,    1193, "a high-quality crab shell",  weight.HEAVY       }, --  1% 11pz
            { rate.VERY_RARE,    1312, "a piece of angel skin",      weight.VERY_LIGHT  }, --  1% 3pz
            { rate.SUPER_RARE,   9427, "an emperor clam",            weight.MODERATE    }, -- .5% 7pz
            { rate.SUPER_RARE,    866, "a handful of wyvern scales", weight.LIGHT       }, -- .5% 6pz
        },
        points =
        {
            -- East beach
            {  241.875,  4.000, -166.803 }, -- !pos 241.875 4.000 -166.803 103
            {  270.648,  4.000, -169.573 }, -- !pos 270.648 4.000 -169.573 103
            {  297.963,  4.000, -163.399 }, -- !pos 297.963 4.000 -163.399 103
            {  330.265,  4.000, -172.011 }, -- !pos 330.265 4.000 -172.011 103

            -- West Beach
            { -233.908,  4.300, -150.931 }, -- !pos -233.908 4.300 -150.931 103
            {  194.887,  4.000, -129.300 }, -- !pos 194.887 4.000 -129.300 103
            { -163.635,  4.000, -129.427 }, -- !pos -163.635 4.000 -129.427 103
            { -122.096,  4.000, -127.194 }, -- !pos -122.096 4.000 -127.194 103
        },
        announcer =
        {
            name = "Clamrox",
            pos  = { 74.006, 0.098, -82.149, 58 }, -- (G-8) !pos 74.006 0.098 -82.149 103
            look = 500,
        },
        npcs =
        {
            {
                name = "Shellnix",
                pos  = { 254.752, 0.125, -102.533, 63 }, -- (H-9) !pos 54.752 0.125 -102.533 103
                look = 496,
            },
            {
                name = "Sandilox",
                pos  = { -205.456, 0.326, -52.786, 58 }, -- (F-8) !pos -205.456 0.326 -52.786 103
                look = 499,
            },
        },
    },

    [xi.zone.BUBURIMU_PENINSULA] =
    {
        name = "Buburimu_Peninsula",
        next = xi.zone.CAPE_TERIGGAN,
        loot =
        {
            { rate.VERY_COMMON,   888, "a seashell",                weight.LIGHT       }, -- 24% 6pz
            { rate.COMMON,      17296, "a pebble",                  weight.LIGHT       }, -- 15% 6pz
            { rate.COMMON,       4400, "a slice of land crab meat", weight.MODERATE    }, -- 15% 7pz
            { rate.UNCOMMON,      868, "a handful of pugil scales", weight.VERY_LIGHT  }, -- 10% 3pz
            { rate.UNCOMMON,      893, "a giant femur",             weight.HEAVY       }, -- 10% 11pz
            { rate.UNCOMMON,      887, "a coral fragment",          weight.LIGHT       }, -- 10% 6pz
            { rate.UNCOMMON,     4484, "a shall shell",             weight.LIGHT       }, -- 10% 6pz
            { rate.RARE,         4361, "a nebimonite",              weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          856, "a rabbit hide",             weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          936, "a chunk of rock salt",      weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          816, "a spool of silk thread",    weight.LIGHT       }, --  5% 6pz
            { rate.VERY_RARE,    3509, "a plate of heavy metal",    weight.VERY_HEAVY  }, --  1% 20pz
            { rate.VERY_RARE,    1311, "a piece of oxblood",        weight.LIGHT       }, --  1% 6pz
            { rate.VERY_RARE,     882, "a sheep tooth",             weight.VERY_LIGHT  }, --  1% 3pz
            { rate.VERY_RARE,    1771, "a dragon bone",             weight.MODERATE    }, --  1% 7pz
            { rate.VERY_RARE,     722, "a divine log",              weight.HEAVY       }, --  1% 11pz
            { rate.SUPER_RARE,   9427, "an emperor clam",           weight.MODERATE    }, -- .5% 7pz
            { rate.SUPER_RARE,    938, "a sprig of papaka grass",   weight.VERY_LIGHT  }, -- .5% 3pz
        },
        points =
        {
            -- East beach
            {  445.149, 20.000,   74.667 }, -- !pos 445.149 20.000 74.667 118
            {  450.261, 20.000,  108.568 }, -- !pos 450.261 20.000 108.568 118
            {  446.551, 20.000,  136.214 }, -- !pos 446.551 20.000 136.214 118
            {  446.667, 20.000,  169.471 }, -- !pos 446.667 20.000 169.471 118

            -- West Beach (H-10)
            { -121.063, 20.000, -329.754 }, -- !pos -121.063 20.000 -329.754 118
            {  -92.214, 20.000, -329.598 }, -- !pos -92.214 20.000 -329.598 118
            {  -63.191, 20.000, -326.841 }, -- !pos -63.191 20.000 -326.841 118
            {  -34.915, 20.000, -328.118 }, -- !pos -34.915 20.000 -328.118 118
        },
        announcer =
        {
            name = "Clamrox",
            pos  = { 153.187, 0.786, -180.691, 43 }, -- (I-9) !pos 153.187 0.786 -180.691 118
            look = 500,
        },
        npcs =
        {
            {
                name = "Shellnix",
                pos  = { 333.914, 0.261, 103.234, 125 }, -- (J-7) !pos 333.914 0.261 103.234 118
                look = 496,
            },
            {
                name = "Sandilox",
                pos  = { -58.070, 16.000, -227.223, 73 }, -- (H-9) !pos -58.070 16.000 -227.223 118
                look = 499,
            },
        },
    },

    [xi.zone.CAPE_TERIGGAN] =
    {
        name = "Cape_Teriggan",
        next = xi.zone.VALKURM_DUNES,
        loot =
        {
            { rate.VERY_COMMON,   888, "a seashell",                 weight.LIGHT       }, -- 24% 6pz
            { rate.COMMON,      17296, "a pebble",                   weight.LIGHT       }, -- 15% 6pz
            { rate.COMMON,       4400, "a slice of land crab meat",  weight.MODERATE    }, -- 15% 7pz
            { rate.UNCOMMON,      868, "a handful of pugil scales",  weight.VERY_LIGHT  }, -- 10% 3pz
            { rate.UNCOMMON,     1193, "a high-quality crab shell",  weight.HEAVY       }, -- 10% 11pz
            { rate.UNCOMMON,      876, "a manta skin",               weight.LIGHT       }, -- 10% 6pz
            { rate.UNCOMMON,     4484, "a shall shell",              weight.LIGHT       }, -- 10% 6pz
            { rate.RARE,         4361, "a nebimonite",               weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          887, "a coral fragment",           weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          866, "a handful of wyvern scales", weight.LIGHT       }, --  5% 6pz
            { rate.RARE,          853, "a raptor skin",              weight.LIGHT       }, --  5% 6pz
            { rate.VERY_RARE,     646, "a chunk of adaman ore",      weight.VERY_HEAVY  }, --  1% 20pz
            { rate.VERY_RARE,     842, "a giant bird feather",       weight.VERY_LIGHT  }, --  1% 3pz
            { rate.VERY_RARE,    1311, "a piece of oxblood",         weight.LIGHT       }, --  1% 6pz
            { rate.VERY_RARE,     908, "an adamantoise shell",       weight.HEAVY       }, --  1% 11pz
            { rate.VERY_RARE,     854, "a cockatrice skin",          weight.MODERATE    }, --  1% 7pz
            { rate.SUPER_RARE,   9427, "an emperor clam",            weight.MODERATE    }, -- .5% 7pz
            { rate.SUPER_RARE,    942, "a philosophers stone",       weight.VERY_LIGHT  }, -- .5% 3pz
        },
        points =
        {
            {  112.838,  4.300, -198.535 }, -- !pos 112.838 4.300 -198.535 113
            {   82.727,  4.000, -171.403 }, -- !pos 82.727 4.000 -171.403 113
            {   56.757,  4.000, -167.667 }, -- !pos 56.757 4.000 -167.667 113
            {   31.734,  4.000, -169.583 }, -- !pos 31.734 4.000 -169.583 113
        },
        announcer =
        {
            name = "Clamrox",
            pos  = { -186.832, 7.312, -55.300, 95 }, -- (G-8) !pos -186.832 7.312 -55.300 113
            look = 500,
        },
        npcs =
        {
            {
                name = "Sandilox",
                pos  = { 86.250, -0.173, -77.149, 96 }, -- (I-8) !pos 86.250 -0.173 -77.149 113
                look = 499,
            },
        },
    },
}

local rewards =
{
    [xi.race.HUME_M] =
    {
        { "Custom Gilet",     11265,  500 },
        { "Custom Gilet +1",  11273, 2000 },
        { "Custom Trunks",    16321,  500 },
        { "Custom Trunks +1", 16329, 2000 },

        { "Hume Gilet",       14450, 1500 },
        { "Hume Gilet +1",    14457, 8000 },
        { "Hume Trunks",      15408, 1500 },
        { "Hume Trunks +1",   15415, 1500 },
    },

    [xi.race.HUME_F] =
    {
        { "Custom Top",        11266,  500 },
        { "Custom Top +1",     11274, 2000 },
        { "Custom Shorts",     16322,  500 },
        { "Custom Shorts +1",  16330, 2000 },

        { "Hume Top",          14451, 1500 },
        { "Hume Top +1",       14458, 8000 },
        { "Hume Shorts",       15409, 1500 },
        { "Hume Shorts +1",    15416, 8000 },
    },

    [xi.race.ELVAAN_M] =
    {
        { "Magna Gilet",       11267,  500 },
        { "Magna Gilet +1",    11275, 2000 },
        { "Magna Trunks",      16323,  500 },
        { "Magna Trunks +1",   16331, 2000 },

        { "Elvaan Gilet",      14452, 1500 },
        { "Elvaan Gilet +1",   14459, 8000 },
        { "Elvaan Trunks",     15410, 1500 },
        { "Elvaan Trunks +1",  15417, 8000 },
    },

    [xi.race.ELVAAN_F] =
    {
        { "Magna Top",         11268,  500 },
        { "Magna Top +1",      11276, 2000 },
        { "Magna Shorts",      16324,  500 },
        { "Magna Shorts +1",   16332, 2000 },

        { "Elvaan Top",        14453, 1500 },
        { "Elvaan Top +1",     14460, 8000 },
        { "Elvaan Shorts",     15411, 1500 },
        { "Elvaan Shorts +1",  15418, 8000 },
    },

    [xi.race.TARU_M] =
    {
        { "Wonder Maillot",    11269,  500 },
        { "Wonder Maillot +1", 11277, 2000 },
        { "Wonder Trunks",     16325,  500 },
        { "Wonder Trunks +1",  16333, 2000 },

        { "Tarutaru Maillot",  14454, 1500 },
        { "Taru. Maillot +1",  14461, 8000 },
        { "Tarutaru Trunks",   15412, 1500 },
        { "Taru. Trunks +1",   15419, 8000 },
    },

    [xi.race.TARU_F] =
    {
        { "Wonder Top",        11270,  500 },
        { "Wonder Top +1",     11278, 2000 },
        { "Wonder Shorts",     16326,  500 },
        { "Wonder Shorts +1",  16334, 2000 },

        { "Tarutaru Top",      14471, 1500 },
        { "Taru. Top +1",      14472, 8000 },
        { "Tarutaru Shorts",   15423, 1500 },
        { "Taru. Shorts +1",   15424, 8000 },
    },

    [xi.race.MITHRA] =
    {
        { "Savage Top",        11271,  500 },
        { "Savage Top +1",     11279, 2000 },
        { "Savage Shorts",     16327,  500 },
        { "Savage Shorts +1",  16335, 2000 },

        { "Mithra Top",        14455, 1500 },
        { "Mithra Top +1",     14462, 8000 },
        { "Mithra Shorts",     15413, 1500 },
        { "Mithra Shorts +1",  15420, 8000 },
    },

    [xi.race.GALKA] = 
    {
        { "Elder Gilet",       11272,  500 },
        { "Elder Gilet +1",    11280, 2000 },
        { "Elder Trunks",      16328,  500 },
        { "Elder Trunks +1",   16336, 2000 },

        { "Galka Gilet",       14456, 1500 },
        { "Galka Gilet +1",    14463, 8000 },
        { "Galka Trunks",      15414, 1500 },
        { "Galka Trunks +1",   15421, 8000 },
    },
}

-----------------------------------
-- Helper functions
-----------------------------------
local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local canUpgrade = function(player)
    local size   = player:getCharVar(vars.SIZE)
    local weight = player:getCharVar(vars.WEIGHT)

    return (
        size ==  50 and weight >=  45 or
        size == 100 and weight >=  95 or
        size == 150 and weight >= 145
    )
end

-----------------------------------
-- Bucket functions
-----------------------------------
local function clearBucket(player)
    local zoneID = player:getZoneID()

    for _, itemInfo in pairs(areas[zoneID].loot) do
        player:setCharVar(fmt(vars.ITEM, itemInfo[2]), 0)
    end

    player:setCharVar(vars.WEIGHT, 0)
    player:setCharVar(vars.SIZE,   0)
end

local function takeBucket(player)
    local zoneID = player:getZoneID()
    local ID     = zones[zoneID]
    local cost   = settings.cost

    if player:isCrystalWarrior() then
        cost = settings.cwCost
    end

    if player:getGil() < cost then
        player:fmt("You do not have enough gil.")
        return
    end

    player:delGil(cost)

    player:setCharVar(vars.BROKEN, 0)
    player:setCharVar(vars.WEIGHT, 0)
    player:setCharVar(vars.SIZE,  50)

    for _, itemInfo in pairs(areas[zoneID].loot) do
        player:setCharVar(fmt(vars.ITEM, itemInfo[2]), 0)
    end

    if player:hasKeyItem(xi.ki.CLAMMING_KIT) then
        player:delKeyItem(xi.ki.CLAMMING_KIT)
    end

    player:addKeyItem(xi.ki.CLAMMING_KIT)
    player:messageSpecial(ID.text.KEYITEM_OBTAINED, xi.ki.CLAMMING_KIT)
end

local function bucketUpgrade(player, npc)
    if canUpgrade(player) then
        local size = player:getCharVar(vars.SIZE) + 50

        cexi.util.dialog(player, {
            "Here's your upgraded bucket.",
            " See what else you can find!",
        }, npc:getPacketName(), { npc = npc })

        player:sys("Your clamming capacity has increased to {} ponzes!", size)
        player:fmt("Now you may be able to dig a...")

        player:setCharVar(vars.SIZE, size)
    end
end

local function bucketHandIn(player, npc)
    local weight = player:getCharVar(vars.WEIGHT)

    if weight == 0 then
        cexi.util.dialog(player, {
            "You don't have anything in that bucket yet!"
        }, npc:getPacketName(), { npc = npc })

        return
    end

    local total = 0

    for _, itemInfo in pairs(areas[zoneID].loot) do
        local quantity = player:getCharVar(fmt(vars.ITEM, itemInfo[2]))
        total = total + quantity
    end

    if total > player:getFreeSlotsCount() then
        player:sys("You do not have enough inventory space.")
        return
    end

    local zoneID  = player:getZoneID()
    local credits = weight * 3

    cexi.util.dialog(player, {
        "All right, let's get these wrapped up.",
        " ...",
        "Here you go! You're welcome back any time.",
        { message = fmt("{} gains {} clamming credits.", player:getName(), credits) }
    }, npc:getPacketName(), { npc = npc })

    player:incrementCharVar(vars.CREDITS, credits)

    -- Delay giving items until after dialog
    player:timer(4000, function(playerArg)
        local emperor = false

        for _, itemInfo in pairs(areas[zoneID].loot) do
            local varName  = fmt(vars.ITEM, itemInfo[2])
            local quantity = player:getCharVar(varName)

            if quantity > 0 then
                if itemInfo[2] == xi.item.EMPEROR_CLAM then
                    emperor = true
                else
                    if npcUtil.giveItem(player,
                        { { itemInfo[2], quantity } },
                        { multiple = true })
                    then
                        player:setCharVar(varName, 0)
                    else
                        cexi.util.dialog(player, { "You can't carry anymore." }, npc:getPacketName(), { npc = npc })
                        return
                    end
                end
            end
        end

        if emperor then
            if GetServerVariable(vars.EMPEROR) == 0 then
                SetServerVariable(vars.EMPEROR, os.time() + 600, NextConquestTally()) -- Clam is caught until Conquest Tally
                SetServerVariable(vars.NEXT, areas[zoneID].next) -- Set next zone in sequence
                xi.zones[player:getZoneName()].Zone.Announced = nil -- Reset this incase the server isn't restarted for more than 1 week for some reason

                local playerName = player:getName()
                player:printToArea(fmt("\129\154 {} found this week's Emperor Clam! \129\154", playerName), xi.msg.channel.SYSTEM_3, 0, "")
                print(fmt("[CLAMMING] Colossal Clamming Competition has ended ({} is the winner)", playerName))

                print(fmt("[CLAMMING] Event finished (Winner: {}), next week: {}", playerName, cexi.zoneName[areas[zoneID].next]))

                cexi.util.dialog(player, { "Incredible! You found the Emperor Clam! And that's the first one this week!" }, npc:getPacketName(), { npc = npc })
                npcUtil.giveCurrency(player, 'gil', 1000000)
            else
                cexi.util.dialog(player, { "Wow, you found an Emperor Clam! Unfortunately, another player got there first. Better luck next week!" }, npc:getPacketName(), { npc = npc })
                npcUtil.giveCurrency(player, 'gil', 5000)
            end
        end
    end)

    player:setCharVar(vars.BROKEN, 0)
    player:setCharVar(vars.WEIGHT, 0)
    player:delKeyItem(xi.ki.CLAMMING_KIT)
    player:messageSpecial(zones[zoneID].text.KEYITEM_LOST, xi.ki.CLAMMING_KIT)
end

local function weighBucket(player, npc)
    local weight = player:getCharVar(vars.WEIGHT)

    if weight == 0 then
        cexi.util.dialog(player, { "There's nothing in your bucket yet." }, npc:getPacketName(), { npc = npc } )
    else
        cexi.util.dialog(player, {
            fmt("Hmm... This weighs around {} ponzes.", weight),
            " Be careful to not add too much, or you'll break the bucket!",
        }, npc:getPacketName(), { npc = npc } )
    end
end

local function bucketUpdate(player, npc)
    if canUpgrade(player) then
        cexi.util.dialog(player, {
            "Wow, you filled the entire bucket!",
            fmt("Now you've proven yourself, how about an upgrade? This new bucket can hold up to {} ponzes!", player:getCharVar(vars.SIZE) + 50),
        }, npc:getPacketName(), { npc = npc })

        delaySendMenu(player, {
            title   = "Move your catch to a new bucket?",
            options =
            {
                {
                    "Yes, keep on clamming!",
                    function()
                        bucketUpgrade(player, npc)
                    end,
                },
                {
                    "No, collect my clams...",
                    function()
                        bucketHandIn(player, npc)
                    end,
                },
            },
        })
        return
    end

    bucketHandIn(player, npc)
end

-----------------------------------
-- Clamming point funtions
-----------------------------------
local function getResult(player)
    local zoneId       = player:getZoneID()
    local bucket       = math.floor(player:getVar(vars.SIZE) / 50)
    local breakRate    = incidents[bucket]

    if bucket > 50 and player:getMod(xi.mod.CLAMMING_REDUCED_INCIDENTS) > 0 then
        breakRate = incidents[bucket - 1]
    end

    -- Chance of clamming incident
    if math.random(0, 100) < breakRate then
        cexi.util.dialog(player, {
            "Something jumps into your bucket and breaks through the bottom!",
            " All your shellfish are washed away...",
        })

        player:setCharVar(vars.BROKEN, 1)
        clearBucket(player)

        return
    end

    local result = cexi.util.pickItem(
        areas[zoneId].loot,
        utils.clamp(bucket + player:getMod(xi.mod.CLAMMING_IMPROVED_RESULTS), 1, 5)
    )

    if result[2] == xi.item.EMPEROR_CLAM then
        if GetServerVariable(vars.EMPEROR) > 0 then
            result[2] = 17296
            result[3] = "a pebble"
            result[4] = weight.MODERATE
        end
    end

    player:incrementCharVar(fmt(vars.ITEM, result[2]), 1)
    player:incrementCharVar(vars.WEIGHT, result[4])

    if settings.DEBUG then
        player:sys("Added: {} pz, Total: {} pz", result[4], player:getVar(vars.WEIGHT))
    end

    -- Bucket breaks due to weight
    if player:getVar(vars.WEIGHT) > player:getCharVar(vars.SIZE) then
        cexi.util.dialog(player, {
            fmt("You find {} and toss it into your bucket...", result[3] ),
            "But the weight is too much for the bucket and its bottom breaks!",
            " All your shellfish are washed away...",
        })

        player:setCharVar(vars.BROKEN, 1)
        clearBucket(player)
    else
        player:fmt("You find {} and toss it into your bucket.", result[3])
    end
end

local function startClamming(player, npc)
    player:setLocalVar(vars.NEXT, os.time() + settings.REPEAT)
    player:setAnimation(settings.ANIMATION)

    -- Player must wait before using the current Clamming Point again
    if math.random(0, 100) < 15 then
        player:setLocalVar(fmt(vars.DELAY, npc:getID()), os.time() + settings.DELAY)
    end

    player:timer(settings.DURATION, function(playerArg)
        getResult(playerArg)
        playerArg:setAnimation(xi.animation.NONE)
    end)
end

-----------------------------------
-- Setup Clamming Points
-----------------------------------
local function onTrigger(player, npc)
    local zoneId = player:getZoneID()

    if not player:hasKeyItem(xi.ki.CLAMMING_KIT) then
        player:fmt("The area is littered with pieces of broken seashells.")
        return
    end

    if player:getCharVar(vars.BROKEN) > 0 then
        player:fmt("You cannot collect any clams with a broken bucket!")
        return
    end

    local t = os.time()

    if t > player:getLocalVar(vars.NEXT) then
        if t < player:getLocalVar(fmt(vars.DELAY, npc:getID())) then
            player:fmt("It looks like someone has been digging here.")
            return
        end

        player:fmt("The area is littered with pieces of broken seashells.")

        delaySendMenu(player, {
            title = "Dig here?",
            options =
            {
                {
                    "Yep",
                    function()
                        startClamming(player, npc)
                    end,
                },
                {
                    "Nope",
                    function()
                    end,
                },
            }
        })
    end
end

-----------------------------------
-- Setup NPCs
-----------------------------------
local function confirmPurchase(player, npc, item)
    local balance = player:getCharVar(vars.CREDITS)

    if item[3] > balance then
        player:sys("You can't afford this purchase.")
        return
    end

    if player:getFreeSlotsCount() < 1 then
        player:sys("You don't have enough space to purchase that.")
        return
    end

    delaySendMenu(player, {
        title   = string.format("Purchase {} for {} credits?", item[1], item[3]),
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    if npcUtil.giveItem(player, item[2]) then
                        player:incrementCharVar(vars.CREDITS, -item[3])
                    end
                end,
            },
        },
    })
end

local function npcOnTrigger(player, npc)
    if player:hasKeyItem(xi.ki.CLAMMING_KIT) then
        if player:getCharVar(vars.BROKEN) == 1 then
            cexi.util.dialog(player, {
                "You broke the bucket!\n That clamming kit is going to need repairs. I'll take it off your hands for now.",
                { message = "You return the clamming kit." },
            }, npc:getPacketName(), { npc = npc })

            player:setCharVar(vars.BROKEN, 0)
            player:delKeyItem(xi.ki.CLAMMING_KIT)
        else
            delaySendMenu(player, {
                title   = "Had enough clamming for today?",
                options =
                {
                    {
                        "I'm done.",
                        function()
                            bucketUpdate(player, npc)
                        end,
                    },
                    {
                        "Keep going.",
                        function()
                            weighBucket(player, npc)
                        end
                    },
                    {
                        "Explain again.",
                        function(player)
                            cexi.util.dialog(player, { "Try looking for seashells by the water." }, npc:getPacketName(), { npc = npc })
                        end,
                    },
                },
            })
        end
    else
        local cost = settings.cost

        if player:isCrystalWarrior() then
            cost = settings.cwCost
        end

        cexi.util.dialog(player, {
            fmt("Would you like to try clamming?\n It'll cost you {} gil.", cost),
        }, npc:getPacketName(), { npc = npc } )

        delaySendMenu(player, {
            title = "Begin clamming?",
            options =
            {
                {
                    "Yes",
                    function()
                        takeBucket(player)
                    end,
                },
                {
                    "No",
                    function()
                        cexi.util.dialog(player, {
                            "I understand. Clamming isn't for everyone.",
                        }, npc:getPacketName(), { npc = npc })
                    end,
                },
                {
                    "Shop",
                    function()
                        local balance = player:getCharVar(vars.CREDITS)
                        cexi.util.simpleShop(player, npc, rewards[player:getRace()], confirmPurchase, fmt("Select an item ({} credits):", balance))
                    end,
                },
                {
                    "Explain",
                    function(player)
                        cexi.util.dialog(player, {
                            "What's clamming? Nearby are prime locations for digging up shellfish and an assortment of other items.",
                            "You'll need a clamming kit to dig. Collect your finds in the bucket which comes with the kit.",
                            "The starter bucket stores up to 50 ponzes. Be careful to not overfill it.",
                            "When you're done, bring everything back here and I'll wrap up your findings to take home.",
                        }, npc:getPacketName(), { npc = npc })
                    end,
                },
            },
        })
    end
end

local function showNPC(zoneEntity, npcName)
    local result = zoneEntity:queryEntitiesByName(npcName)

    if
        result ~= nil and
        result[1] ~= nil
    then
        result[1]:setStatus(xi.status.NORMAL)
    end
end

local function hideNPC(zoneEntity, npcName)
    local result = zoneEntity:queryEntitiesByName(npcName)

    if
        result ~= nil and
        result[1] ~= nil
    then
        result[1]:setStatus(xi.status.DISAPPEAR)
    end
end

local entities = {}

for zoneID, area in pairs(areas) do
    entities[area.name] = entities[area.name] or {}

    table.insert(entities[area.name], {
        name      = area.announcer.name,
        objtype   = xi.objType.NPC,
        x         = area.announcer.pos[1],
        y         = area.announcer.pos[2],
        z         = area.announcer.pos[3],
        rotation  = area.announcer.pos[4],
        look      = area.announcer.look,
        hidden    = true,

        onTrigger = function(player, npc)
            cexi.util.dialog(player, {
                "Have you heard? It's time for the weekly Colossal Clamming Competition!",
                " There's a 1 million gil prize for the first one to catch the elusive Emperor Clam!",
                "Huh, what are you waiting for? Head over to the beaches and get Clamming!",
            }, npc:getPacketName(), { npc = npc })
        end,
    })

    for _, npcInfo in pairs(area.npcs) do
        table.insert(entities[area.name], {
            name      = npcInfo.name,
            objtype   = xi.objType.NPC,
            x         = npcInfo.pos[1],
            y         = npcInfo.pos[2],
            z         = npcInfo.pos[3],
            rotation  = npcInfo.pos[4],
            look      = npcInfo.look,
            hidden    = true,
            onTrade   = npcOnTrade,
            onTrigger = npcOnTrigger,
        })
    end

    for index, point in pairs(area.points) do
        table.insert(entities[area.name], {
            name       = fmt("Clamming {}", index),
            packetName = "Clamming Point",
            objtype    = xi.objType.NPC,
            x          = point[1],
            y          = point[2],
            z          = point[3],
            rotation   = point[3],
            look       = settings.MODEL,
            hidden     = true,
            onTrade    = onTrade,
            onTrigger  = onTrigger,
        })
    end

    local path = fmt("xi.zones.{}.Zone", area.name)

    m:addOverride(path .. ".onZoneIn", function(player, zonePrev)
        super(player, zonePrev)

        if player:hasKeyItem(xi.ki.CLAMMING_KIT) then
            player:setCharVar(vars.BROKEN, 1)
        end

        local currentZone = GetServerVariable(vars.AREA)

        if
            currentZone == zoneID and
            GetServerVariable(vars.EMPEROR) == 0
        then
            -- Player entered the current zone
            player:timer(3000, function()
                player:sys("The \129\154 Colossal Clamming Competition \129\154 is currently taking place!\n ")

                -- Perform announcement / Spawn NPCs if not done since restart
                if xi.zones[area.name].Zone.Announced == nil then
                    local areaName = string.gsub(areas[currentZone].name, "_", " ")

                    print(fmt("[CLAMMING] Colossal Clamming Competition started in {}", areaName))

                    player:printToArea(fmt("It's time for the weekly \129\154 Colossal Clamming Competition \129\154!\n Contestants make your way to {}!", areaName), xi.msg.channel.UNITY, xi.msg.area.SYSTEM, "Sandilox")
                    player:printToArea("Who will find the first Emperor Clam and claim that incredible 1 Million Gil prize!?", xi.msg.channel.UNITY, xi.msg.area.SYSTEM, "Sandilox")

                    xi.zones[area.name].Zone.Announced = true

                    local zoneEntity = player:getZone()

                    -- Spawn NPCs
                    showNPC(zoneEntity, fmt("DE_{}",area.announcer.name))

                    for index, point in pairs(area.points) do
                        showNPC(zoneEntity, fmt("DE_Clamming {}", index))
                    end

                    for _, npcInfo in pairs(area.npcs) do
                        showNPC(zoneEntity, fmt("DE_{}", npcInfo.name))
                    end
                end
            end)
        end
    end)

    m:addOverride(path .. ".onZoneOut", function(player)
        super(player)

        if player:hasKeyItem(xi.ki.CLAMMING_KIT) then
            clearBucket(player)
            player:setCharVar(vars.BROKEN, 1)
        end
    end)

    m:addOverride(path .. ".onGameHour", function(zone)
        super(zone)

        -- Set current zone if due
        local currentZone = GetServerVariable(vars.AREA)

        if currentZone == 0 then
            local nextZone = GetServerVariable(vars.NEXT)

            -- Guarantee a safe default on fresh server
            if nextZone == 0 then
                nextZone = xi.zone.VALKURM_DUNES
            end

            print(fmt("[Clamming] Event will soon spawn in {}", cexi.zoneName[nextZone]))
            SetServerVariable(vars.AREA, nextZone, NextConquestTally())
            return
        end

        -- Don't perform work for other zones
        if GetServerVariable(vars.AREA) ~= zone:getID() then
            return
        end

        -- If the Emperor Clam has already been caught, clean up NPCs for this zone if needed
        local emperorClam = GetServerVariable(vars.EMPEROR)

        if
            emperorClam > 0 and
            os.time() > emperorClam and
            xi.zones[area.name].Zone.Clamming_Done == nil
        then

            hideNPC(zone, fmt("DE_{}", area.announcer.name))

            for index, point in pairs(area.points) do
                hideNPC(zone, fmt("DE_Clamming {}", index))
            end

            for _, npcInfo in pairs(area.npcs) do
                hideNPC(zone, fmt("DE_{}", npcInfo.name))
            end

            xi.zones[area.name].Zone.Clamming_Done = true
        end
    end)
end

cexi.util.liveReload(m, entities)

return m
