-----------------------------------
-- Warp Restrictions based on char type
-----------------------------------
require("modules/module_utils")
require('scripts/globals/player')
require("scripts/globals/homepoint")
require("scripts/globals/teleports")
require("scripts/globals/teleports/survival_guide")
require("scripts/globals/conquest")
require("scripts/globals/unity")
require("scripts/globals/sparkshop")
-----------------------------------

local m = Module:new("warp_restrictions")

local message =
{
    ROE                 = "Your character type cannot access the Records of Eminence system.",
    UNITY_CONCORD       = "Your character type cannot access Unity Concord.",
    SURVIVAL_GUIDE      = "Your character type cannot use Survival Guides.",
}

-------------------------------------------------
-- Import table from scripts/globals/conquest.lua
-------------------------------------------------
local outposts =
{
    [xi.region.RONFAURE]        = { zone = xi.zone.WEST_RONFAURE,          ki = xi.ki.RONFAURE_SUPPLIES,              cp = 10, lvl = 10, fee = 100 },
    [xi.region.ZULKHEIM]        = { zone = xi.zone.VALKURM_DUNES,          ki = xi.ki.ZULKHEIM_SUPPLIES,              cp = 30, lvl = 10, fee = 100 },
    [xi.region.NORVALLEN]       = { zone = xi.zone.JUGNER_FOREST,          ki = xi.ki.NORVALLEN_SUPPLIES,             cp = 40, lvl = 15, fee = 150 },
    [xi.region.GUSTABERG]       = { zone = xi.zone.NORTH_GUSTABERG,        ki = xi.ki.GUSTABERG_SUPPLIES,             cp = 10, lvl = 10, fee = 100 },
    [xi.region.DERFLAND]        = { zone = xi.zone.PASHHOW_MARSHLANDS,     ki = xi.ki.DERFLAND_SUPPLIES,              cp = 40, lvl = 15, fee = 150 },
    [xi.region.SARUTABARUTA]    = { zone = xi.zone.WEST_SARUTABARUTA,      ki = xi.ki.SARUTABARUTA_SUPPLIES,          cp = 10, lvl = 10, fee = 100 },
    [xi.region.KOLSHUSHU]       = { zone = xi.zone.BUBURIMU_PENINSULA,     ki = xi.ki.KOLSHUSHU_SUPPLIES,             cp = 40, lvl = 10, fee = 100 },
    [xi.region.ARAGONEU]        = { zone = xi.zone.MERIPHATAUD_MOUNTAINS,  ki = xi.ki.ARAGONEU_SUPPLIES,              cp = 40, lvl = 15, fee = 150 },
    [xi.region.FAUREGANDI]      = { zone = xi.zone.BEAUCEDINE_GLACIER,     ki = xi.ki.FAUREGANDI_SUPPLIES,            cp = 70, lvl = 35, fee = 350 },
    [xi.region.VALDEAUNIA]      = { zone = xi.zone.XARCABARD,              ki = xi.ki.VALDEAUNIA_SUPPLIES,            cp = 50, lvl = 40, fee = 400 },
    [xi.region.QUFIMISLAND]     = { zone = xi.zone.QUFIM_ISLAND,           ki = xi.ki.QUFIM_SUPPLIES,                 cp = 60, lvl = 15, fee = 150 },
    [xi.region.LITELOR]         = { zone = xi.zone.THE_SANCTUARY_OF_ZITAH, ki = xi.ki.LITELOR_SUPPLIES,               cp = 40, lvl = 25, fee = 250 },
    [xi.region.KUZOTZ]          = { zone = xi.zone.EASTERN_ALTEPA_DESERT,  ki = xi.ki.KUZOTZ_SUPPLIES,                cp = 70, lvl = 30, fee = 300 },
    [xi.region.VOLLBOW]         = { zone = xi.zone.CAPE_TERIGGAN,          ki = xi.ki.VOLLBOW_SUPPLIES,               cp = 70, lvl = 50, fee = 500 },
    [xi.region.ELSHIMOLOWLANDS] = { zone = xi.zone.YUHTUNGA_JUNGLE,        ki = xi.ki.ELSHIMO_LOWLANDS_SUPPLIES,      cp = 70, lvl = 25, fee = 250 },
    [xi.region.ELSHIMOUPLANDS]  = { zone = xi.zone.YHOATOR_JUNGLE,         ki = xi.ki.ELSHIMO_UPLANDS_SUPPLIES,       cp = 70, lvl = 35, fee = 350 },
    [xi.region.TULIA]           = { zone = xi.zone.RUAUN_GARDENS,                                                     cp = 0,  lvl = 70, fee = 500 },
    [xi.region.MOVALPOLOS]      = { zone = xi.zone.OLDTON_MOVALPOLOS,                                                 cp = 40, lvl = 25, fee = 250 },
    [xi.region.TAVNAZIANARCH]   = { zone = xi.zone.LUFAISE_MEADOWS,        ki = xi.ki.TAVNAZIAN_ARCHIPELAGO_SUPPLIES, cp = 70, lvl = 30, fee = 300 },
}

local function hasOutpostOverride(player, region)
    local hasOP = player:hasTeleport(player:getNation(), region + 5)

    if
        not hasOP and
        not player:isCrystalWarrior()
    then
        hasOP = region <= xi.region.ELSHIMOUPLANDS
    end

    return hasOP
end

local function getRegionsMaskOverride(nation)
    local mask = 0

    for region = xi.region.RONFAURE, xi.region.TAVNAZIANARCH do
        if GetRegionOwner(region) == nation then
            mask = bit.bor(mask, bit.lshift(1, region + 5)) -- Region bits start at 5th bit
        end
    end

    return mask
end

local function getAllowedTeleportsOverride(player, nation)
    local allowedTeleports = 0x3F40001F -- All outposts set (0 indicates allowed)

    if not (player:isCrystalWarrior() or player:isClassicMode()) then
        return 0x3FE0001F -- Allow all outposts except for Tulia and Tavnazia
    end

    for region = xi.region.RONFAURE, xi.region.TAVNAZIANARCH do
        if not xi.conquest.canTeleportToOutpost(player, region) then
            allowedTeleports = bit.bor(allowedTeleports, bit.lshift(1, region + 5)) -- Region bits start at 5th bit
        end
    end

    return allowedTeleports
end

m:addOverride("xi.conquest.teleporterOnTrigger", function(player, teleporterNation, teleporterEvent)
    local sandyRegions     = getRegionsMaskOverride(xi.nation.SANDORIA)
    local bastokRegions    = getRegionsMaskOverride(xi.nation.BASTOK)
    local windyRegions     = getRegionsMaskOverride(xi.nation.WINDURST)
    local beastmenRegions  = getRegionsMaskOverride(xi.nation.BEASTMEN)
    local allowedTeleports = getAllowedTeleportsOverride(player, teleporterNation)
    local nationBits       = player:getNation() + bit.lshift(teleporterNation, 8)

    player:startEvent(teleporterEvent, sandyRegions, bastokRegions, windyRegions, beastmenRegions, 0, nationBits, player:getMainLvl(), allowedTeleports)
end)

m:addOverride("xi.conquest.outpostFee", function(player, region)
    if not hasOutpostOverride(player, region) then
        return 0
    end

    local fee = outposts[region].fee

    if GetRegionOwner(region) == player:getNation() then
        return fee
    else
        return fee * 3
    end
end)

m:addOverride("xi.conquest.canTeleportToOutpost", function(player, region)
    local outpost = outposts[region]
    if
        outpost == nil or
        player:getMainLvl() < outpost.lvl or
        not hasOutpostOverride(player, region)
    then
        return false
    end

    return true
end)

-- Block Explorer Moogles
m:addOverride("xi.teleport.explorerMoogleOnTrigger", function(player, event)
    if
        player:isClassicMode() and
        player:getRank(player:getNation()) < 5
    then
        player:printToPlayer("You cannot use Explorer Moogles before Rank 5.", xi.msg.channel.SYSTEM_3)
        return
    end

    if player:isCrystalWarrior() then
        player:printToPlayer("Crystal Warriors cannot use the Explorer Moogles.", xi.msg.channel.SYSTEM_3)
        return
    else
        super(player, event)
    end
end)

m:addOverride("xi.teleport.explorerMoogleOnEventFinish", function(player, csid, option, event)
    if player:isCrystalWarrior() then
        player:printToPlayer("Crystal Warriors cannot use the Explorer Moogles.", xi.msg.channel.SYSTEM_3)
        return
    else
        super(player, csid, option, event)
    end
end)

-- Domenic (BCNM Warps) for all players
m:addOverride("xi.zones.Lower_Jeuno.npcs.Domenic.onTrigger", function(player, npc)
    -- Players unlock use of Domenic by completing custom quest "Digging up Dirt".
    if player:getCharVar("[CW]DOMENIC") == 1 then
        player:startEvent(10115, player:getGil())
    else
        player:startEvent(10116)
    end
end)

-----------------------------------
-- Homepoint Warps
-----------------------------------
local canHomePointWarp = function(player)
    -- Generic will block HP warps if this var exists
    -- TODO set this variable for ACE on death to avoid zombie fights against HNMs?
    if player:getCharVar('HomepointWarpUsed') > 0 then
        return false
    end

    if player:isCrystalWarrior() then
        return false
    end

    -- WEW only allowed HP warps from cities
    if
        player:isClassicMode() and
        player:getZone() and
        player:getZone():getTypeMask() ~= xi.zoneType.CITY
    then
        return false
    end

    return true
end

m:addOverride("xi.homepoint.onTrigger", function(player, csid, index)
    if
        not canHomePointWarp(player)
    then
        player:startEvent(csid, 0, 0, 0, 0, 0, player:getGil(), 4095, index)
        if not player:isCrystalWarrior() then
            -- the default event params above block registering homepoints, too
            player:fmt('Homepoint : You may not use Homepoint teleports at this time.')

            -- register homepoint, still
            local hpBit  = index % 32
            local hpSet  = math.floor(index / 32)

            if not player:hasTeleport(xi.teleport.type.HOMEPOINT, hpBit, hpSet) then
                player:addTeleport(xi.teleport.type.HOMEPOINT, hpBit, hpSet)
                player:fmt('You have registered a new home point!')
            end
        end

        return
    end

    super(player, csid, index)
end)

m:addOverride("xi.homepoint.onEventUpdate", function(player, csid, option)
    if not canHomePointWarp(player) then
        player:updateEvent(-1, -1, -1)
        printf("[warning] player %s initiated xi.homepoint.onEventUpdate when they should not be able to", player:getName())
        return
    end

    super(player, csid, option)
end)

m:addOverride("xi.homepoint.onEventFinish", function(player, csid, option, event)
    local choice = bit.band(option, 0xFF)
    if
        (choice == 2 or choice == 3) and -- event finish choice for teleport or same region
        csid == event
    then
        if not canHomePointWarp(player) then
            printf("[warning] player %s initiated xi.homepoint.onEventFinish when they should not be able to", player:getName())
            return -- TODO this will leave player stuck in blackscreen, zone them in place?
        end

        if player:isClassicMode() then
            player:setCharVar('HomepointWarpUsed', 1, os.time() + 30 * 60) -- HP warp every 30 minutes
        end
    end

    super(player, csid, option, event)
end)

-----------------------------------
-- Survival Guides
-----------------------------------
m:addOverride("xi.survivalGuide.onTrigger", function(player)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.SURVIVAL_GUIDE, xi.msg.channel.SYSTEM_3)
        return
    end

    super(player)
end)

m:addOverride("xi.survivalGuide.onEventUpdate", function(player, csid, option)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        printf("[warning] player %s initiated xi.survivalGuide.onEventUpdate as a Crystal Warrior", player:getName())
        return
    end

    super(player, csid, option)
end)

m:addOverride("xi.survivalGuide.onEventFinish", function(player, eventId, option)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        printf("[warning] player %s initiated xi.survivalGuide.onEventFinish as a Crystal Warrior", player:getName())
        return
    end

    super(player, eventId, option)
end)

-----------------------------------
-- Unity Concord
-----------------------------------
m:addOverride("xi.unity.onTrigger", function(player, csid, option)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.UNITY_CONCORD, xi.msg.channel.SYSTEM_3)
    else
        super(player, csid, option)
    end
end)

m:addOverride("xi.unity.onEventUpdate", function(player, csid, option)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.UNITY_CONCORD, xi.msg.channel.SYSTEM_3)
    else
        super(player, csid, option)
    end
end)

m:addOverride("xi.unity.onEventFinish", function(player, csid, option)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.UNITY_CONCORD, xi.msg.channel.SYSTEM_3)
    else
        super(player, csid, option)
    end
end)

-----------------------------------
-- Sparks Shop
-----------------------------------
m:addOverride("xi.sparkshop.onTrigger", function(player, csid, option)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.ROE, xi.msg.channel.SYSTEM_3)
        return
    else
        super(player, csid, option)
    end
end)

m:addOverride("xi.sparkshop.onEventUpdate", function(player, csid, option, npc)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.ROE, xi.msg.channel.SYSTEM_3)
        return
    else
        super(player, csid, option, npc)
    end
end)

m:addOverride("xi.sparkshop.onTrade", function(player, npc, trade, eventid)
    if
        player:isCrystalWarrior() or
        player:isClassicMode()
    then
        player:printToPlayer(message.ROE, xi.msg.channel.SYSTEM_3)
        return
    else
        super(player, npc, trade, eventid)
    end
end)

return m
