-----------------------------------
-- Campaign Core
-----------------------------------
require("modules/module_utils")
require("scripts/globals/utils")
require("scripts/globals/campaign")
require("modules/catseyexi/lua/additive_overrides/systems/campaign/campaign_data")
-----------------------------------
local m = Module:new("campaign_core")

xi = xi or {}
xi.campaign = xi.campaign or {}

local settings =
{
    CREDITS    = "[CAMPAIGN]CREDITS",   -- Promotion Credits
    EVALUATION = "[CAMPAIGN]EVALUATION" -- Evaluation Cooldown
}

xi.campaign.zoneData = {}

local state =
{
    INACTIVE = 0,
    WAITING  = 1,
    FIGHTING = 2,
}

-- Returns index of the first medal player doesn't have
local function getNextRank(player)
    for index, medal in pairs(xi.campaign.promotions) do
        if not player:hasKeyItem(medal[1]) then
            return index
        end
    end

    return #xi.campaign.promotions
end

local function getCredits(player, participants, win)
    local adjusted = utils.clamp(participants, 1, 18)

    if win then
        return xi.campaign.credits.win[adjusted]
    else
        return xi.campaign.credits.loss[adjusted]
    end
end

local function awardCredits(player, nextRank, credits)
    local rankInfo = xi.campaign.promotions[nextRank]
    local balance  = player:getCharVar(settings.CREDITS)

    player:sys("{} gains {} promotion credits.", player:getName(), credits)

    -- Handle rank up
    if balance + credits >= rankInfo[2] then
        local cooldown = player:getCharVar(settings.EVALUATION)

        if cooldown > 0 then
            player:setCharVar(settings.CREDITS, rankInfo[2])
            player:sys("You must wait {} hours before receiving your next promotion. ({}/{})", math.random((cooldown - os.time()) / 3600), rankInfo[2], rankInfo[2])
        else
            player:sys("{} has been awarded the {}!", player:getName(), rankInfo[3])

            local carryOver = (balance + credits) - rankInfo[2]
            player:setCharVar(settings.CREDITS, carryOver)
            player:setCharVar(settings.EVALUATION, os.time() + utils.hours(72))
            npcUtil.giveKeyItem(player, rankInfo[1])
        end
    else
        player:fmt("Your next promotion is the {} ({}/{})", rankInfo[3], balance + credits, rankInfo[2])
        player:setCharVar(settings.CREDITS, balance + credits)
    end
end

local function removeNPC(npc)
    -- This will send the packet to ensure NPC disappears for player
    npc:setStatus(xi.status.INVISIBLE)

    npc:timer(500, function(npcArg)
        if npcArg ~= nil then
            -- This will delete DE server side on zone tick
            npcArg:setStatus(xi.status.DISAPPEAR)
        end
    end)
end

local function endCampaign(zone, win)
    local zoneID = zone:getID()

    if xi.campaign.zoneData[zoneID] == nil then
        return
    end

    for _, npcID in pairs(xi.campaign.zoneData[zoneID].npcs or {}) do
        DespawnMob(npcID, zone)
    end

    for _, mobID in pairs(xi.campaign.zoneData[zoneID].mobs or {}) do
        DespawnMob(mobID, zone)
    end

    xi.campaign.zoneData[zoneID].state = state.INACTIVE

    local players  = zone:getPlayers()
    local eligible = {}

    for _, player in pairs(players) do
        if win then
            player:messageSpecial(zones[zoneID].text.CAMPAIGN_RESULTS_TALLIED + xi.campaign.messages.CAMPAIGN_WIN)
        else
            player:messageSpecial(zones[zoneID].text.CAMPAIGN_RESULTS_TALLIED + xi.campaign.messages.CAMPAIGN_LOSE)
        end

        player:messageSpecial(zones[zoneID].text.CAMPAIGN_RESULTS_TALLIED + xi.campaign.messages.CAMPAIGN_FINISH)
        player:changeMusic(0, xi.campaign.data[zoneID].music[1])
        player:changeMusic(1, xi.campaign.data[zoneID].music[2])

        if player:hasStatusEffect(xi.effect.ALLIED_TAGS) then
            player:delStatusEffect(xi.effect.ALLIED_TAGS)
            table.insert(eligible, player)
        end
    end

    for _, player in pairs(eligible) do
        local credits  = getCredits(player, #eligible, win)
        local nextRank = getNextRank(player)

        player:timer(3000, function(playerArg)
            awardCredits(playerArg, nextRank, credits)
        end)

        -- TODO: Set some contribution requirement here
        player:setLocalVar("[CAMPAIGN]SPOILS", 1)
    end

    SetServerVariable(fmt("[CAMPAIGN]ACTIVE_{}", zone:getID()), 0)

    zone:setBackgroundMusicDay(xi.campaign.data[zoneID].music[1])
    zone:setBackgroundMusicNight(xi.campaign.data[zoneID].music[2])
    zone:setSoloBattleMusic(xi.campaign.data[zoneID].music[3])
    zone:setPartyBattleMusic(xi.campaign.data[zoneID].music[4])

    if #eligible > 0 then
        local de = zone:insertDynamicEntity({
            name        = "Campaign Coffer",
            objtype     = xi.objType.NPC,
            look        = 961,
            x           = xi.campaign.data[zoneID].chest[1],
            y           = xi.campaign.data[zoneID].chest[2],
            z           = xi.campaign.data[zoneID].chest[3],
            rotation    = xi.campaign.data[zoneID].chest[4],
            widescan    = 1,

            onTrigger   = function(player, npc)
                if player:getLocalVar("[CAMPAIGN]SPOILS") == 0 then
                    player:fmt("You lack the required participation to access this chest.")
                    return
                end

                if player:getFreeSlotsCount() < 5 then
                    player:fmt("You lack the inventory space to collect these spoils.")
                    return
                end

                npc:ceAnimationPacket(player, "open", npc)
                player:setLocalVar("[CAMPAIGN]SPOILS", 0)

                npc:timer(5000, function(npcArg)
                    npcArg:ceAnimationPacket(player, "clos", npcArg)
                end)

                local qty = math.random(2, 5)

                for i = 1, qty do
                    player:addTreasure(cexi.util.pickItem(xi.campaign.data[zoneID].items)[2], npc)
                end
            end,

            releaseIdOnDisappear = true,
        })

        -- Remove chest after 5 minutes
        de:timer(300000, function(npcArg)
            if npcArg ~= nil then
                removeNPC(npcArg)
            end
        end)
    end
end

local function aggroGroups(group1, group2)
    for _, entityId1 in pairs(group1) do
        for _, entityId2 in pairs(group2) do
            local entity1 = GetMobByID(entityId1)
            local entity2 = GetMobByID(entityId2)

            if entity1 == nil or entity2 == nil then
                printf("[warning] Could not apply aggro because either %i or %i are not valid entities", entityId1, entityId2)
            else
                --printf("Applying enmity: %i <-> %i", entityId1, entityId2)
                entity1:addEnmity(entity2, math.random(1, 5), math.random(1, 5))
                entity2:addEnmity(entity1, math.random(1, 5), math.random(1, 5))
            end
        end
    end
end

local function applyModsAndEffects(mob, mobInfo)
    if
        mobInfo.mods ~= nil and
        #mobInfo.mods > 0
    then
        for modID, modValue in pairs(mobInfo.mods) do
            mob:setMod(modID, modValue)
        end
    end

    if
        mobInfo.effects ~= nil and
        #mobInfo.effects > 0
    then
        for effectID, effectTable in pairs(mobInfo.effects) do
            mob:addStatusEffect(effectID, unpack(effectTable))
        end
    end
end

local function applyTemplate(mob, template)
    if template == nil then
        return
    end

    if template.mods ~= nil then
        for modID, modValue in pairs(template.mods) do
            mob:setMod(modID, modValue)
        end
    end

    if template.effects ~= nil then
        for effectID, effectTable in pairs(template.effects) do
            mob:addStatusEffect(effectID, unpack(effectTable))
        end
    end
end

local function getRandomPos(posList)
    return posList[math.random(1, #posList)]
end

local function spawnMob(zone, mob, posList, allied)
    local pos     = getRandomPos(posList)
    local zoneID  = zone:getID()
    local mobName = mob:getName()

    if xi.campaign.mods[mobName] ~= nil then
        applyModsAndEffects(mob, xi.campaign.mods[mobName])
    end

    applyTemplate(mob, xi.campaign.basicResist)

    mob:setSpawn(unpack(pos))
    mob:setMobMod(xi.mobMod.NO_DROPS,  1)
    mob:setMobMod(xi.mobMod.CHARMABLE, 0)
    mob:spawn()

    DisallowRespawn(mob:getID(), true)

    mob:setRoamFlags(xi.roamFlag.SCRIPTED)
    mob:addStatusEffect(xi.effect.ALLIED_TAGS, 1, 0, 0)

    if allied then
        mob:addListener("DEATH", "CAMPAIGN_NPC_DEATH", function(mobArg)
            local zoneID = zone:getID()

            xi.campaign.zoneData[zoneID].defeatedNPCs = (xi.campaign.zoneData[zoneID].defeatedNPCs or 0) + 1

            local defeated = xi.campaign.zoneData[zoneID].defeatedNPCs
            local total    = #xi.campaign.zoneData[zoneID].npcs

            printf("[CAMPAIGN] NPC died %s/%u", defeated, total)

            -- When all NPCs are dead, end campaign
            if defeated >= total then
                endCampaign(mobArg:getZone(), false)
            end
        end)

        table.insert(xi.campaign.zoneData[zoneID].npcs, mob:getID())
    else
        mob:addListener("DEATH", "CAMPAIGN_MOB_DEATH", function(mobArg)
            local zoneID = zone:getID()

            xi.campaign.zoneData[zoneID].defeatedMobs = (xi.campaign.zoneData[zoneID].defeatedMobs or 0) + 1

            local defeated = xi.campaign.zoneData[zoneID].defeatedMobs
            local total    = #xi.campaign.zoneData[zoneID].mobs

            printf("[CAMPAIGN] Mob died %s/%u", defeated, total)

            -- When all mobs are dead, end campaign
            if defeated >= total then
                endCampaign(mobArg:getZone(), true)
            end
        end)

        table.insert(xi.campaign.zoneData[zoneID].mobs, mob:getID())
    end
end

local function spawnMobs(zone, mobs, posList, allied)
    for _, mobName in pairs(mobs) do
        local res = zone:queryEntitiesByName(mobName)

        if res ~= nil then
            for _, mob in pairs(res) do
                spawnMob(zone, mob, posList, allied)
            end
        else
            print(fmt("[CAMPAIGN] Unable to spawn {}", mobs[1]))
        end
    end
end

local function startCampaign(zone, ally, enemy)
    if not xi.settings.main.ENABLE_CAMPAIGN then
        return
    end

    SetServerVariable(fmt("[CAMPAIGN]ACTIVE_{}", zone:getID()), 1, os.time() + utils.hours(2)) -- Add 2 hour expiry incase this doesn't get cleared
    print(fmt("[CAMPAIGN] A battle has begun in {} ({} vs. {})", zone:getName(), ally[1], enemy[1]))

    zone:setBackgroundMusicDay(247)
    zone:setBackgroundMusicNight(247)
    zone:setSoloBattleMusic(247)
    zone:setPartyBattleMusic(247)

    local zoneID  = zone:getID()
    local area    = xi.campaign.data[zoneID]
    local players = zone:getPlayers()

    for _, player in pairs(players) do
        player:messageSpecial(zones[zoneID].text.CAMPAIGN_RESULTS_TALLIED + xi.campaign.messages.CAMPAIGN_BEGIN)
        player:changeMusic(0, 247)
        player:changeMusic(1, 247)
    end

    xi.campaign.zoneData[zoneID]              = xi.campaign.zoneData[zoneID] or {}
    xi.campaign.zoneData[zoneID].state        = state.WAITING
    xi.campaign.zoneData[zoneID].defeatedNPCs = 0
    xi.campaign.zoneData[zoneID].defeatedMobs = 0
    xi.campaign.zoneData[zoneID].npcs         = {}
    xi.campaign.zoneData[zoneID].mobs         = {}

    spawnMobs(zone, ally,  area.defender, true)
    spawnMobs(zone, enemy, area.attacker, false)

    aggroGroups(xi.campaign.zoneData[zoneID].npcs, xi.campaign.zoneData[zoneID].mobs)
end

xi.campaign.start = function(player)
    startCampaign(
        player:getZone(),
        xi.campaign.group.SANDORIA[1],
        xi.campaign.group.ORCISH[1]
    )
end

xi.campaign.stop = function(player)
    endCampaign(player:getZone(), false)
end

xi.campaign.win = function(player)
    endCampaign(player:getZone(), true)
end

for zoneName, zoneInfo in pairs(xi.campaign.npcOverride) do
    for npcIndex, npcName in pairs(zoneInfo) do
        local path = fmt("xi.zones.{}.npcs.{}", zoneName, npcName)
        cexi.util.ensureNPC(zoneName, npcName)

        m:addOverride(path .. ".onTrigger", function(player, npc)
            local zoneID = player:getZoneID()

            if not player:hasCompletedMission(xi.mission.log_id.WOTG, xi.mission.id.wotg.BACK_TO_THE_BEGINNING) then
                player:startEvent(xi.campaign.event[npcIndex][2])
                return
            end

            if GetServerVariable(fmt("[CAMPAIGN]ACTIVE_{}", zoneID)) > 0 then
                if player:hasStatusEffect(xi.effect.ALLIED_TAGS) then
                    player:startEvent(xi.campaign.event[npcIndex][1], 0, 21)
                else
                    player:startEvent(xi.campaign.event[npcIndex][1], 0, 5)
                end
            else
                player:startEvent(xi.campaign.event[npcIndex][1])
            end
        end)

        m:addOverride(path .. ".onEventFinish", function(player, csid, option, npc)
            if
                csid == xi.campaign.event[npcIndex][1] and
                option == 3
            then
                player:addStatusEffect(xi.effect.ALLIED_TAGS, 1, 0, 0)
            end
        end)
    end
end

for zoneName, armyName in pairs(xi.campaign.activeZones) do
    m:addOverride(fmt("xi.zones.{}.Zone.onGameHour", zoneName), function(zone)
        super(zone)

        if
            not xi.settings.main.ENABLE_CAMPAIGN or
            #zone:getPlayers() == 0 or
            GetServerVariable(fmt("[CAMPAIGN]ACTIVE_{}", zone:getID())) == 1
        then
            return
        end

        for i = 1, 5 do
            local deployed = fmt("[CAMPAIGN]DEPLOYED_{}_{}", armyName[2], i)

            if GetServerVariable(deployed) == 0 then
                SetServerVariable(deployed, zone:getID(), os.time() + math.random(utils.hours(2), utils.minutes(4)))
                startCampaign(zone, xi.campaign.group[armyName[1]][math.random(1, 5)], xi.campaign.group[armyName[2]][i])
                return
            end
        end
    end)

    m:addOverride(fmt("xi.zones.{}.Zone.onInitialize", zoneName), function(zone)
        super(zone)

        local zoneID = zone:getID()
        local offset = zones[zoneID].npc.CAMPAIGN_NPC_OFFSET

        for i = 0, 7 do
            GetNPCByID(offset + i):setStatus(xi.status.INVISIBLE)
        end

        -- TODO: Make this based on territory control
        if xi.campaign.activeZones[zoneName][1] == "SANDORIA" then
            GetNPCByID(offset + xi.campaign.npcs.SANDORIA_GUARD):setStatus(xi.status.NORMAL)
            GetNPCByID(offset + xi.campaign.npcs.SANDORIA_FLAG):setStatus(xi.status.NORMAL)
        elseif xi.campaign.activeZones[zoneName][1] == "BASTOK" then
            GetNPCByID(offset + xi.campaign.npcs.BASTOK_GUARD):setStatus(xi.status.NORMAL)
            GetNPCByID(offset + xi.campaign.npcs.BASTOK_FLAG):setStatus(xi.status.NORMAL)
        elseif xi.campaign.activeZones[zoneName][1] == "WINDURST" then
            GetNPCByID(offset + xi.campaign.npcs.WINDURST_GUARD):setStatus(xi.status.NORMAL)
            GetNPCByID(offset + xi.campaign.npcs.WINDURST_FLAG):setStatus(xi.status.NORMAL)
        end
    end)
end

return m
