-----------------------------------
-- Beast Raising - Beastiary Book
-----------------------------------
-- !pos -95.620 -0.100, -202.150 245
-----------------------------------
require("modules/module_utils")
require('scripts/globals/npc_util')
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("beast_raising_beastiary_book")

local vars =
{
    beast    = "[BEAST]{}",
    fed      = "[BEAST]FED",
    activity = "[BEAST]ACTIVITY",
    unlocked = "[BEAST]UNLOCKED",

    -- Dailies
    roll     = "[BEAST]ACTIVITY_ROLL",
    lock     = "[BEAST]ACTIVITY_LOCK",
    treasure = "[BEAST]TREASURE",
    found    = "[BEAST]FOUND",
    mobs     = "[BEAST]MOBS",
}

local areas =
{
    xi.zone.LA_THEINE_PLATEAU,
    xi.zone.KONSCHTAT_HIGHLANDS,
    xi.zone.TAHRONGI_CANYON,
}

-----------------------------------
-- Utilities
-----------------------------------
local delaySendMenu = function(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

function petDialog(tbl, petName)
    return cexi.util.npcSpawner(tbl, petName, "Beastiary Book")
end

function gainLevel(player, pet, amount)
    local beastLevelVar = fmt(vars.beast, pet.name)
    local beastLevel    = player:getCharVar(beastLevelVar)

    if beastLevel < 50 then
        player:sys("{} gains {} experience points.", pet.name, amount * 100)
        player:setCharVar(beastLevelVar, utils.clamp(beastLevel + amount), 0, 50)

        if beastLevel % 10 == 0 then
            player:sys("{} reaches level {}! (Bonus: {})", pet.name, string.rep("\129\154", level), pet.desc)
        end
    end
end

local function getName(str)
    return string.gsub(str, "_", " ")
end

-----------------------------------
-- Daily Activities
-----------------------------------
local function restDay(player, npc, pet)
    cexi.util.dialog(player, petDialog({
        fmt("{} is happy to see you but just wants to rest today.", pet.name),
        { delay = 3000 },
    }, pet.name))

    player:setCharVar(vars.activity, 1, JstMidnight())
    gainLevel(player, pet, 2)
end

local function defeatMobs(player, npc, pet)
    local roll = player:getCharVar(vars.roll)

    if roll == 0 then
        roll = math.random(1, #pet.mobs)
        player:setCharVar(vars.roll, roll, JstMidnight())
        player:setCharVar(vars.mobs, 0)
    end

    local mobQty   = pet.mobs[roll][3]
    local mobName  = getName(pet.mobs[roll][2])
    local zoneName = getName(pet.mobs[roll][1])

    if player:getCharVar(vars.mobs) >= mobQty then
        cexi.util.dialog(player, petDialog({
            { name = pet.name, emotion = "is happy to see you return." },
            { entity = pet.name, animate = 251, mode = 4 }, -- Hearts animation
            { delay = 3000 },
        }, pet.name))

        player:setCharVar(vars.roll, 0)
        player:setCharVar(vars.mobs, 0)
        player:setCharVar(vars.activity, 1, JstMidnight())
        gainLevel(player, pet, 4)
    else
        cexi.util.dialog(player, petDialog({
            { name = pet.name, emotion = fmt("wants you to defeat {} {}s in {}.", mobQty, mobName, zoneName) },
            { delay = 3000 },
        }, pet.name))
    end
end

local function findTreasure(player, npc, pet)
    if player:getCharVar(vars.found) == 1 then
        cexi.util.dialog(player, petDialog({
            { name = pet.name, emotion = fmt("is very happy you were able to find the item!", zoneName) },
            { entity = pet.name, animate = 251, mode = 4 }, -- Hearts animation
            { delay = 3000 },
        }, pet.name))

        player:setCharVar(vars.roll, 0)
        player:setCharVar(vars.treasure, 0)
        player:setCharVar(vars.found, 0)
        player:setCharVar(vars.activity, 1, JstMidnight())
        gainLevel(player, pet, 5)

        return
    end

    local zoneID = player:getCharVar(vars.treasure)

    if zoneID == 0 then
        zoneID = areas[math.random(1, #areas)]
        player:setCharVar(vars.treasure, zoneID, JstMidnight())
        player:setCharVar(vars.roll, 0)
    end

    local zoneName = getName(cexi.zoneName[zoneID])

    cexi.util.dialog(player, petDialog({
        { name    = pet.name, emotion = fmt("wants you to find something buried in {}.", zoneName) },
        "If you don't know where to look, just have a '/think'!",
        { delay   = 3000 },
    }, pet.name))
end

-----------------------------------
-- Menu Options
-----------------------------------
function checkStatus(player, npc, petIndex)
    local pet = cexi.job.bst.beasts[petIndex[2]]
    local options =
    {
        {
            "Status",
            function()
                if player:getCharVar(vars.fed) == 0 then
                    cexi.util.dialog(player, petDialog({
                        fmt("{} has not yet been fed. {} enjoys {}.", pet.name, pet.name, pet.feed.name),
                        { delay = 3000 },
                    }, pet.name))
                else
                    cexi.util.dialog(player, petDialog({
                        { name  = pet.name, emotion = "is satisfied." },
                        { delay = 3000 },
                    }, pet.name))
                end
            end,
        },
        {
            "Check Bonus",
            function()
                local level = math.floor(player:getCharVar(fmt(vars.beast, pet.name)) / 10)
                level = utils.clamp(level, 0, 5)

                if level == 0 then
                    player:sys("{} has not learned any bonuses yet.", pet.name)
                else
                    local tbl =
                    {
                        { name = pet.name, emotion = fmt("has reached level {} ({}).", string.rep("\129\154", level), pet.desc) },
                    }

                    if level >= 5 then
                        table.insert(tbl, { name = pet.name, emotion = fmt("has the ability to inspire alies with {}.", pet.buff.name) })
                    end

                    table.insert(tbl, { delay = 3000 })

                    cexi.util.dialog(player, petDialog(tbl, pet.name))
                end
            end,
        },
    }

    local activity = player:getCharVar(vars.activity)

    if
        activity == 0 or
        (activity > 1 and player:getCharVar(vars.lock) == petIndex[2])
    then
        table.insert(options, {
            "Activity",
            function()
                if activity == 0 then
                    activity = math.random(1, 3) + 1
                    player:setCharVar(vars.activity, activity,    JstMidnight())
                    player:setCharVar(vars.lock,     petIndex[2], JstMidnight())
                end

                if activity == 2 then
                    restDay(player, npc, pet)

                elseif activity == 3 then
                    defeatMobs(player, npc, pet)

                elseif activity == 4 then
                    findTreasure(player, npc, pet)
                end
            end,
        })
    end

    table.insert(options, {
        "Exit",
        function()
        end,
    })

    delaySendMenu(player, {
        title   = fmt("Select an action ({}):", pet.name),
        options = options,
    })
end

local function onTrigger(player, npc)
    local unlocked = player:getCharVar(vars.unlocked)

    if unlocked == 0 then
        player:sys("You haven't unlocked any beasts to raise yet.")
        return
    end

    local list = {}

    -- Only display unlocked pets on the menu
    for index, pet in pairs(cexi.job.bst.beasts) do
        if utils.mask.getBit(unlocked, index) then
            table.insert(list, { pet.name, index })
        end
    end

    cexi.util.simpleMenu(player, npc, list, checkStatus, "Choose a pet:")
end

local function onTrade(player, npc, trade)
    local unlocked = player:getCharVar(vars.unlocked)

    if unlocked == 0 then
        return
    end

    if player:getCharVar(vars.fed) == 0 then
        for index, pet in pairs(cexi.job.bst.beasts) do
            if
                utils.mask.getBit(unlocked, index) and
                npcUtil.tradeHasExactly(trade, pet.feed.item)
            then
                player:tradeComplete()

                cexi.util.dialog(player, petDialog({
                    { name   = pet.name, emotion = fmt("takes the {} and is now satisfied!", pet.feed.name) },
                    { entity = pet.name, animate = 251, mode = 4 }, -- Hearts animation
                    { delay  = 3000},
                }, pet.name))

                player:setCharVar(vars.fed, pet.id, JstMidnight())
                gainLevel(player, pet, 1)
            end
        end
    else
        player:sys("You have already fed a pet today.")
    end
end

local entities =
{
    {
        objtype   = xi.objType.NPC,
        name      = "Beastiary Book",
        look      = 2290,
        x         = -99,522,
        y         = -0.100,
        z         = -204.110,
        rotation  = 112,
        onTrade   = onTrade,
        onTrigger = onTrigger,
    },
}

for _, pet in pairs(cexi.job.bst.beasts) do
    table.insert(entities, {
        objtype  = xi.objType.NPC,
        name     = pet.name,
        look     = pet.look,
        x        = -99,522,
        y        = -0.100,
        z        = -204.110,
        rotation = 240,
        hidden   = true,
    })
end

cexi.util.liveReload(m, {
    ["Lower_Jeuno"] = entities
})

return m
