-----------------------------------
-- CatsEyeXI
-- Aman Trove Global
-- Created by: GM Hax - 12/11/2023
-----------------------------------
require('modules/module_utils')
require('scripts/globals/npc_util')
-----------------------------------
local m = Module:new('aman_trove_global')
local ID = zones[xi.zone.HORLAIS_PEAK]
-----------------------------------
-- Global settings
-----------------------------------

-- Debug mode toggle. Set to true to enable debug messages.
local debugMode = false

m.chestExpBonus = 100  -- Experience bonus per non-mimic chest opened

-- Jackpot settings
m.jackpotReward = 500000 -- Amount of gil for jackpot
m.jackpotChance = 5 -- Chance out of 100 (5%) for jackpot to occur

-- Percent chance settings for different chest outcomes
local chestOutcomeChances = {
    normal = 70, -- Percent chance for normal (NQ) outcome
    hq1    = 20, -- Percent chance for high quality 1 (HQ1) outcome
    hq2    = 10, -- Percent chance for high quality 2 (HQ2) outcome
    rainbowChestChance = 50, -- Percent chance for a rainbow chest to spawn
    entryRainbowChestChance = 20 -- Percent chance for a rainbow chest to spawn on entry
}

-----------------------------------
-- Venus setup
-----------------------------------

-- -- List of original coffer IDs for Venus
local venusOriginalCofferIDs = {
    17347068, 17347069, 17347070, 17347071, 17347072,
    17347073, 17347074, 17347075, 17347076, 17347077,
}

-- Terminal ID for Venus
m.venusTerminalID = 17347078

-- Flag to track if a rainbow chest has spawned in Venus
local venusRainbowChestSpawned = false

-- Resets venusCofferMeta to its original state.
m.venusCofferMeta = {
    reset = function(self)
        -- Empty the table
        for k in pairs(self) do self[k] = nil end
        -- Repopulate with original IDs
        for _, id in ipairs(venusOriginalCofferIDs) do
            table.insert(self, id)
        end
    end
}

-- Assign 'reset' function to venusCofferIDs via metatable.
m.venusCofferIDs = {}
setmetatable(m.venusCofferIDs, {__index = m.venusCofferMeta})

-----------------------------------
-- Mars setup
-----------------------------------

-- List of original coffer IDs for Mars
local marsOriginalCofferIDs = {
    17347035, 17347036, 17347037, 17347038, 17347039,
    17347040, 17347041, 17347042, 17347043, 17347044,
}

-- Terminal ID for Mars
m.marsTerminalID = 17347045

-- Flag to track if a rainbow chest has spawned in Mars
local marsRainbowChestSpawned = false

-- Resets marsCofferMeta to its original state.
m.marsCofferMeta = {
    reset = function(self)
        -- Empty the table
        for k in pairs(self) do self[k] = nil end
        -- Repopulate with original IDs
        for _, id in ipairs(marsOriginalCofferIDs) do
            table.insert(self, id)
        end
    end
}

-- Assign 'reset' function to marsCofferIDs via metatable.
m.marsCofferIDs = {}
setmetatable(m.marsCofferIDs, {__index = m.marsCofferMeta})

-----------------------------------
-- Global functions
-----------------------------------

-- Debug function
m.debugPrint = function(message)
    if debugMode then
        print("[DEBUG] " .. message)
    end
end

-- Changes to a mimic and K.O.s the player
m.mimicAction = function(mob, player)
    m.debugPrint("Mimic action triggered for mob ID: " .. tostring(mob:getID()))

    if player and player:getID() then  -- Check if player is valid
        player:addStatusEffect(xi.effect.STUN, 1, 0, 20)
        m.debugPrint("Player stunned. Player ID: " .. tostring(player:getID()))
    else
        m.debugPrint("Player not valid or not present.")
    end

    mob:setAnimationSub(1)

    mob:timer(3000, function()
        mob:setModelId(259)
        mob:hideName(false)
        mob:setStatus(xi.status.UPDATE)
        -- mob:setAnimationSub(5)
        mob:setAnimationSub(13)

        m.debugPrint("Mimic model updated for mob ID: " .. tostring(mob:getID()))

        if player and player:getID() then  -- Check again before accessing player
            mob:setMobMod(xi.mobMod.NO_AGGRO, 0)
            mob:updateClaim(player)
            m.debugPrint("Mimic claimed by player ID: " .. tostring(player:getID()))
        else
            m.debugPrint("Player not valid or not present at mimic update.")
        end

        mob:setBehaviour(bit.band(mob:getBehaviour(), bit.bnot(xi.behavior.STANDBACK)))
        mob:setBehaviour(bit.band(mob:getBehaviour(), bit.bnot(xi.behavior.NO_TURN)))
        -- mob:useMobAbility(4167)  -- Kathwack
        mob:useMobAbility(588)  -- Death_Trap

        m.debugPrint("Mimic transformation complete. Kathack used on mob ID: " .. tostring(mob:getID()))
    end)
end

m.removeBFStatus = function(battlefield)
    local playerID = battlefield:getInitiator()
    local player = GetPlayerByID(playerID)

    if player then
        player:timer(5000, function(player)
            local playerID = player:getID()
            local zone = player:getZoneID()

            m.debugPrint("Initiator ID: " .. playerID .. " in Zone: " .. zone)

            local alliance = player:getAlliance()
            for _, member in pairs(alliance) do
                m.debugPrint("Checking alliance member ID: " .. member:getID() .. " in Zone: " .. member:getZoneID())

                if member:getZoneID() == zone and member:getID() ~= playerID then
                    local battlefieldEffectID = xi.effect.BATTLEFIELD
                    if member:hasStatusEffect(battlefieldEffectID) then
                        member:delStatusEffect(battlefieldEffectID)
                        m.debugPrint("Explicitly removed battlefield status (ID: " .. battlefieldEffectID .. ") from member ID: " .. member:getID())
                    else
                        m.debugPrint("No battlefield status effect (ID: " .. battlefieldEffectID .. ") to remove for member ID: " .. member:getID())
                    end
                else
                    if member:getID() == playerID then
                        m.debugPrint("Skipping initiator ID: " .. playerID)
                    else
                        m.debugPrint("Skipping member ID: " .. member:getID() .. " (not in the same zone)")
                    end
                end
            end
        end)
    else
        m.debugPrint("Could not find player with ID: " .. playerID)
    end
end

-----------------------------------
-- Venus functions
-----------------------------------

-- Prints Venus battlefield debug info to the player.
m.venusPrintDebugInfo = function(player, battlefield)
    if debugMode then
        local reward = battlefield:getLocalVar("reward")
        local rewardReceived = battlefield:getLocalVar("rewardReceived")
        local round = battlefield:getLocalVar("round")
        local mimicID = battlefield:getLocalVar("mimicID")
        player:printToPlayer("Venus Debug Info:", 28)
        player:printToPlayer("Reward: " .. reward, 28)
        player:printToPlayer("Reward Received: " .. rewardReceived, 28)
        player:printToPlayer("Round: " .. round, 28)
        player:printToPlayer("Mimic ID: " .. mimicID, 28)
    end
end

-- Function to despawn the Venus Terminal Coffer
m.venusTerminalDespawn = function()
    local terminalCoffer = GetNPCByID(m.venusTerminalID)
    terminalCoffer:setStatus(xi.status.DISAPPEAR)
    m.debugPrint("Venus Terminal Coffer despawned with ID: " .. m.venusTerminalID)
end

-- Function to remove a given chest ID from the venusCofferIDs table
m.venusRemoveChestID = function(mob)
    local chestID = mob:getID()
    for i = #m.venusCofferIDs, 1, -1 do
        if m.venusCofferIDs[i] == chestID then
            table.remove(m.venusCofferIDs, i)
            m.debugPrint("Removed chest ID " .. chestID .. " from venusCofferIDs.")
            return true
        end
    end
    m.debugPrint("Chest ID " .. chestID .. " not found in venusCofferIDs.")
    return false
end

-- Function to clean up all coffers except the one that triggered the event
m.venusRemoveALLChests = function(triggerMob)
    local bf = triggerMob:getBattlefield()
    if bf then
        local triggerMobID = triggerMob:getID()
        m.debugPrint("Cleaning up coffers in Battlefield ID: " .. bf:getID())

        for _, cofferID in ipairs(venusOriginalCofferIDs) do
            if cofferID ~= triggerMobID then
                DespawnMob(cofferID)
                m.debugPrint("Despawned Coffer_OPlenty with ID: " .. cofferID)
            end
        end
    else
        m.debugPrint("Error: Battlefield not found.")
    end
end

-- Function to transform a remaining chest into a rainbow chest
m.venusTransformToRainbowChest = function(battlefield)
    local bfID = battlefield:getID()

    m.debugPrint("Transforming to Rainbow Chest in battlefield ID: " .. bfID)

    -- Randomly select an unopened chest to transform
    if #m.venusCofferIDs > 0 then
        local selectedChestID = m.venusCofferIDs[math.random(#m.venusCofferIDs)]
        m.debugPrint("Selected chest ID for transformation: " .. selectedChestID)

        local rainbowChest = GetMobByID(selectedChestID)

        -- Check if the chest object is valid
        if rainbowChest then
            m.debugPrint("Transforming chest ID: " .. selectedChestID .. " into a rainbow chest")

            -- Transform the chest into a rainbow chest
            rainbowChest:setStatus(xi.status.DISAPPEAR)
            rainbowChest:timer(1000, function(mob)
                mob:setModelId(2426) -- Model ID for rainbow chest
                mob:setStatus(xi.status.NORMAL)
            end)

            -- Call the venusRemoveChestID function
            m.venusRemoveChestID(rainbowChest)
            venusRainbowChestSpawned = true
        else
            m.debugPrint("Failed to retrieve chest object for ID: " .. selectedChestID)
        end
    else
        m.debugPrint("No unopened chests available for transformation in battlefield ID: " .. bfID)
    end
end

m.venusEntryRainbowChest = function (battlefield)
    local bfID = battlefield:getID()
    local rainbowModelID = 2426  -- Rainbow chest model ID
    local entryRainbowChestChance = chestOutcomeChances.entryRainbowChestChance

    m.debugPrint("Attempting to transform a chest into a rainbow chest in Battlefield ID: " .. bfID)

    -- Randomly decide if a chest will be transformed into a rainbow chest
    if math.random(100) <= entryRainbowChestChance then
        m.debugPrint("Chance for rainbow chest met. Searching for an unopened chest in Battlefield ID: " .. bfID)

        -- Randomly select an unopened chest to transform
        if #m.venusCofferIDs > 0 then
            local entryRainbowChestID = m.venusCofferIDs[math.random(#m.venusCofferIDs)]
            local rainbowChest = GetMobByID(entryRainbowChestID)

            -- Set the model ID of the selected chest to the rainbow chest model ID
            rainbowChest:setModelId(rainbowModelID)

            -- Remove the selected chest ID from the list of unopened chest IDs
            m.venusRemoveChestID(rainbowChest)

            m.debugPrint("Transformed chest ID " .. entryRainbowChestID .. " into a rainbow chest in Battlefield ID: " .. bfID)
        else
            m.debugPrint("No eligible chest found for transformation in Battlefield ID: " .. bfID)
        end
    else
        m.debugPrint("Rainbow chest transformation chance not met in Battlefield ID: " .. bfID)
    end

    m.removeBFStatus(battlefield)
end

m.venusRandomizeMimic = function (battlefield)
    local bfID = battlefield:getID()
    local remainingChests = m.venusCofferIDs

    m.debugPrint("Randomizing mimic for Battlefield ID: " .. bfID)

    -- If there's more than one chest remaining, randomly select one to be the mimic
    if #remainingChests > 1 then
        local mimicID = remainingChests[math.random(#remainingChests)]
        battlefield:setLocalVar("mimicID", mimicID)
        m.debugPrint("Selected mimic ID for next round: " .. mimicID .. " in Battlefield ID: " .. bfID)
    -- If there's only one chest remaining, treat it as the final chest and don't select a mimic
    elseif #remainingChests == 1 then
        local finalChestMob = GetMobByID(remainingChests[1])
        m.venusFinalBox(finalChestMob)
        battlefield:setLocalVar("mimicID", 0)
        m.debugPrint("No mimic selected for next round as only final chest remains in Battlefield ID: " .. bfID)
    -- If there are no chests remaining, print a message
    else
        m.debugPrint("No remaining chests to be selected as mimic in Battlefield ID: " .. bfID)
    end
end

m.venusFinalBox = function(mob)
    local bfID = mob:getBattlefieldID()  -- Get the battlefield ID
    m.debugPrint("finalBox function triggered for Battlefield ID: " .. bfID)

    local finalChestID = nil  -- This will store the ID of the final chest
    local rainbowModelID = 2426  -- Rainbow chest model ID
    local goldModelID = 969     -- Gold chest model ID

    -- If there's only one chest left, it's the final chest
    if #m.venusCofferIDs == 1 then
        finalChestID = m.venusCofferIDs[1]
        m.debugPrint("Final chest ID determined: " .. finalChestID)

        -- Randomly select the chest type: 1 for rainbow, 2 for gold
        local chestType = math.random(2)
        -- Set the model ID based on the chest type
        local modelID = (chestType == 1) and rainbowModelID or goldModelID

        -- Set the model ID for the final chest
        local finalChest = GetMobByID(finalChestID)
        finalChest:setModelId(modelID)
        m.debugPrint("Final chest transformed. Model ID set to: " .. modelID)

        -- Remove the final chest ID from the venusCofferIDs table
        table.remove(m.venusCofferIDs, 1)
        m.debugPrint("Removed final chest ID " .. finalChestID .. " from venusCofferIDs.")
    else
        m.debugPrint("Not the final chest. No transformation applied.")
    end
end

m.venusRollChestOutcome = function(player, battlefield, mob)
    local roll = math.random(100)
    local reward = battlefield:getLocalVar("reward")
    local bfID = battlefield:getID()
    local chestType = mob:getModelId()
    local chestID = mob:getID()
    local currentRound = battlefield:getLocalVar("round")
    local rainbow = 2426 -- Rainbow chest model ID
    local gold    = 969  -- Gold chest model ID
    local wood    = 966  -- Wooden chest model ID

    -- Function to increment the reward
    local incrementReward = function(amount)
        reward = reward + amount
        battlefield:setLocalVar("reward", reward)
        m.debugPrint("Reward incremented to " .. tostring(reward))
    end

    --Remove the chest ID from the venusCofferIDs table
    m.venusRemoveChestID(mob)

    -- Increment the round
    battlefield:setLocalVar("round", battlefield:getLocalVar("round") + 1)
    -- Check for specific chest types
    if chestType == rainbow then
        player:messageSpecial(ID.text.WHATS_THE_MATTARU + 5)  -- Display a special message for rainbow chest
        incrementReward(15)  -- Increment the reward by 15
        m.debugPrint("Rainbow chest opened. Reward incremented to " .. tostring(reward + 15))
    elseif chestType == gold then
        player:messageSpecial(ID.text.WHATS_THE_MATTARU + 6)  -- Display a special message for gold chest
        incrementReward(10)  -- Increment the reward by 10
        m.debugPrint("Gold chest opened. Reward incremented to " .. tostring(reward + 10))
    elseif chestType == wood then
        m.debugPrint("Wooden chest interaction. Roll: " .. roll)
        if roll <= chestOutcomeChances.normal then
            player:messageSpecial(ID.text.WHATS_THE_MATTARU + 7)  -- Display a special message for normal outcome
            incrementReward(5)  -- Increment the reward by 5
            m.debugPrint("Normal outcome. Reward incremented to " .. tostring(reward))
        elseif roll <= chestOutcomeChances.normal + chestOutcomeChances.hq1 then
            if math.random(100) <= chestOutcomeChances.rainbowChestChance and not venusRainbowChestSpawned then
                m.venusTransformToRainbowChest(battlefield)  -- Transform the chest to rainbow chest
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 9)  -- Display a special message for rainbow chest
                venusRainbowChestSpawned = true  -- Set the flag to true
                incrementReward(10)  -- Increment the reward by 10
                m.debugPrint("Transformed to Rainbow Chest. Reward incremented to " .. tostring(reward))
            else
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 6)  -- Display a special message for HQ1 outcome
                incrementReward(10)  -- Increment the reward by 10
                m.debugPrint("HQ1 outcome. Reward incremented to " .. tostring(reward))
            end
        else
            if math.random(100) <= chestOutcomeChances.rainbowChestChance and not venusRainbowChestSpawned then
                m.venusTransformToRainbowChest(battlefield)  -- Transform the chest to rainbow chest
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 9)  -- Display a special message for rainbow chest
                venusRainbowChestSpawned = true  -- Set the flag to true
                incrementReward(15)  -- Increment the reward by 15
                m.debugPrint("Transformed to Rainbow Chest. Reward incremented to " .. tostring(reward))
            else
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 5)  -- Display a special message for HQ2 outcome
                incrementReward(15)  -- Increment the reward by 15
                m.debugPrint("HQ2 outcome. Reward incremented to " .. tostring(reward))
            end
        end

        if currentRound == 6 then
            venusRainbowChestSpawned = true  -- Set the flag to true
            m.debugPrint("NO LONGER eligible for rainbow chest spawn in Battlefield ID: " .. bfID)
        end
    end
    m.venusRandomizeMimic(battlefield)  -- Randomize the mimic in the battlefield
    m.venusPrintDebugInfo(player, battlefield)  -- Print the debug info
end

m.venusDespawnCoffersWin = function()
    for _, cofferID in pairs(venusOriginalCofferIDs) do
        DespawnMob(cofferID)
    end
end

m.venusAmanTroveCleanup = function (battlefield)
    local bfID = battlefield:getID()
    m.debugPrint("Resetting rainbowChestSpawned to false for Battlefield ID: " .. bfID)
    venusRainbowChestSpawned = false  -- Reset the flag for rainbow chest spawn
    m.venusCofferIDs:reset()  -- Reset the coffer IDs
    m.venusTerminalDespawn()
end

-----------------------------------
-- Mars functions
-----------------------------------

-- Prints Mars battlefield debug info to the player.
m.marsPrintDebugInfo = function(player, battlefield)
    if debugMode then
        local reward = battlefield:getLocalVar("reward")
        local rewardReceived = battlefield:getLocalVar("rewardReceived")
        local round = battlefield:getLocalVar("round")
        local mimicID = battlefield:getLocalVar("mimicID")
        player:printToPlayer("Mars Debug Info:", 28)
        player:printToPlayer("Reward: " .. reward, 28)
        player:printToPlayer("Reward Received: " .. rewardReceived, 28)
        player:printToPlayer("Round: " .. round, 28)
        player:printToPlayer("Mimic ID: " .. mimicID, 28)
    end
end

-- Function to despawn the Mars Terminal Coffer
m.marsTerminalDespawn = function()
    local terminalCoffer = GetNPCByID(m.marsTerminalID)
    terminalCoffer:setStatus(xi.status.DISAPPEAR)
    m.debugPrint("Mars Terminal Coffer despawned with ID: " .. m.marsTerminalID)
end

-- Function to remove a given chest ID from the marsCofferIDs table
m.marsRemoveChestID = function(mob)
    local chestID = mob:getID()
    for i = #m.marsCofferIDs, 1, -1 do
        if m.marsCofferIDs[i] == chestID then
            table.remove(m.marsCofferIDs, i)
            m.debugPrint("Removed chest ID " .. chestID .. " from marsCofferIDs.")
            return true
        end
    end
    m.debugPrint("Chest ID " .. chestID .. " not found in marsCofferIDs.")
    return false
end

-- Function to clean up all coffers except the one that triggered the event
m.marsRemoveALLChests = function(triggerMob)
    local bf = triggerMob:getBattlefield()
    if bf then
        local triggerMobID = triggerMob:getID()
        m.debugPrint("Cleaning up chest in Battlefield ID: " .. bf:getID())

        for _, cofferID in ipairs(marsOriginalCofferIDs) do
            if cofferID ~= triggerMobID then
                DespawnMob(cofferID)
                m.debugPrint("Despawned Chest_OPlenty with ID: " .. cofferID)
            end
        end
    else
        m.debugPrint("Error: Battlefield not found.")
    end
end

-- Function to transform a remaining chest into a rainbow chest
m.marsTransformToRainbowChest = function(battlefield)
    local bfID = battlefield:getID()

    m.debugPrint("Transforming to Rainbow Chest in battlefield ID: " .. bfID)

    -- Randomly select an unopened chest to transform
    if #m.marsCofferIDs > 0 then
        local selectedChestID = m.marsCofferIDs[math.random(#m.marsCofferIDs)]
        m.debugPrint("Selected chest ID for transformation: " .. selectedChestID)

        local rainbowChest = GetMobByID(selectedChestID)

        -- Check if the chest object is valid
        if rainbowChest then
            m.debugPrint("Transforming chest ID: " .. selectedChestID .. " into a rainbow chest")

            -- Transform the chest into a rainbow chest
            rainbowChest:setStatus(xi.status.DISAPPEAR)
            rainbowChest:timer(1000, function(mob)
                mob:setModelId(2426) -- Model ID for rainbow chest
                mob:setStatus(xi.status.NORMAL)
            end)

            -- Call the marsRemoveChestID function
            m.marsRemoveChestID(rainbowChest)
            marsRainbowChestSpawned = true
        else
            m.debugPrint("Failed to retrieve chest object for ID: " .. selectedChestID)
        end
    else
        m.debugPrint("No unopened chests available for transformation in battlefield ID: " .. bfID)
    end
end

m.marsEntryRainbowChest = function (battlefield)
    local bfID = battlefield:getID()
    local rainbowModelID = 2426  -- Rainbow chest model ID
    local entryRainbowChestChance = chestOutcomeChances.entryRainbowChestChance

    m.debugPrint("Attempting to transform a chest into a rainbow chest in Battlefield ID: " .. bfID)

    -- Randomly decide if a chest will be transformed into a rainbow chest
    if math.random(100) <= entryRainbowChestChance then
        m.debugPrint("Chance for rainbow chest met. Searching for an unopened chest in Battlefield ID: " .. bfID)

        -- Randomly select an unopened chest to transform
        if #m.marsCofferIDs > 0 then
            local entryRainbowChestID = m.marsCofferIDs[math.random(#m.marsCofferIDs)]
            local rainbowChest = GetMobByID(entryRainbowChestID)

            -- Set the model ID of the selected chest to the rainbow chest model ID
            rainbowChest:setModelId(rainbowModelID)

            -- Remove the selected chest ID from the list of unopened chest IDs
            m.marsRemoveChestID(rainbowChest)

            m.debugPrint("Transformed chest ID " .. entryRainbowChestID .. " into a rainbow chest in Battlefield ID: " .. bfID)
        else
            m.debugPrint("No eligible chest found for transformation in Battlefield ID: " .. bfID)
        end
    else
        m.debugPrint("Rainbow chest transformation chance not met in Battlefield ID: " .. bfID)
    end

    m.removeBFStatus(battlefield)
end

m.marsRandomizeMimic = function (battlefield)
    local bfID = battlefield:getID()
    local remainingChests = m.marsCofferIDs

    m.debugPrint("Randomizing mimic for Battlefield ID: " .. bfID)

    -- If there's more than one chest remaining, randomly select one to be the mimic
    if #remainingChests > 1 then
        local mimicID = remainingChests[math.random(#remainingChests)]
        battlefield:setLocalVar("mimicID", mimicID)
        m.debugPrint("Selected mimic ID for next round: " .. mimicID .. " in Battlefield ID: " .. bfID)
    -- If there's only one chest remaining, treat it as the final chest and don't select a mimic
    elseif #remainingChests == 1 then
        local finalChestMob = GetMobByID(remainingChests[1])
        m.marsFinalBox(finalChestMob)
        battlefield:setLocalVar("mimicID", 0)
        m.debugPrint("No mimic selected for next round as only final chest remains in Battlefield ID: " .. bfID)
    -- If there are no chests remaining, print a message
    else
        m.debugPrint("No remaining chests to be selected as mimic in Battlefield ID: " .. bfID)
    end
end

m.marsFinalBox = function(mob)
    local bfID = mob:getBattlefieldID()  -- Get the battlefield ID
    m.debugPrint("finalBox function triggered for Battlefield ID: " .. bfID)

    local finalChestID = nil  -- This will store the ID of the final chest
    local rainbowModelID = 2426  -- Rainbow chest model ID
    local goldModelID = 969     -- Gold chest model ID

    -- If there's only one chest left, it's the final chest
    if #m.marsCofferIDs == 1 then
        finalChestID = m.marsCofferIDs[1]
        m.debugPrint("Final chest ID determined: " .. finalChestID)

        -- Randomly select the chest type: 1 for rainbow, 2 for gold
        local chestType = math.random(2)
        -- Set the model ID based on the chest type
        local modelID = (chestType == 1) and rainbowModelID or goldModelID

        -- Set the model ID for the final chest
        local finalChest = GetMobByID(finalChestID)
        finalChest:setModelId(modelID)
        m.debugPrint("Final chest transformed. Model ID set to: " .. modelID)

        -- Remove the final chest ID from the marsCofferIDs table
        table.remove(m.marsCofferIDs, 1)
        m.debugPrint("Removed final chest ID " .. finalChestID .. " from marsCofferIDs.")
    else
        m.debugPrint("Not the final chest. No transformation applied.")
    end
end

m.marsRollChestOutcome = function(player, battlefield, mob)
    local roll = math.random(100)
    local reward = battlefield:getLocalVar("reward")
    local bfID = battlefield:getID()
    local chestType = mob:getModelId()
    local chestID = mob:getID()
    local currentRound = battlefield:getLocalVar("round")
    local rainbow = 2426 -- Rainbow chest model ID
    local gold    = 969  -- Gold chest model ID
    local wood    = 966  -- Wooden chest model ID

    -- Function to increment the reward
    local incrementReward = function(amount)
        reward = reward + amount
        battlefield:setLocalVar("reward", reward)
        m.debugPrint("Reward incremented to " .. tostring(reward))
    end

    --Remove the chest ID from the marsCofferIDs table
    m.marsRemoveChestID(mob)

    -- Increment the round
    battlefield:setLocalVar("round", battlefield:getLocalVar("round") + 1)
    -- Check for specific chest types
    if chestType == rainbow then
        player:messageSpecial(ID.text.WHATS_THE_MATTARU + 5)  -- Display a special message for rainbow chest
        incrementReward(15)  -- Increment the reward by 15
        m.debugPrint("Rainbow chest opened. Reward incremented to " .. tostring(reward + 15))
    elseif chestType == gold then
        player:messageSpecial(ID.text.WHATS_THE_MATTARU + 6)  -- Display a special message for gold chest
        incrementReward(10)  -- Increment the reward by 10
        m.debugPrint("Gold chest opened. Reward incremented to " .. tostring(reward + 10))
    elseif chestType == wood then
        m.debugPrint("Wooden chest interaction. Roll: " .. roll)
        if roll <= chestOutcomeChances.normal then
            player:messageSpecial(ID.text.WHATS_THE_MATTARU + 7)  -- Display a special message for normal outcome
            incrementReward(5)  -- Increment the reward by 5
            m.debugPrint("Normal outcome. Reward incremented to " .. tostring(reward))
        elseif roll <= chestOutcomeChances.normal + chestOutcomeChances.hq1 then
            if math.random(100) <= chestOutcomeChances.rainbowChestChance and not marsRainbowChestSpawned then
                m.marsTransformToRainbowChest(battlefield)  -- Transform the chest to rainbow chest
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 9)  -- Display a special message for rainbow chest
                marsRainbowChestSpawned = true  -- Set the flag to true
                incrementReward(10)  -- Increment the reward by 10
                m.debugPrint("Transformed to Rainbow Chest. Reward incremented to " .. tostring(reward))
            else
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 6)  -- Display a special message for HQ1 outcome
                incrementReward(10)  -- Increment the reward by 10
                m.debugPrint("HQ1 outcome. Reward incremented to " .. tostring(reward))
            end
        else
            if math.random(100) <= chestOutcomeChances.rainbowChestChance and not marsRainbowChestSpawned then
                m.marsTransformToRainbowChest(battlefield)  -- Transform the chest to rainbow chest
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 9)  -- Display a special message for rainbow chest
                marsRainbowChestSpawned = true  -- Set the flag to true
                incrementReward(15)  -- Increment the reward by 15
                m.debugPrint("Transformed to Rainbow Chest. Reward incremented to " .. tostring(reward))
            else
                player:messageSpecial(ID.text.WHATS_THE_MATTARU + 5)  -- Display a special message for HQ2 outcome
                incrementReward(15)  -- Increment the reward by 15
                m.debugPrint("HQ2 outcome. Reward incremented to " .. tostring(reward))
            end
        end

        if currentRound == 6 then
            marsRainbowChestSpawned = true  -- Set the flag to true
            m.debugPrint("NO LONGER eligible for rainbow chest spawn in Battlefield ID: " .. bfID)
        end
    end
    m.marsRandomizeMimic(battlefield)  -- Randomize the mimic in the battlefield
    m.marsPrintDebugInfo(player, battlefield)  -- Print the debug info
end

m.marsDespawnCoffersWin = function()
    for _, cofferID in pairs(marsOriginalCofferIDs) do
        DespawnMob(cofferID)
    end
end

m.marsAmanTroveCleanup = function (battlefield)
    local bfID = battlefield:getID()
    m.debugPrint("Resetting rainbowChestSpawned to false for Battlefield ID: " .. bfID)
    marsRainbowChestSpawned = false  -- Reset the flag for rainbow chest spawn
    m.marsCofferIDs:reset()  -- Reset the coffer IDs
    m.marsTerminalDespawn()
end

return m
