-----------------------------------
-- Scrying Skill
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
require('scripts/globals/player')
require('scripts/globals/npc_util')
-----------------------------------
local m = Module:new("skill_scrying")

local rate =
{
    GUARANTEED  = { 1000, 1000, 1000, 1000 }, -- 100% -> 100% -> 100% -> 100%
    VERY_COMMON = {  240,  150,  100,   50 }, --  24% ->  15% ->  10% ->   5%
    COMMON      = {  150,  100,   50,   10 }, --  15% ->  10% ->   5% ->   1%
    UNCOMMON    = {  100,  100,  100,  100 }, --  10% ->  10% ->  10% ->  10%
    RARE        = {   50,  100,  150,  240 }, --   5% ->  10% ->  15% ->  24%
    VERY_RARE   = {   10,   50,  100,  150 }, --   1% ->   5% ->  10% ->  15%
    SUPER_RARE  = {    5,   10,   50,  100 }, -- 0.5% ->   1% ->   5% ->  10%
    ULTRA_RARE  = {    1,    5,   10,   50 }, -- 0.1% -> 0.5% ->   1% ->   5%
}

local shardLevel =
{
     15, -- Faint
     30, -- Minor
     45, -- Solid
     60, -- Large
     75, -- Major
     90, -- Radial
    105, -- Mystic
    110, -- Divine
}

local shardName =
{
    "Shard",
    "Fragment",
    "Jewel",
}

local shards =
{
    {
        name = "Faint",
        cap  = { 5, 10, 15 }, -- Copper/Tin + Zinc
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_COPPER_ORE, }, -- 24%
            { rate.COMMON,      xi.item.BORNITE,            }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_TIN_ORE,    }, -- 15%
            { rate.UNCOMMON,    xi.item.AZURITE,            }, -- 10%
            { rate.UNCOMMON,    xi.item.CHUNK_OF_ZINC_ORE,   }, -- 10%
            { rate.RARE,        xi.item.STANNITE,           }, --  5%
            { rate.VERY_RARE,   xi.item.CYLINDRITE,         }, --  1%
        },
    },
    {
        name = "Minor",
        cap  = { 20, 25, 30 }, -- Tin/Zinc + Silver
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_TIN_ORE,    }, -- 24%
            { rate.COMMON,      xi.item.STANNITE,           }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_ZINC_ORE,   }, -- 15%
            { rate.UNCOMMON,    xi.item.CYLINDRITE,         }, -- 10%
            { rate.UNCOMMON,    xi.item.CHUNK_OF_SILVER_ORE, }, -- 10%
            { rate.RARE,        xi.item.SPHALERITE,         }, --  5%
            { rate.VERY_RARE,   xi.item.WURTZITE,           }, --  1%
        },
    },
    {
        name = "Solid",
        cap  = { 35, 40, 45 }, -- Zinc/Silver + Iron
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_ZINC_ORE,   }, -- 24%
            { rate.COMMON,      xi.item.SPHALERITE,         }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_SILVER_ORE, }, -- 15%
            { rate.UNCOMMON,    xi.item.WURTZITE,           }, -- 10%
            { rate.UNCOMMON,    xi.item.CHUNK_OF_IRON_ORE,   }, -- 10%
            { rate.RARE,        xi.item.BROMYRITE,          }, --  5%
            { rate.VERY_RARE,   xi.item.ARGENTITE,          }, --  1%
        },
    },
    {
        name = "Large",
        cap  = { 50, 55, 60 }, -- Silver/Iron + Darksteel
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_SILVER_ORE,    }, -- 24%
            { rate.COMMON,      xi.item.BROMYRITE,             }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_IRON_ORE,      }, -- 15%
            { rate.UNCOMMON,    xi.item.ARGENTITE,             }, -- 10%
            { rate.UNCOMMON,    xi.item.CHUNK_OF_DARKSTEEL_ORE, }, -- 10%
            { rate.RARE,        xi.item.HEMATITE,              }, --  5%
            { rate.VERY_RARE,   xi.item.MAGNETITE,             }, --  1%
        },
    },
    {
        name = "Major",
        cap  = { 65, 70, 75 }, -- Iron/Darksteel + Gold
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_IRON_ORE,      }, -- 24%
            { rate.COMMON,      xi.item.HEMATITE,              }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_DARKSTEEL_ORE, }, -- 15%
            { rate.UNCOMMON,    xi.item.MAGNETITE,             }, -- 10%
            { rate.UNCOMMON,    xi.item.CHUNK_OF_GOLD_ORE,      }, -- 10%
            { rate.RARE,        xi.item.SIDERITE,              }, --  5%
            { rate.VERY_RARE,   xi.item.GOETHITE,              }, --  1%
        },
    },
    {
        name = "Radial",
        cap  = { 80, 85, 90 }, -- Darksteel/Gold + Platinum
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_DARKSTEEL_ORE, }, -- 24%
            { rate.COMMON,      xi.item.SIDERITE,              }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_GOLD_ORE,      }, -- 15%
            { rate.UNCOMMON,    xi.item.GOETHITE,              }, -- 10%
            { rate.UNCOMMON,    xi.item.CHUNK_OF_PLATINUM_ORE,  }, -- 10%
            { rate.RARE,        xi.item.PETZITE,               }, --  5%
            { rate.VERY_RARE,   xi.item.SYLVANITE,             }, --  1%
        },
    },
    {
        name = "Mystic",
        cap  = { 95, 100, 105 }, -- Gold/Platinum + Hibonite
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_GOLD_ORE,       }, -- 24%
            { rate.COMMON,      xi.item.PETZITE,                }, -- 15%
            { rate.COMMON,      xi.item.CHUNK_OF_PLATINUM_ORE,   }, -- 15%
            { rate.UNCOMMON,    xi.item.SYLVANITE,              }, -- 10%
            { rate.UNCOMMON,    xi.item.HIBONITE,               }, -- 10%
            { rate.RARE,        xi.item.SPERRYLITE,             }, --  5%
            { rate.VERY_RARE,   xi.item.MALANITE,               }, --  1%
        },
    },
    {
        name = "Divine",
        cap  = { 110 }, -- Platinum/Hibonite + Taenite/Kamacite + Gems
        drop =
        {
            { rate.VERY_COMMON, xi.item.CHUNK_OF_PLATINUM_ORE,   }, -- 24%
            { rate.COMMON,      xi.item.SPERRYLITE,             }, -- 15%
            { rate.COMMON,      xi.item.HIBONITE,               }, -- 15%
            { rate.UNCOMMON,    xi.item.MALANITE,               }, -- 10%
            { rate.UNCOMMON,    xi.item.TAENITE,                }, -- 10%
            { rate.RARE,        xi.item.KAMACITE,               }, --  5%
            { rate.VERY_RARE,   xi.item.ROUGH_SAPPHIRE,         }, --  1%
            { rate.VERY_RARE,   xi.item.ROUGH_JADEITE,          }, --  1%
            { rate.VERY_RARE,   xi.item.ROUGH_RUBY,             }, --  1%
        },
    },
}

local sizeAnim =
{
    597,
    599,
    596,
}

local function removeNPC(npc)
    -- This will send the packet to ensure NPC disappears for player
    npc:setStatus(xi.status.INVISIBLE)

    npc:timer(500, function(npcArg)
        if npcArg ~= nil then
            -- This will delete DE server side on zone tick
            npcArg:setStatus(xi.status.DISAPPEAR)
        end
    end)
end

m.onTrigger = function(player, npc, tier, size)
    local busy = npc:getLocalVar("[SCRY]Busy")

    if busy > 0 then
        if player:getID() ~= busy then
            player:fmt("Another player is currently using that.")
        end

        return
    end

    -- Unable to use if more than 10 levels below
    if player:getCharVar(cexi.skill.scrying.varSkill) + 100 < (tier.cap[size] * 10) then
        player:fmt("You don't have enough skill to use this.")
        return
    end

    npc:setLocalVar("[SCRY]Busy", player:getID())

    local result = cexi.util.pickItem(tier.drop, size)
    local anim   = player:getAnimation()

    player:setAnimation(48) -- Kneeling

    player:timer(1500, function()
        player:setAnimation(anim) -- Reset animation

        if npcUtil.giveItem(player, result[2]) then
            if tier.cap == 110 then
                player:injectActionPacket(player:getID(), 6, 598, 0, 0, 0, 0, 0)
            else
                player:injectActionPacket(player:getID(), 6, sizeAnim[size], 0, 0, 0, 0, 0)
            end

            cexi.skill.skillUp(player, cexi.skill.scrying, tier.cap[size])
            player:incrementCharVar(cexi.skill.scrying.varTally, 1)
            removeNPC(npc)
        end
    end)
end

m.spawnShard = function(zone, tier, pos)
    local size = 1

    if #shards[tier].cap == 3 then
        local roll = math.random(1, 100)

        if roll < 40 then
            if roll < 15 then
                size = 3
            else
                size = 2
            end
        end
    end

    local dynamicEntity =
    {
        name        = string.format("%s %s", shards[tier].name, shardName[size]),
        objtype     = xi.objType.NPC,
        look        = 1525,
        x           = pos.x,
        y           = pos.y,
        z           = pos.z,
        rotation    = pos.rot,
        widescan    = 1,

        onTrigger   = function(playerArg, npcArg)
            m.onTrigger(playerArg, npcArg, shards[tier], size)
        end,

        releaseIdOnDisappear = true,
    }

    local npc = zone:insertDynamicEntity(dynamicEntity)

    npc:hideName(true)
    npc:hideHP(true)

    npc:timer(30000, function(npcArg)
        if npcArg ~= nil then
            removeNPC(npcArg)
        end
    end)
end

return m
