-----------------------------------
-- Grand Trial (Lv50)
-----------------------------------
-- Cassie  !pos -93.685 -27.000 -39.942 26
-- !setvar [CQ]GRAND_TRIAL 0
-- !setvar [CQ]GRAND_TRIAL_ITEM 0
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
require('scripts/globals/player')
require('scripts/globals/npc_util')
local cq = require("modules/catseyexi/lua/additive_overrides/utils/custom_quest")
----------------------------------------------------------
local m = Module:new("cq_grand_trial")

local info =
{
    author  = "Loxley",
    name    = "Grand Trial",
    var     = "[CQ]GRAND_TRIAL",
    itemVar = "[CQ]GRAND_TRIAL_ITEM",
    progVar = "[CQ]GRAND_TRIAL_{}_{}"
}

local CASSIE = "CASSIE"

local entity =
{
    {
        id     = CASSIE,
        name   = "Cassie",
        type   = xi.objType.NPC,
        look   = cexi.util.look({
            race = xi.race.TARU_F,
            face = cexi.face.A5,
            head = 116, -- Lilac Corsage
            body = 136, -- Barone
            hand = 136, -- Barone
            legs = 136, -- Barone
            feet = 136, -- Barone
            main = 378, -- Mezraq
        }),
        area   = "Tavnazian_Safehold",
        pos    = { -93.685, -27.000, -39.942, 128 }, -- !pos -93.685 -27.000 -39.942 26
        dialog =
        {
            NAME     = true,
            DEFAULT  = { "Do I know you?" },

            QUESTION =
            {
                { emote = xi.emote.WAVE },
                "Hey you! Looking for an extral special boost to your existing armor or weapons?",
                { face = "player" },
                " Bring me an item and if it's eligible, I'll offer a series of trials to upgrade it!",
                { emote = xi.emote.YES },
            },

            REJECTED =
            {
                { emote = xi.emote.NO },
                "Sorry, I can't offer upgrades for that item right now."
            },

            CANCELLED =
            {
                { emote = xi.emote.SIGH },
                "Oh, I see. Well, if wishes were fishes then we'd all have a fry.",
            },

            REMINDER_MOBS =
            {
                { emote = xi.emote.HUH },
                { face = "player" },
                "Need a reminder for your %s trial? Here's what you'll need to do for this step:",
                { emote = xi.emote.LAUGH },
            },

            REMINDER_BOSS =
            {
                { emote = xi.emote.THINK },
                "Need a reminder for your {} trial?",
                { face = "player" },
                " For this step, you'll need to find the {} in {} and defeat {}.",
                { emote = xi.emote.LAUGH },
            },

            REMINDER_MARK =
            {
                { emote = xi.emote.THINK },
                "Need a reminder for your {} trial?",
                " For this step, you'll need to find the {} in {}."
            },

            REMINDER_ITEM  = { "Need a reminder for your {} trial? For this step, you'll need to trade me some items:" },
            ACCEPTED_LIST  = { "To upgrade {} with {}:" },
            ACCEPTED_LIST2 = { "Once you're done with all that, bring the {} back to me and I'll complete the upgrade for you." },

            PASSED   =
            {
                { face = "player" },
                "Congratulations on completing the trial!",
                { emote = xi.emote.CLAP },
                " Hand over the %s and I'll complete the upgrade for you!",
            },
        },
    },
}

-----------------------------------
-- Create entities
-----------------------------------
local function nameToID(str)
    return string.upper(string.gsub(str, " ", "_"))
end

local entities = {}

-- Create reused boss entities
for _, bossInfo in pairs(cexi.trials.grand.boss) do
    table.insert(entity, {
        id          = nameToID(bossInfo.pop),
        name        = bossInfo.pop,
        marker      = cq.SIDE_QUEST,
        area        = bossInfo.area,
        pos         = bossInfo.pos,
        look        = bossInfo.popLook,
        dialog      =
        {
            DEFAULT = cq.NOTHING,
            AFTER   = cq.NOTHING_ELSE,
        },
    })

    table.insert(entity, {
        id          = nameToID(bossInfo.mob),
        name        = bossInfo.mob,
        type        = xi.objType.MOB,
        groupId     = bossInfo.groupId,
        groupZoneId = bossInfo.groupZoneId,
        look        = bossInfo.look,
        flags       = bossInfo.flags,
        area        = bossInfo.area,
        pos         = bossInfo.pos,
        level       = bossInfo.level,
    })

    entities[bossInfo.mob] = {}
end

-- Create reused mark entities
for _, markInfo in pairs(cexi.trials.grand.mark) do
    table.insert(entity, {
        id          = nameToID(markInfo.name),
        name        = markInfo.name,
        marker      = cq.SIDE_QUEST,
        area        = markInfo.area,
        pos         = markInfo.locs[math.random(1, #markInfo.locs)],
        look        = markInfo.look,
        dialog      =
        {
            DEFAULT = cq.NOTHING,
            AFTER   = cq.NOTHING_ELSE,
        },
    })

    entities[markInfo.name] = {}
end

-- Create step entities
for itemID, trialInfo in pairs(cexi.trials.grand.list) do
    for stepIndex, stepInfo in pairs(trialInfo.step) do
        if stepInfo.boss ~= nil then
            if entities[stepInfo.boss.mob] == nil then
                entities[stepInfo.boss.mob] = {}

                table.insert(entity, {
                    id          = nameToID(stepInfo.boss.pop),
                    name        = stepInfo.boss.pop,
                    marker      = cq.SIDE_QUEST,
                    area        = stepInfo.boss.area,
                    pos         = stepInfo.boss.pos,
                    look        = stepInfo.boss.popLook,
                    dialog      =
                    {
                        DEFAULT = cq.NOTHING,
                        AFTER   = cq.NOTHING_ELSE,
                    },
                })

                table.insert(entity, {
                    id          = nameToID(stepInfo.boss.mob),
                    name        = stepInfo.boss.mob,
                    type        = xi.objType.MOB,
                    groupId     = stepInfo.boss.groupId,
                    groupZoneId = stepInfo.boss.groupZoneId,
                    look        = stepInfo.boss.look,
                    flags       = stepInfo.boss.flags,
                    area        = stepInfo.boss.area,
                    pos         = stepInfo.boss.pos,
                    level       = stepInfo.boss.level,
                })
            end

            -- Skip first step (Accept quest)
            local stepVar = stepIndex + 1
            entities[stepInfo.boss.mob][stepVar] = entities[stepInfo.boss.mob][stepVar] or {}
            table.insert(entities[stepInfo.boss.mob][stepVar], itemID)
        end

        if stepInfo.mark ~= nil then
            if entities[stepInfo.mark.name] == nil then
                entities[stepInfo.mark.name] = {}

                table.insert(entity, {
                    id          = nameToID(stepInfo.mark.name),
                    name        = stepInfo.mark.name,
                    marker      = cq.SIDE_QUEST,
                    area        = stepInfo.mark.area,
                    pos         = stepInfo.mark.locs[math.random(1, #stepInfo.mark.locs)],
                    look        = stepInfo.mark.look,
                    dialog      =
                    {
                        DEFAULT = cq.NOTHING,
                        AFTER   = cq.NOTHING_ELSE,
                    },
                })
            end

            -- Skip first step (Accept quest)
            local stepVar = stepIndex + 1
            entities[stepInfo.mark.name][stepVar] = entities[stepInfo.mark.name][stepVar] or {}
            table.insert(entities[stepInfo.mark.name][stepVar], itemID)
        end
    end
end

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

-----------------------------------
-- Trial Quest Step Functions
-----------------------------------
local tq = {}

tq.trialOffer = function(questName, itemVar)
    return function(player, npc, trade, tbl, var, step)
        -- Offer to transfer stats to HQ
        for nq, hq in pairs(cexi.trials.grand.upgrade) do
            if cexi.util.tradeHasExactly(trade, { { nq, 1 }, { hq[2], 1 } }) then
                delaySendMenu(player, {
                    title   = string.format("Transfer stats to %s?", hq[1]),
                    options =
                    {
                        {
                            "No",
                            function()
                            end,
                        },
                        {
                            "Yes",
                            function()
                                npc:independentAnimation(player, 16, 2)
                                cq.transferAugments(player, trade, nq, hq[2])
                                player:tradeComplete()
                            end,
                        },
                    },
                })

                return
            end
        end

        -- List trial requirements for NQ
        for itemID, trial in pairs(cexi.trials.grand.list) do
            if npcUtil.tradeHasExactly(trade, itemID) then
                local message =
                {
                    { emote = xi.emote.THINK },
                    fmt(tbl.dialog.ACCEPTED_LIST[1], trial.name, trial.desc),
                }

                for _, stepInfo in pairs(trial.step) do
                    table.insert(message, " " .. stepInfo.desc)
                end

                table.insert(message, { emote = xi.emote.YES })
                table.insert(message, fmt(tbl.dialog.ACCEPTED_LIST2[1], trial.name))

                cexi.util.dialog(player, message, npc:getPacketName(), { npc = npc })

                local delay = cexi.util.dialogDelay(message)

                player:timer(delay, function()
                    delaySendMenu(player, {
                        title   = fmt("Begin {} trial?", trial.name),
                        options =
                        {
                            {
                                "Not yet",
                                function()
                                end,
                            },
                            {
                                "I'm ready",
                                function()
                                    player:printToPlayer(string.format("\129\158 Quest Accepted: %s (%s)", questName, trial.name), xi.msg.channel.SYSTEM_3)
                                    player:setCharVar(info.var, step)
                                    player:setCharVar(info.itemVar, itemID)
                                end,
                            },
                        },
                    })
                end)

                return
            end
        end

        cexi.util.dialog(player, tbl.dialog.REJECTED, tbl.name, { npc = npc })
    end
end

-----------------------------------
-- Cassie reminder dialog
-----------------------------------
local function stepDialog(player, npc, tbl, progressVar, itemID, step)
    local delay     = 0
    local trialInfo = cexi.trials.grand.list[itemID]
    local trial     = cexi.trials.grand.list[itemID].step[step - 1]

    if trial.mobs ~= nil then
        local pass    = true
        local message = {}

        for _, row in pairs(tbl.dialog.REMINDER_MOBS) do
            table.insert(message, row)
        end

        for index, mobInfo in pairs(trial.mobs) do
            local result = player:getCharVar(fmt(progressVar, step - 1, index))

            if result < mobInfo[3] then
                pass = false
            end

            local mobName  = string.gsub(mobInfo[1], "_", " ")
            local zoneName = string.gsub(mobInfo[2], "_", " ")

            table.insert(message, string.format(" %u/%u (%s in %s)", result, mobInfo[3], mobName, zoneName))
        end

        -- Shouldn't be needed as the objectives advance the quest step
        -- but we can leave it here as a fall-through method
        if pass then
            cexi.util.dialog(player, tbl.dialog[passed], tbl.name, { [1] = trialInfo.name, npc = npc })
            player:setCharVar(info.var, step)

            return 0
        else
            delay = cexi.util.dialogDelay(message)
            cexi.util.dialog(player, message, tbl.name, { [1] = trialInfo.name, npc = npc })
        end

    elseif trial.boss ~= nil then
        local message =
        {
            tbl.dialog.REMINDER_BOSS[1],
            fmt(tbl.dialog.REMINDER_BOSS[2], trialInfo.name),
            fmt(tbl.dialog.REMINDER_BOSS[3], trial.boss.pop, string.gsub(trial.boss.area, "_", " "), trial.boss.mob),
        }

        delay = cexi.util.dialogDelay(message)
        cexi.util.dialog(player, message, tbl.name, { npc = npc })

    elseif trial.mark ~= nil then
        local message =
        {
            tbl.dialog.REMINDER_MARK[1],
            fmt(tbl.dialog.REMINDER_MARK[2], trialInfo.name),
            fmt(tbl.dialog.REMINDER_MARK[3], trial.mark.name, string.gsub(trial.mark.area, "_", " ")),
        }

        delay = cexi.util.dialogDelay(message)
        cexi.util.dialog(player, message, tbl.name, { npc = npc })

    elseif trial.item ~= nil then
        local message =
        {
            { emote = xi.emote.THINK },
            fmt(tbl.dialog.REMINDER_ITEM[1], trialInfo.name),
            " " .. trial.desc,
        }

        delay = cexi.util.dialogDelay(message)
        cexi.util.dialog(player, message, npc:getPacketName(), { npc = npc })
    end

    return delay
end

-- Used when cancelling or completing a trial
local function deleteTrial(player, itemID)
    -- Reset the quest
    player:setCharVar(info.var, 0)
    player:setCharVar(info.itemVar, 0)

    -- Wipe credit for any mobs killed
    for stepIndex, stepInfo in pairs(cexi.trials.grand.list[itemID].step) do
        if stepInfo.mobs ~= nil then
            for index, _ in pairs(stepInfo.mobs) do
                player:setCharVar(fmt(info.progVar, stepIndex, index), 0)
            end
        end
    end
end

-- Used when completing each step
local function finishStep(player, trial, stepArg, noIncrement)
    player:sys("{} Step Complete! {}", "\129\154", "\129\154")

    if noIncrement == nil then
        player:incrementCharVar(info.var, 1)
    end

    if stepArg < #trial.step then
        player:sys("Next step: {}", trial.step[stepArg + 1].desc)
    else
        player:sys("Trial complete: Speak to Cassie for your reward.")
    end
end

-- Dialog to cancel trial
local function cancelTrial(player, npc, tbl, progressVar, itemID, questName)
    delaySendMenu(player, {
        title = "Are you sure?",
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    local delay = cexi.util.dialogDelay(tbl.dialog.CANCELLED)
                    cexi.util.dialog(player, tbl.dialog.CANCELLED, tbl.name, { npc = npc })

                    player:timer(delay, function()
                        player:sys("\129\158 Quest Cancelled: {} ({})", questName, cexi.trials.grand.list[itemID].name)
                    end)

                    deleteTrial(player, itemID)
                end,
            }
        },
    })
end

-----------------------------------
-- Cassie finish trial / perform upgrade
-----------------------------------
tq.trialOnTrade = function(progressVar, itemVar, questName, music)
    return function(player, npc, trade, tbl, var, step)
        local itemID  = player:getCharVar(itemVar)
        local trial   = cexi.trials.grand.list[itemID]
        local current = step - 1

        if cexi.trials.grand.list[itemID].step[current] ~= nil then
            if cexi.trials.grand.list[itemID].step[current].item ~= nil then
                if npcUtil.tradeHasExactly(trade, cexi.trials.grand.list[itemID].step[current].item) then
                    player:confirmTrade()
                    finishStep(player, trial, current)
                end
            end

           return
        end

        if npcUtil.tradeHasExactly(trade, itemID) then
            if player:getFreeSlotsCount() > 0 then
                player:tradeComplete()
                player:addItem(itemID, 1, unpack(trial.augs))
                npc:independentAnimation(player, 4, 2)

                npc:timer(2000, function()
                    local ID = zones[player:getZoneID()]
                    player:messageSpecial(ID.text.ITEM_OBTAINED, itemID)
                end)

                deleteTrial(player, itemID)

                player:printToPlayer(string.format("\129\159 Quest Completed: %s (%s)", questName, trial.name), xi.msg.channel.SYSTEM_3)

                player:changeMusic(0, 67)
                player:changeMusic(1, 67)
                player:timer(5000, function(playerArg)
                    player:changeMusic(0, music or 0)
                    player:changeMusic(1, music or 0)
                end)
            else
                local ID = zones[player:getZoneID()]
                player:tradeRelease()
                player:messageSpecial(ID.text.ITEM_CANNOT_BE_OBTAINED, itemID)
            end
        end
    end
end

-----------------------------------
-- Cassie check status
-----------------------------------
tq.trialOnTrigger = function(progressVar, itemVar, questName)
    return function(player, npc, tbl, var, step)
        local itemID = player:getCharVar(itemVar)
        local trial  = cexi.trials.grand.list[itemID]

        if trial == nil then
            print(string.format("[CQ] %s attempted to get task list for Grand Trial with itemID: %u", player:getName(), itemID))
            return
        end

        if step > #trial.step + 1 then
            cexi.util.dialog(player, tbl.dialog.PASSED, tbl.name, { [1] = trial.name, npc = npc })
            return
        end

        local delay = stepDialog(player, npc, tbl, progressVar, itemID, step)

        player:timer(delay, function()
            delaySendMenu(player, {
                title   = fmt("Continue {} trial?", trial.name),
                options =
                {
                    {
                        "Continue",
                        function()
                        end,
                    },
                    {
                        "Quit trial",
                        function()
                            cancelTrial(player, npc, tbl, progressVar, itemID, questName)
                        end,
                    },
                },
            })
        end)
    end
end

-----------------------------------
-- Create steps
-----------------------------------
local step =
{
    {
        [CASSIE] =
        {
            onTrigger = cq.talkOnly("QUESTION"),
            onTrade   = tq.trialOffer(info.name, info.itemVar),
        },
    },
}

-- Create placeholder steps
for i = 1, 5 do
    table.insert(step, {
        [CASSIE] =
        {
            onTrigger = tq.trialOnTrigger(info.progVar, info.itemVar, info.name),
            onTrade   = tq.trialOnTrade(info.progVar, info.itemVar, info.name, cexi.music.TAVNAZIAN_SAFEHOLD),
        },
    })
end

-- Automatically create steps from trials table
for itemID, trial in pairs(cexi.trials.grand.list) do
    for stepIndex, stepInfo in pairs(trial.step) do
        if stepInfo.mobs ~= nil then
            for index, mobInfo in pairs(stepInfo.mobs) do
                cexi.util.ensureMob(mobInfo[2], mobInfo[1])

                m:addOverride(string.format("xi.zones.%s.mobs.%s.onMobDeath", mobInfo[2], mobInfo[1]), function(mob, player, optParams)
                    super(mob, player, optParams)

                    if
                        player ~= nil and
                        player:isPC() and
                        player:getCharVar("[CQ]GRAND_TRIAL_ITEM") == itemID and
                        player:getCharVar(info.var) == stepIndex
                    then
                        local lookup = fmt(info.progVar, stepIndex, index)
                        local result = player:getCharVar(lookup)

                        if result < mobInfo[3] then
                            local mobName  = string.gsub(mobInfo[1], "_", " ")
                            local zoneName = string.gsub(mobInfo[2], "_", " ")
                            local message  = string.format("Grand Trial [%s] %u/%u (%s in %s)", trial.name, result + 1, mobInfo[3], mobName, zoneName)

                            player:printToPlayer(message, xi.msg.channel.SYSTEM_3)
                            player:incrementCharVar(lookup, 1)

                            if result + 1 >= mobInfo[3] then
                                -- this particular mob is done for this step, check if others are done, too
                                local fullyComplete = true

                                -- re-init stepInfo to avoid relying on table inheritance long-term
                                local trial2 = cexi.trials.grand.list[player:getCharVar("[CQ]GRAND_TRIAL_ITEM")]
                                local stepInfo2 = trial2.step[player:getCharVar(info.var)]
                                for index2, mobInfo2 in pairs(stepInfo2.mobs) do
                                    local lookup2 = fmt(info.progVar, stepIndex, index2)
                                    local result2 = player:getCharVar(lookup2)
                                    if index2 ~= index and result2 < mobInfo2[3] then
                                        -- tell player other objectives remain on this step
                                        if fullyComplete then
                                            player:printToPlayer(string.format("%s %s Complete! %s", "\129\154", mobName, "\129\154"), xi.msg.channel.SYSTEM_3)
                                        end

                                        local mobName2  = string.gsub(mobInfo2[1], "_", " ")
                                        local zoneName2 = string.gsub(mobInfo2[2], "_", " ")
                                        local message2  = string.format("Grand Trial [%s] %u/%u (%s in %s)", trial2.name, result2, mobInfo2[3], mobName2, zoneName2)

                                        player:printToPlayer(message2, xi.msg.channel.SYSTEM_3)

                                        fullyComplete = false
                                    end
                                end

                                if fullyComplete then
                                    finishStep(player, trial, stepIndex)
                                end
                            end
                        end
                    end
                end)
            end
        end

        if stepInfo.boss ~= nil then
            local popID     = nameToID(stepInfo.boss.pop)
            local mobID     = nameToID(stepInfo.boss.mob)
            local stepVar   = 1 + stepIndex
            local stepCheck = { varin = { info.itemVar, entities[stepInfo.boss.mob][stepVar] } }

            if step[stepVar][popID] == nil then
                -- Spawn mob from item trade if specified
                if stepInfo.boss.item ~= nil then
                    step[stepVar][popID] =
                    {
                        check     = cq.checks(stepCheck),
                        onTrade   = cq.tradeSpawn(mobID, stepInfo.boss.item[2], { levelCap = stepInfo.boss.capped, hideSpawner = true }),
                        onTrigger = cq.talkString(fmt("Something may happen here if you have {}.", stepInfo.boss.item[1])),
                    }

                -- Otherwise use ready check menu to spawn
                else
                    step[stepVar][popID] =
                    {
                        check     = cq.checks(stepCheck),
                        onTrigger = cq.menuSpawn(mobID, "Are you ready?", { { "Not yet." }, { "I'm ready." } }, 2, true, { levelCap = stepInfo.boss.capped }),
                    }
                end
            end

            if step[stepVar][mobID] == nil then
                step[stepVar][mobID]     =
                {
                    check      = cq.checks(stepCheck),
                    onMobDeath = cq.killStep(popID, { mobID }, nil, {
                        func  = function(player, stepArg)
                            finishStep(player, trial, stepArg, false)
                        end
                    }),
                }
            end
        end

        if stepInfo.mark ~= nil then
            local npcID     = nameToID(stepInfo.mark.name)
            local stepVar   = 1 + stepIndex
            local stepCheck = { varin = { info.itemVar, entities[stepInfo.mark.name][stepVar] } }
            local markIndex = cexi.trials.grand.markLookup[stepInfo.mark.name]

            if step[stepVar][npcID] == nil then
                step[stepVar][npcID] =
                {
                    check     = cq.checks(stepCheck),
                    onTrigger = cq.markStep(cexi.trials.grand.mark[markIndex].locs, stepInfo.mark.text,
                        function(player, stepArg)
                            finishStep(player, trial, stepArg, false)
                        end
                    ),
                }
            end

            -- TODO: This needs to be rolled into markStep/check to prevent overwriting other steps
            --[[
            if step[stepVar + 1][npcID] == nil then
                step[stepVar + 1][npcID] =
                {
                    check     = cq.checks(stepCheck),
                    onTrigger = cq.talkOnly("AFTER"), -- "There is nothing else to do here."
                }
            end
            ]]
        end
    end
end

cq.add(m, {
    info   = info,
    entity = entity,
    step   = step,
})

return m
