-----------------------------------
-- Wings Library
-----------------------------------
require("modules/module_utils")
require('scripts/globals/npc_util')
require("scripts/globals/shop")
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("wings-library")
-----------------------------------
local heroes =
{
    {
        title = "Legendary Fisherman",
        name  = "AnimeRed",
        desc  =
        {
            "Hailing from Windurst, there are stories of a legendary fisherman.",
            "From the streams of Lufaise meadows, to the high Bastore Seas, to the large cliffs of Qufim, this legendary fisherman has mastered them all.",
            "If you happened to find yourself in his presence he'd inform you that it's always a good day for fishing!"
        },
        look  = cexi.util.look({
            race = xi.race.MITHRA,
            face = cexi.face.B4, -- should be large
            head = 145, -- yagudo_headgear
            body = 153, -- fisherman's smock
            hand = 102, -- angler's
            legs = 102, -- angler's
            feet = 122, -- waders
            main = 408, -- thalassocrat_+1
        }),
        anim = 50, -- Fishing
    },
    {
        title = "Close Encounters of the Cerberus",
        name  = "Lovelady",
        desc  =
        {
            "It was an early morning cerberus.",
            "There were 4 people, and it was smooth to 25%.",
            "There are many spectators. Lovelady is solo tanking.",
            "A Gates of Hades goes off at 2%, taking out the RDM and the BLU.",
            "Next the bard dies and Lovelady is at 1k TP.",
            "It only drops to 1% from Atonement. It readies Gates again. Lovelady uses reprisal.",
            "Cerberus swings a successful block triggers reprisal damage.",
            "Lovelady is at 50hp, and Cerberus is dead.",
        },
        look  = cexi.util.look({
            race = xi.race.GALKA,
            face = cexi.face.B7,
            main = 320, -- excalibur
            offh = 48, -- aegis
            head = 95, -- kaiser
            body = 95, -- kaiser
            hand = 95, -- kaiser
            legs = 95, -- kaiser
            feet = 95, -- kaiser
        }),
        emote = xi.emote.KNEEL,
    },
    {
        title = "Octo-Strike Odyssey",
        name  = "Goren",
        desc  =
        {
            "In the vast world of FFXI, a player named Goren caused a stir by spending nearly two years relentlessly camping The Lord of Onzozo.",
            "In pursuit of the desired Kraken Club, a Fatal mishap led to the loss of one, intensifying server-wide drama.",
            "Accusations of botting and dual-boxing ensued, creating an enduring legend within the WingsXI community. ",
            "Goren's pursuit left an indelible mark on the server's history, blurring the lines between dedication and obsession.",
            "...",
            {message = "You do not meet the requirements to obtain the kraken club."},
            {message = "Kraken club lost."},
        },
        look  = cexi.util.look({
            race = xi.race.GALKA,
            face = cexi.face.B6,
            main = 110, -- KRAKEN_CLUB
            offh = 110, -- KRAKEN_CLUB
            head = cexi.model.HECATOMB_HARNESS,
            body = 35, -- RAPPAREE_HARNESS
            hand = cexi.model.DUSK_JERKIN,
            legs = cexi.model.HOMAM_CORAZZA,
            feet = cexi.model.DUSK_JERKIN,
        }),
        jobability = 181, -- animated_flourish
    },
    {
        -- possibly include Patieto as well if we can do 2 npcs: race - 6, face - 11, treat staff, traveller's hat, and whm AF
        title = "GoodMorningEveryone!",
        name  = "Ramzah",
        desc  =
        {
            "One cannot talk about Wings without talking about GME...",
            "Founded by Ramzah and Patieto, GME (short for GoodMorningEveryone) was intended to be the opposite of what most other linkshells strive to be.",
            "While other linkshells always had plans for their future, be it endgame or otherwise; GME's strength was in the lack of one.",
            "Indeed, the linkshell was always intended to be nothing more than a fun and casual place for friendly people to gather and hang out. ",
            "A place no one would have to fear getting kicked from. For newbies to ask for help.",
            "For more experienced friends to help them. There was no endgame, no loot chase. No rules other than be civil, be cool, be kind.",
            "There was simply family.",
            " --Siiru",
        },
        look  = cexi.util.look({
            race = xi.race.MITHRA,
            face = cexi.face.B7,
            main = 176, -- thiefs_knife
            offh = 259, -- ridill
            head = 152, -- maat's cap
            body = cexi.model.HECATOMB_HARNESS,
            hand = cexi.model.HECATOMB_HARNESS,
            legs = cexi.model.HECATOMB_HARNESS,
            feet = cexi.model.HECATOMB_HARNESS,
        }),
        emote = xi.emote.HURRAY,
    },
    {
        title = "Smash! A Benevolent Menace!",
        name  = "Riko Kupenreich",
        desc  =
        {
            { entity = "Riko Kupenreich", packet = "shbk" },
            "Behind schedule and over budget, Flibe finally finished coding the first 13 missions of the new questline: A Moogle Kupo d'etat.",
            "But there was no time to rest!",
            "The unexpected early release of the first 13 missions added real urgency to completing the final boss fight",
            " with Riko Kupenreich in the Throne Room.",
            { entity = "Riko Kupenreich", packet = "corp" },
            "Many tireless nights later, and endless moogles killed with gm commands, the ultimate BCNM battle was complete.",
            "The cursed greedy moogle developer with a large purple gem lodged in his belly like some bedazzled Tele-tubby was released to Vana'diel",
            "This forced many adventurers to trek through the northlands over and over.",
            "As for Flibe, it wasn't until the eve of the end of the world that, with some of his oldest friends on the server,",
            "he was able to fight the battle himself, not for treasure, but for the challenge of the adventure.",
            "...and also the treasure.",
        },
        look = 2366,
        anim = 1505,
    },
}

local haiku =
{
    {
        title  = "WingsXI Rebirth",
        author = "Siiru",
        text   =
        {
            "An attempt was made,",
            "We learned from many mistakes,",
            "Now it's time to rest",
        },
    },
    {
        title  = "Impermanence",
        author = "Pooter",
        text   =
        {
            "Video games fade,",
            "Nothing lasts forever boys,",
            "Wings will die Friday.",
        },
    },
    {
        title  = "Leaden Salute",
        author = "pizzapocket",
        text   =
        {
            "Salute to the dead,",
            "A path leaden with mistakes,",
            "Darkness will free us.",
        },
    },
    {
        title  = "Fafnir Waits",
        author = "Lovelady",
        text   =
        {
            "Fafnir waits in time,",
            "Spawn's freedom, not forced nor bound,",
            "Nature's rhythm flows."
        },
    },
    {
        title  = "3 Relic Updates",
        author = "Keiba",
        text   =
        {
            "The first vote was close,",
            "Ragnarok, Gungnir, Apoc,",
            "Should only be fixed!",
        },
    },
    {
        title  = "Cait sith's 2hr",
        author = "Badtoad",
        text   =
        {
            "Altana's Favor,",
            "Narrow range, raise one, for dead,",
            "Depressing meow.",
        },
    },
    {
        title  = "Explorer Moogles",
        author = "Teehsho",
        text   =
        {
            "explorer moogles,",
            "should be pretty darn awesome,",
            "but they kinda suck.",
        },
    },
    {
        title  = "King Behemoth roams",
        author = "Lovelady",
        text   =
        {
            "Majestic giant,",
            "King Behemoth, fierce and wild,",
            "Roams with primal might.",
        },
    },
    {
        title  = "Rename",
        author = "Billean",
        text   =
        {
            "I suggest changing,",
            "crawlers in Buburimu,",
            "to 'Bubu Crawlers'.",
        },
    },
    {
        title  = "Ix'DRK QOL change",
        author = "Nicholi",
        text   =
        {
            "Ix'Drks still alive,",
            "All of our spirits have died,",
            "Cap re-raise at five.",
        },
    },
    {
        title  = "Ninja Needs...",
        author = "Teehsho",
        text   =
        {
            "shikanofuda,",
            "inoshishinofuda,",
            "and chonofuda.",
        },
    },
    {
        title  = "Fireworks vendor",
        author = "Flink",
        text   =
        {
            "Fireworks are so fun,",
            "Add Kindlix to Port Jeuno,",
            "Exciting new toys.",
        },
    },
    {
        title  = "Rare/Ex version in BCNM",
        author = "Opaline",
        text   =
        {
            "We have family,",
            "Camping 30hr not healthy,",
            "They want their daddy.",
        },
    },
    {
        title  = "No Rare/Ex version in BCNM",
        author = "Concepcion",
        text   =
        {
            "Some people live for,",
            "The adrenaline of claims,",
            "This hurts their profits.",
        },
    },
}

-----------------------------------
-----------------------------------
local heroList = {}

for index, hero in pairs(heroes) do
    table.insert(heroList, { hero.title, index })
end

local haikuList = {}

for index, text in pairs(haiku) do
    table.insert(haikuList, { text.title, index })
end

local function npcDialog(hero)
    local result = {}

    table.insert(result, { spawn   = { hero.name } })
    table.insert(result, { delay   = 2000          })

    table.insert(result, " \n== " .. hero.title .. " ==\n")

    if hero.emote ~= nil then
        table.insert(result, { entity = hero.name, emote = hero.emote })
    end

    if hero.jobability ~= nil then
        table.insert(result, { entity = hero.name, animate = hero.jobability, mode = 2 })
    end

    for _, line in pairs(hero.desc) do
        table.insert(result, line)
    end

    table.insert(result, { delay   = 2000          })
    table.insert(result, { despawn = { hero.name } })

    return result
end

local function learnMore(player, npc, heroIndex)
    local hero = heroes[heroIndex[2]]
    cexi.util.dialog(player, npcDialog(hero))
end

local function onTrigger(player, npc)
    cexi.util.simpleMenu(player, npc, heroList, learnMore, "Select a Chapter:", 0)
end

local npcs =
{
    {
        name      = "Book of Heroes",
        objtype   = xi.objType.NPC,
        look      = 2809,
        hideName  = true,
        x         = -105.812,
        y         = -3.150,
        z         = -101.364,
        rotation  = 245,
        onTrigger = onTrigger,
    },
}

for _, hero in pairs(heroes) do
    table.insert(npcs, {
        name     = hero.name,
        objtype  = xi.objType.NPC,
        look     = hero.look,
        namevis  = 0x80, -- Ghost
        x        = -103.968,
        y        = -2.150,
        z        = -98.395,
        rotation = 52,
        hidden   = true,
        anim     = hero.anim,
    })
end

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function readHaiku(player, npc, haikuIndex)
    local result = {}

    local haikuInfo = haiku[haikuIndex[2]]

    table.insert(result, " \n== " .. haikuInfo.title .. " ==\n")

    for _, line in pairs(haikuInfo.text) do
        table.insert(result, line)
    end

    table.insert(result, " -" .. haikuInfo.author)

    cexi.util.dialog(player, result)
end

local function onTriggerHaiku(player, npc)
    delaySendMenu(player, {
        title = "Table of Contents:",
        options =
        {
            {
                "Random Haiku",
                function()
                    readHaiku(player, npc, haikuList[math.random(1, #haikuList)])
                end,
            },
            {
                "Full Index",
                function()
                    cexi.util.simpleMenu(player, npc, haikuList, readHaiku, "Select a Haiku:", 0)
                end,
            },
        },
    })
end

table.insert(npcs, {
    name      = "Book of Haiku",
    objtype   = xi.objType.NPC,
    look      = 2810,
    hideName  = true,
    x         = -108.665,
    y         = -3.150,
    z         = -89.8392,
    rotation  = 231,
    onTrigger = onTriggerHaiku,
})

-- If a check returns true, will award that many slots to wardrobe 7 then flag on the player that it's completed
-- ORDER OF THIS TABLE MATTERS. ADD NEW ITEMS TO THE END
local additionalSlots =
{
    {
        name = "Nyzul climb to 100",
        slots = 10,
        check = function(player)
            local climb = player:getCharVar('NyzulFloorProgress')
            if
                climb >= 100 or
                player:getCharVar('NyzulClimbNumber') > 0 -- Old climb reset from WingsXI:https://github.com/Wings-XI/LegacyWingsXI/blob/master/scripts/zones/Aht_Urhgan_Whitegate/npcs/Zaranf.lua
            then
                return true
            end

            return false, climb
        end,
    },
    {
        name = "Fishing at 100",
        slots = 5,
        check = function(player)
            local skill = player:getCharSkillLevel(xi.skill.FISHING)
            if
                skill >= 1000
            then
                return true
            end

            return false, skill / 10
        end,
    },
    {
        name = "Digging at 100",
        slots = 5,
        check = function(player)
            local skill = player:getCharSkillLevel(xi.skill.DIG)
            if
                skill >= 1000
            then
                return true
            end

            return false, skill / 10
        end,
    },
    {
        name = "Max. Merit Points rank 15",
        slots = 5,
        check = function(player)
            local meritPoints = player:getMerit(xi.merit.MAX_MERIT)
            if player:getMerit(xi.merit.MAX_MERIT) >= 15 then
                return true
            end

            return false, meritPoints
        end,
    },
    {
        name = "6 jobs at lvl 75",
        slots = 10,
        check = function(player)
            local jobsAt75 = 0
            for _, job in pairs(xi.job) do
                if job > 0 and player:getJobLevel(job) >= 75 then
                    jobsAt75 = jobsAt75 + 1
                end
            end

            if jobsAt75 >= 6 then
                return true
            end

            return false, jobsAt75
        end,
    },
    {
        name = "1,000,000 Yalms Travelled",
        slots = 5,
        check = function(player)
            local history = player:getHistory(xi.history.DISTANCE_TRAVELLED)

            if history >= 1000000 then
                return true
            end

            return false, history
        end,
    },
    {
        name = "5000 Enemies Defeated",
        slots = 5,
        check = function(player)
            local history = player:getHistory(xi.history.ENEMIES_DEFEATED)

            if history >= 5000 then
                return true
            end

            return false, history
        end,
    },
    {
        name = "Defeat Dynamis Lord",
        slots = 10,
        check = function(player)
            if
                -- Dynamis Lord defeated
                player:hasTitle(xi.title.LIFTER_OF_SHADOWS) or
                player:hasKeyItem(xi.keyItem.HYDRA_CORPS_BATTLE_STANDARD)
            then
                return true
            end

            return false
        end,
    },
    {
        name = "Complete Wardrobe 1-4",
        slots = 10,
        check = function(player)
            local totalSize = 0
            totalSize = totalSize + player:getContainerSize(xi.inv.WARDROBE)
            totalSize = totalSize + player:getContainerSize(xi.inv.WARDROBE2)
            totalSize = totalSize + player:getContainerSize(xi.inv.WARDROBE3)
            totalSize = totalSize + player:getContainerSize(xi.inv.WARDROBE4)

            if
                totalSize >= 4 * 80
            then
                return true
            end

            return false, totalSize
        end,
    },
    {
        name = "Complete Wardrobe 5-6",
        slots = 10,
        check = function(player)
            local totalSize = 0
            totalSize = totalSize + player:getContainerSize(xi.inv.WARDROBE5)
            totalSize = totalSize + player:getContainerSize(xi.inv.WARDROBE6)

            if
                totalSize >= 2 * 80
            then
                return true
            end

            return false, totalSize
        end,
    },
    {
        name = "Redeem Spatial Anomaly",
        slots = 5,
        check = function(player)
            if
                player:getCharVar('New_Player_Reward') > 0
            then
                return true
            end

            return false
        end,
    },
}

local additionalSlotsVar = '[WEW]WardSlotStatus'

local checkWardrobe = function(player, reportStatus)
    -- TODO retroactive mission checks? (not many players should have issues here as completing a mission does most of the checks)

    -- Satchel and Sack share capacity
    local satchel = player:getContainerSize(xi.inv.MOGSATCHEL)
    local sack = player:getContainerSize(xi.inv.MOGSACK)
    local changeAmt = sack - satchel
    if changeAmt ~= 0 then
        -- Note that weird things happen if the size is adjusted below what's in the sack, some players might have satchel and no sack
        -- This would cause them to get the message that it reduces every time they talk to C.A.
        -- Hence, the message about the Artisan Moogle
        player:changeContainerSize(xi.inv.MOGSATCHEL, changeAmt)
        player:sys('\129\154 Your Satchel adjusted {} slots to sync with your Mog Sack. Speak to an Artisan Moogle to adjust. \129\154', changeAmt)
    end

    for idx, task in ipairs(additionalSlots) do
        local additionalSlotsVarStatus = player:getCharVar(additionalSlotsVar)
        local taskComplete = utils.mask.getBit(additionalSlotsVarStatus, idx)
        if taskComplete then
            -- no checks needed, report if requested
            if reportStatus then
                player:sys("You've already received {} Wardrobe 7 slots for: {}", task.slots, task.name)
            end
        else
            local result1, result2 = task.check(player)
            if result1 == true then
                local bag = xi.inv.WARDROBE7
                local currSize = player:getContainerSize(bag)
                local bagIncrease = math.min(task.slots, 80 - currSize)
                if bagIncrease > 0 then
                    player:changeContainerSize(bag, bagIncrease)
                    player:sys('\129\154 You gain {} Wardrobe 7 slots for: {} \129\154', task.slots, task.name)
                end

                player:setCharVar(additionalSlotsVar, utils.mask.setBit(additionalSlotsVarStatus, idx, true))
            else
                if reportStatus then
                    if result2 ~= nil then
                        player:fmt("Your progress for ({}) is: {}", task.name, result2)
                    else
                        player:fmt("You have not completed ({}) yet.", task.name)
                    end
                end
            end
        end
    end
end

local function onTriggerCA(player)
    player:timer(500, function(playerArg)
        -- run this as a timer to avoid errors in the checkWardrobe function blocking player travel
        checkWardrobe(playerArg, false)
    end)

    delaySendMenu(player, {
        title = "Going somewhere?",
        options =
        {
            {
                "Lower Jeuno",
                function()
                    player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                    player:timer(3000, function(playerArg)
                        playerArg:setPos(unpack({ -51, 6, -66.7, 162, xi.zone.LOWER_JEUNO}))
                    end)
                end,
            },
            {
                "Retrace",
                function()
                    if player:getCampaignAllegiance() == 0 then
                        player:printToPlayer("You are not a member of a Past Nation.", xi.msg.channel.SYSTEM_3)
                    else
                        player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                        player:addStatusEffectEx(xi.effect.TELEPORT, 0, xi.teleport.id.RETRACE, 0, 3)
                    end
                end,
            },
            {
                "Repatriate",
                function()
                    player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                    player:addStatusEffectEx(xi.effect.TELEPORT, 0, xi.teleport.id.HOME_NATION, 0, 3)
                end,
            },
            {
                "Warp",
                function()
                    player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                    player:addStatusEffectEx(xi.effect.TELEPORT, 0, xi.teleport.id.WARP, 0, 3)
                end,
            },
            {
                "Aht Urhgan",
                function()
                    if player:getCurrentMission(xi.mission.log_id.TOAU) > xi.mission.id.toau.IMMORTAL_SENTRIES then
                        player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                        player:timer(3000, function(playerArg)
                            xi.teleport.to(playerArg, xi.teleport.id.WHITEGATE)
                        end)
                    else
                        player:printToPlayer("You must complete more ToAU missions to unlock this.", xi.msg.channel.SYSTEM_3)
                    end
                end,
            },
            {
                "Set homepoint",
                function()
                    player:setHomePoint()
                    player:printToPlayer("Homepoint set here.", xi.msg.channel.NS_SAY)
                end,
            },
            {
                "Check Wardrobe unlocks",
                function()
                    checkWardrobe(player, true)
                end,
            },
        },
    })
end

local function onTradeCA(player, npc, trade)
    if not xi.settings.main.ENABLE_WEW_TO_ACE then
        player:sys("That feature is not currently available.")
        return
    end

    if cexi.util.tradeHasExactly(trade, { { xi.item.TARTARUS_PLATEMAIL, 1 }, { xi.item.TALARIA, 1 } }) then
        if #player:getAceChars() > 0 then
            cexi.util.dialog(player, {
                "You've already joined ACE with another character on this account.",
                " There's nothing else I can do for you."
            })

            return
        end

        cexi.util.dialog(player, {
            "So you've decided to retire from Wings-Era Warrior and join the Accelerated CatsEyeXI Experience (ACE)?",
            " Remember, there is absolutely no turning back. This is a one time deal.",
            " You will no longer be able to access the Library or unique Wings-Era features.",
        }, npc:getPacketName(), { npc = npc })

        delaySendMenu(player, {
            title = "Leave Wings-Era Warrior?",
            options =
            {
                {
                    "No way!",
                    function()
                    end,
                },
                {
                    "Join ACE",
                    function()
                        player:tradeComplete()
                        player:setClassicMode(false)

                        -- Raise all bags to 80 slots
                        for i = xi.inv.MOGSATCHEL, xi.inv.WARDROBE8 do
                            local containerSize = player:getContainerSize(i)
                            local difference    = 80 - containerSize

                            if difference > 0 then
                                player:changeContainerSize(i, difference)
                            end
                        end

                        player:setHomePointAt(41.200, -4.998, 84.000, 85, xi.zone.LOWER_JEUNO)
                        player:warp()
                    end,
                },
            }
        })
    end
end

table.insert(npcs, {
    name      = "Wings C.A.",
    objtype   = xi.objType.NPC,
    look      = cexi.util.look({
        race = xi.race.HUME_M,
        face = cexi.face.B4,
        head = cexi.model.ALUMINE_HAUBERT,
        body = cexi.model.NOBLES_TUNIC,
        hand = cexi.model.ALUMINE_HAUBERT,
        legs = cexi.model.ALUMINE_HAUBERT,
        feet = cexi.model.ALUMINE_HAUBERT,
    }),
    hideName  = false,
    x         = -94.4,
    y         = -2.6,
    z         = -84.3,
    rotation  = 95,
    onTrigger = onTriggerCA,
    onTrade   = onTradeCA,
})


local function onTriggerParrot(player)
    delaySendMenu(player, {
        title = "Squawk!",
        options =
        {
            {
                "Feed the Parrot",
                function()
                    if player:getCharSkillLevel(xi.skill.FISHING) < 800 then
                        player:printToPlayer("You need at least 80 Fishing skill for the Parrot to acknowledge you.", xi.msg.channel.SYSTEM_3)
                    else
                        player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)
                        player:timer(3000, function(playerArg)
                            playerArg:setPos(unpack({ -73.513, -0.654, 30.992, 196, xi.zone.NORG})) -- position copied from eyepatch
                        end)
                    end
                end,
            },
            {
                "Back away slowly",
                function()
                end,
            },
        },
    })
end

table.insert(npcs, {
    name      = "Calico Jones",
    objtype   = xi.objType.NPC,
    look      = 2071,
    hideName  = false,
    x         = -102.55,
    y         = -3.15,
    z         = -89.112,
    rotation  = 12,
    onTrigger = onTriggerParrot,
})

local function giveFoodEffect(player, mods, type)
    player:addStatusEffect(xi.effect.FOOD, 0, 0, 3600)
    local effect = player:getStatusEffect(xi.effect.FOOD)
    if effect then
        for _, mod in pairs(mods) do
            -- onEffectGain processes mods on an effect, so we have to add the mod to the player and the effect
            -- this gives them the mod now and ensures it is lost when food buff is removed
            player:addMod(mod[1], mod[2])
            effect:addMod(mod[1], mod[2])
        end

        -- Note that the vector of mods on an effect isn't saved to the db, so rather than get clever with lua we will restrict the buff to wear off on zone
        -- this flag will remove the buff on zone in instead of on zone out. that way the buff is gone even if you d/c
        effect:addEffectFlag(xi.effectFlag.ON_ZONE_PATHOS)

        -- ITEM_FINISH action with animation 29, to match kitron macaron
        player:injectActionPacket(player:getID(), 5, 29, 0, 0, 0, 0, 0)

        player:fmt('You sense this knowledge of {} will be lost if you leave the Library.', type)
    end
end

local function onTriggerSynthBook(player, npc)
    if player:hasStatusEffect(xi.effect.FOOD) then
        player:printToPlayer('Come back when your belly is empty and your mind is open', xi.msg.channel.SYSTEM_3)
        return
    end

    player:printToPlayer('This book looks very old and seems to contain countless generations of knowledge in the art of Crystal Synthesis', xi.msg.channel.SYSTEM_3)

    delaySendMenu(player, {
        title = "Gain mastery of:",
        options =
        {
            -- Give a 1-hour food buff that has one type of effects from macaron, rusk, etc
            -- This of course will not work if the player already has food, so you can't double-up the buff
            {
                "Skillups",
                function()
                    -- SYNTH_SKILL_GAIN mod
                    -- Shaper's shawl gives 25
                    -- kitron macaron gives 7
                    giveFoodEffect(player,
                                {
                                    { xi.mod.SYNTH_SKILL_GAIN , 25 },
                                },
                                'increased skillups')
                end,
            },
            {
                "Success",
                function()
                    -- SYNTH_SUCCESS
                    -- kitron macaron gives 7, which is 7% improved success rate
                    giveFoodEffect(player,
                                {
                                    { xi.mod.SYNTH_FAIL_RATE, -10 },
                                    { xi.mod.SYNTH_SUCCESS, 20 },
                                    { xi.mod.DESYNTH_SUCCESS, 40 },
                                },
                                'higher success rate and less material loss')
                end,
            },
            {
                "Perfection",
                function()
                    -- SYNTH_HQ_RATE
                    -- Coconut rusk gives 3, which converts to 300/512 +.5% hq rate
                    giveFoodEffect(player,
                                {
                                    { xi.mod.SYNTH_HQ_RATE , 9 },
                                },
                                'improved HQ chance')
                end,
            },
        },
    })
end

table.insert(npcs, {
    name      = "Synthesis Vol42",
    objtype   = xi.objType.NPC,
    look      = 2809,
    hideName  = true,
    -- awesome placement at the top of the ladder, but can't interact
    -- x         = -110.8,
    -- y         = -7.5,
    -- z         = -105.6,
    x         = -93.3,
    y         = -1.1,
    z         = -90.9,
    rotation  = 50,
    onTrigger = onTriggerSynthBook,
})

local function onTriggerTome(player, npc)
    if player:getCharVar("[CQ]WINGS_ERA_WARRIOR") < 13 then
        player:printToPlayer("You see nothing out of the ordinary.", xi.msg.channel.NS_SAY)
        return
    end

    delaySendMenu(player, {
        title = "Enter the library?",
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                "Yes",
                function()
                    player:independentAnimation(player, 96, 4)
                    player:injectActionPacket(player:getID(), 6, 600, 0, 0, 0, 0, 0)

                    player:timer(3000, function()
                        player:setPos(unpack({ -97.338, -2.150, -102.563, 160, 284 }))
                    end)
                end,
            },
        },
    })
end

cexi.util.liveReload(m, {
    ["Celennia_Memorial_Library"] = npcs,
    ["Upper_Jeuno"] =
    {
        {
            name      = "Timelost Tome", -- !pos -79.249 -2.210 58.438 244
            objtype   = xi.objType.NPC,
            look      = 1382,
            hideName  = true,
            x         = -79.249,
            y         = -2.210,
            z         = 58.438,
            rotation  = 234,
            onTrigger = onTriggerTome,
        },
    },
})

return m
