-----------------------------------
-- Squire
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("npc_squire")

-----------------------------------
-- NPC Tables
-----------------------------------
local areas =
{
    ["Northern_San_dOria"] =
    {
        name = "Squire Knight", -- (K-9)
        pos  = { 58.317, -0.199, 11.526, 190 }, -- !pos 58.317 -0.199 11.526 231
        look = cexi.util.look({
            race = xi.race.ELVAAN_M,
            face = cexi.face.B3,
            head = cexi.model.MYTHRIL_BREASTPLATE,
            body = cexi.model.MYTHRIL_BREASTPLATE,
            hand = cexi.model.MYTHRIL_BREASTPLATE,
            legs = cexi.model.MYTHRIL_BREASTPLATE,
            feet = cexi.model.MYTHRIL_BREASTPLATE,
            main = 267, -- Darksteel Sword
            offh = 26,  -- Heater Shield
        }),
    },

    ["Bastok_Markets"] =
    {
        name = "Squire Decurion", -- (H-7)
        pos  = { -206.703, -10.000, -17.017, 254 }, -- !pos -206.703 -10.000 -17.017 235
        look = cexi.util.look({
            race = xi.race.HUME_M,
            face = cexi.face.B3,
            head = cexi.model.MYTHRIL_BREASTPLATE,
            body = cexi.model.MYTHRIL_BREASTPLATE,
            hand = cexi.model.MYTHRIL_BREASTPLATE,
            legs = cexi.model.MYTHRIL_BREASTPLATE,
            feet = cexi.model.MYTHRIL_BREASTPLATE,
            main = 267, -- Darksteel Sword
            offh = 26,  -- Heater Shield
        }),
    },

    ["Windurst_Waters"] =
    {
        name = "Squire Warlock", -- (L-10)
        pos  = { 181.420, -0.690, -2.787, 93 }, -- !pos 181.420 -0.690 -2.787 238
        look = cexi.util.look({
            race = xi.race.TARU_M,
            face = cexi.face.B3,
            head = cexi.model.MYTHRIL_BREASTPLATE,
            body = cexi.model.MYTHRIL_BREASTPLATE,
            hand = cexi.model.MYTHRIL_BREASTPLATE,
            legs = cexi.model.MYTHRIL_BREASTPLATE,
            feet = cexi.model.MYTHRIL_BREASTPLATE,
            main = 267, -- Darksteel Sword
            offh = 26,  -- Heater Shield
        }),
    },

    ["RuLude_Gardens"] =
    {
        name = "Squire Guard", -- (I-10)
        pos  = { 55.276, 10.000, -66.790, 117 }, -- !pos 55.276 10.000 -66.790 243
        look = cexi.util.look({
            race = xi.race.GALKA,
            face = cexi.face.B3,
            head = cexi.model.MYTHRIL_BREASTPLATE,
            body = cexi.model.MYTHRIL_BREASTPLATE,
            hand = cexi.model.MYTHRIL_BREASTPLATE,
            legs = cexi.model.MYTHRIL_BREASTPLATE,
            feet = cexi.model.MYTHRIL_BREASTPLATE,
            main = 267, -- Darksteel Sword
            offh = 26,  -- Heater Shield
        }),
    },
}

-----------------------------------
-- Load Armor Lists
-----------------------------------
local tradeable = {}

for itemID, itemInfo in pairs(cexi.augments.dynamis_augments) do
    tradeable[itemID] = { itemInfo.name, "Relic +1", xi.jobNames[itemInfo.job][1] }
end

for itemType, typeInfo in pairs(cexi.trials.novice.storage) do
    for _, itemInfo in pairs(typeInfo) do
        tradeable[itemInfo[2]] = { itemInfo[1], "Novice Trial", itemType, true }
    end
end

for itemType, typeInfo in pairs(cexi.trials.grand.storage) do
    for _, itemInfo in pairs(typeInfo) do
        tradeable[itemInfo[2]] = { itemInfo[1], "Grand Trial", itemType, true }
    end
end

for itemID, itemInfo in pairs(cexi.augments.crystal_warrior) do
    tradeable[itemID] = { itemInfo.name, "Crystal Warrior", xi.jobNames[itemInfo.job][1] }
end

for itemID, itemInfo in pairs(cexi.augments.arena_augments) do
    tradeable[itemID]         = { itemInfo.name,  "Yagudo Arena", itemInfo.set }
    tradeable[itemInfo.hq[1]] = { itemInfo.hq[2], "Yagudo Arena", itemInfo.hq[3] }
end

-- Reverse order eg. { Warrior, WAR }
local jobNames = utils.map(xi.jobNames, function(_, jobName) return { jobName[2], jobName[1] } end)

-----------------------------------
-- Dialog
-----------------------------------
local function alreadyHolding(player, npc)
    local dialog = "Miss, I'm afraid I can't carry any more of those."

    if player:getGender() == 1 then
        dialog = "Sir, I'm already holding one of those."
    end

    cexi.util.dialog(player, { dialog, { emote = xi.emote.NO } }, npc:getPacketName(),{ npc = npc })
end

local function cannotStore(player, npc)
    local dialog = "Sorry miss, I'm not able to store those."

    if player:getGender() == 1 then
        dialog = "Sir, you're not allowed to store that here."
    end

    cexi.util.dialog(player, { dialog, { emote = xi.emote.NO } }, npc:getPacketName(),{ npc = npc })
end

local function notAugmented(player, npc)
    local dialog = "Sorry miss, I can't store that unless it's augmented."

    if player:getGender() == 1 then
        dialog = "Sir, you're not allowed to store that unless it's augmented."
    end

    cexi.util.dialog(player, { dialog, { emote = xi.emote.NO } }, npc:getPacketName(),{ npc = npc })
end

local function finishedTrade(player, npc)
    cexi.util.dialog(player, { "At your service!", { emote = xi.emote.SALUTE } }, npc:getPacketName(),{ npc = npc })
end

local function retrievedItem(player, npc)
    cexi.util.dialog(player, { { emote = xi.emote.YES } }, npc:getPacketName(),{ npc = npc })
end

-----------------------------------
-- Storage Utils
-----------------------------------
local slots =
{
    ["Head"]  = 1,
    ["Body"]  = 2,
    ["Hands"] = 3,
    ["Legs"]  = 4,
    ["Feet"]  = 5,
}

local replacePrefix =
{
    ["Novice Trial"]    = "NT",
    ["Grand Trial"]     = "GT",
    ["Crystal Warrior"] = "CW",
    ["Yagudo Arena"]    = "YA",
}

local function getPathValue(player, itemType, itemSubtype)
    local itype   = itemType
    local subtype = string.gsub(itemSubtype, " ", "_")
    subtype       = string.gsub(subtype, "+", "P")

    if replacePrefix[itype] ~= nil then
        itype = replacePrefix[itype]
    end

    local path  = fmt("[SQUIRE]{}_{}", itype, subtype)
    local value = player:getCharVar(path)

    return path, value
end

local function dynamisGetSlot(itemID)
    return cexi.augments.dynamis_slot[cexi.augments.dynamis_augments[itemID].item]
end

local function noviceGetTypes()
    local types = {}

    for itemType, _ in pairs(cexi.trials.novice.storage) do
        table.insert(types, { itemType, itemType } )
    end

    return types
end

local function grandGetTypes()
    local types = {}

    for itemType, _ in pairs(cexi.trials.grand.storage) do
        table.insert(types, { itemType, itemType } )
    end

    return types
end

local function getArenaSlot(itemID)
    if cexi.augments.arena_hq[itemID] ~= nil then
        return cexi.augments.arena_hq[itemID][2]
    else
        return cexi.augments.arena_augments[itemID].slot
    end
end

-----------------------------------
-- Storage Type Handlers
-----------------------------------
local storageType =
{
    ["Relic +1"] =
    {
        subtypes     = jobNames,

        isStored     = function(player, itemID, itemType, itemSubtype)
            local _, value = getPathValue(player, itemType, itemSubtype)
            local slot     = dynamisGetSlot(itemID)

            return cexi.util.getByte(value, slot) > 0
        end,

        getItemList  = function(jobName)
            return cexi.augments.dynamis_jobs[jobName]
        end,

        storeItem    = function(player, itemID, augs, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local slot        = dynamisGetSlot(itemID)

            -- If not augmented, simply set as R0
            if augs[1] == nil then
                player:setCharVar(path, cexi.util.setByte(value, slot, 1))
                return "R0"
            end

            -- Update stored byte with augment tier
            local tier = cexi.augments.dynamisGetTier(itemID, augs)
            player:setCharVar(path, cexi.util.setByte(value, slot, tier))

            return fmt("R{}", tier)
        end,

        retrieveItem = function(player, npc, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local slot        = dynamisGetSlot(itemID)
            local tier        = cexi.util.getByte(value, slot)

            -- Item is not augmented
            if tier <= 1 then
                if player:addItem(itemID, 1) then
                    player:setCharVar(path, cexi.util.setByte(value, slot, 0))
                    player:timer(100, function()
                        player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, itemID)
                    end)
                end

                return
            end

            -- Skip first cell of table (augment ID)
            tier       = tier + 1
            local augs = cexi.augments.dynamis_augments[itemID].augs

            if player:addItem(itemID, 1, augs[1][1], augs[1][tier], augs[2][1], augs[2][tier], augs[3][1], augs[3][tier]) then
                player:setCharVar(path, cexi.util.setByte(value, slot, 0))
                player:timer(100, function()
                    player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, itemID)
                end)
            end
        end,
    },

    ["Crystal Warrior"] =
    {
        cwOnly       = true,
        subtypes     = jobNames,

        isStored     = function(player, itemID, itemType, itemSubtype)
            local _, value = getPathValue(player, itemType, itemSubtype)
            local slot     = slots[cexi.augments.crystal_warrior[itemID].slot]

            return cexi.util.getByte(value, slot) > 0
        end,

        getItemList  = function(jobName)
            return cexi.augments.cw_jobs[jobName]
        end,

        storeItem    = function(player, itemID, augs, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local slot        = slots[cexi.augments.crystal_warrior[itemID].slot]

            player:setCharVar(path, cexi.util.setByte(value, slot, 1))
            return ""
        end,

        retrieveItem = function(player, npc, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local slot        = slots[cexi.augments.crystal_warrior[itemID].slot]

            if player:addItem(itemID, 1) then
                player:setCharVar(path, cexi.util.setByte(value, slot, 0))
                player:timer(100, function()
                    player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, itemID)
                end)
            end
        end,
    },

    ["Yagudo Arena"] =
    {
        cwOnly       = true,
        subtypes     = cexi.augments.arena_sets,

        isStored     = function(player, itemID, itemType, itemSubtype)
            local _, value = getPathValue(player, itemType, itemSubtype)
            local slot     = getArenaSlot(itemID)

            return cexi.util.getByte(value, slot) > 0
        end,

        getItemList  = function(setName)
            return cexi.augments.arena_info[setName]
        end,

        storeItem    = function(player, itemID, augs, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local slot        = getArenaSlot(itemID)
            local tier        = 0

            -- If not augmented, simply set as R0
            if augs[1] == nil then
                player:setCharVar(path, cexi.util.setByte(value, slot, 1))
                return "A0"
            end

            -- Update stored byte with augment tier

            if cexi.augments.arena_hq[itemID] ~= nil then
                tier = cexi.augments.arenaGetTier(cexi.augments.arena_hq[itemID][1], augs)
            else
                tier = cexi.augments.arenaGetTier(itemID, augs)
            end

            player:setCharVar(path, cexi.util.setByte(value, slot, tier))

            return fmt("A{}", tier)
        end,

        retrieveItem = function(player, npc, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local slot        = getArenaSlot(itemID)
            local tier        = cexi.util.getByte(value, slot)

            -- Item is not augmented
            if tier <= 1 then
                if player:addItem(itemID, 1) then
                    player:setCharVar(path, cexi.util.setByte(value, slot, 0))
                    player:timer(100, function()
                        player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, itemID)
                    end)
                end

                return
            end

            player:setCharVar(path, cexi.util.setByte(value, slot, 0))

            local id = itemID

            if cexi.augments.arena_hq[itemID] ~= nil then
                cexi.augments.arenaGiveTier(player, cexi.augments.arena_hq[itemID][1], tier, true)
            else
                cexi.augments.arenaGiveTier(player, id, tier)
            end
        end,
    },

    ["Novice Trial"] =
    {
        subtypes  = noviceGetTypes(),

        isStored     = function(player, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local index = cexi.trials.novice.items[itemID] - 1

            return utils.mask.getBit(value, index)
        end,

        getItemList  = function(itemType)
            return cexi.trials.novice.storage[itemType]
        end,

        storeItem    = function(player, itemID, augs, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local index = cexi.trials.novice.items[itemID] - 1

            player:setCharVar(path, utils.mask.setBit(value, index, 1))

            return ""
        end,

        retrieveItem = function(player, npc, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local index = cexi.trials.novice.items[itemID] - 1
            local augs  = {}

            if cexi.trials.novice.list[itemID] == nil then
                local nq = cexi.trials.novice.hq_lookup[itemID]
                augs     = cexi.trials.novice.list[nq].augs
            else
                augs     = cexi.trials.novice.list[itemID].augs
            end


            if player:addItem(itemID, 1, unpack(augs)) then
                player:setCharVar(path, utils.mask.setBit(value, index, 0))
                player:timer(100, function()
                    player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, itemID)
                end)
            end
        end,
    },

    ["Grand Trial"] =
    {
        subtypes  = grandGetTypes(),

        isStored     = function(player, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local index = cexi.trials.grand.items[itemID] - 1

            return utils.mask.getBit(value, index)
        end,

        getItemList  = function(itemType)
            return cexi.trials.grand.storage[itemType]
        end,

        storeItem    = function(player, itemID, augs, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local index = cexi.trials.grand.items[itemID] - 1

            player:setCharVar(path, utils.mask.setBit(value, index, 1))

            return ""
        end,

        retrieveItem = function(player, npc, itemID, itemType, itemSubtype)
            local path, value = getPathValue(player, itemType, itemSubtype)
            local index = cexi.trials.grand.items[itemID] - 1
            local augs  = {}

            if cexi.trials.grand.list[itemID] == nil then
                local nq = cexi.trials.grand.hq_lookup[itemID]
                augs     = cexi.trials.grand.list[nq].augs
            else
                augs     = cexi.trials.grand.list[itemID].augs
            end


            if player:addItem(itemID, 1, unpack(augs)) then
                player:setCharVar(path, utils.mask.setBit(value, index, 0))
                player:timer(100, function()
                    player:messageSpecial(zones[player:getZoneID()].text.ITEM_OBTAINED, itemID)
                end)
            end
        end,
    },
}

-----------------------------------
-- Trade Interaction
-----------------------------------
local function onTrade(player, npc, trade)
    local storing  = {}
    local lastSlot = trade:getSlotCount() - 1

    -- Catch incorrect items before updating any variables
    for i = 0, lastSlot do
        local itemID = trade:getItemId(i)

        if tradeable[itemID] == nil then
            cannotStore(player, npc)
            return
        end

        local itemType    = tradeable[itemID][2]
        local itemSubtype = tradeable[itemID][3]

        if storageType[itemType].isStored(player, itemID, itemType, itemSubtype) then
            alreadyHolding(player, npc)
            return
        end

        if tradeable[itemID][4] then
            local item = trade:getItem(i)
            local augs = item:getAugment(0)

            if augs[1] == 0 then
                notAugmented(player, npc)
                return
            end
        end
    end

    for i = 0, lastSlot do
        local itemID      = trade:getItemId(i)
        local item        = trade:getItem(i)
        local itemName    = tradeable[itemID][1]
        local itemType    = tradeable[itemID][2]
        local itemSubtype = tradeable[itemID][3]
        local itemAugs    = cexi.util.augment.getAugments(player, item)
        local extra       = storageType[itemType].storeItem(player, itemID, itemAugs, itemType, itemSubtype)

        table.insert(storing, fmt("You store {}: {} \129\168 {} {}", itemName, itemType, itemSubtype, extra))
    end

    -- This will empty the trade container, which makes the first loop necessary
    player:tradeComplete()

    -- Small delay after interaction to feel more game-like
    player:timer(200, function()
        for _, message in pairs(storing) do
            player:fmt(message)
        end

        finishedTrade(player, npc)
    end)
end

-----------------------------------
-- Menu Functions
-----------------------------------
local function selectItem(player, npc, itemSet, selectedType)
    local selectedSubType = itemSet[2]
    local total  = storageType[selectedType].getItemList(itemSet[2])
    local stored = {}

    -- Only display the items stored
    for _, storedItem in pairs(total) do
        if storageType[selectedType].isStored(player, storedItem[2], selectedType, itemSet[2]) then
            table.insert(stored, storedItem)
        end
    end

    local func = function(player, npc, item)
        if player:getFreeSlotsCount() < 1 then
            player:messageSpecial(zones[player:getZoneID()].text.ITEM_CANNOT_BE_OBTAINED, item[2])
            return
        end

        storageType[selectedType].retrieveItem(player, npc, item[2], selectedType, itemSet[2])
        retrievedItem(player, npc)

        -- Re-open menu after each item is retrieved
        selectItem(player, npc, itemSet, selectedType)
    end

    cexi.util.simpleMenu(player, npc, stored, func, "Select an item:")
end

local function selectSubtype(player, npc, selectedType)
    local func = function(player, npc, itemSet)
        selectItem(player, npc, itemSet, selectedType[1])
    end

    cexi.util.simpleMenu(player, npc, storageType[selectedType[1]].subtypes, func, "Select a set:", 0, { cycle = true })
end

-- Create main menu from Storage Types
local categories   = {}
local categoriesCW = {}

for storageName, info in pairs(storageType) do
    if info.cwOnly == nil then
        table.insert(categories, { storageName })
    end
end

for storageName, info in pairs(storageType) do
    table.insert(categoriesCW, { storageName })
end

local function onTrigger(player, npc)
    if player:isCrystalWarrior() then
        cexi.util.simpleMenu(player, npc, categoriesCW, selectSubtype, "Select a type:")
    else
        cexi.util.simpleMenu(player, npc, categories, selectSubtype, "Select a type:")
    end
end

-- Create reload list from NPC table
local npcs = {}

for zoneName, npcInfo in pairs(areas) do
    npcs[zoneName] =
    {
        {
            name      = npcInfo.name,
            objtype   = xi.objType.NPC,
            look      = npcInfo.look,
            x         = npcInfo.pos[1],
            y         = npcInfo.pos[2],
            z         = npcInfo.pos[3],
            rotation  = npcInfo.pos[4],
            onTrigger = onTrigger,
            onTrade   = onTrade,
        },
    }
end

cexi.util.liveReload(m, npcs)

return m
