-----------------------------------
-- Ephemeral Box - Store Crafting Materials
-----------------------------------
require("modules/module_utils")
require('scripts/globals/utils')
require('scripts/globals/player')
require('scripts/globals/npc_util')

local cq      = require("modules/catseyexi/lua/additive_overrides/utils/custom_quest")
local unlocks = require("modules/catseyexi/lua/additive_overrides/utils/custom_unlocks")
-----------------------------------
local m = Module:new("npc_cw_ephemeral_box")

local settings =
{
    name   = "Ephemeral Box",
    lookup = "Ephemeral_Box",
    look   = 2331,
}

local EB_PAGE  = "[EB]PAGE"
local EB_ITEM  = "[EB]%u"

local mainMenu =
{
    "Alchemy",
    "Bonecraft",
    "Clothcraft",
    "Cooking",

    "Goldsmithing",
    "Leathercraft",
    "Smithing",
    "Woodworking",

    "Fishing",
    "Lapidary",
    "Ninjutsu",
    "Food",
    "Incursion",
    "Spawners",
    "Other",
}

local function delayMenu(player, menuNext)
    local menu = menuNext

    player:timer(200, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

function boxGiveItem(player, item, qty)
    local ID = zones[player:getZoneID()]
    local givenItems = { { item, qty } }

    if player:getFreeSlotsCount() < #givenItems then
        player:messageSpecial(ID.text.ITEM_CANNOT_BE_OBTAINED, givenItems[1][1])
        return false
    end

    local messagedItems = {}
    for _, v in pairs(givenItems) do
        if player:addItem(v[1], v[2], true) then
            if not messagedItems[v[1]] then
                player:messageSpecial(ID.text.ITEM_OBTAINED + 9, v[1], v[2])
            end

            messagedItems[v[1]] = true
        elseif #givenItems == 1 then
            player:messageSpecial(ID.text.ITEM_CANNOT_BE_OBTAINED, givenItems[1][1])
            return false
        end
    end

    return true
end

local function quantityPage(player, npc, material)
    local tbl = {}
    local curr = player:getCharVar(string.format(EB_ITEM, material[1]))

    if material[3] == 1 then
        if curr > 0 then
            delayMenu(player, {
                title   = "Select a quantity:",
                options =
                {
                    {
                        "x1",
                        function()
                            npc:ceAnimate(player, player, 75, 4)

                            player:timer(500, function()
                                if boxGiveItem(player, material[1], 1) then
                                    player:setCharVar(string.format(EB_ITEM, material[1]), curr - 1)

                                    if (curr - 1) > 0 then
                                        quantityPage(player, npc, material)
                                    end
                                end
                            end)
                        end,
                    },
                },
            })
        end

        return
    end

    for k, v in pairs(cexi.crystal_warrior.ephemeral_box.qtyList) do
        if v[2] <= curr then
            table.insert(tbl, {
                v[1],
                function()
                    npc:ceAnimate(player, player, 75, 4)

                    player:timer(500, function()
                        if boxGiveItem(player, material[1], v[2]) then
                            player:setCharVar(string.format(EB_ITEM, material[1]), curr - v[2])

                            if (curr - v[2]) > 0 then
                                quantityPage(player, npc, material)
                            end
                        end
                    end)
                end,
            })
        end
    end

    delayMenu(player, {
        title   = "Select a quantity:",
        options = tbl,
    })
end

local function materialPage(player, npc, material)
    local tbl  = {}
    local page = player:getLocalVar(EB_PAGE)
    local max  = 1

    local lastPage = math.floor((#material - 1) / 4)

    if page > 0 then
        table.insert(tbl, {
            "(Prev)",
            function(player)
                local p = player:getLocalVar(EB_PAGE)
                player:setLocalVar(EB_PAGE, p - 1)
                materialPage(player, npc, material)
            end,
        })
    else
        table.insert(tbl, {
            "(Prev)",
            function(player)
                local p = player:getLocalVar(EB_PAGE)
                player:setLocalVar(EB_PAGE, lastPage)
                materialPage(player, npc, material)
            end,
        })
    end

    for i = 1, 4 do
        local m = material[page * 4 + i]

        if m ~= nil then
            local currentQty = player:getCharVar(string.format(EB_ITEM, m[1]))
            table.insert(tbl, {
                string.format("%s (%u)", m[2], currentQty),
                function(player)
                    quantityPage(player, npc, m)
                end
            })

            max = page * 4 + i
        end
    end

    if max < #material then
        table.insert(tbl, {
            "(Next)",
            function(player)
                local p = player:getLocalVar(EB_PAGE)
                player:setLocalVar(EB_PAGE, p + 1)
                materialPage(player, npc, material)
            end,
        })
    else
        table.insert(tbl, {
            "(Next)",
            function(player)
                local p = player:getLocalVar(EB_PAGE)
                player:setLocalVar(EB_PAGE, 0)
                materialPage(player, npc, material)
            end,
        })
    end

    delayMenu(player, {
        title   = "Select an item:",
        options = tbl,
    })
end

local function skillPage(player, npc, skill)
    local tbl = {}

    for k, v in pairs(skill.category) do
        table.insert(tbl, {
            v,
            function(playerArg)
                playerArg:setLocalVar(EB_PAGE, 0)
                materialPage(playerArg, npc, skill.list[k])
            end
        })
    end

    delayMenu(player, {
        title   = "Select a type:",
        options = tbl,
    })
end

local function mainPage(player, npc, craftList, page)
    local tbl = {}

    if page > 0 then
        table.insert(tbl, {
            "(Prev)",
            function()
                mainPage(player, npc, craftList, page - 1)
            end,
        })
    else
        table.insert(tbl, {
            "(Prev)",
            function()
                mainPage(player, npc, craftList, math.floor(#craftList / 4))
            end,
        })
    end

    for i = 1, 4 do
        local currentCraft = craftList[i + (page * 4)]

        if currentCraft ~= nil then
            table.insert(tbl, {
                currentCraft,
                function(playerArg)
                    skillPage(playerArg, npc, cexi.crystal_warrior.ephemeral_box.list[currentCraft])
                end
            })
        end
    end

    if page < math.floor(#craftList / 4) then
        table.insert(tbl, {
            "(Next)",
            function()
                mainPage(player, npc, craftList, page + 1)
            end,
        })
    else
        table.insert(tbl, {
            "(Next)",
            function()
                mainPage(player, npc, craftList, 0)
            end,
        })
    end

    delayMenu(player, {
        title   = "Select a category:",
        options = tbl,
    })
end

local function setGuildFirst(player, npc, craftType)
    if craftType == nil then
        return mainMenu
    end

    local craftList = {}

    table.insert(craftList, craftType)

    for k, v in pairs(mainMenu) do
        if v ~= craftType then
            if
                cexi.crystal_warrior.ephemeral_box.list[v].condition == nil or
                player:getCharVar(cexi.crystal_warrior.ephemeral_box.list[v].condition) > 0
            then
                table.insert(craftList, v)
            end
        end
    end

    return craftList
end

local function onTriggerBox(player, npc, craftType)
    if
        not player:isCrystalWarrior() or
        GetServerVariable("[EB]DISABLE") == 1 or
        player:getCharVar("[CW]BOXED_UP") < 5
    then
        player:printToPlayer("It's locked.", xi.msg.channel.NS_SAY)
        return
    end

    local craftList = setGuildFirst(player, npc, craftType)

    mainPage(player, npc, craftList, 0)
end

local function onTradeBox(player, npc, trade)
    if
        not player:isCrystalWarrior() or
        GetServerVariable("[EB]DISABLE") == 1 or
        player:getCharVar("[CW]BOXED_UP") < 5
    then
        player:printToPlayer("It's locked.", xi.msg.channel.NS_SAY)
        return
    end

    local storing  = {}
    local lastSlot = trade:getSlotCount() - 1
    local sold     = 0

    for i = 0, lastSlot do
        local itemID = trade:getItemId(i)

        if
            cexi.crystal_warrior.scrolls_tradeable[itemID]          == nil and
            cexi.crystal_warrior.ephemeral_box.tradeable[itemID]    == nil and
            (cexi.crystal_warrior.ephemeral_box.conditional[itemID] == nil or
            player:getCharVar(cexi.crystal_warrior.ephemeral_box.conditional[itemID][1]) == 0)
        then
            player:printToPlayer("You cannot store this item.", xi.msg.channel.NS_SAY)
            return
        end
    end

    for i = 0, lastSlot do
        local itemID    = trade:getItemId(i)
        local itemQty   = trade:getSlotQty(i)

        if cexi.crystal_warrior.ephemeral_box.conversion[itemID] ~= nil then
            itemQty = cexi.crystal_warrior.ephemeral_box.conversion[itemID][2] * itemQty
            itemID  = cexi.crystal_warrior.ephemeral_box.conversion[itemID][1]
        end

        local itemName  = "???"
        local craftType = "???"
        local itemType  = "???"

        if cexi.crystal_warrior.ephemeral_box.tradeable[itemID] == nil then
            if cexi.crystal_warrior.scrolls_tradeable[itemID] == nil then
                itemName  = cexi.crystal_warrior.ephemeral_box.conditional[itemID][2]
                craftType = cexi.crystal_warrior.ephemeral_box.conditional[itemID][3]
                itemType  = cexi.crystal_warrior.ephemeral_box.conditional[itemID][4]
            end
        else
            itemName  = cexi.crystal_warrior.ephemeral_box.tradeable[itemID][1]
            craftType = cexi.crystal_warrior.ephemeral_box.tradeable[itemID][2]
            itemType  = cexi.crystal_warrior.ephemeral_box.tradeable[itemID][3]
        end

        if cexi.crystal_warrior.ephemeral_box.seals[itemID] ~= nil then
            player:addSeals(itemQty, cexi.crystal_warrior.ephemeral_box.seals[itemID][1])

            local totalSeals = player:getSeals(cexi.crystal_warrior.ephemeral_box.seals[itemID][1])
            table.insert(storing, string.format("You store %s x%u (Total: %u)", itemName, itemQty, totalSeals))

        elseif itemID == xi.item.ANCIENT_BEASTCOIN then
            local balance = player:getCurrency("ancient_beastcoin")
            local total   = balance + itemQty

            player:setCurrency("ancient_beastcoin", total)
            table.insert(storing, string.format("You store %s x%u (Total: %u)", itemName, itemQty, total ))

        elseif cexi.crystal_warrior.ephemeral_box.organs[itemID] ~= nil then
            local orgVar  = fmt("[Sea]{}", itemID)
            local balance = player:getCharVar(orgVar)
            local total   = balance + itemQty

            player:setCharVar(orgVar, total)
            table.insert(storing, string.format("You store %s x%u (Total: %u)", itemName, itemQty, total))

        elseif cexi.crystal_warrior.scrolls_tradeable[itemID] ~= nil then
            local index    = cexi.crystal_warrior.scrolls_tradeable[itemID][1]
            local itemName = cexi.crystal_warrior.scrolls_tradeable[itemID][2]
            local category = cexi.crystal_warrior.scrolls_tradeable[itemID][3]
            local tier     = cexi.crystal_warrior.scrolls_tradeable[itemID][4]
            local varName  = fmt("[CW]SCROLL_{}_{}", cexi.crystal_warrior.scrolls_short[category], cexi.numeral[tier])
            local value    = player:getCharVar(varName)

            if utils.mask.getBit(value, index) then
                local itemInfo = GetItemByID(itemID)

                if itemInfo ~= nil then
                    local value = itemInfo:getBasePrice()
                    sold = sold + value
                    table.insert(storing, fmt("You scrap a scroll of {} for {} gil (Already stored)", itemName, value))
                end
            else
                player:setCharVar(varName, utils.mask.setBit(value, index, true))
                table.insert(storing, fmt("You store a scroll of {} ({} \129\168 {})", itemName, category, cexi.numeral[tier]))
            end
        else
            local itemVar   = string.format(EB_ITEM, itemID)
            local storedQty = player:getCharVar(itemVar)

            player:setCharVar(itemVar, storedQty + itemQty)
            table.insert(storing, string.format("You store %s x%u (Total: %u) %s \129\168 %s", itemName, itemQty, storedQty + itemQty, craftType, itemType))
        end
    end

    if sold > 0 then
        player:addGil(sold)
    end

    player:tradeComplete()
    npc:ceAnimate(player, player, 46, 4)

    player:timer(500, function()
        for k, v in pairs(storing) do
            player:printToPlayer(v, xi.msg.channel.NS_SAY)
        end
    end)
end

-- Create reload list from NPC table
local npcs = {}

for zoneName, npcInfo in pairs(cexi.crystal_warrior.ephemeral_box.areas) do
    for craftType, pos in pairs(npcInfo) do
        npcs[zoneName] = npcs[zoneName] or {}

        table.insert(npcs[zoneName], {
            name       = fmt("{}_{}", settings.lookup, craftType),
            packetName = settings.name,
            objtype    = xi.objType.NPC,
            look       = settings.look,
            namevis    = 0x80,      -- Ephemeral
            x          = pos[1],
            y          = pos[2],
            z          = pos[3],
            rotation   = pos[4],
            onTrade    = onTradeBox,
            onTrigger  = function(player, npc)
                onTriggerBox(player, npc, craftType)
            end,
        })
    end
end

cexi.util.liveReload(m, npcs)

return m
