-----------------------------------
-- Aether Traveler
-----------------------------------
require("modules/module_utils")
-----------------------------------
local m = Module:new("npc_aether_traveler")

local settings =
{
    -- Char Vars
    points = "[AETHER]Fragments",
    weekly = "[AETHER]Weekly",
    traded = "[AETHER]Traded",

    -- Server/Char Vars
    equipment  = "[AETHER]Equipment",
    currencies = "[AETHER]Currencies",
    materials  = "[AETHER]Materials",
    spawners   = "[AETHER]Spawners",
    mystery    = "[AETHER]Mystery",
}

-- A list of these items is randomly selected each week
local rewards =
{
    equipment =
    {
        { "Animator +1",       17857, 10 },
        { "Gleeman's Cape",    16229, 10 },
        { "Ritter Gorget",     16267, 10 },
        { "Kubira Beads",      16268, 10 },
        { "Morgana's Choker",  16269, 10 },
        { "Buccaneer's Belt",  15911, 10 },
        { "Iota Ring",         15799, 10 },
        { "Omega Ring",        15800, 10 },
        { "Delta Earring",     15990, 10 },
        { "Rose Strap",        19041, 10 },
        { "Aesir Torque",      11589, 10 },
        { "Aesir Mantle",      11546, 10 },
        { "Aesir Ear Pendant", 16057, 10 },
        { "Ifrit's Bow",       17192, 10 },
        { "Leviathan's Couse", 18109, 10 },
        { "Shiva's Shotel",    17711, 10 },
        { "Ramuh's Mace",      18404, 10 },
        { "Garuda's Sickle",   18063, 10 },
        { "Carbuncle's Cuffs", 14931, 10 },
        { "Titan's Baselarde", 18021, 10 },
        { "Strigoi Ring",      11628, 10 },
        { "Zilant Ring",       11629, 10 },
        { "Corneus Ring",      11630, 10 },
        { "Blobnag Ring",      11631, 10 },
        { "Karka Ring",        11632, 10 },
        { "Galdr Ring",        11633, 10 },
        { "Veela Ring",        11634, 10 },
        { "Succor Ring",       15859, 10 },
    },

    currencies =
    {
        { "100 Byne Bill",     xi.item.ONE_HUNDRED_BYNE_BILL,  2 },
        { "M. Silverpiece",    xi.item.MONTIONT_SILVERPIECE,   2 },
        { "L. Jadeshell",      xi.item.LUNGO_NANGO_JADESHELL,  2 },
        { "Lin. Purse (Alx.)", xi.item.LINEN_COIN_PURSE,       2 },
        { "Muiridian Sack",    xi.item.MUIRIDIAN_SACK,         3 },
        { "Wyrm Gold",         xi.item.WYRM_GOLD,             10 },
    },

    materials =
    {
        { "Fable Cloth",       xi.item.FABLE_CLOTH,        5 }, -- Venture NMs
        { "Titanium Ingot",    xi.item.TITANIUM_INGOT,     5 }, -- Venture NMs
        { "Wyrm Scale",        xi.item.WYRM_SCALE,         5 }, -- Dragonslaying
        { "Wyrm Tooth",        xi.item.WYRM_TOOTH,         5 }, -- Dragonslaying
        { "Embersteel Ingot",  xi.item.EMBERSTEEL_INGOT,  15 }, -- Dragonslaying
        { "Prismatic Cluster", xi.item.PRISMATIC_CLUSTER,  5 }, -- Prismatic Cluster
        { "Star Sapphire",     2359,                       5 }, -- ToAU HNM
        { "Yoichi's Sash",     2330,                      10 }, -- ToAU HNM
        { "Cerberus Hide",     2169,                      10 }, -- Cerberus
        { "Khimaira Tail",     2373,                      10 }, -- Khimaira
        { "Hydra Fang",        2158,                      10 }, -- Hydra
        { "Hydra Scale",       2172,                      10 }, -- Hydra
        { "Spectral Ore",      9075,                      15 }, -- Fomor NM x5 Battle
        { "Fine Ochre",        9515,                      15 }, -- Dragonslaying Wyrms
        { "Serica Cloth",      3545,                      25 }, -- Brjota
    },

    spawners =
    {
        { "Diorite",        xi.item.CHUNK_OF_DIORITE,               2 }, -- Ullikummi
        { "Ro'Maeve Water", xi.item.FLASK_OF_ROMAEVE_SPRING_WATER,  2 }, -- Olla
        { "Goblin Brew",    xi.item.GOBLIN_BREW,                    2 }, -- Venture NMs
        { "Goblin Brew +1", xi.item.GOBLIN_BREW_P1,                 3 }, -- Venture NMs
        { "Goblin Brew +2", xi.item.GOBLIN_BREW_P2,                10 }, -- Venture NMs
        { "Golden Bangle",  xi.item.GOLDEN_BANGLE,                  5 }, -- Dragonslaying Lv60
        { "Golden Rings",   xi.item.GOLDEN_RINGS,                  15 }, -- Dragonslaying Wyrms
        { "Honey Wine",     xi.item.JUG_OF_HONEY_WINE,             15 }, -- Fafnir
        { "Sweet Tea",      xi.item.CUP_OF_SWEET_TEA,              10 }, -- Nidhogg
        { "Savory Shank",   xi.item.SAVORY_SHANK,                  10 }, -- King Behemoth
        { "Red Pondweed",   xi.item.CLUMP_OF_RED_PONDWEED,         10 }, -- Red Pondweed
        { "Cloud Evoker",   xi.item.CLOUD_EVOKER,                   5 }, -- Ouryu
        { "Monarch's Orb",  xi.item.MONARCHS_ORB,                  20 }, -- Bahamut
        { "Atropos Orb",    1180,                                   5 }, -- KSNM30
        { "Clotho Orb",     1175,                                   5 }, -- KSNM30
        { "Lachesis Orb",   1178,                                   5 }, -- KSNM30
        { "Themis Orb",     1553,                                  15 }, -- KSNM99
        { "Epic Lure",      xi.item.EPIC_LURE,                     15 }, -- 30,000 Junknix scraps
    },
}

local mysteryItems = {}

for categoryName, category in pairs(rewards) do
    for index, item in pairs(category) do
        table.insert(mysteryItems, item)
        rewards[categoryName][index][4] = categoryName
    end
end

local requests =
{
    { "three divine logs",                { { xi.item.DIVINE_LOG,                3 } } },
    { "three locks of siren's hair",      { { xi.item.LOCK_OF_SIRENS_HAIR,       3 } } },
    { "three squares of shining cloth",   { { xi.item.SQUARE_OF_SHINING_CLOTH,   3 } } },
    { "three squares of damascene cloth", { { xi.item.SQUARE_OF_DAMASCENE_CLOTH, 3 } } },
    { "three squares of eltoro leather",  { { xi.item.SQUARE_OF_ELTORO_LEATHER,  3 } } },
    { "three squares of galateia",        { { xi.item.SQUARE_OF_GALATEIA,        3 } } },
    { "three pots of viridian urushi",    { { xi.item.POT_OF_VIRIDIAN_URUSHI,    3 } } },
    { "three dragon bones",               { { xi.item.DRAGON_BONE,               3 } } },
    { "three pieces of angel skin",       { { xi.item.PIECE_OF_ANGEL_SKIN,       3 } } },
    { "three damascus ingots",            { { xi.item.DAMASCUS_INGOT,            3 } } },
    { "three orichalcum ingots",          { { xi.item.ORICHALCUM_INGOT,          3 } } },
    { "three venomous claws",             { { xi.item.VENOMOUS_CLAW,             3 } } },
}

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function confirmPurchase(player, npc, item)
    if player:getCharVar(settings.points) < item[3] then
        player:sys("You don't have enough stored fragments to purchase that.")
        return
    end

    delaySendMenu(player, {
        title   = fmt("Spend {} fragments?", item[3]),
        options =
        {
            {
                "No",
                function()
                end,
            },
            {
                fmt("Purchase: {}", item[1]),
                function()
                    if item[2] == "Mystery" then
                        local result = mysteryItems[math.random(1, #mysteryItems)]

                        if npcUtil.giveItem(player, result[2]) then
                            player:setCharVar(settings.mystery, 1, NextConquestTally())
                            player:incrementCharVar(settings.points, -15)
                            cexi.util.dialog(player, { fmt("Yes... This is your mystery item.", item[3]) }, npc:getPacketName(), { npc = npc })
                        end
                    else
                        if npcUtil.giveItem(player, item[2]) then
                            player:setCharVar(settings[item[4]], 1, NextConquestTally())
                            player:incrementCharVar(settings.points, -item[3])
                            cexi.util.dialog(player, { fmt("Yes... I'll take those {} fragments.", item[3]) }, npc:getPacketName(), { npc = npc })
                        end
                    end
                end,
            },
        },
    })
end

local function onTrade(player, npc, trade)
    if npcUtil.tradeHasOnly(trade, xi.item.AETHER_FRAGMENT) then
        local balance = player:getCharVar(settings.points)
        local qty     = trade:getItemQty(xi.item.AETHER_FRAGMENT)

        player:setCharVar("[AETHER]Fragments", balance + qty)
        player:tradeComplete()

        cexi.util.dialog(player, { fmt("Yes... {} fragments. That makes {} total.", qty, balance + qty) }, npc:getPacketName(), { npc = npc })

        return
    end

    local roll = player:getCharVar(settings.weekly)

    if
        roll > 0 and
        npcUtil.tradeHasExactly(trade, requests[roll][2])
    then
        if npcUtil.giveItem(player, { { xi.item.AETHER_FRAGMENT, 5 }}) then
            cexi.util.dialog(player, { "Well done... See you next week." }, npc:getPacketName(), { npc = npc })
            player:tradeComplete()
            player:setCharVar(settings.traded, 1, NextConquestTally())
        end

        return
    end
end

local qtyList =
{
    {  "x1",  1 },
    {  "x2",  2 },
    {  "x3",  3 },
    {  "x6",  6 },
    { "x12", 12 },
    { "x24", 24 },
    { "x36", 36 },
    { "x99", 99 },
}

local function selectQty(player, npc)
    local balance = player:getCharVar(settings.points)
    local tbl     = {}

    for _, qtyInfo in pairs(qtyList) do
        if qtyInfo[2] <= balance then
            table.insert(tbl, {
                qtyInfo[1],
                function()
                    if npcUtil.giveItem(player, { { xi.item.AETHER_FRAGMENT, qtyInfo[2] } }) then
                        player:incrementCharVar(settings.points, -qtyInfo[2])

                        if balance - qtyInfo[2] > 0 then
                            selectQty(player, npc)
                        end
                    end
                end,
            })
        end
    end

    delaySendMenu(player, {
        title   = fmt("Withdraw Fragments ({}):", balance),
        options = tbl,
    })
end

local function onTrigger(player, npc)
    local balance = player:getCharVar("[AETHER]Fragments")
    local options =
    {
        {
            "Spend fragments",
            function()
                local rolls =
                {
                    equipment  = GetServerVariable(settings.equipment),
                    currencies = GetServerVariable(settings.currencies),
                    materials  = GetServerVariable(settings.materials),
                    spawners   = GetServerVariable(settings.spawners),
                }

                for category, _ in pairs(rolls) do
                    if rolls[category] == 0 then
                        local result    = math.random(1, #rewards[category])
                        rolls[category] = result
                        SetServerVariable(settings[category], result, NextConquestTally())
                    end
                end

                local items = {}

                if player:getCharVar(settings.equipment) == 0 then
                    table.insert(items, rewards.equipment[rolls.equipment])
                end

                if player:getCharVar(settings.currencies) == 0 then
                    table.insert(items, rewards.currencies[rolls.currencies])
                end

                if player:getCharVar(settings.materials) == 0 then
                    table.insert(items, rewards.materials[rolls.materials])
                end

                if player:getCharVar(settings.spawners) == 0 then
                    table.insert(items, rewards.spawners[rolls.spawners])
                end

                if player:getCharVar(settings.mystery) == 0 then
                    table.insert(items, { "??? Mystery Item", "Mystery", 15 })
                end

                cexi.util.simpleShop(player, npc, items, confirmPurchase, fmt("Select an item ({} Aeth. Fragments):", balance))
            end,
        },
        {
            "This week's special",
            function()
                if player:getCharVar(settings.traded) > 0 then
                    cexi.util.dialog(player, { "Hmm... You've already completed my request for this week." }, npc:getPacketName(), { npc = npc })
                    return
                end

                local roll = player:getCharVar(settings.weekly)

                if roll == 0 then
                    roll = math.random(1, #requests)
                    player:setCharVar(settings.weekly, roll, NextConquestTally())
                end

                cexi.util.dialog(player, { fmt("Yes... This week I require {}. Bring them to me and you shall have the fragments.", requests[roll][1]) }, npc:getPacketName(), { npc = npc })
            end,
        },
    }

    if balance > 0 then
        table.insert(options, {
            "Retrieve fragments",
            function()
                selectQty(player, npc)
            end,
        })
    end

    delaySendMenu(player, {
        title   = "What is it, adventurer?",
        options = options
    })
end

cexi.util.liveReload(m, {
    ["Port_Jeuno"] =
    {
        {
            name      = "Aether Traveler",
            objtype   = xi.objType.NPC,
            look      = cexi.util.look({
                race = xi.race.TARU_M,
                face = cexi.face.A1,
                head = 407,
                body = 407,
                hand = 407,
                legs = 407,
                feet = 407,
                main = 66,
            }),
            x         = -160.339,
            y         = 11.000,
            z         = 90.873,
            rotation  = 135,
            hidden    = true,
            onTrigger = onTrigger,
            onTrade   = onTrade,
        },
    },
})

m:addOverride("xi.zones.Port_Jeuno.Zone.onGameHour", function(zone)
    super(zone)

    local nextConquest = NextConquestTally() - os.time()
    local result       = zone:queryEntitiesByName("DE_Aether Traveler")

    if
        result == nil or
        result[1] == nil
    then
        return
    end

    -- Aether Traveler only appears between Friday - Saturday
    if
        nextConquest > 86400 and
        nextConquest < 172800
    then
        if result[1]:getStatus() == xi.status.DISAPPEAR then
            -- TODO: Can we do an announcement without char reference?
            result[1]:setStatus(xi.status.NORMAL)
        end
    else
        if result[1]:getStatus() == xi.status.NORMAL then
            result[1]:setStatus(xi.status.DISAPPEAR)
        end
    end
end)

return m
