require("modules/module_utils")
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("beastmaster_adjustments")

-- percent boost to _bst pets_, starts at 0
local bstBonus = function(master)
    -- Get Player Stats
    local bstIsMain     = master:getMainJob() == xi.job.BST
    local bstIsSub      = master:getSubJob() == xi.job.BST
    local mainJobLevel  = master:getMainLvl()
    local subJobLevel   = master:getSubLvl()
    local playerCHR     = master:getMod(xi.mod.CHR) + 2 * master:getMerit(xi.merit.CHR)
    local scalerValue   = 1.0
    local chrBoost      = (playerCHR / 4) / 100 -- 1% bonus per 4 chr over cap

    -- Damage Calculations
    if bstIsSub then
        scalerValue = subJobLevel / mainJobLevel
    end

    return utils.clamp(chrBoost * scalerValue, 0, 1) -- max 100% bonus
end

-- give pet buff to effect, so it wears off naturally
local function buffPet(master, pet)
    local dmgMultiplier = bstBonus(master)
    local dmgMult100 = math.floor(100 * dmgMultiplier)
    local dmgMult1000 = math.floor(1000 * dmgMultiplier)
    -- pet:setLocalVar("bstDmgMult1000", dmgMult1000)

    pet:delStatusEffectSilent(xi.effect.BOOST)
    if pet:addStatusEffect(xi.effect.BOOST, 0, 0, 0) then
        local effect = pet:getStatusEffect(xi.effect.BOOST)
        if effect then
            -- ensure the buff doesn't wear from the normal reasons "Boost" would wear
            effect:delEffectFlag(xi.effectFlag.ATTACK)

            -- apply buffs to the status effect along with the player, so they go away automatically when de-charmed
            local boostEffects = {
                xi.mod.ATTP,
                xi.mod.ACC,
                xi.mod.MATT,
                xi.mod.MACC,
                xi.mod.DEFP,
                xi.mod.COUNTER,
                xi.mod.SKILLCHAINBONUS,
            }
            for _, modId in ipairs(boostEffects) do
                pet:addMod(   modId,        dmgMult100)
                effect:addMod(modId,        dmgMult100)
            end

            master:printToPlayer(fmt("{}'s charisma provides {} with a {}% bonus!", master:getName(), string.gsub(pet:getName(), '_', ' '), dmgMult1000 / 10), xi.msg.channel.SYSTEM_3)
        end
    end
end

m:addOverride("xi.actions.abilities.call_beast.onUseAbility", function(player, target, ability)
    super(player, target, ability)

    local pet = player:getPet()
    if pet then
        buffPet(player, pet)
    end
end)

m:addOverride("xi.actions.abilities.bestial_loyalty.onUseAbility", function(player, target, ability)
    super(player, target, ability)

    local pet = player:getPet()
    if pet then
        buffPet(player, pet)
    end
end)

-- native boost to jug pets
m:addOverride("xi.actions.abilities.charm.onUseAbility", function(player, target, ability)
    super(player, target, ability)

    local pet = player:getPet()
    if pet then
        buffPet(player, pet)
    end
end)

-- TODO make the dmg bonus global and only set skillchain properties if set in the table
-- need a way to know the damage type of other skills in that case
-- a workaround might be to just increase the return and do target:takeDamage with the _difference_ between original damage and new damage
for _, bstAbility in pairs(cexi.job.bst.petAbilities) do
    -- no need for ensurepath, as super would fail if this didn't exist anyway
    local skillPath = string.format("xi.actions.mobskills.%s.onMobWeaponSkill", bstAbility[1])
    m:addOverride(skillPath, function(target, mob, skill)
        local master = mob:getMaster()
        if
            master and
            master:isPC()
        then
            -- Configuration
            local attackType    = bstAbility[2]
            local damageType    = bstAbility[3]
            -- set sc properties
            skill:setSkillchainProps(bstAbility[4], bstAbility[5], bstAbility[6])
            -- print(fmt("{}-{}-{}", skill:getSkillchainProps()))

            -- calculate the chr bonus separately from charm/call beast
            local dmgMultiplier = bstBonus(master)
            local dmgMult100    = math.floor(100 * dmgMultiplier)
            local originalDmg   = super(target, mob, skill)
            local finalDmg      = math.floor(originalDmg * (1 + dmgMultiplier))
            -- print(fmt("originalDmg: {}, finalDmg: {}", originalDmg, finalDmg))
            if target:getHP() > 0 then
                -- Undo damage dealt
                target:takeDamage(-originalDmg, mob, attackType, damageType)

                -- Deal new damage
                target:takeDamage(finalDmg, mob, attackType, damageType)
            end
            
            return finalDmg
        else
            return super(target, mob, skill)
        end
    end)
end

for zone, mobs in pairs(cexi.job.bst.charmMobs) do
    for _, mob in pairs(mobs) do
        m:addOverride(string.format("xi.zones.%s.mobs.%s.onMobInitialize", zone, mob), function(mob)
            mob:setMobMod(xi.mobMod.CHARMABLE, 1)

            super(mob)
        end)
    end
end

-- allow snarl with charmed pets
m:addOverride("xi.actions.abilities.snarl.onAbilityCheck", function(player, target, ability)
    local result1, result2 = super(player, target, ability)

    if
        result1 == xi.msg.basic.PET_CANNOT_DO_ACTION and
        player:getPet() -- should already be confirmed that player has pet, but just in case
    then
        return 0, 0
    else
        return result1, result2
    end
end)

return m
