-----------------------------------
require("scripts/globals/weaponskills")
-----------------------------------
local m = Module:new("ws_modifications")

-- All WS with non-retail adjustments to params
-- This allows fine-tuning the WS but preserving the majority of content from LSB
-- Also we can define explicit multipliers at the end to allow scaling on weapon dmg/stats/etc but still keep them on par with others

local wsMods =
{
    -----------------------------------
    -- Axe
    -----------------------------------
    [xi.weaponskill.DECIMATION] = {
        ['params'] = {
            ['canCrit'] = true,
        },
        ['multiplier'] = 1.40,
    },
    [xi.weaponskill.ONSLAUGHT] = {
        ['params'] = {
            ['dex_wsc'] = 0.80,
            ['ftpMod'] = {
                5.75,
                5.75,
                5.75,
            },
        },
    },
    [xi.weaponskill.PRIMAL_REND] = {
        ['params'] = {
            ['dex_wsc'] = 0.70,
            ['chr_wsc'] = 0.70,
            ['ftpMod'] = {
                4.75,
                4.85,
                4.95,
            },
        },
    },
    [xi.weaponskill.RUINATOR] = {
        ['params'] = {
            ['str_wsc'] = 0.75,
            ['ftpMod'] = {
                5.00,
                5.00,
                5.00,
            },
            ['atkVaries'] = {
                2.0,
                2.1,
                2.2,
            },
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Archery
    -----------------------------------
    [xi.weaponskill.FLAMING_ARROW] = {
         ['params'] = {
            ['ftpMod'] = {
                2.0, -- 1.0
                2.5, -- 1.0
                3.0, -- 1.0
            },
            ['str_wsc'] = 0.30, -- 0.16
            ['agi_wsc'] = 0.40, -- 0.25
            ['effect'] =
            {
                status  = xi.effect.BURN,
                element = xi.element.FIRE,
                amount  = 4,
            },
        },
    },
    [xi.weaponskill.EMPYREAL_ARROW] = {
        ['params'] = {
            ['str_wsc'] = 0.60,
            ['agi_wsc'] = 0.60,
            ['ftpMod'] = {
                3.10,
                3.15,
                3.20,
            },
        },
    },
    [xi.weaponskill.NAMAS_ARROW] = {
        ['params'] = {
            ['str_wsc'] = 0.50,
            ['agi_wsc'] = 0.50,
        },
        ['multiplier'] = 1.20, -- ftp100 = 2.75 -> 3.3
    },
    [xi.weaponskill.APEX_ARROW] = {
        ['params'] = {
            ['agi_wsc'] = 0.95,
            ['ftpMod'] = {
                5.65,
                5.65,
                5.65,
            },
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Club
    -----------------------------------
    [xi.weaponskill.BLACK_HALO] = {
        ['params'] = {
            ['ftpMod'] = {
                3.25,
                3.30,
                3.35,
            },
            ['str_wsc'] = 0.75,
            ['mnd_wsc'] = 0.75,
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
        },
    },
    [xi.weaponskill.RANDGRITH] = {
        ['params'] = {
            ['ftpMod'] = {
                5.5,
                5.5,
                5.5,
            },
            ['str_wsc'] = 0.85,
            ['mnd_wsc'] = 0.85,
        },
    },
    [xi.weaponskill.REALMRAZER] = {
        ['params'] = {
            ['ftpMod'] = {
                6.0,
                6.0,
                6.0,
            },
            ['mnd_wsc'] = 0.85,
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
        },
        ['variableMultiplier'] = true,
    },
    [xi.weaponskill.MYSTIC_BOON] = { -- loses `player:addMP(damage * math.random(35,45)/100)` but fairly certain this was still giving infinite mp
        ['params'] = {
            ['ftpMod'] = {
                5.5,
                5.6,
                5.7,
            },
            ['str_wsc'] = 0.95,
            ['mnd_wsc'] = 0.95,
            ['atkVaries'] = {
                1.25,
                1.25,
                1.25,
            },
        },
    },

    -----------------------------------
    -- Dagger
    -----------------------------------
    [xi.weaponskill.WASP_STING] = {
        ['params'] = {
            ['dex_wsc'] = 0.30,
            ['ftpMod'] = {
                1.0,
                1.0,
                1.0,
            },
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
        },
    },
    [xi.weaponskill.VIPER_BITE] = {
        ['params'] = {
            ['dex_wsc'] = 0.50,
            ['ftpMod'] = {
                1.5,
                1.75,
                2.0,
            },
            ['atkVaries'] = {
                2.0,
                2.0,
                2.0,
            },
        },
    },
    [xi.weaponskill.EVISCERATION] = {
        ['params'] = {
            ['dex_wsc'] = 0.50,
        },
        ['multiplier'] = 1.25,
    },
    [xi.weaponskill.EXENTERATOR] = {
        ['params'] = {
            ['ftpMod'] = {
                5.5,
                5.5,
                5.5,
            },
            ['agi_wsc'] = 0.95,
            ['atkVaries'] = {
                1.66,
                1.66,
                1.66,
            },
        },
        ['variableMultiplier'] = true,
    },
    [xi.weaponskill.SHARK_BITE] = {
        ['params'] = {
            ['dex_wsc'] = 0.6,
            ['agi_wsc'] = 0.6,
        },
        ['multiplier'] = 1.5,
    },
    [xi.weaponskill.MANDALIC_STAB] = {
        ['params'] = {
            ['dex_wsc'] = 0.75,
        },
        ['multiplier'] = 2.12, -- ftp300 = 2.5 -> 5.30
    },
    [xi.weaponskill.MERCY_STROKE] = {
        ['params'] = {
            ['str_wsc'] = 0.80,
        },
        ['multiplier'] = 1.83,
    },
    [xi.weaponskill.MORDANT_RIME] = {
        ['params'] = {
            ['ftpMod'] = {
                5,
                5,
                5,
            },
            ['dex_wsc'] = 0.95,
            ['chr_wsc'] = 0.95,
        },
    },
    [xi.weaponskill.PYRRHIC_KLEOS] = {
        ['params'] = {
            ['ftpMod'] = {
                3.5,
                3.5,
                3.5,
            },
            ['str_wsc'] = 0.80,
            ['dex_wsc'] = 0.80,
        },
    },

    -----------------------------------
    -- Great Axe
    -----------------------------------
    [xi.weaponskill.FULL_BREAK] = {
        ['multiplier'] = 1.5,
    },
    [xi.weaponskill.KINGS_JUSTICE] = {
        ['params'] = {
            ['ftpMod'] = {
                3.00,
                3.15,
                3.30,
            },
            ['str_wsc'] = 0.70,
        },
    },
    [xi.weaponskill.RAGING_RUSH] = {
        ['params'] = {
            ['str_wsc'] = 0.5,
        },
    },
    [xi.weaponskill.STEEL_CYCLONE] = {
        ['params'] = {
            ['str_wsc'] = 0.60,
            ['vit_wsc'] = 0.60,
        },
        ['multiplier'] = 1.5, -- ftp100 = 1.5 -> 2.25
    },
    [xi.weaponskill.UPHEAVAL] = {
        ['params'] = {
            ['ftpMod'] = {
                4.5,
                4.5,
                4.5,
            },
            ['vit_wsc'] = 0.60,
            ['str_wsc'] = 0.60,
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Great Katana
    -----------------------------------
    [xi.weaponskill.TACHI_GEKKO] = {
        ['params'] = {
            ['str_wsc'] = 0.95,
        },
        ['multiplier'] = 1.1,
    },
    [xi.weaponskill.TACHI_KAITEN] = {
        ['params'] = {
            ['ftpMod'] = {
                5,
                5,
                5,
            },
            ['str_wsc'] = 0.80,
            ['atkVaries'] = {
                1.75,
                1.75,
                1.75,
            },
        },
    },
    [xi.weaponskill.TACHI_KASHA] = {
        ['params'] = {
            ['str_wsc'] = 0.85,
        },
        ['multiplier'] = 2.1,
    },
    [xi.weaponskill.TACHI_RANA] = {
        ['params'] = {
            ['ftpMod'] = {
                5.5,
                5.5,
                5.5,
            },
            ['str_wsc'] = 0.90,
        },
    },
    [xi.weaponskill.TACHI_SHOHA] = {
        ['params'] = {
            ['ftpMod'] = {
                5.8,
                5.8,
                5.8,
            },
            ['str_wsc'] = 0.75,
            ['atkVaries'] = {
                1.75,
                1.75,
                1.75,
            },
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Great Sword
    -----------------------------------
    [xi.weaponskill.FROSTBITE] = {
         ['params'] = {
            ['ftpMod'] = {
                2.0, -- 1.0
                2.5, -- 2.0
                3.0, -- 2.5
            },
            ['str_wsc'] = 0.40, -- 0.20
            ['int_wsc'] = 0.40, -- 0.20
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
            ['effect'] =
            {
                status  = xi.effect.FROST,
                element = xi.element.ICE,
                amount  = 3,
            },
        },
    },
    [xi.weaponskill.FREEZEBITE] = {
         ['params'] = {
            ['ftpMod'] = {
                2.0,
                2.5,
                3.0,
            },
            ['str_wsc'] = 0.50, -- 0.30
            ['int_wsc'] = 0.50, -- 0.20
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
            ['effect'] =
            {
                status  = xi.effect.FROST,
                element = xi.element.ICE,
                amount  = 5,
            },
        },
    },
    [xi.weaponskill.DIMIDIATION] = {
        ['params'] = {
            ['ftpMod'] = {
                5.30,
                5.45,
                5.60,
            },
            ['dex_wsc'] = 0.95,
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
        },
    },
    [xi.weaponskill.GROUND_STRIKE] = {
        ['params'] = {
            ['str_wsc'] = 0.80,
            ['int_wsc'] = 0.80,
        },
    },
    [xi.weaponskill.RESOLUTION] = {
        ['params'] = {
            ['ftpMod'] = {
                2.0,
                2.0,
                2.0,
            },
            ['str_wsc'] = 0.75,
        },
        ['variableMultiplier'] = true,
    },
    [xi.weaponskill.SCOURGE] = {
        ['params'] = {
            ['ftpMod'] = {
                3.5,
                3.5,
                3.5,
            },
            ['str_wsc'] = 0.50,
            ['vit_wsc'] = 0.50,
            ['mnd_wsc'] = 0.0,
            ['chr_wsc'] = 0.0,
        },
    },

    -----------------------------------
    -- Hand-to-Hand
    -----------------------------------
    [xi.weaponskill.ASURAN_FISTS] = {
        ['params'] = {
            ['str_wsc'] = 0.50,
            ['vit_wsc'] = 0.50,
        },
    },
    [xi.weaponskill.FINAL_HEAVEN] = {
        ['params'] = {
            ['ftpMod'] = {
                5.75,
                5.75,
                5.75,
            },
            ['vit_wsc'] = 0.80,
        },
    },
    [xi.weaponskill.ASCETICS_FURY] = {
        ['params'] = {
            ['ftpMod'] = {
                4.0,
                4.0,
                4.0,
            },
            ['str_wsc'] = 0.75,
            ['vit_wsc'] = 0.75,
        },
    },
    [xi.weaponskill.STRINGING_PUMMEL] = {
        ['params'] = {
            ['ftpMod'] = {
                2.0,
                2.0,
                2.0,
            },
            ['str_wsc'] = 0.75,
            ['vit_wsc'] = 0.75,
        },
    },
    [xi.weaponskill.SHIJIN_SPIRAL] = {
        ['params'] = {
            ['ftpMod'] = {
                6.5,
                6.5,
                6.5,
            },
            ['dex_wsc'] = 0.95,
            ['atkVaries'] = {
                1.25,
                1.25,
                1.25,
            },
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Katana
    -----------------------------------
    [xi.weaponskill.BLADE_JIN] = {
        ['params'] = {
            ['str_wsc'] = 0.5,
            ['dex_wsc'] = 0.5,
        },
    },
    [xi.weaponskill.BLADE_KAMU] = {
        ['params'] = {
            ['ftpMod'] = {
                4.75,
                4.75,
                4.75,
            },
            ['str_wsc'] = 0.80,
            ['int_wsc'] = 0.80,
        },
    },
    [xi.weaponskill.BLADE_KU] = {
        ['params'] = {
            ['ftpMod'] = {
                1.5,
                1.75,
                2,
            },
            ['str_wsc'] = 0.50,
            ['dex_wsc'] = 0.50,
        },
    },
    [xi.weaponskill.BLADE_METSU] = {
        ['params'] = {
            ['ftpMod'] = {
                5.75,
                5.75,
                5.75,
            },
            ['dex_wsc'] = 0.85,
        },
    },
    [xi.weaponskill.BLADE_SHUN] = {
        ['params'] = {
            ['ftpMod'] = {
                4.5,
                4.5,
                4.5,
            },
            ['dex_wsc'] = 0.95,
            ['atkVaries'] = {
                1.25,
                1.75,
                2,
            },
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Marksmanship
    -----------------------------------
    [xi.weaponskill.HOT_SHOT] = {
         ['params'] = {
            ['ftpMod'] = {
                2.0, -- 1.0
                2.5, -- 1.0
                3.0, -- 1.0
            },
            ['str_wsc'] = 0.30, -- 0.16
            ['agi_wsc'] = 0.40, -- 0.25
            ['effect'] =
            {
                status  = xi.effect.BURN,
                element = xi.element.FIRE,
                amount  = 4,
            },
        },
    },
    [xi.weaponskill.CORONACH] = {
        ['params'] = {
            ['dex_wsc'] = 0.4,
            ['agi_wsc'] = 0.4,
        },
    },
    [xi.weaponskill.DETONATOR] = {
        ['params'] = {
            ['agi_wsc'] = 0.60,
        },
        ['multiplier'] = 2.1, -- ftp300 = 2.5 -> 4.1
    },
    [xi.weaponskill.LAST_STAND] = {
        ['params'] = {
            ['ftpMod'] = {
                2.5,
                3.0,
                3.5,
            },
            ['agi_wsc'] = 0.87,
        },
    },
    [xi.weaponskill.LEADEN_SALUTE] = {
        ['params'] = {
            ['ftpMod'] = {
                4.5,
                4.8,
                5.1,
            },
            ['agi_wsc'] = 0.95,
        },
    },
    [xi.weaponskill.TRUEFLIGHT] = {
        ['params'] = {
            ['ftpMod'] = {
                5.50,
                5.75,
                6.0,
            },
            ['agi_wsc'] = 0.75,
        },
    },

    -----------------------------------
    -- Polearm
    -----------------------------------
    [xi.weaponskill.DRAKESBANE] = {
        ['params'] = {
            ['ftpMod'] = {
                1.5,
                1.5,
                1.5,
            },
            ['str_wsc'] = 0.75,
        },
    },
    [xi.weaponskill.GEIRSKOGUL] = {
        ['params'] = {
            ['agi_wsc'] = 0.75,
        },
        ['multiplier'] = 1.2, -- ftp300 = 3 -> 3.5
    },
    [xi.weaponskill.IMPULSE_DRIVE] = {
        ['params'] = {
            ['ftpMod'] = {
                2.5,
                2.75,
                3.0,
            },
        },
    },
    [xi.weaponskill.STARDIVER] = {
        ['params'] = {
            ['ftpMod'] = {
                1.92, -- was 3.55, but LSB has "multiHitfTP = true" so we adjusted
                1.92,
                1.92,
            },
            ['str_wsc'] = 0.75,
        },
        ['variableMultiplier'] = true,
    },

    -----------------------------------
    -- Scythe
    -----------------------------------
    [xi.weaponskill.SHADOW_OF_DEATH] = {
        ['params'] = {
            ['ftpMod'] = {
                2.5,  -- 1.0
                2.75, -- 2.5
                3.0,  -- 3.0
            },
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
            ['str_wsc'] = 0.60, -- 0.30
            ['int_wsc'] = 0.60, -- 0.30
        },
    },
    [xi.weaponskill.CATASTROPHE] = {
        ['params'] = {
            ['str_wsc'] = 0.60,
            ['agi_wsc'] = 0.0,
            ['int_wsc'] = 0.60,
        },
        ['multiplier'] = 1.25,
    },
    [xi.weaponskill.ENTROPY] = {
        ['params'] = {
            ['ftpMod'] = {
                5.2,
                5.2,
                5.2,
            },
            ['int_wsc'] = 0.95,
            ['str_wsc'] = 0.2,
        },
        ['variableMultiplier'] = true,
    },
    [xi.weaponskill.INSURGENCY] = {
        ['params'] = {
            ['ftpMod'] = {
                3.30,
                3.55,
                3.80,
            },
            ['str_wsc'] = 0.30,
            ['int_wsc'] = 0.30,
        },
    },
    [xi.weaponskill.SPIRAL_HELL] = {
        ['params'] = {
            ['str_wsc'] = 0.65,
            ['int_wsc'] = 0.65,
        },
        ['multiplier'] = 1.8, -- ftp100 = 1.375 -> 2.5
    },

    -----------------------------------
    -- Staff
    -----------------------------------
    [xi.weaponskill.GARLAND_OF_BLISS] = {
        ['params'] = {
            ['mnd_wsc'] = 0.85,
        },
        ['multiplier'] = 2, -- ftp300 = 2 -> 5.50
    },
    [xi.weaponskill.GATE_OF_TARTARUS] = {
        ['params'] = {
            ['chr_wsc'] = 0.90,
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
        },
        ['multiplier'] = 1.8, -- ftp300 = 3 -> 5.5
    },
    [xi.weaponskill.OMNISCIENCE] = {
        ['params'] = {
            ['mnd_wsc'] = 0.75,
        },
        ['multiplier'] = 2, -- ftp300 = 2 -> 5
    },
    [xi.weaponskill.RETRIBUTION] = {
        ['params'] = {
            ['str_wsc'] = 0.90,
            ['mnd_wsc'] = 0.90,
        },
        ['multiplier'] = 1.6, -- ftp300 = 3 -> 5.0
    },
    [xi.weaponskill.SHATTERSOUL] = {
        ['params'] = {
            ['ftpMod'] = {
                5.5,
                5.5,
                5.5,
            },
            ['int_wsc'] = 0.95,
            ['atkVaries'] = {
                2,
                2,
                2,
            },
        },
        ['variableMultiplier'] = true,
    },
    [xi.weaponskill.VIDOHUNIR] = {
        ['params'] = {
            ['int_wsc'] = 0.85,
        },
        ['multiplier'] = 2, -- ftp300 = 1.75 -> 3.75
    },

    -----------------------------------
    -- Sword
    -----------------------------------
    [xi.weaponskill.DEATH_BLOSSOM] = {
        ['params'] = {
            ['mnd_wsc'] = 0.85,
        },
        ['multiplier'] = 2,
    },
    [xi.weaponskill.ATONEMENT] = {
        ['takeWSDMGMultiplier'] = 1.34, -- max 750 -> 1000
    },
    [xi.weaponskill.EXPIACION] = {
        ['params'] = {
            ['ftpMod'] = {
                6.0,
                6.25,
                6.50,
            },
            ['str_wsc'] = 0.95,
            ['int_wsc'] = 0.95,
        },
    },
    [xi.weaponskill.KNIGHTS_OF_ROUND] = {
        ['params'] = {
            ['str_wsc'] = 0.65,
            ['mnd_wsc'] = 0.65,
            ['atkVaries'] = {
                1.5,
                1.5,
                1.5,
            },
        },
        ['multiplier'] = 1.2, -- ftp300 = 3 -> 3.5
    },
    [xi.weaponskill.REQUIESCAT] = {
        ['params'] = {
            ['ftpMod'] = {
                6.0,
                6.0,
                6.0,
            },
            ['mnd_wsc'] = 0.85,
            ['atkVaries'] = {
                1.75,
                1.75,
                1.75,
            },
        },
        ['variableMultiplier'] = true,
    },
    [xi.weaponskill.SAVAGE_BLADE] = {
        ['params'] = {
            ['ftpMod'] = {
                3.5,
                3.6,
                3.7,
            },
            ['str_wsc'] = 0.80,
            ['mnd_wsc'] = 0.80,
        },
    },
}

local adjustParams = function(attacker, tp, wsID, wsParams)
    if
        wsMods[wsID] and
        wsMods[wsID]['params']
    then
        for k, v in pairs(wsMods[wsID]['params']) do
            wsParams[k] = v
        end
    end

    if
        wsMods[wsID] and
        wsMods[wsID]['multiplier']
    then
        wsParams['customMultiplier'] = wsMods[wsID]['multiplier']
    elseif
        wsMods[wsID] and
        wsMods[wsID]['variableMultiplier']
    then
        -- FTP stays high for these, but raw damage varies by real tp (unaffected by martial weapons)
        -- reduce total dmg by 20% at 1k tp, linearly up to full dmg at 2k+ (since tp_bonus still caps at 3k, we can't know how far above 2k they were)
        local tpMod = utils.clamp(tp - attacker:getMod(xi.mod.TP_BONUS), 1000, 2000) - 1000 -- Value scale  1000 - Range:    0, 1000
        tpMod       = tpMod / 1000                                                          -- Value scale  1.f  - Range:    0,    1
        tpMod       = 100 - 20 * (1 - tpMod)                                                -- Value scale  100  - Range:   80,  100
        tpMod       = tpMod / 100                                                           -- Value scale  1.f  - Range: 0.80, 1.00

        wsParams['customMultiplier'] = tpMod
    end
end

m:addOverride("xi.weaponskills.takeWeaponskillDamage", function(defender, attacker, wsParams, primaryMsg, attack, wsResults, action)
    if
        wsMods[wsResults.wsID] and
        wsMods[wsResults.wsID]['takeWSDMGMultiplier']
    then
        -- special multiplier only for certain ws to adjust damage just before explicit takeWeaponskillDamage binding is called when the 3 base aren't explicitly called
        wsResults.finalDmg = wsResults.finalDmg * wsMods[wsResults.wsID]['takeWSDMGMultiplier']
    elseif
        wsParams['customMultiplier'] and
        not (wsParams['customMultiplier'] < 1 and -- don't nerf damage if the ws would be killing the mob, since weaponskills.lua doesn't process all attacks if the mob would die
        wsResults.finalDmg > defender:getHP())
    then
        -- we create a new key in wsParams in the 3 individual overrides, then hook it here to multiply the final damage
        wsResults.finalDmg = wsResults.finalDmg * wsParams['customMultiplier']
    end

    -- All weapons that grant a weaponskill will boost that weaponskill by 10%
    local wsWeapon = attacker:getEquippedItem(attack.slot)
    local addsWeaponskill = wsWeapon and (wsWeapon:getMod(xi.mod.ADDS_WEAPONSKILL) + wsWeapon:getMod(xi.mod.ADDS_WEAPONSKILL_DYN)) or 0
    if
        addsWeaponskill > 0 and
        addsWeaponskill == wsResults.wsID
    then
        -- print('boosted')
        wsResults.finalDmg = wsResults.finalDmg * 1.10
    end

    return super(defender, attacker, wsParams, primaryMsg, attack, wsResults, action)
end)

local function applyAdditionalEffect(attacker, target, tp, wsID, finaldmg)
    if
        finaldmg == 0 or
        wsMods[wsID] == nil or
        wsMods[wsID].params.effect == nil
    then
        return
    end

    local effect = wsMods[wsID].params.effect

    if not target:hasStatusEffect(effect.status) then
        local duration = (30 + (tp / 1000 * 30)) * applyResistanceAddEffect(attacker, target, effect.element, 0, 0)
        target:addStatusEffect(effect.status, effect.amount, 0, duration)
    end
end

m:addOverride("xi.weaponskills.doPhysicalWeaponskill", function(attacker, target, wsID, wsParams, tp, action, primaryMsg, taChar)
    adjustParams(attacker, tp, wsID, wsParams)
    local finaldmg, criticalHit, tpHitsLanded, extraHitsLanded, shadowsAbsorbed = super(attacker, target, wsID, wsParams, tp, action, primaryMsg, taChar)
    applyAdditionalEffect(attacker, target, tp, wsID, finaldmg)
    return finaldmg, criticalHit, tpHitsLanded, extraHitsLanded, shadowsAbsorbed
end)

m:addOverride("xi.weaponskills.doRangedWeaponskill", function(attacker, target, wsID, wsParams, tp, action, primaryMsg)
    adjustParams(attacker, tp, wsID, wsParams)
    local finaldmg, criticalHit, tpHitsLanded, extraHitsLanded, shadowsAbsorbed = super(attacker, target, wsID, wsParams, tp, action, primaryMsg)
    applyAdditionalEffect(attacker, target, tp, wsID, finaldmg)
    return finaldmg, criticalHit, tpHitsLanded, extraHitsLanded, shadowsAbsorbed
end)

m:addOverride("xi.weaponskills.doMagicWeaponskill", function(attacker, target, wsID, wsParams, tp, action, primaryMsg)
    adjustParams(attacker, tp, wsID, wsParams)
    local dmg, criticalHit, tpHitsLanded, extraHitsLanded, shadowsAbsorbed = super(attacker, target, wsID, wsParams, tp, action, primaryMsg)
    applyAdditionalEffect(attacker, target, tp, wsID, finaldmg)
    return dmg, criticalHit, tpHitsLanded, extraHitsLanded, shadowsAbsorbed
end)

return m
