----------------------------------
-- func: !box (item name or ID)
-- desc: Check the quantity of item stored in Ephemeral Box
-----------------------------------
local commandObj = {}
local tradeable  = cexi.crystal_warrior.ephemeral_box.tradeable

commandObj.cmdprops =
{
    permission = 0,
    parameters = "s"
}

local function error(player, msg)
    player:printToPlayer(msg)
    player:printToPlayer("!box (item name or ID)")
end

local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function getItemInfo(player, itemID)
    if
        itemID == nil or
        itemID == 0
    then
        return nil
    end

    if cexi.crystal_warrior.ephemeral_box.tradeable[itemID] == nil then
        if
            cexi.crystal_warrior.ephemeral_box.conditional[itemID] ~= nil and
            player:getCharVar(cexi.crystal_warrior.ephemeral_box.conditional[itemID][1]) ~= 0
        then
            return {
                itemName  = cexi.crystal_warrior.ephemeral_box.conditional[itemID][2],
                craftType = cexi.crystal_warrior.ephemeral_box.conditional[itemID][3],
                itemType  = cexi.crystal_warrior.ephemeral_box.conditional[itemID][4],
            }
        else
            return nil
        end
    else
        return {
            itemName  = cexi.crystal_warrior.ephemeral_box.tradeable[itemID][1],
            craftType = cexi.crystal_warrior.ephemeral_box.tradeable[itemID][2],
            itemType  = cexi.crystal_warrior.ephemeral_box.tradeable[itemID][3],
        }
    end
end

local function getBox(player)
    local zoneName = player:getZoneName()

    if cexi.crystal_warrior.ephemeral_box.areas[zoneName] ~= nil then
        for craftName, pos in pairs(cexi.crystal_warrior.ephemeral_box.areas[zoneName]) do
            if player:checkDistance(pos[1], pos[2], pos[3]) <= 5 then
                local zone = player:getZone()
                local npc  = zone:queryEntitiesByName(fmt("DE_Ephemeral_Box_{}", craftName))

                if
                    npc ~= nil and
                    npc[1] ~= nil
                then
                    return npc[1]
                end
            end
        end
    end
end

local function selectClusters(player, npc, material)
    local tbl       = {}
    local varName   = fmt("[EB]{}", material[2])
    local current   = player:getCharVar(varName)
    local available = math.floor(current / 12)

    for _, quantity in pairs(cexi.crystal_warrior.ephemeral_box.qtyList) do
        if quantity[2] <= available then
            table.insert(tbl, {
                quantity[1],
                function()
                    npc:ceAnimate(player, player, 75, 4)

                    player:timer(500, function()
                        if cexi.crystal_warrior.ephemeral_box.boxGiveItem(player, material[3], quantity[2]) then
                            local updated = current - (quantity[2] * 12)
                            player:setCharVar(varName, updated)

                            if updated >= 12 then
                                selectClusters(player, npc, material)
                            end
                        end
                    end)
                end,
            })
        end
    end

    delaySendMenu(player, {
        title   = "Select a quantity:",
        options = tbl,
    })
end

local crystals =
{
    { "Fire Cluster",      xi.item.FIRE_CRYSTAL,      xi.item.FIRE_CLUSTER      },
    { "Ice Cluster",       xi.item.ICE_CRYSTAL,       xi.item.ICE_CLUSTER       },
    { "Wind Cluster",      xi.item.WIND_CRYSTAL,      xi.item.WIND_CLUSTER      },
    { "Earth Cluster",     xi.item.EARTH_CRYSTAL,     xi.item.EARTH_CLUSTER     },
    { "Lightning Cluster", xi.item.LIGHTNING_CRYSTAL, xi.item.LIGHTNING_CLUSTER },
    { "Water Cluster",     xi.item.WATER_CRYSTAL,     xi.item.WATER_CLUSTER     },
    { "Light Cluster",     xi.item.LIGHT_CRYSTAL,     xi.item.LIGHT_CLUSTER     },
    { "Dark Cluster",      xi.item.DARK_CRYSTAL,      xi.item.DARK_CLUSTER      },
}

local feature =
{
    cluster = function(player, arg, box)
        if box == nil then
            return
        end

        local list = {}

        for _, crystal in pairs(crystals) do
            local varName = fmt("[EB]{}", crystal[2])
            local balance = player:getCharVar(varName)

            if balance >= 12 then
                table.insert(list, {
                    fmt("({}) {}", math.floor(balance / 12), crystal[1]),
                    crystal[2],
                    crystal[3],
                })
            end
        end

        cexi.util.simpleMenu(player, box, list, selectClusters, "Choose type:")
    end,
}

commandObj.onTrigger = function(player, arg)
    -- Block command if player is not CW or hasn't completed quest
    if
        not player:isCrystalWarrior() or
        player:getCharVar("[CW]BOXED_UP") < 5
    then
        return
    end

    if feature[arg] ~= nil then
        feature[arg](player, arg, getBox(player))

        return
    end

    local qty      = 0
    local args     = arg
    local firstArg = string.find(args, " ")

    if firstArg ~= nil then
        local firstNum = tonumber(string.sub(args, 1, firstArg))

        if firstNum ~= nil then
            qty  = firstNum
            args = string.sub(args, firstArg + 1)
        end
    end

    local itemID = tonumber(args)

    if itemID == nil then
        local str = string.gsub(string.lower(args), " ", "_")

        if cexi.crystal_warrior.ephemeral_box.duplicateItem[str] ~= nil then
            itemID = cexi.crystal_warrior.ephemeral_box.duplicateItem[str]
        else
            itemID = GetItemIDByName(str)
        end
    end

    local itemInfo = getItemInfo(player, itemID)

    if itemInfo == nil then
        player:printToPlayer("Ephemeral Box : I don't recognize that item.", xi.msg.channel.NS_SAY)
        return
    end

    local amount = player:getCharVar(string.format("[EB]%u", itemID))

    player:fmt("Ephemeral Box : I have {} {}. ({} \129\168 {})", amount, itemInfo.itemName, itemInfo.craftType, itemInfo.itemType)

    if amount == 0 then
        return
    end

    local box = getBox(player)

    if box ~= nil then
        if qty == 0 then
            cexi.crystal_warrior.ephemeral_box.quantityPage(player, box, { itemID, itemInfo.itemName })
        else
            cexi.crystal_warrior.ephemeral_box.giveQuantity(player, box, itemID, qty)
        end
    end
end

return commandObj
