-----------------------------------
-- func: !item
-- Allows viewing and modification of item properties
-----------------------------------
local commandObj = {}

commandObj.cmdprops =
{
    permission = 3,
    parameters = "ssi"
}

-- Allow command to be used read-only but
-- only the following level GM can update properties
local setPropertiesGMLevel = 5

local function error(player, msg)
    player:printToPlayer(msg)
    player:printToPlayer("!item (itemID) (param) (value)")
end

local function getItem(itemID)
    local id = tonumber(itemID)

    if id == nil then
        id = GetItemIDByName(itemID)
    end

    return GetItemByID(id)
end

local function getJobList(jobs)
    local result = ""

    for jobID, jobName in pairs(xi.jobNames) do
        if
            jobID > 0 and
            utils.mask.getBit(jobs, jobID - 1)
        then
            result = fmt("{}/{}", result, jobName[1])
        end
    end

    if string.len(result) > 0 then
        result = string.sub(result, 2, -1)
    end

    return result
end

local function getFlagList(flags)
    local result = ""

    for flagID, flagName in pairs(xi.itemFlagID) do
        if utils.mask.getBit(flags, flagID - 1) then
            result = fmt("{}, {}", result, flagName)
        end
    end

    if string.len(result) > 0 then
        result = string.sub(result, 3, -1)
    end

    return result
end

local params =
{
    price =
    {
        get = function(item)
            return item:getBasePrice()
        end,

        set = function(item, amount)
            item:setBasePrice(amount)
        end,
    },

    flags =
    {
        get = function(item)
            local flags = item:getFlags()
            return fmt("{} ({})", flags, getFlagList(flags))
        end,

        set = function(item, flags)
            item:setFlags(flags)
        end,
    },

    stack =
    {
        get = function(item)
            return item:getStackSize()
        end,

        set = function(item, size)
            item:setStackSize(size)
        end,
    },

    mods =
    {
        get = function(item)
            local mods   = item:getMods()
            local result = ""

            for _, modInfo in pairs(mods) do
                result = fmt("{}\n{} ({}) = {}", result, modInfo[1], xi.modID[modInfo[1]], modInfo[2])
            end

            return result
        end,
    },

    level =
    {
        get = function(item)
            return item:getLevel()
        end,

        set = function(item, level)
            item:setLevel(level)
        end,
    },

    ilvl =
    {
        get = function(item)
            return item:getIlvl()
        end,

        set = function(item, ilvl)
            item:setIlvl(ilvl)
        end,
    },

    jobs =
    {
        get = function(item)
            local jobs = item:getJobs()
            return fmt("{} ({})", jobs, getJobList(jobs))
        end,

        set = function(item, jobs)
            item:setJobs(jobs)
            return getJobList(jobs)
        end,
    },

    model =
    {
        get = function(item)
            return item:getModelID()
        end,

        set = function(item, modelID)
            item:setModelID(modelID)
        end,
    },

    shieldSize =
    {
        get = function(item)
            return item:getShieldSize()
        end,

        set = function(item, shieldSize)
            item:setShieldSize(shieldSize)
        end,
    },

    slot =
    {
        get = function(item)
            return item:getSlot()
        end,

        set = function(item, slot)
            item:setSlot(slot)
        end,
    },

    skillType =
    {
        get = function(item)
            return item:getSkillType()
        end,

        set = function(item, skillType)
            item:setSkillType(skillType)
        end,
    },

    subSkillType =
    {
        get = function(item)
            return item:getSubSkillType()
        end,

        set = function(item, subSkillType)
            item:setSubSkillType(subSkillType)
        end,
    },

    ilvlSkill =
    {
        get = function(item)
            return item:getIlvlSkill()
        end,

        set = function(item, skillLevel)
            item:setIlvlSkill(skillLevel)
        end,
    },

    ilvlParry =
    {
        get = function(item)
            return item:getIlvlParry()
        end,

        set = function(item, skillLevel)
            item:setIlvlParry(skillLevel)
        end,
    },

    ilvlMacc =
    {
        get = function(item)
            return item:getIlvlMacc()
        end,

        set = function(item, skillLevel)
            item:setIlvlMacc(skillLevel)
        end,
    },

    delay =
    {
        get = function(item)
            return item:getDelay()
        end,

        set = function(item, delay)
            item:setDelay(delay)
        end,
    },

    baseDelay =
    {
        get = function(item)
            return item:getBaseDelay()
        end,

        set = function(item, baseDelay)
            item:setBaseDelay(baseDelay)
        end,
    },

    damage =
    {
        get = function(item)
            return item:getDamage()
        end,

        set = function(item, damage)
            item:setDamage(damage)
        end,
    },

    damageType =
    {
        get = function(item)
            return item:getDamageType()
        end,

        set = function(item, damageType)
            item:setDamageType(damageType)
        end,
    },

    additionalEffect =
    {
        get = function(item)
            return item:getAdditionalEffect()
        end,

        set = function(item, additionalEffect)
            item:setAdditionalEffect(additionalEffect)
        end,
    },

    hitCount =
    {
        get = function(item)
            return item:getHitCount()
        end,

        set = function(item, hitCount)
            item:setHitCount(hitCount)
        end,
    },
}

commandObj.onTrigger = function(player, itemID, param, value)
    if itemID == nil then
        error(player, "itemID not specified")
        return
    end

    if param == nil then
        error(player, "Param not specified")

        for paramName, _ in pairs(params) do
            player:fmt(paramName)
        end

        return
    end

    if value ~= nil then
        if type(value) ~= "number"then
            error(player, "Value must be a number")
            return
        end

        if player:getGMLevel() < setPropertiesGMLevel then
            player:sys("Your GM level cannot set item properties")
            return
        end
    end

    local item = getItem(itemID)

    if item == nil then
        error(player, fmt("Item {} not found!", itemID))
        return
    end

    if params[param] == nil then
        error(player, fmt("Parameter {} not found!", param))
        return
    end

    -- Assume get
    if value == nil then
        player:fmt("Item ({}): {} is currently set to {}", itemID, param, params[param].get(item))

    -- Assume set
    elseif params[param].set ~= nil then
        local result = params[param].set(item, value)
        local msg    = fmt("Item ({}): {} has been updated to {}", itemID, param, value)

        if result ~= nil then
            player:fmt("{} ({})", msg, result)
        else
            player:fmt(msg)
        end
    end
end

return commandObj
