-----------------------------------
-- Holiday Event
-----------------------------------
require("modules/module_utils")
require('scripts/globals/npc_util')
require("scripts/globals/shop")
require("scripts/globals/utils")
-----------------------------------
local m = Module:new("event-holiday")

local settings =
{
    enabled      = true,
    pointsVar    = "[EVENT]STARLIGHT_CHEER",
    milestoneVar = "[EVENT]STARLIGHT_MILESTONE",
}

-- TODO: Need missing models (Find with !costume <num>)
local model =
{
    TARU_GIRL_1   = 170,
    TARU_GIRL_2   = 172,
    TARU_BOY_1    = 168,
    TARU_BOY_2    = 167,
    GALKA_BOY_1   = 954,
    GALKA_BOY_2   =  72,
    MITHRA_GIRL_1 = 182,
    MITHRA_GIRL_2 = 183,
    HUME_BOY_1    = 114,
    HUME_BOY_2    =   0,
    HUME_GIRL_1   = 118,
    HUME_GIRL_2   =   0,
    ELVAAN_BOY_1  = 154,
    ELVAAN_BOY_2  =   0,
    ELVAAN_GIRL_1 = 158,
    ELVAAN_GIRL_2 =   0,
}

local kids =
{
    [xi.zone.BASTOK_MINES] =
    {
        ["Bastok_Markets"] =
        {
            { "Jolie",     model.HUME_GIRL_1,   { -161.164,  -4.840, -114.035,  14 } }, -- (I-9)  !pos -161.164 -4.840 -114.035 235
            { "Wumbah",    model.GALKA_BOY_1,   { -112.391,  -4.000,  -76.985, 121 } }, -- (J-8)  !pos -112.391 -4.000 -76.985 235
            { "Egbert",    model.HUME_BOY_1,    { -172.608,  -4.824,   74.206,  80 } }, -- (I-5)  !pos -172.608 -4.824 74.206 235
            { "Dhi Jahya", model.MITHRA_GIRL_1, { -310.300, -16.319, -147.492,  28 } }, -- (F-10) !pos -310.300 -16.319 -147.492 235
        },
    },

    [xi.zone.WINDURST_WATERS] =
    {
        ["Windurst_Waters"] =
        {
            { "Hapi-Chapea", model.TARU_BOY_1,  {  -56.035,  -9.250,   24.686,  75 } }, -- (F-10) !pos -56.035 -9.250 24.686 238
            { "Elulu",       model.TARU_GIRL_1, {  -23.292, -10.750,   92.447,  85 } }, -- (G-8) !pos -23.292 -10.750 92.447 238
            { "Gifu-Geru",   model.TARU_BOY_2,  {   -9.109,  -4.000, -154.555,  34 } }, -- South (I-8) !pos -9.109 -4.000 -154.555 238
            { "Mimima",      model.TARU_GIRL_2, { -227.793,  -5.306, -309.448, 115 } }, -- South (D-11) !pos -227.793 -5.306 -309.448 238
        },
    },

    [xi.zone.SOUTHERN_SAN_DORIA] =
    {
        ["Southern_San_dOria"] =
        {
            { "Excelliard", model.ELVAAN_BOY_1,  {  154.669, -6.760, 64.846, 143 } }, -- !pos 154.669 -6.760 64.846 230
            { "Magnelie",   model.ELVAAN_GIRL_1, { -152.579, -2.000, 51.514, 137 } }, -- !pos -152.579 -2.000 51.514 230
            { "Ralaufaux",  model.ELVAAN_BOY_1,  { -188.185, -8.760, 99.520, 163 } }, -- !pos -188.185 -8.760 99.520 230
            { "Alfene",     model.ELVAAN_GIRL_1, {   13.641,  2.100, 12.058,  78 } }, -- !pos 13.641 2.100 12.058 230
        },
    },
}

local npcAreas =
{
    [xi.zone.BASTOK_MINES]       = "Bastok_Markets",
    [xi.zone.WINDURST_WATERS]    = "Windurst_Waters",
    [xi.zone.SOUTHERN_SAN_DORIA] = "Southern_San_dOria",
}

local presentAreas =
{
    [xi.zone.BASTOK_MINES]       = "South_Gustaberg",
    [xi.zone.WINDURST_WATERS]    = "West_Sarutabaruta",
    [xi.zone.SOUTHERN_SAN_DORIA] = "East_Ronfaure",
}

local responses =
{
    "I wish somebody would bring me a present!",
    "I wonder what surprises this year will bring?",
    "I hope I've been good enough this year!",
    "It's finally time for the Starlight Celebration!",
}

local presents =
{
    ["East_Ronfaure"] =
    {
        { 163.3564, -50.0000,   83.3971,  29 }, -- e ron 1 !pos 163.3564 -50.0000 83.3971 101
        { 356.4953, -49.5000,   89.9441, 196 }, -- e ron 2 (H-7) !pos 356.4953 -49.5000 89.9441 101
        { 450.1159, -48.6477,  105.3853, 223 }, -- e ron 3 (I-7) !pos 450.1159 -48.6477 105.3853 101
        { 585.8430, -51.2171,  223.8555, 238 }, -- e ron 4 (J-6) !pos 585.8430 -51.2171 223.8555 101
        { 499.6635, -60.0101,  470.6425, 206 }, -- e ron 5 (I-5) !pos 499.6635 -60.0101 470.6425 101
        { 742.8979, -62.8113,  418.2257,  77 }, -- e ron 6 (K-5) !pos 742.8979 -62.8113 418.2257 101
        { 415.2681, -29.7315,  -85.2698,  31 }, -- e ron 7 (I-8) !pos 415.2681 -29.7315 -85.2698 101
        { 502.8862, -19.7735, -165.9880,  28 }, -- e ron 8 (I-9) !pos 502.8862 -19.7735 -165.9880 101
        { 558.3507, -10.0000, -480.4775,   5 }, -- e ron 9 (J-11) !pos 558.3507 -10.0000 -480.4775 101
        { 262.5101,   0.0000, -502.2738,  25 }, -- e ron 10 (H-11) !pos 262.5101 0.0000 -502.2738 101
        { 258.0971, -19.9600, -336.0596, 105 }, -- e ron 10 (H-10) !pos 258.0971 -19.9600 -336.0596 101
    },

    ["South_Gustaberg"] =
    {
        { 522.3029,   0.0000, -479.6435,  85 }, -- s gusta 1 (K-9) !pos 522.3029 0.0000 -479.6435 107
        { 715.9815,  -1.0944, -673.5373, 187 }, -- s gusta 2 (L-10) !pos 715.9815 -1.0944 -673.5373 107
        { 522.3799,  -3.0766, -737.9913,  51 }, -- s gusta 3 (K-10) !pos 522.3799 -3.0766 -737.9913 107
        { 458.8300, -21.0146, -581.0390,  39 }, -- s gusta 4 (K-9) !pos 458.8300 -21.0146 -581.0390 107
        { 160.1786,  -0.2495, -680.6891, 183 }, -- s gusta 5 (I-10) !pos 160.1786 -0.2495 -680.6891 107
        { 78.2750,  -20.2976, -442.0895, 153 }, -- s gusta 6 (I-8) !pos 78.2750 -20.2976 -442.0895 107
        { -35.1277,  10.0103, -360.1960,  31 }, -- s gusta 7 (H-8) !pos -35.1277 10.0103 -360.1960 107
        { -203.4587, 10.0000, -363.1026, 128 }, -- s gusta 8 (G-8) !pos -203.4587 10.0000 -363.1026 107
        { -325.4352, 30.3192, -395.0072, 118 }, -- s gusta 9 (F-8) !pos -325.4352 30.3192 -395.0072 107
        { -463.1472, 45.0401, -432.6719, 225 }, -- s gusta 10 (E-8) !pos -463.1472 45.0401 -432.6719 107
    },

    ["West_Sarutabaruta"] =
    {
        {  337.8182,  -5.8494,  137.4522, 158 }, -- w saruta 1 (J-7) !pos 337.8182 -5.8494 137.4522 115
        {  347.6525, -18.9619,  360.1057, 107 }, -- w saruta 2 (J-6) !pos 347.6525 -18.9619 360.1057 115
        {  157.8896, -40.6968,  346.4700, 203 }, -- w saruta 3 (I-6) !pos 157.8896 -40.6968 346.4700 115
        {  153.6929, -19.7957,  505.1304, 248 }, -- w saruta 4 (I-5) !pos 153.6929 -19.7957 505.1304 115
        {  -28.2504, -13.0161,  368.0145,  26 }, -- w saruta 5 (H-6) !pos -28.2504 -13.0161 368.0145 115
        { -406.8958, -29.0706,  381.1554, 204 }, -- w saruta 6 (F-6) !pos -406.8958 -29.0706 381.1554 115
        { -106.9160, -21.5743,  180.0128,  42 }, -- w saruta 7 (G-7) !pos -106.9160 -21.5743 180.0128 115
        { -189.3972,  -8.3705, -143.3038,  47 }, -- w saruta 8 (G-9) !pos -189.3972 -8.3705 -143.3038 115
        { -369.7472,   4.7443, -397.1181, 126 }, -- w saruta 9 (F-11) !pos -369.7472 4.7443 -397.1181 115
        {  136.0918,   9.1555, -461.3716,  41 }, -- w saruta 10 (I-11) !pos 136.0918 9.1555 -461.3716 115
    },
}

local boxes =
{
    ["South_Gustaberg"] =
    {
        { 270.0078, 1.2500, -214.3061, 167 }, -- !pos 270.0078 1.2500 -214.3061 107
        { 290.7633, -1.1458, -227.0008, 24 }, -- !pos 290.7633 -1.1458 -227.0008 107
        { 314.3199, -0.2864, -244.0028, 11 }, -- !pos 314.3199 -0.2864 -244.0028 107
        { 301.8465, -0.6019, -265.3467, 97 }, -- !pos 301.8465 -0.6019 -265.3467 107
        { 279.3623, 0.0381, -269.4350, 121 }, -- !pos 279.3623 0.0381 -269.4350 107
        { 263.0362, 0.8722, -253.4302, 243 }, -- !pos 263.0362 0.8722 -253.4302 107
        { 241.4115, -0.0716, -257.8537, 226 }, -- !pos 241.4115 -0.0716 -257.8537 107
        { 227.8139, 0.0691, -238.5950, 166 }, -- !pos 227.8139 0.0691 -238.5950 107
        { 216.1309, 2.3606, -214.0885, 172 }, -- !pos 216.1309 2.3606 -214.0885 107
        { 237.9284, 0.0000, -202.6433, 7 }, -- !pos 237.9284 0.0000 -202.6433 107
    },

    ["East_Ronfaure"] =
    {
        { 129.5458, -59.6279, 255.9033, 194 }, -- !pos 129.5458 -59.6279 255.9033 101
        { 160.9503, -60.0000, 282.0847, 252 }, -- !pos 160.9503 -60.0000 282.0847 101
        { 193.9606, -50.1291, 241.9034, 183 }, -- !pos 193.9606 -50.1291 241.9034 101
        { 195.1330, -50.0560, 197.3397, 21 }, -- !pos 195.1330 -50.0560 197.3397 101
        { 160.1559, -60.0000, 158.8263, 123 }, -- !pos 160.1559 -60.0000 158.8263 101
        { 122.7250, -50.0798, 125.2559, 119 }, -- !pos 122.7250 -50.0798 125.2559 101
        { 89.3586, -59.5000, 158.3665, 205 }, -- !pos 89.3586 -59.5000 158.3665 101
        { 88.5073, -59.5335, 199.6485, 124 }, -- !pos 88.5073 -59.5335 199.6485 101
        { 62.8139, -59.9277, 202.0789, 134 }, -- !pos 62.8139 -59.9277 202.0789 101
        { 114.0670, -59.3671, 211.3130, 167 }, -- !pos 114.0670 -59.3671 211.3130 101
    },

    ["West_Sarutabaruta"] =
    {
        { 303.3889, -5.0398, -0.3985, 167 }, -- !pos 303.3889 -5.0398 -0.3985 115
        { 332.0411, -4.5864, 4.3531, 18 }, -- !pos 332.0411 -4.5864 4.3531 115
        { 345.6804, -4.7968, 0.1343, 6 }, -- !pos 345.6804 -4.7968 0.1343 115
        { 278.2001, -4.1585, 6.5098, 152 }, -- !pos 278.2001 -4.1585 6.5098 115
        { 268.9892, -4.8670, 27.4864, 198 }, -- !pos 268.9892 -4.8670 27.4864 115
        { 300.4330, -4.5000, 57.1294, 238 }, -- !pos 300.4330 -4.5000 57.1294 115
        { 328.5707, -4.1291, 73.4432, 24 }, -- !pos 328.5707 -4.1291 73.4432 115
        { 366.8175, -4.5971, 86.8643, 43 }, -- !pos 366.8175 -4.5971 86.8643 115
        { 396.2771, -4.0000, 80.6374, 15 }, -- !pos 396.2771 -4.0000 80.6374 115
        { 386.8145, -5.2370, 53.0685, 89 }, -- !pos 386.8145 -5.2370 53.0685 115
    },
}

-- Given for each activity in lieu of Dream +1 items
local randomRewards =
{
    { 5922, { 6, 12 } }, -- Walnut Cookie x6-12
    { 4216, { 6, 12 } }, -- Brilliant Snow x6-12
    5621, -- Candy Ring
    5622, -- Candy Cane
    5552, -- Black Pudding
    5620, -- Roast Turkey
    5542, -- Gat. aux Fraises
    5616, -- Lebkuchen House
    { 5779, { 1, 2 } }, -- Cherry Macaron x1-2
    { 5782, { 1, 2 } }, -- Sugar Rusk x1-2
}

-- Earned with total Starlight Cheer
local cheerRewards =
{
    { "Dream Bell",         18863,  5 },
    { "Dream Mittens",      10382,  5 },
    { "Dream Trousers",     11965,  5 }, -- Male
    { "Dream Pants",        11967,  5 }, -- Female
    { "Dream Bell +1",      18864, 10 },
    { "Hagoita",            21117, 15 },
    { "Dream Mittens +1",   10383, 20 },
    { "Snow Bunny Hat",     10875, 25 },
    { "Snowman Cap",        10875, 30 },
    { "Dream Trousers +1",  11966, 35 }, -- Male
    { "Dream Pants +1",     11968, 35 }, -- Female
    { "Poele Classique",      365, 40 },
    { "Kanonenofen",          366, 40 },
    { "Pot Topper",           367, 40 },
    { "Starlight Cake",      3634, 45 },
    { "Cour. des Etoiles",   3619, 50 },
    { "Silberkranz",         3620, 50 },
    { "Leafberry Wreath",    3621, 50 },
    { "S. Bunny Hat +1",    11491, 70 },
    { "Caliber Ring",       26164, 75 },
    { "Caliber Ring",       26164, 85 },
    { "Caliber Ring",       26164, 100 },
    -- TODO: Need Cheer milestone rewards
}

local treeRewards =
{
    { 4394, { 15, 25 } }, -- Ginger Cookie x15-25
    { 4218, {  5, 10 } }, -- Air Rider x5-10
     21097, -- Leafkin Bopper
      1875, -- Anct. Beastcoin
    { 1452, { 1,   3 } }, -- O.Bronzepiece
    { 1455, { 1,   3 } }, -- 1 Byne Bill
    { 1449, { 1,   3 } }, -- T.Whiteshell
    { 2488, { 2,   4 } }, -- Alexandrite
    176, -- Snowman Knight
    177, -- Snowman Miner
    178, -- Snowman Mage
}

-- Distributes rewards earned with Starlight Cheer
local smilebringers =
{
    ["Bastok_Mines"]       = {   7.297,  0.000, -83.982,   0 }, -- (I-9) !pos -34.792, 0.000, -98.707 234
    ["Southern_San_dOria"] = { -30.685,  2.000, -29.985, 233 }, -- (H-9) !pos -30.685 2.000 -29.985 230
    ["Windurst_Waters"]    = {   2.731, -1.000,  39.075, 100 }, -- (H-9) !pos 2.731 -1.000 39.075 238
}

local smilehelpers =
{
    ["Bastok_Mines"] =
    {
        delivery = { 5.186, 0.000, -87.623,  24 }, -- (I-9) !pos 5.186 0.000 -87.623 234
        courier  = { 6.915, 0.000, -82.017, 232 }, -- (I-9) !pos 6.915 0.000 -82.017 234
        wrapping = { 6.594, 0.000, -85.782,  21 }, -- (I-9) !pos 6.594 0.000 -85.782 234
    },

    ["Southern_San_dOria"] =
    {
        delivery = { -31.773, 2.000, -28.685, 223 }, -- (H-9) !pos -31.773 2.000 -28.685 230
        courier  = { -33.854, 2.000, -28.453, 171 }, -- (H-9) !pos -33.854 2.000 -28.453 230
        wrapping = { -31.027, 2.000, -32.197,  22 }, -- (H-9) !pos -31.027 2.000 -32.197 230
    },

    ["Windurst_Waters"] =
    {
        delivery = {  1.797, -1.000, 43.788, 148 }, -- (H-9) !pos 1.797 -1.000 43.788 238
        courier  = {  4.576, -1.000, 34.767, 124 }, -- (H-9) !pos 4.576 -1.000 34.767 238
        wrapping = { -6.414, -1.000, 45.125,   0 }, -- (H-9) !pos -6.414 -1.000 45.125 238
    },
}

local moogles =
{
    ["North_Gustaberg"] =
    {
        { 649.065,  0.591, 318.777,  12 }, -- (L-8)  !pos 649.065 0.591 318.777 106
        { 158.724, -0.451, 800.793,  33 }, -- (I-5)  !pos 158.724 -0.451 800.793 106
        {  36.033,  0.000,  81.764, 159 }, -- (H-10) !pos 36.033,  0.000,  81.764 106
    },

    ["West_Ronfaure"] =
    {
        { -611.915, -49.786,  285.548, 252 }, -- (F-6)  !pos -611.915 -49.786 285.548 100
        { -524.593, -10.287, -356.190,  29 }, -- (G-10) !pos -524.593 -10.287 -356.190 100
        { -152.388, -60.021,  276.123, 121 }, -- (I-6)  !pos -152.388 -60.021 276.123 100
    },

    ["East_Sarutabaruta"] =
    {
        { -105.443,  -5.486, -507.228,  25 }, -- (G-11) !pos -105.443 -5.486 -507.228 116
        { -257.544,  -5.555,   88.104,  55 }, -- (F-7)  !pos -257.544 -5.555 88.104 116
        {  291.441, -27.114,  584.670, 102 }, -- (J-4)  !pos 291.441 -27.114 584.670 116
    },
}

local treants =
{
    ["North_Gustaberg"] =
    {
        {  770.832,  -8.383, 462.304, 127 }, -- (M-7) !pos 770.832 -8.383 462.304 106
        {  362.225,  -0.226, 167.677, 198 }, -- (J-9) !pos 362.225 -0.226 167.677 106
        {  -31.606,   0.250, 249.076, 118 }, -- (H-9) !pos -31.606 0.250 249.076 106
        { -289.970,  -1.392, 163.862, 253 }, -- (F-9) !pos -289.970 -1.392 163.862 106
        {  -31.773,  -0.534, 632.899,  49 }, -- (H-6) !pos -31.773 -0.534 632.899 106
    },

    ["West_Ronfaure"] =
    {
        { -562.977, -50.167,  366.654,  26 }, -- (F-5)  !pos -562.977 -50.167 366.654 100
        { -456.809, -31.273,   32.595, 238 }, -- (G-8)  !pos -456.809 -31.273 32.595 100
        { -665.854, -27.920,   20.375, 253 }, -- (F-8)  !pos -665.854 -27.920 20.375 100
        { -397.491, -20.000, -157.320,  93 }, -- (G-9)  !pos -397.491 -20.000 -157.320 100
        {  -37.846, -10.701, -348.439,  95 }, -- (J-10) !pos -37.846 -10.701 -348.439 100
        { -201.060,   0.000, -522.925, 136 }, -- (I-11) !pos -201.060 0.000 -522.925 100
    },

    ["East_Sarutabaruta"] =
    {
        {  194.359, -17.040, -418.233, 202 }, -- (I-11) !pos 194.359 -17.040 -418.233 116
        {  473.876,   7.728, -122.559, 118 }, -- (K-9)  !pos 473.876 7.728 -122.559 116
        { -342.994,  -0.505,  -89.034, 249 }, -- (F-9)  !pos -342.994 -0.505 -89.034 116
        { -237.601,  -8.000,  317.016,  64 }, -- (J-6)  !pos -237.601 -8.000 317.016 116
        { -270.909, -27.370,  642.212,  22 }, -- (F-4)  !pos -270.909 -27.370 642.212 116
        {   46.228, -36.000,  561.358,  43 }, -- (H-5)  !pos 46.228 -36.000 561.358 116
    },
}

local treeID =
{
    ["East_Ronfaure"]     = 1,
    ["West_Ronfaure"]     = 1,
    ["North_Gustaberg"]   = 2,
    ["South_Gustaberg"]   = 2,
    ["West_Sarutabaruta"] = 3,
    ["East_Sarutabaruta"] = 3,
}

local trees =
{
    86,  -- San d'Orian Tree
    115, -- Bastokan Tree
    116, -- Windurstian Tree
    138, -- Jeunoan Tree
}

local gifts =
{
    {  25,  155 }, -- ( 25 Treants) Dream Stocking
    {  50,  140 }, -- ( 50 Treants) Dream Platter
    {  75,  141 }, -- ( 75 Treants) Dream Coffer
    { 100, 3693 }, -- (100 Treants) Lamb Carving
    { 200, 3750 }, -- (200 Treants) Qiqirn Sack
    { 300, 3584 }, -- (300 Treants) Panetiere
}

local giftboxes =
{
    ["Windurst_Waters"] =
    {
        { -39.009, -5.000, 80.640, 228 }, -- (G-8) !pos -39.009 -5.000 80.640 238
        { -41.229, -5.000, 80.577, 154 },
        { -40.111, -5.000, 78.286,  66 },
        { -40.224, -5.000, 82.315, 194 },
        { -40.269, -5.000, 79.699, 194 },
        { -37.816, -5.000, 78.877,  20 },
    },

    ["Bastok_Mines"] =
    {
        {  3.200, 0.000, -87.344,  26 }, -- (I-9) !pos 3.200 0.000 -87.344 234
        {  3.985, 0.000, -84.078,   3 },
        {  2.384, 0.000, -81.345, 231 },
        { -2.833, 0.000, -81.383, 161 },
        { -2.794, 0.000, -86.410,  97 },
        {  0.005, 0.000, -87.637,  62 },
    },

    ["Southern_San_dOria"] =
    {
        { -0.825, -3.000, -30.526, 103 }, -- (I-9) !pos -0.825 -3.000 -30.526 230
        { -0.104, -3.000, -29.450, 193 },
        {  0.780, -3.000, -30.458,  20 },
        {  2.186, -3.000, -29.270, 231 },
        { -2.073, -3.000, -29.440, 109 },
        {  0.134, -3.000, -32.050,  59 },
    },
}

local decorations =
{
    ["Bastok_Mines"] =
    {
        { 1240, -0.1938, 0.5, -84.0268, 0 }, -- canopy
        { 1241, -0.1938, 0.00, -84.0268, 0 }, -- tree
        { 1242, 23.3901, 0.00, -65.2250, 0 }, -- smallTree1
        { 1242, -23.3901, 0.00, -65.2250, 0 }, -- smallTree2
        { 1242, -18.0587, -3.00, -58.1178, 0 }, -- smallTree3
        { 1242, 18.0587, -3.00, -58.1178, 0 }, -- smallTree4
        { 1242, 44.3039, 0, -47.8915, 0 }, -- smallTree5
        { 1242, 63.6522, 0, -94.4761, 0 }, -- smallTree6
        { 1242, 74.1042, 0, -84.4704, 0 }, -- smallTree6
        { 1242, -5.8772, 0, -69.4263, 0 }, -- smallTree7
        { 1242, -5.9314, 0, -69.2842, 0 }, -- smallTree8
        {  764, 103.1067, 0.9944, -72.0866, 0 }, -- mediumTree1
        { 1273, 0.0487, -3.00, -57.4482, 0 }, -- streetlight1
        { 1273, 8.4837, -3.00, -56.8379, 0 }, -- streetlight2
        { 1273, -8.4837, -3.00, -56.8379, 0 }, -- streetlight3
        { 1273, -8.6930, -3.00, -64.7668, 0 }, -- streetlight4
        { 1273, -21.9640, -1.00, -126.8323, 0 }, -- streetlight4
        { 1273, -9.9605, -1.00, -126.8323, 0 }, -- streetlight5
        { 1273, 8.6930, -3.00, -64.7668, 0 }, -- streetlight5
        { 1273, 82.1272, 0, -65.4064, 0 }, -- streetlight6
        { 1273, 82.1272, 0, -78.7164, 0 }, -- streetlight7
        { 1273, 87.1380, 0.9944, -60.1172, 0 }, -- streetlight8
        { 1273, 87.1380, 0.9944, -84.1575, 0 }, -- streetlight9
        {  736, -12.0785, 0.0000, -95.9336, 90 }, -- kadomatsu1
        {  736, 12.0785, 0.0000, -95.9336, 20 }, -- kadomatsu2
        {  736, 12.0785, 0.0000, -71.9957, -45 }, -- kadomatsu3
        {  736, -12.0785, 0.0000, -71.9957, -90 }, -- kadomatsu4
        {  766, 45.0891, 0.00, -83.7432, 0 }, -- newYearShrub1
        {  766, 35.1695, 0.00, -83.7432, 0 }, -- newYearShrub2
        {  766, 1.3298, -1.00, -127.9548, 0 }, -- newYearShrub3
        {  766, -33.1322, -1.00, -127.9548, 0 }, -- newYearShrub4
        {  766, 61.1790, 0, -62.9457, 0 }, -- newYearShrub5
        {  766, 71.1627, 0, -62.9457, 0 }, -- newYearShrub6
        {  766, 118.7277, 0, -69.0521, 0 }, -- newYearShrub7
        {  766, 118.7277, 0, -74.9729, 0 }, -- newYearShrub8
        {  768, 50.4685, 0.8720, -91.0164, 0 }, -- skeletonTree
        {  767, -40.5901, 0, -73.8390,  }, -- twoTrees
    },
    ["Southern_San_dOria"] =
    {
        { 765, 119.6729, 0.0000, 87.3900, 30 }, -- StarlightCeleb31
        { 1243, 0.4057, 0.00, 24.9993, 193 }, -- doubletree1
        { 1242, 155.2721, -2.0000, 162.6213, 59 }, -- babytree1
        { 1242, 162.9608, -2.0000, 155.8975, 158 }, -- babytree2
        { 1242, 156.1904, -2.0000, 131.3109, 245 }, -- babytree3
        { 1242, 162.9013, -2.0000, 123.8494, 210 }, -- babytree4
        { 1242, 152.3975, -2.0000, 112.0942, 142 }, -- babytree5
        { 1242, 143.9548, -2.0000, 120.2859, 101 }, -- babytree6
        { 1242, 130.9028, -2.2000, 115.0713, 250 }, -- babytree7
        { 1242, 146.1112, -2.2000, 97.9584, 192 }, -- babytree8
        { 766, 151.7581, -2.0000, 149.4076, 13 }, -- balltree1
        { 766, 169.9008, -2.0000, 129.5512, 23 }, -- balltree2
        { 766, 151.3943, -2.0000, 101.1043, 172 }, -- balltree3
        { 766, 133.3308, -2.0000, 118.9150, 23 }, -- balltree4
        { 766, 106.7165, 1.0000, 110.4071, 76 }, -- balltree5
        { 766, 93.8941, 1.0000, 97.7647, 106 }, -- balltree6
        { 768, 137.1933, 0.0000, 80.9723, 192 }, -- sticktree1
        { 1242, 131.1598, 0.0000, 64.1406, 211 }, -- babytree9
        { 1242, 154.3784, 0.0000, 50.2646, 203 }, -- babytree10
        { 768, 152.2039, 0.0000, 34.2839, 125 }, -- sticktree2
        { 1242, 137.0035, 0.0000, 13.8518, 173 }, -- babytree10
        { 1242, 119.4911, 2.0000, 17.5109, 50 }, -- babytree11
        { 1242, 119.3114, 2.0000, 10.2657, 234 }, -- babytree12
        { 766, 113.2405, 2.0000, 10.1227, 229 }, -- balltree7
        { 766, 112.9329, 2.0000, 17.9826, 114 }, -- balltree8
        { 1242, 105.7935, 4.0000, 60.0139, 35 }, -- babytree13
        { 768, 93.6888, 4.0000, 47.9103, 78 }, -- sticktree3
        { 766, 93.3359, 4.0000, 61.7121, 70 }, -- balltree10
        { 768, 102.1748, 4.0000, 40.9274, 121 }, -- sticktree4
        { 768, 91.3439, 4.0000, 23.6167, 101 }, -- sticktree5
        { 1245, 93.1564, 2.0000, 2.2200, 24 }, -- lightsticktree1
        { 1245, 82.2983, 2.0000, 13.1442, 96 }, -- lightsticktree2
        { 1243, 98.6304, 1.0000, -42.6780, 32 }, -- doubletree2
        { 1242, 103.0865, 4.0000, 24.2808, 116 }, -- babytree14
        { 766, 69.1980, 2.0000, -5.1459, 13 }, -- balltree11
        { 1242, 98.2672, 1.0000, -51.2460, 154 }, -- babytree15
        { 1242, 107.0176, 1.0000, -42.4312, 215 }, -- babytree16
        { 1245, 115.2921, 1.0000, -28.4851, 14 }, -- bigsticktree1
        { 1245, 83.4468, 1.0000, -61.0037, 140 }, -- bigsticktree2
        { 1245, 75.2472, 2.0000, -42.4589, 150 }, -- bigsticktree3
        { 1245, 98.9895, 2.0000, -19.1048, 74 }, -- bigsticktree4
        { 766, 80.0945, 2.0000, -32.1865, 128 }, -- balltree12
        { 766, 87.7558, 2.0000, -23.7318, 113 }, -- balltree13
        --working below
        { 1245, 55.0764, 2.0000, -28.7028, 156 }, -- bigsticktree5
        { 1245, 55.9577, 2.0000, -12.4291, 223 }, -- bigsticktree6
        { 1242, 44.5582, 2.0000, -8.3803, 44 }, -- babytree17
        { 1243, 0.3678, 2.0000, -7.7877, 65 }, -- bigdoubletree
        { 1245, -0.0403, -3.0000, -30.1518, 192 }, -- ahtoptree
        { 1244, -22.4517, 2.1000, 6.1705, 59 }, -- courttree1
        { 1244, 22.2798, 2.1000, 5.7971, 115 }, -- courttree2
        { 1242, -45.4796, 2.0000, -9.5471, 48 }, -- babytree18
        { 1243, -95.3600, 1.0000, -39.5266, 97 }, -- sparks1
        { 1244, -109.5766, 2.0000, -19.7092, 84 }, -- sparks2
        { 1244, -76.9359, 2.0000, -46.5288, 181 }, -- sparks3
        { 768, -58.0030, 2.0000, -12.5504, 104 }, -- regionven1
        { 766, -69.5827, 2.0000, -5.5240, 238 }, -- regionven2
        { 1242, -78.1815, 2.0000, 2.7377, 114 }, -- regionven3
        { 1244, -95.3141, -2.0000, 22.4361, 68 }, -- regionven4
        -------------
        { 1245, 98.9895, 2.0000, -19.1048, 74 }, -- bigsticktree4
        { 766, 80.0945, 2.0000, -32.1865, 128 }, -- balltree12
        { 766, 87.7558, 2.0000, -23.7318, 113 }, -- balltree13
    },

    ["Windurst_Waters"] =
    {
        { 1247, -56.8214, -10.7500, 95.2693, 0 }, -- canopy1
        { 1247, -23.1692, -10.7500, 95.2693, 0 }, -- canopy2
        { 763, -40.1371, -5, 80.0112, 0 }, -- canopy2
        { 766, -35.1772, -5, 101.8268, 0 }, -- newYearShrub1
        { 766, -44.8707, -5, 101.8268, 0 }, -- newYearShrub2
        { 766, -44.8707, -5, 108.3983, 0 }, -- newYearShrub3
        { 766, -35.1772, -5, 108.3983, 0 }, -- newYearShrub4
        { 766, -62.5534, -5, 83.5578, 0 }, -- newYearShrub5
        { 766, -58.2720, -3.5, 67.1799, 0 }, -- newYearShrub6
        { 766, -52.9816, -3.5, 61.6787, 0 }, -- newYearShrub7
        { 766, -115.1555, -2, 34.1878, 0 }, -- newYearShrub8
        { 766, -88.8935, -2, 45.7852, 0 }, -- newYearShrub9
        { 766, -94.0420, -2, 62.8659, 0 }, -- newYearShrub10
        { 766, -72.0016, -5, 101.5145, 0 }, -- newYearShrub11
        { 766, -8.5680, -1, 69.2314, 0 }, -- newYearShrub12
        { 766, -3.03333, -1, 17.2329, 0 }, -- newYearShrub13
        { 766, -12, -1, 17.2173, 0 }, -- newYearShrub14
        { 766, -62.3252, -3.5, 39.3378, 0 }, -- newYearShrub15
        { 766, -54.4763, -3.5, 35.6573, 0 }, -- newYearShrub16
        { 766, -76.4078, -3.5, 22.2108, 0 }, -- newYearShrub17
        { 766, -61.1850, -3.5, 0.7307, 0 }, -- newYearShrub18
        { 1248, -17.3691, -5, 83.1543, 206 }, -- palmTree1
        { 1248, -31.3175, -3.5, 59.4233, 181 }, -- palmTree2
        { 1248, -78.9725, -3.5, 54.4054, 169 }, -- palmTree3
        { 1248, -85.4278, -3.5, 70.7953, 39 }, -- palmTree4
        { 1248, -47.7215, -3.5, 59.9022, 249 }, -- palmTree5
        { 1248, -30.6700, -3.5, 42.1922, 115 }, -- palmTree6
        { 1248, 2.5418, -1, 58.6250, 62 }, -- palmTree7
        { 1248, 1.9200, -1, 41.8816, 181 }, -- palmTree8
        { 1248, -47.2811, -3.5, 43.7694, 254 }, -- palmTree9
        { 768, -103.7146, -2, 46.6989, 57 }, -- skeletonTree1
        { 735, -114.8496, -2, 53.9881, 49 }, -- komadatsu1
        { 735, -101.7088, -2, 58.8969, 49 }, -- komadatsu2
        { 1390, -38.8415, -3.5, 33.2178, 71 }, -- holidayTree
        { 1390, -13.9731, -1, 29.2651, 254 }, -- holidayTree2
        { 1304, -30.1915, -5, 78.7058, 143 }, -- dancingTaruM
        { 1305, -31.0215, -5, 77.0948, 168 }, -- dancingTaruF
    },
}

-----------------------------------
-- Entity Functions
-----------------------------------
local function delaySendMenu(player, menu)
    player:timer(100, function(playerArg)
        playerArg:customMenu(menu)
    end)
end

local function onTradeNPC(player, npc, trade, index)
    if
        player:getCharVar("[EVENT]STARLIGHT_DELIVER") == index and
        npcUtil.tradeHasExactly(trade, 1742)
    then
        player:tradeComplete()
        cexi.util.dialog(player, { "Wow, thank you! This is just what I wanted!" }, npc:getPacketName(), { npc = npc })

        player:printToPlayer(string.format("%s gains 1 starlight cheer.", player:getName()), xi.msg.channel.SYSTEM_3)
        player:incrementCharVar(settings.pointsVar, 1)
        player:setCharVar("[EVENT]STARLIGHT_DELIVERY", 2)
        
    end
end

local function onTriggerNPC(player, npc)
    cexi.util.dialog(player, { responses[math.random(1, #responses)] }, npc:getPacketName(), { npc = npc })
end

local function onTriggerPresent(player, npc)
    if player:getLocalVar("[EVENT]STARLIGHT_BUSY") > 0 then
        return
    end

    if player:getCharVar("[EVENT]STARLIGHT_COURIER") ~= 1 then
        player:printToPlayer("You see nothing out of the ordinary.", xi.msg.channel.NS_SAY)
        return
    end

    player:setLocalVar("[EVENT]STARLIGHT_BUSY", 1)
    player:setAnimation(48)

    player:timer(3000, function()
        player:setLocalVar("[EVENT]STARLIGHT_BUSY", 0)
        player:setAnimation(0)

        player:printToPlayer("You found a lost present!", xi.msg.channel.NS_SAY)
        player:printToPlayer(string.format("%s gains 1 starlight cheer.", player:getName()), xi.msg.channel.SYSTEM_3)
        player:incrementCharVar(settings.pointsVar, 1)
        player:setCharVar("[EVENT]STARLIGHT_COURIER", 2)

        local zoneName = player:getZoneName()
        local selected = presents[zoneName][math.random(1, #presents[zoneName])]

        npc:setStatus(xi.status.DISAPPEAR)

        npc:timer(3000, function()
            npc:setPos(selected[1], selected[2], selected[3])
            npc:setStatus(xi.status.NORMAL)
        end)
    end)
end

local function getIndex(itemName)
    for itemIndex, item in pairs(cheerRewards) do
        if item[1] == itemName then
            return itemIndex
        end
    end
end

local function selectReward(player, npc, item)
    delaySendMenu(player, {
        title   = string.format("Claim your %s?", item[1]),
        options =
        {
            {
                "Yes",
                function()
                    local points = player:getCharVar(settings.pointsVar)

                    if points < item[3] then
                        npc:facePlayer(player, true)
                        player:printToPlayer("Smilebringer : Hoho~ You haven't been good enough for that one yet!", xi.msg.channel.NS_SAY)
                        return
                    end

                    if npcUtil.giveItem(player, item[2]) then
                        local milestones = player:getCharVar(settings.milestoneVar)
                        local itemIndex  = getIndex(item[1])
                        local result     = utils.mask.setBit(milestones, itemIndex, true)

                        player:setCharVar(settings.milestoneVar, result)
                    end
                end,
            },
            {
                "Not yet",
                function()
                end,
            },
        },
    })
end

local function onTriggerSmilebringer(player, npc)
    local helperPoints = player:getCharVar(settings.pointsVar)
    local points       = string.format("Starlight Cheer (%u):", helperPoints)

    npc:facePlayer(player, true)

    cexi.util.dialog(player, { "Help us bring smiles to the children of Vana'diel, so we can all share in the festivities!" }, npc:getPacketName(), { npc = npc })
    cexi.util.simpleShop(player, npc, cheerRewards, selectReward, points, 0, { milestone = settings.milestoneVar })
end

local function rewardOrRandom(player, nq, hq, progVar, resetVar)
    if not player:hasItem(nq) then
        if npcUtil.giveItem(player, nq) then
            player:setCharVar(progVar, 0)
            player:setCharVar(resetVar, JstMidnight())
        end
    elseif not player:hasItem(hq) then
        if npcUtil.giveItem(player, hq) then
            player:setCharVar(progVar, 0)
            player:setCharVar(resetVar, JstMidnight())
        end
    else
        local selected = randomRewards[math.random(1, #randomRewards)]

        if type(selected) == "table" then
            if npcUtil.giveItem(player, { { selected[1], math.random(selected[2][1], selected[2][2]) } }) then
                player:setCharVar(progVar, 0)
                player:setCharVar(resetVar, JstMidnight())
            end
        else
            if npcUtil.giveItem(player, selected) then
                player:setCharVar(progVar, 0)
                player:setCharVar(resetVar, JstMidnight())
            end
        end
    end
end

local function onTriggerDelivery(player, npc, index)
    local prog     = player:getCharVar("[EVENT]STARLIGHT_DELIVERY")
    local zoneID   = player:getZoneID()
    local assigned = kids[zoneID][npcAreas[zoneID]]
    local areaName = string.gsub(npcAreas[zoneID], "_", " ")

    if prog == 0 then
        if os.time() < player:getCharVar("[EVENT]STARLIGHT_DELIVERY_NEXT") then
            cexi.util.dialog(player, { "Sorry, we're not ready to deliver another present yet!" }, npc:getPacketName(), { npc = npc })
            return
        end

        local roll = math.random(1, 4)
        local tbl  =
        {
            "We're running late this year and we need the help of adventurers to deliver all these presents!",
            string.format(" Please help us out by delivering this present to %s in %s!", assigned[roll][1], areaName),
        }

        cexi.util.dialog(player, tbl, npc:getPacketName(), { npc = npc })

        local delay = cexi.util.dialogDelay(tbl)

        player:timer(delay, function()
            if npcUtil.giveItem(player, 1742) then
                player:setCharVar("[EVENT]STARLIGHT_DELIVERY", 1)
                player:setCharVar("[EVENT]STARLIGHT_DELIVER", roll)
            end
        end)

    elseif prog == 1 then
        local roll = player:getCharVar("[EVENT]STARLIGHT_DELIVER")

        cexi.util.dialog(player, {
            string.format(" Please help us out by delivering the present to %s in %s!", assigned[roll][1], areaName),
        }, npc:getPacketName(), { npc = npc })

    elseif prog == 2 then
        cexi.util.dialog(player, { "Thank you for delivering the present. Here's something for you!" }, npc:getPacketName(), { npc = npc })
        rewardOrRandom(player, 14519, 14520, "[EVENT]STARLIGHT_DELIVERY", "[EVENT]STARLIGHT_DELIVERY_NEXT")
    end
end

local function onTriggerCourier(player, npc, index)
    local prog     = player:getCharVar("[EVENT]STARLIGHT_COURIER")
    local zoneID    = player:getZoneID()
    local areaName = string.gsub(presentAreas[zoneID], "_", " ")

    if prog == 0 then
        if os.time() < player:getCharVar("[EVENT]STARLIGHT_COURIER_NEXT") then
            cexi.util.dialog(player, { "Thanks to you, all of today's presents are accounted for!" }, npc:getPacketName(), { npc = npc })
            return
        end

        cexi.util.dialog(player, {
            { emote = xi.emote.PANIC },
            string.format("On the way here, we lost one of the presents over in %s! Please help us find it!", areaName),
        }, npc:getPacketName(), { npc = npc })

        player:setCharVar("[EVENT]STARLIGHT_COURIER", 1)

    elseif prog == 1 then
        cexi.util.dialog(player, {
            string.format("Please help us out by finding the lost present in %s!", areaName),
        }, npc:getPacketName(), { npc = npc })

    elseif prog == 2 then
        cexi.util.dialog(player, { "Thank you for finding the present. Here's something for you!" }, npc:getPacketName(), { npc = npc })
        rewardOrRandom(player, 15752, 15753, "[EVENT]STARLIGHT_COURIER", "[EVENT]STARLIGHT_COURIER_NEXT")
    end
end

local function onTriggerWrapping(player, npc, index)
    local prog     = player:getCharVar("[EVENT]STARLIGHT_WRAPPING")
    local zoneID    = player:getZoneID()
    local areaName = string.gsub(presentAreas[zoneID], "_", " ")

    if prog == 0 then
        if os.time() < player:getCharVar("[EVENT]STARLIGHT_WRAPPING_NEXT") then
            cexi.util.dialog(player, { "Thanks to you, all of today's presents are now being wrapped!" }, npc:getPacketName(), { npc = npc })
            return
        end

        cexi.util.dialog(player, {
            { emote = xi.emote.STAGGER },
            string.format("You see those boxes out there in %s? Open them and try to find the present so I can wrap it!", areaName),
        }, npc:getPacketName(), { npc = npc })

        player:setCharVar("[EVENT]STARLIGHT_WRAPPING", 1)

    elseif prog == 1 then
        cexi.util.dialog(player, {
            string.format("Open the boxes in %s and try to find the present!", areaName),
        }, npc:getPacketName(), { npc = npc })

    elseif prog == 2 then
        cexi.util.dialog(player, { "Thank you for finding the present. Here's something for you!" }, npc:getPacketName(), { npc = npc })
        rewardOrRandom(player, 15178, 15179, "[EVENT]STARLIGHT_WRAPPING", "[EVENT]STARLIGHT_WRAPPING_NEXT")
    end
end

local function boxReward(player, npc, boxNum)
    local counter = GetServerVariable("[EVENT]STARLIGHT_COUNTER")

    if gifts[boxNum] == nil then
        print(string.format("[EVENT]STARLIGHT: Box %u isn't assigned to a gift item", boxNum))
        return
    end

    local claimed = player:getCharVar("[EVENT]STARLIGHT_CLAIMED")

    if counter >= gifts[boxNum][1] then
        if utils.mask.getBit(claimed, boxNum) then
            player:printToPlayer(string.format("%s : You've already claimed this reward!", npc:getPacketName()), xi.msg.channel.NS_SAY)
        else
            if npcUtil.giveItem(player, gifts[boxNum][2]) then
                player:printToPlayer(string.format("Congratulations to the community for defeating %u Treants!", gifts[boxNum][1]), xi.msg.channel.NS_SAY)
                player:setCharVar("[EVENT]STARLIGHT_CLAIMED", utils.mask.setBit(claimed, boxNum, true))
            end
        end
    else
        player:printToPlayer(string.format("%s : This reward will unlock once the community defeats %u Treants!", npc:getPacketName(), gifts[boxNum][1]), xi.msg.channel.NS_SAY)
    end
end

local function onTriggerMoogle(player, npc)
    if player:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
        cexi.util.dialog(player, { "What are you doing back here so soon!?" }, npc:getPacketName(), { npc = npc })

        delaySendMenu(player, {
            title   = "Continue the fight?",
            options =
            {
                {
                    "Keep going!",
                    function()
                    end,
                },
                {
                    "I give up!",
                    function()
                        player:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
                        player:setLocalVar("NO_TRUSTS", 0)
                    end,
                },
            },
        })

        return
    end

    local tbl =
    {
        "Terrible Twinkle Treants are threating to take over the Starlight Celebration!",
        " We need brave adventurers to come together and cut down these unruly trees!",
    }

    cexi.util.dialog(player, tbl, npc:getPacketName(), { npc = npc })

    local delay = cexi.util.dialogDelay(tbl)

    player:timer(delay, function()
        delaySendMenu(player, {
            title   = "Agree to help?",
            options =
            {
                {
                    "No way!",
                    function()
                    end,
                },
                {
                    "I'm ready",
                    function()
                        player:addStatusEffectEx(
                            xi.effect.LEVEL_RESTRICTION,
                            xi.effect.LEVEL_RESTRICTION,
                            20,
                            0,
                            0,
                            0,
                            0,
                            0,
                            xi.effectFlag.ON_ZONE + xi.effectFlag.CONFRONTATION
                        )
                        player:setLocalVar("NO_TRUSTS", 1)
                    end,
                },
            },
        })
    end)
end

local function removeNPC(npc)
    -- This will send the packet to ensure NPC disappears for player
    npc:setStatus(xi.status.INVISIBLE)

    npc:timer(500, function(npcArg)
        if npcArg ~= nil then
            -- This will delete DE server side on zone tick
            npcArg:setStatus(xi.status.DISAPPEAR)
        end
    end)
end

local function onBoxDeath(mob, player, optParams)
    if player:getCharVar("[EVENT]STARLIGHT_WRAPPING") == 1 then
        if math.random(0, 10) < 3 then
            player:printToPlayer("You found the present!", xi.msg.channel.NS_SAY)
            player:printToPlayer(string.format("%s gains 1 starlight cheer.", player:getName()), xi.msg.channel.SYSTEM_3)
            player:incrementCharVar(settings.pointsVar, 1)
            player:setCharVar("[EVENT]STARLIGHT_WRAPPING", 2)
        else
            player:printToPlayer("You find nothing of interest inside the box.", xi.msg.channel.NS_SAY)
        end
    end
end

local function onTreantFight(mob, target)
    if mob:getHPP() < 2 then
        local list     = mob:getEnmityList()
        local counter  = GetServerVariable("[EVENT]STARLIGHT_COUNTER")

        SetServerVariable("[EVENT]STARLIGHT_COUNTER", counter + 1)

        mob:setUnkillable(false)
        mob:setHP(0)

        local zoneName = mob:getZoneName()
        local treant = treants[zoneName]
        local pos = treant[math.random(1, #treant)]

        -- Don't move treant after zone restart
        -- mob:setSpawn(pos[1], pos[2], pos[3], pos[4])

        local announce = false

        if (counter + 1) % 25 == 0 then
            announce = true
        end

        for _, p in pairs(list) do
            local player = p.entity

            if player:hasStatusEffect(xi.effect.LEVEL_RESTRICTION) then
                player:delStatusEffect(xi.effect.LEVEL_RESTRICTION)
                player:setLocalVar("[EVENT]STARLIGHT_GIFT", 1)
                player:setLocalVar("NO_TRUSTS", 0)
            end

            -- Has to be done here because printToArea only works on player entity and target can't be guaranteed
            if announce then
                player:printToArea(string.format("\129\154 %u Twinkle Treants have been defeated! \129\154", counter + 1), xi.msg.channel.SYSTEM_3, 0, "") -- Sends announcement via ZMQ to all processes and zones
                announce = false
            end
        end
    end
end

local function onTreantDeath(mob, player, optParams)
    if mob:getLocalVar("GIFT_PLACED") == 0 then
        mob:setLocalVar("GIFT_PLACED", 1)

        local zone = mob:getZone()

        local dynamicEntity =
        {
            name        = "Glittering Gift",
            objtype     = xi.objType.NPC,
            look        = 1382,
            x           = mob:getXPos(),
            y           = mob:getYPos(),
            z           = mob:getZPos(),
            rotation    = mob:getRotPos(),
        
            onTrigger   = function(player)
                if
                    player:getLocalVar("[EVENT]STARLIGHT_GIFT") > 0 and
                    player:getLocalVar("[EVENT]STARLIGHT_BUSY") == 0
                then
                    player:setLocalVar("[EVENT]STARLIGHT_BUSY", 1)
                    player:setAnimation(48)

                    player:timer(3000, function()
                        player:setLocalVar("[EVENT]STARLIGHT_BUSY", 0)
                        player:setAnimation(0)

                        if npcUtil.giveItem(player, 5269) then
                            local zoneName = mob:getZoneName()

                            player:setCharVar("[EVENT]STARLIGHT_TREE", treeID[zoneName])
                            player:setLocalVar("[EVENT]STARLIGHT_GIFT", 0)

                            local areaVar = string.format("[EVENT]STARLIGHT_TREANT_%u", player:getZoneID())

                            if os.time() > player:getCharVar(areaVar) then
                                player:printToPlayer(string.format("%s gains 1 starlight cheer.", player:getName()), xi.msg.channel.SYSTEM_3)
                                player:incrementCharVar(settings.pointsVar, 1)
                                player:setCharVar(areaVar, JstMidnight())
                            end
                        end
                    end)
                end
            end,

            releaseIdOnDisappear = true,
        }
        
        local npc = zone:insertDynamicEntity(dynamicEntity)
        
        npc:hideName(true)
        npc:hideHP(true)
        
        npc:timer(30000, function(npcArg)
            if npcArg ~= nil then
                removeNPC(npcArg)
            end
        end)
    end
end

xi.module.ensureTable("xi.items.special_present")

m:addOverride("xi.items.special_present.onItemCheck", function(target)
    local result = 0

    if target:getFreeSlotsCount() == 0 then
        result = xi.msg.basic.ITEM_NO_USE_INVENTORY
    end

    return result
end)

m:addOverride("xi.items.special_present.onItemUse", function(target)
    local treeArea = target:getCharVar("[EVENT]STARLIGHT_TREE")

    if not target:hasItem(trees[treeArea]) then
        -- Give nation tree if player doesn't have
        npcUtil.giveItem(target, trees[treeArea])
    else
        if
            target:hasItem(trees[1]) and
            target:hasItem(trees[2]) and
            target:hasItem(trees[3]) and
            not target:hasItem(trees[4])
        then
            -- Give Jeunoan tree if player has all 3 trees
            npcUtil.giveItem(target, trees[4])
        else
            -- Give random reward (random quantity range) if player has both trees
            local selected = treeRewards[math.random(1, #treeRewards)]

            if type(selected) == "table" then
                npcUtil.giveItem(target, { { selected[1], math.random(selected[2][1], selected[2][2]) } })
            else
                npcUtil.giveItem(target, selected)
            end
        end
    end
end)

-----------------------------------
-- Create Entities
-----------------------------------
local de = {}

for _, zoneInfo in pairs(kids) do
    for areaName, areaKids in pairs(zoneInfo) do
        de[areaName] = de[areaName] or {}

        for index, kid in pairs(areaKids) do
            table.insert(de[areaName], {
                objtype   = xi.objType.NPC,
                name      = kid[1],
                look      = kid[2],
                x         = kid[3][1],
                y         = kid[3][2],
                z         = kid[3][3],
                rotation  = kid[3][4],
                widescan  = 1,
                onTrigger = onTriggerNPC,
                onTrade   = function(player, npc, trade)
                    onTradeNPC(player, npc, trade, index)
                end,
            })
        end
    end
end

for areaName, areaPresents in pairs(presents) do
    de[areaName] = de[areaName] or {}

    local selected = areaPresents[math.random(1, #areaPresents)]

    table.insert(de[areaName], {
        objtype   = xi.objType.NPC,
        name      = "Lost Present",
        look      = 1382,
        x         = selected[1],
        y         = selected[2],
        z         = selected[3],
        rotation  = 0,
        onTrigger = onTriggerPresent,
    })
end

for areaName, pos in pairs(smilebringers) do
    de[areaName] = de[areaName] or {}

    table.insert(de[areaName], {
        objtype   = xi.objType.NPC,
        name      = "Smilebringer",
        look      = cexi.util.look({
            race = xi.race.GALKA,
            face = cexi.face.A1,
            head = 140, -- dream robe set
            body = 140, -- dream robe set
            hand = 122, -- dream robe set
            legs = 244, -- dream robe set
            feet = 140, -- dream robe set
        }),
        x         = pos[1],
        y         = pos[2],
        z         = pos[3],
        rotation  = pos[4],
        onTrigger = onTriggerSmilebringer,
    })
end

for areaName, areaHelper in pairs(smilehelpers) do
    de[areaName] = de[areaName] or {}

    table.insert(de[areaName], {
        objtype   = xi.objType.NPC,
        name      = "Delivery Helper",
        look      = cexi.util.look({
            race = xi.race.TARU_F,
            face = cexi.face.A1,
            head = 140, -- dream robe set
            body = 140, -- dream robe set
            hand = 122, -- dream robe set
            legs = 244, -- dream robe set
            feet = 140, -- dream robe set
        }),
        x         = areaHelper.delivery[1],
        y         = areaHelper.delivery[2],
        z         = areaHelper.delivery[3],
        rotation  = areaHelper.delivery[4],
        onTrigger = function(player, npc)
            onTriggerDelivery(player, npc)
        end,
    })

    table.insert(de[areaName], {
        objtype   = xi.objType.NPC,
        name      = "Courier Helper",
        look      = cexi.util.look({
            race = xi.race.TARU_F,
            face = cexi.face.B5,
            head = 140, -- dream robe set
            body = 140, -- dream robe set
            hand = 122, -- dream robe set
            legs = 244, -- dream robe set
            feet = 140, -- dream robe set
        }),
        x         = areaHelper.courier[1],
        y         = areaHelper.courier[2],
        z         = areaHelper.courier[3],
        rotation  = areaHelper.courier[4],
        onTrigger = function(player, npc)
            onTriggerCourier(player, npc)
        end,
    })

    table.insert(de[areaName], {
        objtype   = xi.objType.NPC,
        name      = "Wrapping Helper",
        look      = cexi.util.look({
            race = xi.race.TARU_M,
            face = cexi.face.B5,
            head = 140, -- dream robe set
            body = 140, -- dream robe set
            hand = 122, -- dream robe set
            legs = 244, -- dream robe set
            feet = 140, -- dream robe set
        }),
        x         = areaHelper.wrapping[1],
        y         = areaHelper.wrapping[2],
        z         = areaHelper.wrapping[3],
        rotation  = areaHelper.wrapping[4],
        onTrigger = function(player, npc)
            onTriggerWrapping(player, npc)
        end,
    })
end

for areaName, areaBoxes in pairs(giftboxes) do
    de[areaName] = de[areaName] or {}

    for boxNum, boxPos in pairs(areaBoxes) do
        table.insert(de[areaName], {
            objtype   = xi.objType.NPC,
            name      = string.format("Mystery Gift #%u", boxNum),
            look      = 2331,
            x         = boxPos[1],
            y         = boxPos[2],
            z         = boxPos[3],
            rotation  = boxPos[4],
            hideName  = true,
            onTrigger = function(player, npc)
                boxReward(player, npc, boxNum)
            end,
        })
    end
end

for areaName, areaMoogles in pairs(moogles) do
    de[areaName] = de[areaName] or {}

    for index, moogle in pairs(areaMoogles) do
        table.insert(de[areaName], {
            objtype    = xi.objType.NPC,
            name       = string.format("Moogle #%u", index),
            packetName = "Moogle",
            look       = 82,
            x          = moogle[1],
            y          = moogle[2],
            z          = moogle[3],
            rotation   = moogle[4],
            onTrigger  = onTriggerMoogle,
        })
    end
end

for areaName, treant in pairs(treants) do
    m:addOverride(string.format("xi.zones.%s.Zone.onInitialize", areaName), function(zone)
        super(zone)

        local pos = treant[math.random(1, #treant)]

        local dynamicEntity =
        {
            name        = "Twinkle Treant",
            objtype     = xi.objType.MOB,
            look        = 389,
            groupZoneId = 104,
            groupId     = 15,
            flags       = 159,
            x           = pos[1],
            y           = pos[2],
            z           = pos[3],
            rotation    = pos[4],
            widescan    = 1,
            onMobDeath  = onTreantDeath,
        }

        local mobLevel   = GetServerVariable("[EVENT]STARLIGHT_TREANT_LV")
        local mobHealth  = GetServerVariable("[EVENT]STARLIGHT_TREANT_HP")
        local mobMagres  = GetServerVariable("[EVENT]STARLIGHT_TREANT_MR")
        local mobAttack  = GetServerVariable("[EVENT]STARLIGHT_TREANT_ATT")
        local mobDefense = GetServerVariable("[EVENT]STARLIGHT_TREANT_DEF")
        local mobPDT     = GetServerVariable("[EVENT]STARLIGHT_TREANT_PDT")

        if mobLevel == 0 then
            mobLevel = 18
        end

        if mobHealth == 0 then
            mobHealth = 2500
        end

        if mobMagres == 0 then
            mobMagres = 50
        end

        if mobAttack == 0 then
            mobAttack = -100
        end

        if mobDefense == 0 then
            mobDefense = 200
        end

        if mobPDT == 0 then
            mobPDT = 300
        end

        dynamicEntity.onMobInitialize = function(mob)
            mob:setMobMod(xi.mobMod.DETECTION,  0x08)
            mob:setMobMod(xi.mobMod.CHARMABLE,     0)
            mob:setMobMod(xi.mobMod.CHECK_AS_NM,   1)

            -- TODO: Need to reduce the damage this mob deals
            mob:setMod(xi.mod.STR, -50)
            mob:setMod(xi.mod.AGI, -50)
            mob:setMod(xi.mod.ATT, mobAttack)
            mob:setMod(xi.mod.DEFP, mobDefense)
            mob:setMod(xi.mod.HP, mobHealth)

            mob:setMod(xi.mod.SLASH_SDT,  mobPDT)
            mob:setMod(xi.mod.PIERCE_SDT, mobPDT)
            mob:setMod(xi.mod.IMPACT_SDT, mobPDT)
            mob:setMod(xi.mod.HTH_SDT,    mobPDT)

            -- TODO: Need to reduce magic damage taken even more
            for n = 1, #xi.magic.resistMod, 1 do
                mob:setMod(xi.magic.resistMod[n], mobMagres)
            end
        end

        dynamicEntity.onMobSpawn = function(mob)
            mob:setMobLevel(mobLevel)
            mob:setHP(mob:getMaxHP())
            mob:setMP(mob:getMaxMP())
            mob:updateHealth()
            mob:addStatusEffectEx(
                xi.effect.LEVEL_RESTRICTION,
                xi.effect.LEVEL_RESTRICTION,
                20,
                0,
                0,
                0,
                0,
                0,
                xi.effectFlag.ON_ZONE + xi.effectFlag.CONFRONTATION
            )

            mob:setClaimable(false)
            mob:setUnkillable(true)
        end

        dynamicEntity.onMobDeath = onTreantDeath
        dynamicEntity.onMobFight = onTreantFight

        local entity = zone:insertDynamicEntity(dynamicEntity)

        entity:setSpawn(pos[1], pos[2], pos[3], pos[4])
        entity:setRespawnTime(1200) -- 20 minutes
        entity:setDropID(0)

        entity:spawn()
        entity:setLocalVar("NO_CASKET", 1)
    end)
end

for areaName, boxInfo in pairs(boxes) do
    m:addOverride(string.format("xi.zones.%s.Zone.onInitialize", areaName), function(zone)
        super(zone)

        for _, pos in pairs(boxInfo) do
            local dynamicEntity =
            {
                name        = "Starlight Box",
                objtype     = xi.objType.MOB,
                look        = 2331,
                groupZoneId = 104,
                groupId     = 15,
                flags       = 159,
                x           = pos[1],
                y           = pos[2],
                z           = pos[3],
                rotation    = pos[4],
                widescan    = 1,
                onMobDeath  = onBoxDeath,
            }

            local mobLevel   = GetServerVariable("[EVENT]STARLIGHT_BOX_LV")
            local mobHealth  = GetServerVariable("[EVENT]STARLIGHT_BOX_HP")
            local mobMagres  = GetServerVariable("[EVENT]STARLIGHT_BOX_MR")
            local mobPDT     = GetServerVariable("[EVENT]STARLIGHT_BOX_PDT")

            if mobLevel == 0 then
                mobLevel = 10
            end

            if mobHealth == 0 then
                mobHealth = 0
            end

            if mobMagres == 0 then
                mobMagres = 50
            end

            if mobPDT == 0 then
                mobPDT = 500
            end

            dynamicEntity.onMobInitialize = function(mob)
                mob:setMobMod(xi.mobMod.DETECTION,  0x08)
                mob:setMobMod(xi.mobMod.CHARMABLE,     0)
                mob:setMobMod(xi.mobMod.CHECK_AS_NM,   1)
                mob:setMobMod(xi.mobMod.NO_MOVE,       1)
                mob:setMobMod(xi.mobMod.NO_DROPS,      1)

                mob:setMod(xi.mod.SLASH_SDT,     mobPDT)
                mob:setMod(xi.mod.PIERCE_SDT,    mobPDT)
                mob:setMod(xi.mod.IMPACT_SDT,    mobPDT)
                mob:setMod(xi.mod.HTH_SDT,       mobPDT)
                mob:setMod(xi.mod.HP,         mobHealth)

                -- TODO: Need to reduce magic damage taken even more
                for n = 1, #xi.magic.resistMod, 1 do
                    mob:setMod(xi.magic.resistMod[n], mobMagres)
                end
            end

            dynamicEntity.onMobSpawn = function(mob)
                mob:setMobLevel(mobLevel)
                mob:setHP(mob:getMaxHP())
                mob:setMP(mob:getMaxMP())
            end

            local entity = zone:insertDynamicEntity(dynamicEntity)

            entity:setSpawn(pos[1], pos[2], pos[3], pos[4])
            entity:setRespawnTime(math.random(30, 60)) -- 30-60 seconds
            entity:setDropID(0)

            entity:spawn()
            entity:setLocalVar("NO_CASKET", 1)
            entity:setBehaviour(bit.bor(entity:getBehaviour(), xi.behavior.NO_TURN))
            entity:setAutoAttackEnabled(false)
            entity:setMobAbilityEnabled(false)
        end
    end)
end

if settings.enabled then
    cexi.util.liveReload(m, de)

    -- Set up decorations
    for areaName, decs in pairs(decorations) do
        m:addOverride(string.format("xi.zones.%s.Zone.onInitialize", areaName), function(zone)
            super(zone)

            for _, decoration in pairs(decs) do
                local npc = zone:insertDynamicEntity({
                    objtype  = xi.objType.NPC,
                    name     = " ",
                    look     = decoration[1],
                    x        = decoration[2],
                    y        = decoration[3],
                    z        = decoration[4],
                    rotation = decoration[5],
                })

                npc:setUntargetable(true)
            end
        end)
    end
end

return m
