﻿/************************************************************************
* Crystal Warrior Mode
*************************************************************************
* Crystal Warriors can only party with other Crystal Warriors
* Trading, Delivery Box, Bazaar and Auction House are restricted
************************************************************************/

#include "map/utils/moduleutils.h"

#include "common/sql.h"
#include "map/utils/blueutils.h"
#include "map/utils/charutils.h"
#include "map/entities/battleentity.h"
#include "map/job_points.h"
#include "map/packets/char_jobs.h"
#include "map/packets/char_stats.h"
#include "map/packets/char_skills.h"
#include "map/packets/char_recast.h"
#include "map/packets/char_abilities.h"
#include "map/packets/char_update.h"
#include "map/packets/menu_merit.h"
#include "map/packets/monipulator1.h"
#include "map/packets/monipulator2.h"
#include "map/packets/char_sync.h"
#include "map/packets/message_combat.h"
#include "map/status_effect_container.h"
#include "map/status_effect.h"

class CrystalWarriorUtilModule : public CPPModule
{
    std::string CW_STATUS = "CHAR_TYPE"; // CharVar for Crystal Warrior status

    void OnInit() override
    {
        TracyZoneScoped;

        // Returns true if the player is a Crystal Warrior
        lua["CBaseEntity"]["isCrystalWarrior"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype == TYPE_PC)
            {
                auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

                return PChar->getCharVar(CW_STATUS) == 1 || PChar->getCharVar(CW_STATUS) == 2;
            }

            return false;
        };

        // Returns true if the player is an Unbreakable Crystal Warrior
        lua["CBaseEntity"]["isUnbreakable"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype == TYPE_PC)
            {
                auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

                return PChar->getCharVar(CW_STATUS) == 2;
            }

            return false;
        };

        // Get account type (ie. CW/Regular) of the player
        lua["CBaseEntity"]["getAccountType"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> uint32 {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            auto        query = "SELECT type FROM accounts INNER JOIN chars ON chars.accid = accounts.id WHERE chars.charid = %u;";

            if (sql->Query(query, PChar->id) != SQL_ERROR && sql->NumRows() > 0 && sql->NextRow() != SQL_ERROR)
            {
                return sql->GetUIntData(0);
            }
            else
            {
                return -1;
            }
        };

        lua["CBaseEntity"]["getTotalLevel"] = [](CLuaBaseEntity* PLuaBaseEntity) -> uint32 {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            uint32 total = 0;

            for (uint8 i = 0; i < MAX_JOBTYPE; i++)
            {
                total += PChar->jobs.job[i];
            }

            return total;
        };

        lua["CBaseEntity"]["getTotalCraft"] = [](CLuaBaseEntity* PLuaBaseEntity) -> uint32 {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            uint32 total = 0;

            for (uint8 i = SKILL_FISHING; i < SKILL_SYNERGY; i++)
            {
                total += PChar->GetSkill(i);
            }

            return total;
        };

        lua["CBaseEntity"]["recordDeath"] = [this](CLuaBaseEntity* PLuaBaseEntity, uint8 level, uint32 expLost) -> void {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            auto* PBattle = static_cast<CBattleEntity*>(PEntity);

            auto query = "INSERT INTO char_deaths SET charid = %u, charname = '%s', charjob = %u, charlevel = %u, charlost = %u, mobname = '%s', mobfamily = %u, moblevel = %u;";

            if (PBattle == nullptr)
            {
                return;
            }

            if (PBattle->PLastAttacker != nullptr)
            {
                auto* last = dynamic_cast<CMobEntity*>(PBattle->PLastAttacker);

                if (last != nullptr)
                {
                    sql->Query(query, PBattle->id, PBattle->getName(), PBattle->GetMJob(), level, expLost, last->getName(), last->m_SuperFamily, last->GetMLevel());
                }
            }
            else
            {
                sql->Query(query, PBattle->id, PBattle->getName(), PBattle->GetMJob(), level, expLost, "None", 0, 0);
            }
        };

        lua["CBaseEntity"]["getDeaths"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> sol::table {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();
            auto         table   = lua.create_table();

            if (PEntity->objtype != TYPE_PC)
            {
                return table;
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            auto        query = "SELECT charjob, charlevel FROM char_deaths WHERE charid = %u ORDER BY charlost DESC LIMIT 5;";

            if (sql->Query(query, PChar->id) == SQL_ERROR)
            {
                return table;
            }

            while (sql->NextRow() == SQL_SUCCESS)
            {
                auto row = lua.create_table();
                row.add(sql->GetUIntData(0), sql->GetUIntData(1));
                table.add(row);
            }

            return table;
        };

        lua["CBaseEntity"]["getLastAttacker"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> const std::string {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return "";
            }

            auto*       PBattle = static_cast<CBattleEntity*>(PEntity);

            if (PBattle != nullptr && PBattle->PLastAttacker != nullptr)
            {
                return PBattle->PLastAttacker->getName();
            }

            return "";
        };

        // Add reset function to lua_baseentity
        lua["CBaseEntity"]["resetCrystalWarrior"] = [](CLuaBaseEntity* PLuaBaseEntity) -> uint32 {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            if (PChar)
            {
                // Remove dead party member to prevent level sync dropping to Lv1
                if (PChar->StatusEffectContainer->HasStatusEffect(EFFECT_LEVEL_SYNC))
                {
                    if (PChar->PParty != nullptr)
                    {
                        PChar->PParty->RemoveMember(PChar);
                    }
                }

                // Record lost EXP from current level
                uint32 lostEXP   = PChar->jobs.exp[PChar->GetMJob()];
                uint8  mainLevel = PChar->jobs.job[PChar->GetMJob()];

                // Add EXP from lost levels
                for (uint8 i = 0; i < mainLevel; i++)
                {
                    lostEXP += charutils::GetExpNEXTLevel(i);
                }

                // Set level 1 with 0 EXP
                PChar->SetMLevel(1);
                PChar->jobs.job[PChar->GetMJob()] = 1;
                PChar->SetSLevel(PChar->jobs.job[PChar->GetSJob()]);
                PChar->jobs.exp[PChar->GetMJob()] = 0;

                // Refresh stats
                charutils::SetStyleLock(PChar, false);
                blueutils::ValidateBlueSpells(PChar);
                charutils::CalculateStats(PChar);
                charutils::CheckValidEquipment(PChar);
                jobpointutils::RefreshGiftMods(PChar);
                charutils::BuildingCharSkillsTable(PChar);
                charutils::BuildingCharAbilityTable(PChar);
                charutils::BuildingCharTraitsTable(PChar);

                PChar->UpdateHealth();
                PChar->health.hp = 0;
                PChar->health.mp = 0;

                charutils::SaveCharStats(PChar);
                charutils::SaveCharJob(PChar, PChar->GetMJob());
                charutils::SaveCharExp(PChar, PChar->GetMJob());
                PChar->updatemask |= UPDATE_HP;

                PChar->pushPacket(new CCharJobsPacket(PChar));
                PChar->pushPacket(new CCharStatsPacket(PChar));
                PChar->pushPacket(new CCharSkillsPacket(PChar));
                PChar->pushPacket(new CCharRecastPacket(PChar));
                PChar->pushPacket(new CCharAbilitiesPacket(PChar));
                PChar->pushPacket(new CCharUpdatePacket(PChar));
                PChar->pushPacket(new CMenuMeritPacket(PChar));
                PChar->pushPacket(new CMonipulatorPacket1(PChar));
                PChar->pushPacket(new CMonipulatorPacket2(PChar));
                PChar->pushPacket(new CCharSyncPacket(PChar));

                // Reset level sync
                if (PChar->PParty != nullptr)
                {
                    if (PChar->PParty->GetSyncTarget() == PChar)
                    {
                        PChar->PParty->RefreshSync();
                    }
                    PChar->PParty->ReloadParty();
                }

                // Level down message
                PChar->loc.zone->PushPacket(PChar, CHAR_INRANGE_SELF, new CMessageCombatPacket(PChar, PChar, PChar->jobs.job[PChar->GetMJob()], 0, 11));
                luautils::OnPlayerLevelDown(PChar);
                PChar->updatemask |= UPDATE_HP;

                return lostEXP;
            }
            else
            {
                return 0;
            }
        };

        lua["CBaseEntity"]["setCrystalWarrior"] = [this](CLuaBaseEntity* PLuaBaseEntity, bool value) -> void {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar   = dynamic_cast<CCharEntity*>(PEntity);

            if (value)
            {
                PChar->setCharVar(CW_STATUS, 1);
            }
            else
            {
                PChar->setCharVar(CW_STATUS, 0);
            }

            PChar->updatemask |= UPDATE_HP;
        };

        lua["CBaseEntity"]["setUnbreakable"] = [this](CLuaBaseEntity* PLuaBaseEntity, bool value) -> void {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar   = dynamic_cast<CCharEntity*>(PEntity);

            if (value)
            {
                PChar->setCharVar(CW_STATUS, 2);
            }
            else
            {
                PChar->setCharVar(CW_STATUS, 1);
            }

            PChar->updatemask |= UPDATE_HP;
        };

        lua["CBaseEntity"]["updateLevelSync"] = [](CLuaBaseEntity* PLuaBaseEntity) -> void
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            if (PChar->PParty != nullptr && PChar->StatusEffectContainer->HasStatusEffect(EFFECT_LEVEL_SYNC))
            {
                PChar->PParty->RefreshSync();
            }
        };

        lua["CBaseEntity"]["removeItemsCW"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            auto const query =
                "DELETE FROM char_inventory WHERE charid = %u AND itemId IN ("
                "23757, 23764, 23771, 23778, 23785, " // Sakpata
                "26793, 26949, 27099, 27284, 27459, " // Naga
                "23760, 23767, 23774, 23781, 23788, " // Bunzi
                "26797, 26953, 27103, 27288, 27463, " // Vanya
                "23732, 23733, 23734, 23735, 23736, " // Malignance
                "25613, 25686, 27117, 27302, 27473, " // Adhemar
                "27764, 27910, 28049, 28191, 28330, " // Founder
                "23756, 23763, 23770, 23777, 23784, " // Gleti
                "25641, 25717, 27139, 25841, 27495, " // Valorous
                "26795, 26951, 27101, 27286, 27461, " // Pursuer
                "25642, 25718, 27140, 25842, 27496, " // Herculean
                "25611, 25684, 27115, 27300, 27471, " // Ryuo
                "23758, 23765, 23772, 23779, 23786, " // Mpaca
                "25640, 25716, 27138, 25840, 27494, " // Odyssean
                "25644, 25720, 27142, 25844, 27498, " // Chironic
                "23759, 23766, 23773, 23780, 23787, " // Agwu
                "23755, 23762, 23769, 23776, 23783, " // Ikenga
                "27784, 27924, 28064, 28204, 28344, " // Thurandaut
                "26794, 26950, 27100, 27285, 27460, " // Rawhide
                "25643, 25719, 27141, 25843, 27497, " // Merlinic
                "25615, 25688, 27119, 27304, 27475, " // Amalric
                "26791, 26947, 27097, 27282, 27457, " // Eschite
                "27627, "                             // Svalinn
                "10806, "                             // Adamas
                "18909, "                             // Claritas
                "18627, "                             // Xingzhe
                "22281);";                            // Verutum 

            return sql->Query(query, PChar->id) != SQL_ERROR;
        };
    }
};

REGISTER_CPP_MODULE(CrystalWarriorUtilModule);
