﻿/************************************************************************
* Crystal Warrior Mode
*************************************************************************
* Crystal Warriors can only party with other Crystal Warriors
* Trading, Delivery Box, Bazaar and Auction House are restricted
************************************************************************/

#include "map/utils/moduleutils.h"

#include "map/packets/chat_message.h"
#include "map/packets/auction_house.h"
#include "map/packets/menu_mog.h"
#include "map/packets/shop_menu.h"
#include "map/packets/shop_items.h"
#include "map/packets/bazaar_purchase.h"
#include "map/packets/message_standard.h"
#include "map/packets/trade_action.h"

#include "map/utils/charutils.h"
#include "map/utils/jailutils.h"
#include "map/utils/zoneutils.h"
#include "map/lua/lua_baseentity.h"
#include "map/item_container.h"
#include "map/universal_container.h"
#include "map/trade_container.h"

#include "packets/party_define.h"
#include "packets/party_invite.h"
#include "packets/party_map.h"
#include "packets/party_search.h"
#include "packets/message_basic.h"
#include "packets/message_combat.h"
#include "packets/message_system.h"
#include "message.h"
#include "status_effect_container.h"

extern uint8                                                                             PacketSize[512];
extern std::function<void(map_session_data_t* const, CCharEntity* const, CBasicPacket&)> PacketParser[512];

class CrystalWarriorModule : public CPPModule
{
    std::string MSG_AUCTION_HOUSE    = "You cannot use the Auction House as a Crystal Warrior.";
    std::string MSG_DELIVERY_BOX     = "You cannot use the Delivery Box as a Crystal Warrior.";
    std::string MSG_TRADE_PLAYER     = "You cannot trade as a Crystal Warrior.";
    std::string MSG_TRADE_OTHER      = "You cannot trade with a Crystal Warrior.";
    std::string MSG_EMINENCE_RECORDS = "You cannot undertake Records of Eminence as a Crystal Warrior.";

    std::string MSG_INVITE_PLAYER    = "You cannot invite regular players as a Crystal Warrior.";
    std::string MSG_INVITE_OTHER     = "You cannot invite a Crystal Warrior as a regular player.";
    std::string MSG_INVITE_CONTAINS  = "You cannot invite a Crystal Warrior to a regular alliance.";

    std::string MSG_BAZAAR_SELLING   = "You cannot buy items from a Crystal Warrior.";
    std::string MSG_BAZAAR_BUYING    = "You cannot buy bazaar items as a Crystal Warrior.";

    std::string CHAR_TYPE            = "CHAR_TYPE"; // CharVar for Crystal Warrior status
    uint16 tradePermitted[10];                      // IDs of items tradeable between Crystal Warriors

    bool isCrystalWarrior(CCharEntity* PChar)
    {
        uint16 charType = charutils::GetCharVar(PChar, CHAR_TYPE);
        return charType == 1 || charType == 2;
    }
 
    void OnInit() override
    {
        TracyZoneScoped;

        tradePermitted[0] = 4237; // Perpetual Hourglass
        tradePermitted[1] =  515; // Linkpearl
        tradePermitted[2] =  558; // Tarut: The Fool
        tradePermitted[3] =  559; // Tarut: Death
        tradePermitted[4] =  561; // Tarut: The King
        tradePermitted[5] =  562; // Tarut: The Hermit

        /************************************************************************
        *                        PacketParser methods
        ************************************************************************/

        // Update trade slot
        {
            auto updateTradeSlot           = PacketParser[0x034];
            auto updateTradeSlotRestricted = [this, updateTradeSlot](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                uint16       itemID  = data.ref<uint16>(0x08);
                CCharEntity* PTarget = (CCharEntity*)PChar->GetEntity(PChar->TradePending.targid, TYPE_PC);

                if (PTarget != nullptr && PTarget->id == PChar->TradePending.id)
                {
                    if (isCrystalWarrior(PChar) && isCrystalWarrior(PTarget))
                    {
                        for (int i = 0; i < 10; i++)
                        {
                            if (tradePermitted[i] > 0 && tradePermitted[i] == itemID)
                            {
                                updateTradeSlot(PSession, PChar, data);
                                return;
                            }
                        }

                        return;
                    }
                }

                updateTradeSlot(PSession, PChar, data);
            };
            PacketParser[0x034] = updateTradeSlotRestricted;
        }

        // Party invite
        {
            auto partyInvite           = PacketParser[0x06E];
            auto partyInviteRestricted = [this, partyInvite](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                uint32 charid = data.ref<uint32>(0x04);
                uint16 targid = data.ref<uint16>(0x08);
                uint8 invType = data.ref<uint8>(0x0A);

                // cannot invite yourself
                if (PChar->id == charid)
                {
                    return;
                }

                if (jailutils::InPrison(PChar))
                {
                    // Initiator is in prison.  Send error message.
                    PChar->pushPacket(new CMessageBasicPacket(PChar, PChar, 0, 0, 316));
                    return;
                }

                switch (invType)
                {
                    case 0: // party - must by party leader or solo
                        if (PChar->PParty == nullptr || PChar->PParty->GetLeader() == PChar)
                        {
                            if (PChar->PParty && PChar->PParty->IsFull())
                            {
                                PChar->pushPacket(new CMessageStandardPacket(PChar, 0, 0, MsgStd::CannotInvite));
                                break;
                            }
                            CCharEntity* PInvitee = nullptr;
                            if (targid != 0)
                            {
                                CBaseEntity* PEntity = PChar->GetEntity(targid, TYPE_PC);
                                if (PEntity && PEntity->id == charid)
                                {
                                    PInvitee = (CCharEntity*)PEntity;
                                }
                            }
                            else
                            {
                                PInvitee = zoneutils::GetChar(charid);
                            }
                            if (PInvitee)
                            {
                                ShowDebug("%s sent party invite to %s", PChar->getName(), PInvitee->getName());
                                // make sure invitee isn't dead or in jail, they aren't a party member and don't already have an invite pending, and your party is not full
                                if (PInvitee->isDead() || jailutils::InPrison(PInvitee) || PInvitee->InvitePending.id != 0 || PInvitee->PParty != nullptr)
                                {
                                    ShowDebug("%s is dead, in jail, has a pending invite, or is already in a party", PInvitee->getName());
                                    PChar->pushPacket(new CMessageStandardPacket(PChar, 0, 0, MsgStd::CannotInvite));
                                    break;
                                }
                                // check /blockaid
                                if (PInvitee->getBlockingAid())
                                {
                                    ShowDebug("%s is blocking party invites", PInvitee->getName());
                                    // Target is blocking assistance
                                    PChar->pushPacket(new CMessageSystemPacket(0, 0, MsgStd::TargetIsCurrentlyBlocking));
                                    // Interaction was blocked
                                    PInvitee->pushPacket(new CMessageSystemPacket(0, 0, MsgStd::BlockedByBlockaid));
                                    // You cannot invite that person at this time.
                                    PChar->pushPacket(new CMessageSystemPacket(0, 0, MsgStd::CannotInvite));
                                    break;
                                }
                                if (PInvitee->StatusEffectContainer->HasStatusEffect(EFFECT_LEVEL_SYNC))
                                {
                                    ShowDebug("%s has level sync, unable to send invite", PInvitee->getName());
                                    PChar->pushPacket(new CMessageStandardPacket(PChar, 0, 0, MsgStd::CannotInviteLevelSync));
                                    break;
                                }

                                bool inviterIsCrystal = isCrystalWarrior(PChar);
                                bool inviteeIsCrystal = isCrystalWarrior(PInvitee);

                                // Regular player invites regular player
                                // OR
                                // Restricted player invites restricted player
                                if ((!inviterIsCrystal && !inviteeIsCrystal) || (inviterIsCrystal && inviteeIsCrystal))
                                {
                                    // Send the party invite packet
                                    PInvitee->pushPacket(new CPartyInvitePacket(charid, targid, PChar, INVITE_PARTY));
                                    ShowDebug("Sent party invite packet to %s", PInvitee->getName());
                                    if (PChar->PParty && PChar->PParty->GetSyncTarget())
                                    {
                                        PInvitee->pushPacket(new CMessageStandardPacket(PInvitee, 0, 0, MsgStd::LevelSyncWarning));
                                    }
                                }
                                // Restricted player invites regular player
                                else if (inviterIsCrystal && !inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_PLAYER));
                                    return;
                                }
                                // Regular player invites restricted player
                                else if (!inviterIsCrystal && inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_OTHER));
                                    return;
                                }

                                PInvitee->InvitePending.id     = PChar->id;
                                PInvitee->InvitePending.targid = PChar->targid;
                            }
                            else
                            {
                                ShowDebug("Building invite packet to send to lobby server from %s to (%d)", PChar->getName(), charid);
                                // on another server (hopefully)
                                uint8 packetData[12]{};
                                ref<uint32>(packetData, 0)  = charid;
                                ref<uint16>(packetData, 4)  = targid;
                                ref<uint32>(packetData, 6)  = PChar->id;
                                ref<uint16>(packetData, 10) = PChar->targid;
                                std::string query = "SELECT v.value FROM char_vars v WHERE v.charid = %u AND v.varname = 'CHAR_TYPE'";
                                bool inviterIsCrystal = isCrystalWarrior(PChar);
                                bool inviteeIsCrystal = false;
                                int inviteeRet = sql->Query(query.c_str(),charid);
                                if (inviteeRet != SQL_ERROR && sql->NumRows() > 0)
                                {
                                    while (sql->NextRow() == SQL_SUCCESS)
                                    {
                                        auto charType = sql->GetUIntData(0);
                                        if (charType > 0 && charType != 3)
                                        {
                                                inviteeIsCrystal = true;
                                        }
                                    }
                                }
                                // Regular player invites regular player
                                // OR
                                // Restricted player invites restricted player
                                if ((!inviterIsCrystal && !inviteeIsCrystal) || (inviterIsCrystal && inviteeIsCrystal))
                                {
                                    // Send the party invite packet
                                    message::send(MSG_PT_INVITE, packetData, sizeof packetData, new CPartyInvitePacket(charid, targid, PChar, INVITE_PARTY));

                                    ShowDebug("Sent invite packet to lobby server from %s to (%d)", PChar->getName(), charid);
                                }
                                // Restricted player invites regular player
                                else if (inviterIsCrystal && !inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_PLAYER));
                                    return;
                                }
                                // Regular player invites restricted player
                                else if (!inviterIsCrystal && inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_OTHER));
                                    return;
                                }
                            }
                        }
                        else // in party but not leader, cannot invite
                        {
                            ShowDebug("%s is not party leader, cannot send invite", PChar->getName());
                            PChar->pushPacket(new CMessageStandardPacket(PChar, 0, 0, MsgStd::NotPartyLeader));
                        }
                        break;

                    case 5: // alliance - must be unallied party leader or alliance leader of a non-full alliance
                        if (PChar->PParty && PChar->PParty->GetLeader() == PChar &&
                            (PChar->PParty->m_PAlliance == nullptr ||
                            (PChar->PParty->m_PAlliance->getMainParty() == PChar->PParty && !PChar->PParty->m_PAlliance->isFull())))
                        {
                            CCharEntity* PInvitee = nullptr;
                            if (targid != 0)
                            {
                                CBaseEntity* PEntity = PChar->GetEntity(targid, TYPE_PC);
                                if (PEntity && PEntity->id == charid)
                                {
                                    PInvitee = (CCharEntity*)PEntity;
                                }
                            }
                            else
                            {
                                PInvitee = zoneutils::GetChar(charid);
                            }

                            if (PInvitee)
                            {
                                ShowDebug("%s sent alliance invite to %s", PChar->getName(), PInvitee->getName());
                                // check /blockaid
                                if (PInvitee->getBlockingAid())
                                {
                                    ShowDebug("%s is blocking alliance invites", PInvitee->getName());
                                    // Target is blocking assistance
                                    PChar->pushPacket(new CMessageSystemPacket(0, 0, MsgStd::TargetIsCurrentlyBlocking));
                                    // Interaction was blocked
                                    PInvitee->pushPacket(new CMessageSystemPacket(0, 0, MsgStd::BlockedByBlockaid));
                                    // You cannot invite that person at this time.
                                    PChar->pushPacket(new CMessageSystemPacket(0, 0, MsgStd::CannotInvite));
                                    break;
                                }
                                // make sure intvitee isn't dead or in jail, they are an unallied party leader and don't already have an invite pending
                                if (PInvitee->isDead() || jailutils::InPrison(PInvitee) || PInvitee->InvitePending.id != 0 || PInvitee->PParty == nullptr ||
                                    PInvitee->PParty->GetLeader() != PInvitee || PInvitee->PParty->m_PAlliance)
                                {
                                    ShowDebug("%s is dead, in jail, has a pending invite, or is already in a party/alliance", PInvitee->getName());
                                    PChar->pushPacket(new CMessageStandardPacket(PChar, 0, 0, MsgStd::CannotInvite));
                                    break;
                                }
                                if (PInvitee->StatusEffectContainer->HasStatusEffect(EFFECT_LEVEL_SYNC))
                                {
                                    ShowDebug("%s has level sync, unable to send invite", PInvitee->getName());
                                    PChar->pushPacket(new CMessageStandardPacket(PChar, 0, 0, MsgStd::CannotInviteLevelSync));
                                    break;
                                }

                                bool inviterIsCrystal = isCrystalWarrior(PChar);
                                bool inviteeIsCrystal = isCrystalWarrior(PInvitee);

                                // Regular player invites regular player
                                // OR
                                // Restricted player invites restricted player
                                if ((!inviterIsCrystal && !inviteeIsCrystal) || (inviterIsCrystal && inviteeIsCrystal))
                                {
                                    // Send the party invite packet
                                    PInvitee->pushPacket(new CPartyInvitePacket(charid, targid, PChar, INVITE_ALLIANCE));
                                    ShowDebug("Sent party invite packet to %s", PInvitee->getName());
                                }
                                // Restricted player invites regular player
                                else if (inviterIsCrystal && !inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_PLAYER));
                                    return;
                                }
                                // Regular player invites restricted player
                                else if (!inviterIsCrystal && inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_OTHER));
                                    return;
                                }

                                PInvitee->InvitePending.id     = PChar->id;
                                PInvitee->InvitePending.targid = PChar->targid;

                            }
                            else
                            {
                                ShowDebug("(Alliance)Building invite packet to send to lobby server from %s to (%d)", PChar->getName(), charid);
                                // on another server (hopefully)
                                uint8 packetData[12]{};
                                ref<uint32>(packetData, 0)  = charid;
                                ref<uint16>(packetData, 4)  = targid;
                                ref<uint32>(packetData, 6)  = PChar->id;
                                ref<uint16>(packetData, 10) = PChar->targid;
                                std::string query = "SELECT v.value FROM char_vars v WHERE v.charid = %u AND v.varname = 'CHAR_TYPE'";
                                bool inviterIsCrystal = isCrystalWarrior(PChar);
                                bool inviteeIsCrystal = false;
                                int inviteeRet = sql->Query(query.c_str(),charid);
                                if (inviteeRet != SQL_ERROR && sql->NumRows() > 0)
                                {
                                    while (sql->NextRow() == SQL_SUCCESS)
                                    {
                                        auto charType = sql->GetUIntData(0);
                                        if (charType > 0 && charType != 3)
                                        {
                                                inviteeIsCrystal = true;
                                        }
                                    }
                                }
                                // Regular player invites regular player
                                // OR
                                // Restricted player invites restricted player
                                if ((!inviterIsCrystal && !inviteeIsCrystal) || (inviterIsCrystal && inviteeIsCrystal))
                                {
                                    // Send the party invite packet
                                    message::send(MSG_PT_INVITE, packetData, sizeof packetData, new CPartyInvitePacket(charid, targid, PChar, INVITE_ALLIANCE));

                                    ShowDebug("(Alliance)Sent invite packet to lobby server from %s to (%d)", PChar->getName(), charid);
                                }
                                // Restricted player invites regular player
                                else if (inviterIsCrystal && !inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_PLAYER));
                                    return;
                                }
                                // Regular player invites restricted player
                                else if (!inviterIsCrystal && inviteeIsCrystal)
                                {
                                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_INVITE_OTHER));
                                    return;
                                }
                            }
                        }
                        break;

                    default:
                        ShowError("SmallPacket0x06E : unknown byte <%.2X>", data.ref<uint8>(0x0A));
                        break;
                }
            };
            PacketParser[0x06E] = partyInviteRestricted;
        }

        // Trade request
        {
            auto tradeRequest           = PacketParser[0x032];
            auto tradeRequestRestricted = [this, tradeRequest](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                uint16       targid   = data.ref<uint16>(0x08);
                CCharEntity* PTarget  = (CCharEntity*)PChar->GetEntity(targid, TYPE_PC);
                int32 charIsCrystal   = isCrystalWarrior(PChar);
                int32 targetIsCrystal = isCrystalWarrior(PTarget);

                if (charIsCrystal && !targetIsCrystal)
                {
                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_TRADE_PLAYER));
                }
                else
                {
                    if (!charIsCrystal && targetIsCrystal)
                    {
                        PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_TRADE_OTHER));
                    }
                    else
                    {
                        tradeRequest(PSession, PChar, data);
                    }
                }
            };
            PacketParser[0x032] = tradeRequestRestricted;
        }

        // Bazaar Purchase
        {
            auto bazaarPurchase           = PacketParser[0x106];
            auto bazaarPurchaseRestricted = [this, bazaarPurchase](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;
                CCharEntity* PTarget = (CCharEntity*)PChar->GetEntity(PChar->BazaarID.targid, TYPE_PC);

                if (PTarget == nullptr || PTarget->id != PChar->BazaarID.id)
                {
                    return;
                }

                int32 playerIsCrystal = isCrystalWarrior(PChar);
                int32 targetIsCrystal = isCrystalWarrior(PTarget);

                if (playerIsCrystal) // Buying as a Crystal Warrior
                {
                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_BAZAAR_BUYING));
                    PChar->pushPacket(new CBazaarPurchasePacket(PTarget, false));
                    return;
                }
                if (targetIsCrystal) // Buying from a Crystal Warrior
                {
                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_BAZAAR_SELLING));
                    PChar->pushPacket(new CBazaarPurchasePacket(PTarget, false));
                    return;
                }
                else
                {
                    bazaarPurchase(PSession, PChar, data);
                }
            };
            PacketParser[0x106] = bazaarPurchaseRestricted;
        }

        // Delivery Box (Mog House)
        {
            auto deliveryBox = PacketParser[0x04D];
            auto deliveryBoxRestricted = [this, deliveryBox](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                if (isCrystalWarrior(PChar))
                {
                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_DELIVERY_BOX));
                }
                else
                {
                    deliveryBox(PSession, PChar, data);
                }
            };
            PacketParser[0x04D] = deliveryBoxRestricted;
        }

        /************************************************************************
        *                   Block Auction House
        *   (These packets should be impossible without cheating)
        ************************************************************************/
        {
            auto auctionPackets      = PacketParser[0x04E];
            auto blockAuctionPackets = [this, auctionPackets](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                if (isCrystalWarrior(PChar))
                {
                    ShowWarning("Player %s is trying to use the Auction House as a Crystal Warrior", PChar->getName());
                }
                else
                {
                    auctionPackets(PSession, PChar, data);
                }
            };
            PacketParser[0x04E] = blockAuctionPackets;
        }

        /************************************************************************
        *                        Lua functions
        ************************************************************************/

        // Lua Auction House
        lua["CBaseEntity"]["sendMenu"] = [this](CLuaBaseEntity* PLuaBaseEntity, uint32 menu) {
            TracyZoneScoped;

            if (PLuaBaseEntity == nullptr)
            {
                return;
            }

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            if (auto* PChar = static_cast<CCharEntity*>(PEntity))
            {
                switch (menu)
                {
                    case 1:
                        PChar->pushPacket(new CMenuMogPacket());
                        break;
                    case 2:
                        PChar->pushPacket(new CShopMenuPacket(PChar));
                        PChar->pushPacket(new CShopItemsPacket(PChar));
                        break;
                    case 3:
                        if (isCrystalWarrior(PChar))
                        {
                            PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_AUCTION_HOUSE));
                        }
                        else
                        {
                            PChar->pushPacket(new CAuctionHousePacket(2));
                        }
                        break;
                    default:
                        ShowDebug("Menu %i not implemented, yet.", menu);
                        break;
                }
            }
        };

        // Lua Delivery Box (Delivery NPC)
        lua["CBaseEntity"]["openSendBox"] = [this](CLuaBaseEntity* PLuaBaseEntity) {
            TracyZoneScoped;

            if (PLuaBaseEntity == nullptr)
            {
                return;
            }

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            if (auto* PChar = static_cast<CCharEntity*>(PEntity))
            {
                if (isCrystalWarrior(PChar))
                {
                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_DELIVERY_BOX));
                }
                else
                {
                    charutils::OpenSendBox(PChar, 0x0D, 2);
                }
            }
        };
    }
};

REGISTER_CPP_MODULE(CrystalWarriorModule);
