﻿/************************************************************************
* Packet System Overrides
************************************************************************/

#include "map/utils/moduleutils.h"

#include "map/packets/chat_message.h"

#include "map/utils/charutils.h"
#include "map/lua/lua_baseentity.h"
#include "map/item_container.h"
#include "map/universal_container.h"

#include "blue_spell.h"
#include "recast_container.h"
#include "packets/message_basic.h"
#include "packets/message_combat.h"
#include "packets/message_system.h"
#include "message.h"

extern uint8                                                                             PacketSize[512];
extern std::function<void(map_session_data_t* const, CCharEntity* const, CBasicPacket&)> PacketParser[512];

class PacketSystemModule : public CPPModule
{
    std::string MSG_EMINENCE_RECORDS = "Your character type cannot undertake Records of Eminence.";

    std::string CHAR_TYPE = "CHAR_TYPE"; // CharVar for Crystal Warrior status

    bool isCrystalWarrior(CCharEntity* PChar)
    {
        uint16 charType = charutils::GetCharVar(PChar, CHAR_TYPE);
        return charType == 1 || charType == 2;
    }

    bool isUCrystalWarrior(CCharEntity* PChar)
    {
        uint16 charType = charutils::GetCharVar(PChar, CHAR_TYPE);
        return charType == 2;
    }

    bool isClassicMode(CCharEntity* PChar)
    {
        return charutils::GetCharVar(PChar, CHAR_TYPE) == 3;
    }

    bool isACE(CCharEntity* PChar)
    {
        uint16 charType = charutils::GetCharVar(PChar, CHAR_TYPE);
        return charType == 0 ;
    }
 
    void OnInit() override
    {
        TracyZoneScoped;
        /************************************************************************
        *                        PacketParser methods
        ************************************************************************/

        // Records of Eminence (Start)
        {
            auto eminenceRecordStart           = PacketParser[0x10C];
            auto eminenceRecordStartRestricted = [this, eminenceRecordStart](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                if (isCrystalWarrior(PChar) || isClassicMode(PChar))
                {
                    PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, MSG_EMINENCE_RECORDS));
                }
                else
                {
                    eminenceRecordStart(PSession, PChar, data);
                }
            };
            PacketParser[0x10C] = eminenceRecordStartRestricted;
        }

        /************************************************************************
        *                   Block Selling to Guild
        ************************************************************************/
        {
            auto guildSellPackets      = PacketParser[0x0AC];
            auto blockguildSellPackets = [this, guildSellPackets](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, "You are unable to sell items to the guild."));
            };
            PacketParser[0x0AC] = blockguildSellPackets;
        }

        /************************************************************************
        *                   Logging for GMs
        ************************************************************************/
        {
            auto deliveryBoxAction       = PacketParser[0x04D];
            auto deliveryBoxActionLogged = [this, deliveryBoxAction](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;
                uint8 action  = data.ref<uint8>(0x04);

                if (action == 0x0A)
                {
                    uint8 slotID = data.ref<uint8>(0x06);

                    if (!PChar->UContainer->IsSlotEmpty(slotID))
                    {
                        CItem*      PItem    = PChar->UContainer->GetItem(slotID);
                        auto        item_id  = PItem->getID();
                        auto        quantity = PItem->getQuantity();

                        // Get sender of delivery record
                        int32 ret = sql->Query("SELECT sender FROM delivery_box WHERE charid = %u AND slot = %u AND box = 1 LIMIT 1", PChar->id, slotID);

                        if (ret != SQL_ERROR && sql->NumRows() > 0 && sql->NextRow() == SQL_SUCCESS)
                        {
                            if (quantity > 1)
                            {
                                ShowInfo(fmt::format("Player '{}' removes {} ({}) x{} from the delivery box (Sender: '{}')", PChar->getName(), item_id, PItem->getName(), quantity, sql->GetStringData(0)));
                            }
                            else
                            {
                                ShowInfo(fmt::format("Player '{}' removes {} ({}) from the delivery box (Sender: '{}')", PChar->getName(), item_id, PItem->getName(), sql->GetStringData(0)));
                            }
                        }
                    }
                }
                else if (action == 0x02)
                {
                    uint8  invslot  = data.ref<uint8>(0x07);
                    uint32 quantity = data.ref<uint32>(0x08);

                    CItem* PItem = PChar->getStorage(LOC_INVENTORY)->GetItem(invslot);

                    if (quantity == 0 || !PItem)
                    {
                        return;
                    }

                    // Block all EX items including MAIL2ACCOUNT
                    // Prevents converted ACE sending items to WEW characters on the same account
                    if (isACE(PChar) && (PItem->getFlag() & ITEM_FLAG_EX))
                    {
                        return;
                    }
                }

                deliveryBoxAction(PSession, PChar, data);
            };
            PacketParser[0x04D] = deliveryBoxActionLogged;
        }

        /************************************************************************
        *              Block Mentor/New Adventurer/Autoparty (To repurpose flags)
        *                 Unnecessary with new profile.charType
        ************************************************************************/
//        {
//            auto updateFlags           = PacketParser[0x0DC];
//            auto updateFlagsRestricted = [this, updateFlags](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
//                TracyZoneScoped;
//
//                auto dataRef = data.ref<uint32>(0x04);
//                if (dataRef != 0x2000000 && dataRef != 0x04000000 && dataRef != 0x8000)
//                {
//                    updateFlags(PSession, PChar, data);
//                }
//            };
//            PacketParser[0x0DC] = updateFlagsRestricted;
//        }

        /************************************************************************
        *              Update char-type name flags (via char info request)
        *                   Unnecessary with new profile.charType
        ************************************************************************/
//        {
//            auto originalHandler  = PacketParser[0x00C];
//            auto newHandler       = [this, originalHandler](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket& data) -> void
//            {
//                originalHandler(PSession, PChar, data);
//
//                // ShowDebug("Setting Flags");
//                // CW:  0x02000000
//                PChar->playerConfig.MentorFlg = false;
//                // UCW: 0x00010000
//                PChar->playerConfig.NewAdventurerOffFlg = true;
//                // WEW: 0x01000000
//                PChar->playerConfig.AutoPartyFlg = false;
//
//                // set CW icon
//                if (isCrystalWarrior(PChar) && !isUCrystalWarrior(PChar))
//                    PChar->playerConfig.MentorFlg = true;
//
//                if (isUCrystalWarrior(PChar))
//                    PChar->playerConfig.NewAdventurerOffFlg = false;
//
//                if (isClassicMode(PChar))
//                    PChar->playerConfig.AutoPartyFlg = true;
//
//                charutils::SavePlayerSettings(PChar);
//            };
//
//            PacketParser[0x00C] = newHandler;
//        }

        // Despawn all Trusts when WEW joins party
        {
            auto partyJoin           = PacketParser[0x074];
            auto partyJoinRestricted = [this, partyJoin](map_session_data_t* const PSession, CCharEntity* const PChar, CBasicPacket data) -> void {
                TracyZoneScoped;

                partyJoin(PSession, PChar, data);
                if (data.ref<uint8>(0x04) == 1 && isClassicMode(PChar))
                {
                    if (PChar->PParty != nullptr)
                    {
                        for (auto* PMemberEntity : PChar->PParty->members)
                        {
                            if (auto* PMember = dynamic_cast<CCharEntity*>(PMemberEntity))
                            {
                                PMember->ClearTrusts();
                            }
                        }
                    }
                }
            };
            PacketParser[0x074] = partyJoinRestricted;
        }
    }
};

REGISTER_CPP_MODULE(PacketSystemModule);
