﻿/************************************************************************
* CatsEye Util
*************************************************************************
* Lua util functions for CatsEye XI
************************************************************************/

#include "map/utils/moduleutils.h"

#include "common/sql.h"
#include "common/lua.h"
#include "map/entities/battleentity.h"
#include "map/utils/charutils.h"
#include "map/utils/itemutils.h"
#include "map/item_container.h"
#include "map/conquest_system.h"
#include "utils/zoneutils.h"

#include "map/merit.h"
#include "map/packets/char_abilities.h"
#include "map/packets/menu_merit.h"
#include "map/packets/merit_points_categories.h"
#include "map/packets/inventory_finish.h"
#include "map/packets/chat_message.h"
#include "map/trade_container.h"

class CatsEyeUtilModule : public CPPModule
{
    std::string CHAR_TYPE = "CHAR_TYPE";

    void OnInit() override
    {
        TracyZoneScoped;

        // Returns true if the player is a Classic Mode character
        lua["CBaseEntity"]["isClassicMode"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype == TYPE_PC)
            {
                auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

                return PChar->getCharVar(CHAR_TYPE) == 3;
            }

            return false;
        };

        // Returns true if the player is in a party with a Classic Mode character
        lua["CBaseEntity"]["isClassicModeInParty"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype == TYPE_PC)
            {
                // left joins char_vars
                auto query = "SELECT ap.charid FROM accounts_parties ap left join char_vars cv on ap.charid = cv.charid and varname = 'CHAR_TYPE' WHERE value = 3 and partyid = (select partyid from accounts_parties where charid = %u);";

                if (sql->Query(query, PEntity->id) != SQL_ERROR && sql->NumRows() > 0 && sql->NextRow() != SQL_ERROR)
                {
                    return true;
                }
            }

            return false;
        };

        lua["CBaseEntity"]["setClassicMode"] = [this](CLuaBaseEntity* PLuaBaseEntity, bool value) -> void
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            if (value)
            {
                PChar->setCharVar(CHAR_TYPE, 3);
            }
            else
            {
                PChar->setCharVar(CHAR_TYPE, 0);
            }

            PChar->updatemask |= UPDATE_HP;
        };

        lua["CBaseEntity"]["getVerificationCode"] = [this](CLuaBaseEntity* PLuaBaseEntity, const std::string accountName) -> const std::string
        {
            TracyZoneScoped;

            if (sql->Query("SELECT verification_code FROM accounts WHERE login = '%s';", accountName) != SQL_ERROR && sql->NumRows() > 0 && sql->NextRow() != SQL_ERROR)
            {
                return sql->GetStringData(0);
            }

            return "";
        };

        lua["CBaseEntity"]["getReferral"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> const std::string
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return "";
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            auto        query = "SELECT referral FROM accounts INNER JOIN chars ON chars.accid = accounts.id WHERE chars.charid = %u;";

            if (sql->Query(query, PChar->id) != SQL_ERROR && sql->NumRows() > 0 && sql->NextRow() != SQL_ERROR)
            {
                return sql->GetStringData(0);
            }

            return "";
        };

        lua["CBaseEntity"]["hasReferrals"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return false;
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            auto        query = "SELECT referral FROM accounts WHERE referral = '%s';";

            if (sql->Query(query, PChar->getName()) != SQL_ERROR && sql->NumRows() > 0 && sql->NextRow() != SQL_ERROR)
            {
                if (sql->NumRows() > 0)
                {
                    return true;
                }
            }

            return false;
        };

        lua["CBaseEntity"]["getReferralPoints"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> uint32
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return 0;
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            auto        query = "SELECT value FROM char_vars INNER JOIN chars ON char_vars.charid = chars.charid INNER JOIN accounts ON chars.accid = accounts.id WHERE accounts.referral = '%s' AND char_vars.varname = '[RE]MILESTONE';";
            uint32      total = 0;

            if (sql->Query(query, to_lower(PChar->name)) != SQL_ERROR && sql->NumRows() > 0)
            {
                while (sql->NextRow() == SQL_SUCCESS)
                {
                    uint32 val = sql->GetUIntData(0);

                    if ((val & 0x10) == 0x10)
                    {
                        total += 5;
                    }
                    else if ((val & 0x04) == 0x04)
                    {
                        total += 3;
                    }
                    else if ((val & 0x01) == 0x01)
                    {
                        total += 1;
                    }
                }
            }

            return total;
        };

        lua["CBaseEntity"]["getReferrals"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> sol::table
        {
            TracyZoneScoped;

            auto table = lua.create_table();

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return table;
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            auto        query = "SELECT chars.charname, accounts.type, char_vars.value FROM char_vars INNER JOIN chars ON char_vars.charid = chars.charid INNER JOIN accounts ON chars.accid = accounts.id WHERE accounts.referral = '%s' AND char_vars.varname = '[RE]MILESTONE';";

            if (sql->Query(query, to_lower(PChar->name)) != SQL_ERROR && sql->NumRows() > 0)
            {
                while (sql->NextRow() == SQL_SUCCESS)
                {
                    auto row = lua.create_table();
                    row.add(sql->GetStringData(0), sql->GetUIntData(1), sql->GetUIntData(2));
                    table.add(row);
                }
            }

            return table;
        };

        // Conquest System
        lua["CBaseEntity"]["addInfluencePoints"] = [this](CLuaBaseEntity* PLuaBaseEntity, uint16 amount) -> void
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);
            conquest::GainInfluencePoints(PChar, amount);
        };

        lua["GetWingsAccount"] = [this](const std::string catseyeAccountName) -> sol::table
        {
            TracyZoneScoped;

            auto query =
                "SELECT ffxiwings.ww_chars.name, ww_chars.main_job_lv, ww_chars.main_job, ffxiwings.ww_accounts.username FROM ffxiwings.ww_chars "
                "INNER JOIN ffxiwings.ww_contents ON ffxiwings.ww_chars.content_id = ffxiwings.ww_contents.content_id "
                "INNER JOIN ffxiwings.ww_accounts ON ffxiwings.ww_contents.account_id = ffxiwings.ww_accounts.id "
                "WHERE ffxiwings.ww_accounts.id = "
                "(SELECT wings_accid FROM tpzdb.wings_accounts WHERE tpzdb.wings_accounts.catseye_accid = "
                "(SELECT id FROM tpzdb.accounts WHERE tpzdb.accounts.login = '%s')) "
                "AND ffxiwings.ww_accounts.status <> 2 AND ffxiwings.ww_chars.world_id = 100;";

            auto table = lua.create_table();

            if (sql->Query(query, catseyeAccountName) == SQL_ERROR)
            {
                return table;
            }

            while (sql->NextRow() == SQL_SUCCESS)
            {
                auto row = lua.create_table();
                row.add(sql->GetStringData(0), sql->GetUIntData(1), sql->GetUIntData(2), sql->GetStringData(3));
                table.add(row);
            }

            return table;
        };

        lua["DeleteEmptyAccount"] = [this](const std::string catseyeAccountName) -> bool
        {
            TracyZoneScoped;

            if (sql->Query("SELECT * FROM chars INNER JOIN accounts ON accounts.id = chars.accid WHERE accounts.login = '%s';", catseyeAccountName) == SQL_ERROR || sql->NumRows() > 0)
            {
                // Already existing characters for account
                return false;
            }

            // If account is linked to a WingsXI account, we first need to delete account linking
            if (sql->Query("DELETE wings_accounts FROM wings_accounts INNER JOIN accounts ON accounts.id = wings_accounts.catseye_accid WHERE accounts.login = '%s';", catseyeAccountName) == SQL_ERROR)
            {
                // Unable to delete account link
                return false;
            }

            if (sql->Query("DELETE FROM accounts WHERE accounts.login = '%s';", catseyeAccountName) != SQL_ERROR)
            {
                // Deleted account with no associated chars
                return true;
            }

            // Some other error
            return false;
        };

        lua["CBaseEntity"]["partyHasTrusts"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> bool
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return false;
            }

            if (((CCharEntity*)PEntity)->PParty == nullptr)
            {
                return ((CCharEntity*)PEntity)->PTrusts.size() > 0;
            }

            return ((CCharEntity*)PEntity)->PParty->HasTrusts();
        };

        lua["CBaseEntity"]["partyNumTrusts"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> uint8
        {
            TracyZoneScoped;

            // leader can't create if the leader is in another cluster
            return 6;

            //            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();
            //
            //            if (PEntity->objtype != TYPE_PC)
            //            {
            //                return 0;
            //            }
            //
            //            if (((CCharEntity*)PEntity)->PParty == nullptr)
            //            {
            //                return (uint8)((CCharEntity*)PEntity)->PTrusts.size();
            //            }
            //
            //            CParty* party = ((CCharEntity*)PEntity)->PParty;
            //            CCharEntity* leader = dynamic_cast<CCharEntity*>(party->GetLeader());
            //
            //            return (uint8)leader->PTrusts.size();
        };

        lua["CBaseEntity"]["partyHighestLevel"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> uint8
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return 0;
            }

            CCharEntity* PPlayer = (CCharEntity*)PEntity;
            uint8        highest = PPlayer->GetMLevel();

            if (PPlayer->PParty != nullptr)
            {
                uint16 zoneID = PPlayer->getZone();

                for (auto member : PPlayer->PParty->members)
                {
                    if (member->getZone() == zoneID && member->GetMLevel() > highest)
                    {
                        highest = member->GetMLevel();
                    }
                }
            }

            return highest;
        };

        lua["CBaseEntity"]["getLinkedCharsInZone"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> sol::table
        {
            TracyZoneScoped;

            auto         table   = lua.create_table();
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return table;
            }

            CCharEntity* PPlayer = (CCharEntity*)PEntity;

            auto query =
                "select accid from ip_exceptions left join chars using (accid) where chars.charid = %u;";

            // ip exception, bypass restriction checks altogether for now
            if (sql->Query(query, PPlayer->id) == SQL_ERROR || sql->NumRows() > 0)
            {
                return table;
            }

            query =
                "SELECT charid from accounts_sessions WHERE client_addr = (select client_addr from accounts_sessions where charid = %u);";

            // char is ghosted, this is a separate problem
            if (sql->Query(query, PPlayer->id, PPlayer->id) == SQL_ERROR || sql->NumRows() == 0)
            {
                return table;
            }

            while (sql->NextRow() == SQL_SUCCESS)
            {
                // PPlayer->loc.zone nullptr during zoning
                auto zoneEntity = zoneutils::GetZone(PPlayer->getZone());
                if (auto associatedChar = zoneEntity->GetCharByID(sql->GetIntData(0)))
                {
                    auto row = lua.create_table();
                    table.add(CLuaBaseEntity(associatedChar));
                }
            }

            return table;
        };

        lua["CBaseEntity"]["getChars"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> sol::table
        {
            TracyZoneScoped;

            auto         table   = lua.create_table();
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return table;
            }

            CCharEntity* PPlayer = (CCharEntity*)PEntity;

            auto        query =
                "SELECT chars.charname, (SELECT value FROM char_vars WHERE varname = 'CHAR_TYPE' AND charid = chars.charid) FROM chars"
                " WHERE chars.accid = (SELECT accid FROM chars AS c WHERE c.charid = %u);";

            if (sql->Query(query, PPlayer->id) == SQL_ERROR || sql->NumRows() == 0)
            {
                return table;
            }

            while (sql->NextRow() == SQL_SUCCESS)
            {
                auto row = lua.create_table();
                row.add(sql->GetStringData(0));
                row.add(sql->GetIntData(1));
                table.add(row);
            }

            return table;
        };

        lua["CBaseEntity"]["getAceChars"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> sol::table
        {
            TracyZoneScoped;

            auto         table   = lua.create_table();
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return table;
            }

            CCharEntity* PPlayer = (CCharEntity*)PEntity;

            auto        query =
                "SELECT chars.charname FROM chars INNER JOIN char_vars ON chars.charid = char_vars.charid WHERE"
                " char_vars.charid = chars.charid AND"
                " char_vars.varname = 'CHAR_TYPE' AND"
                " char_vars.value = 0 AND"
                " chars.accid = (SELECT accid FROM chars AS c WHERE c.charid = %u);";

            if (sql->Query(query, PPlayer->id) == SQL_ERROR || sql->NumRows() == 0)
            {
                return table;
            }

            while (sql->NextRow() == SQL_SUCCESS)
            {
                table.add(sql->GetStringData(0));
            }

            return table;
        };

        lua["CBaseEntity"]["tradeRelease"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> void
        {
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            CCharEntity* PChar = (CCharEntity*)PEntity;

            for (uint8 slotID = 0; slotID < TRADE_CONTAINER_SIZE; ++slotID)
            {
                if (PChar->TradeContainer->getInvSlotID(slotID) != 0xFF)
                {
                    CItem* PItem = PChar->TradeContainer->getItem(slotID);
                    if (PItem)
                    {
                        PItem->setReserve(0);
                    }
                }
            }
            PChar->TradeContainer->Clean();
            PChar->pushPacket(new CInventoryFinishPacket());
        };

        // clears a particular merit item and returns how many merits it had cost to be at that tier
        lua["CBaseEntity"]["lowerMeritToZero"] = [this](CLuaBaseEntity* PLuaBaseEntity, MERIT_TYPE merit) -> uint8
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return 0;
            }

            if (auto* const PChar = dynamic_cast<CCharEntity*>(PEntity))
            {
                auto* const PMerit = PChar->PMeritPoints->GetMerit(merit);
                if (PMerit != nullptr)
                {
                    uint8 total = 0;
                    while (PMerit->count > 0)
                    {
                        PChar->PMeritPoints->LowerMerit(merit);
                        total += PMerit->next;
                    }

                    charutils::SaveCharExp(PChar, PChar->GetMJob());
                    PChar->PMeritPoints->SaveMeritPoints(PChar->id);
                    PChar->pushPacket(new CMenuMeritPacket(PChar));
                    PChar->pushPacket(new CMeritPointsCategoriesPacket(PChar, merit));
                    return total;
                }
            }

            return 0;
        };

        lua["GetPlayersLFG"] = [this]() -> sol::table
        {
            TracyZoneScoped;

            auto        seekers = lua.create_table();
            const char* Query   = "SELECT c.charname, a.type, cv2.value, cv3.value, z.name, s.seacom_type, s.seacom_message "
                                  "FROM char_vars cv "
                                  "LEFT JOIN chars          c     ON c.charid  = cv.charid "
                                  "LEFT JOIN accounts a ON c.accid = a.id "
                                  "LEFT JOIN accounts_sessions s ON c.accid = s.accid "
                                  "INNER JOIN char_stats stats ON stats.charid = c.charid "
                                  "INNER JOIN zone_settings z ON z.zoneid = c.pos_zone "
                                  "INNER JOIN char_vars cv2 ON cv2.charid = cv.charid AND cv2.varname = '[LFG]Job' "
                                  "INNER JOIN char_vars cv3 ON cv3.charid = cv.charid AND cv3.varname = '[LFG]Level' "
                                  "WHERE cv.varname = '[LFG]SeekTime'"
                                  "ORDER BY z.name, stats.mlvl;";

            if (sql->Query(Query) != SQL_ERROR && sql->NumRows() != 0)
            {
                while (sql->NextRow() == SQL_SUCCESS)
                {
                    auto row = lua.create_table();
                    // auto charId    = sql->GetUIntData(0);
                    row["playerName"] = sql->GetStringData(0);
                    row["charType"]   = sql->GetIntData(1);
                    row["mJob"]       = sql->GetUIntData(2);
                    row["mJobLvl"]    = sql->GetUIntData(3);
                    row["zoneName"]   = sql->GetStringData(4);
                    row["type"]       = sql->GetUIntData(5);
                    row["comment"]    = sql->GetStringData(6);
                    seekers.add(row);
                }
            }

            return seekers;
        };

        lua["CBaseEntity"]["say"] = [this](CLuaBaseEntity* PLuaBaseEntity, sol::object const& nameObj, std::string const& message, sol::variadic_args va) -> void
        {
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            CCharEntity* PChar = (CCharEntity*)PEntity;
            auto         name  = (nameObj == sol::lua_nil) ? "" : nameObj.as<std::string>();
            PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SAY, lua_fmt(message, va).c_str(), ""));
        };

        lua["CBaseEntity"]["fmt"] = [this](CLuaBaseEntity* PLuaBaseEntity, std::string const& message, sol::variadic_args va) -> void
        {
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            CCharEntity* PChar = (CCharEntity*)PEntity;
            PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_NS_SAY, lua_fmt(message, va).c_str(), ""));
        };

        lua["CBaseEntity"]["sys"] = [this](CLuaBaseEntity* PLuaBaseEntity, std::string const& message, sol::variadic_args va) -> void
        {
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return;
            }

            CCharEntity* PChar = (CCharEntity*)PEntity;
            PChar->pushPacket(new CChatMessagePacket(PChar, MESSAGE_SYSTEM_3, lua_fmt(message, va).c_str(), ""));
        };

        lua["CBaseEntity"]["reloadWeaponskillList"] = [this](CLuaBaseEntity* PLuaBaseEntity) -> void
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype == TYPE_PC)
            {
                auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

                charutils::BuildingCharWeaponSkills(PChar);
                PChar->pushPacket(new CCharAbilitiesPacket(PChar));
            }
        };

        lua["CBaseEntity"]["getPlayersInRange"] = [this](CLuaBaseEntity* PLuaBaseEntity, uint32 dist = 0) -> sol::table
        {
            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();
            auto         players = lua.create_table();

            // clang-format off
            zoneutils::GetZone(PEntity->getZone())->ForEachChar([&](CCharEntity* PChar)
            {
                if (!dist || distance(PChar->loc.p, PEntity->loc.p) < dist)
                {
                    players.add(CLuaBaseEntity(PChar));
                }
            });
            // clang-format on

            return players;
        };

        lua["CBaseEntity"]["canObtainItem"] = [this](CLuaBaseEntity* PLuaBaseEntity, uint16 itemID) -> bool
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();

            if (PEntity->objtype != TYPE_PC)
            {
                return false;
            }

            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            if (PChar->getStorage(LOC_INVENTORY)->GetFreeSlotsCount() == 0)
            {
                return false;
            }

            CItem* PItem = itemutils::GetItem(itemID);

            if (PItem == nullptr)
            {
                return false;
            }

            // Cannot obtain if item is RARE and player already has item
            return !((PItem->getFlag() & ITEM_FLAG_RARE) && charutils::HasItem(PChar, itemID));
        };

        // Allows dynamic menus to lock the client such as a CS event
        // player:setClientLock(true) will lock the client
        // player:setClientLock(false) will unlock the client
        // this requires "023-Char_Lock.patch" to be applied
        lua["CBaseEntity"]["setClientLock"] = [this](CLuaBaseEntity* PLuaBaseEntity, bool lockState) -> bool
        {
            TracyZoneScoped;

            CBaseEntity* PEntity = PLuaBaseEntity->GetBaseEntity();
            auto* const PChar = dynamic_cast<CCharEntity*>(PEntity);

            if (PEntity->objtype != TYPE_PC)
            {
                return false;
            }

            PChar->isClientLocked = lockState;

            bool lockedState = PChar->isClientLocked;

            PChar->updatemask |= UPDATE_HP;

            return lockedState;
        };
    }
};

REGISTER_CPP_MODULE(CatsEyeUtilModule);
